/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.validator;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;

import junit.framework.TestCase;

public class CountValidatorTest extends TestCase {

	public void testConstructor(){
		try{
			new CountValidator(null);
			fail();
		}catch(final NullPointerException expected){
		}
	}

	public void testInterpretCountTrue(){
		final CountValidator counter1 = new CountValidator(ValidatorConstants.TRUE);
		final CountValidator counter0 = new CountValidator(counter1);
		final LinkedList<ValidatorContext> expectedList = new LinkedList<ValidatorContext>();
		assertEquals(0, counter1.getIntepretCount());
		assertEquals(0, counter0.getIntepretCount());
		assertEquals(expectedList, counter1.getContextList());
		assertEquals(expectedList, counter0.getContextList());
		
		expectedList.add(new ValidatorContext());
		assertTrue(counter0.interpret(expectedList.getLast()));
		assertEquals(1, counter1.getIntepretCount());
		assertEquals(1, counter0.getIntepretCount());
		assertEquals(expectedList, counter1.getContextList());
		assertEquals(expectedList, counter0.getContextList());
		
		expectedList.add(new ValidatorContext());
		assertTrue(counter0.interpret(expectedList.getLast()));
		assertEquals(2, counter1.getIntepretCount());
		assertEquals(2, counter0.getIntepretCount());
		assertEquals(expectedList, counter1.getContextList());
		assertEquals(expectedList, counter0.getContextList());
		
		assertEquals(0, counter1.getAppendToCount());
		assertEquals(0, counter0.getAppendToCount());
		assertEquals(0, counter1.getAddMessageToCount());
		assertEquals(0, counter0.getAddMessageToCount());
	}

	public void testInterpretCountFalse(){
		final CountValidator counter1 = new CountValidator(ValidatorConstants.FALSE);
		final CountValidator counter0 = new CountValidator(counter1);
		final LinkedList<ValidatorContext> expectedList = new LinkedList<ValidatorContext>();
		assertEquals(0, counter1.getIntepretCount());
		assertEquals(0, counter0.getIntepretCount());
		assertEquals(expectedList, counter1.getContextList());
		assertEquals(expectedList, counter0.getContextList());

		expectedList.add(new ValidatorContext());
		assertFalse(counter0.interpret(expectedList.getLast()));
		assertEquals(1, counter1.getIntepretCount());
		assertEquals(1, counter0.getIntepretCount());
		assertEquals(expectedList, counter1.getContextList());
		assertEquals(expectedList, counter0.getContextList());

		expectedList.add(new ValidatorContext());
		assertFalse(counter0.interpret(expectedList.getLast()));
		assertEquals(2, counter1.getIntepretCount());
		assertEquals(2, counter0.getIntepretCount());
		assertEquals(expectedList, counter1.getContextList());
		assertEquals(expectedList, counter0.getContextList());

		assertEquals(0, counter1.getAppendToCount());
		assertEquals(0, counter0.getAppendToCount());
		assertEquals(0, counter1.getAddMessageToCount());
		assertEquals(0, counter0.getAddMessageToCount());
	}

	public void testInterpretZero(){
		final CountValidator counter1 = new CountValidator(0, ValidatorConstants.FALSE);
		final LinkedList<ValidatorContext> expectedList = new LinkedList<ValidatorContext>();
		assertEquals(0, counter1.getContextList().size());
		
		expectedList.add(new ValidatorContext());
		assertFalse(counter1.interpret(expectedList.getLast()));
		assertEquals(0, counter1.getContextList().size());
		
		expectedList.add(new ValidatorContext());
		assertFalse(counter1.interpret(expectedList.getLast()));
		assertEquals(0, counter1.getContextList().size());
	}

	public void testInterpretMax(){
		final CountValidator counter0 = new CountValidator(4, ValidatorConstants.FALSE);
		final LinkedList<ValidatorContext> expectedList = new LinkedList<ValidatorContext>();
		
		expectedList.add(new ValidatorContext());
		assertFalse(counter0.interpret(expectedList.getLast()));
		expectedList.add(new ValidatorContext());
		assertFalse(counter0.interpret(expectedList.getLast()));
		expectedList.add(new ValidatorContext());
		assertFalse(counter0.interpret(expectedList.getLast()));
		expectedList.add(new ValidatorContext());
		assertFalse(counter0.interpret(expectedList.getLast()));
		assertEquals(expectedList, counter0.getContextList());

		expectedList.add(new ValidatorContext());
		assertFalse(counter0.interpret(expectedList.getLast()));
		expectedList.removeFirst();
		assertEquals(expectedList, counter0.getContextList());
	}

	public void testAppendToCount() throws IOException{
		final CountValidator counter1 = new CountValidator(ValidatorConstants.FALSE);
		final CountValidator counter0 = new CountValidator(counter1);
		final LinkedList<Appendable> expectedList = new LinkedList<Appendable>();
		assertEquals(0, counter1.getAppendToCount());
		assertEquals(0, counter0.getAppendToCount());
		assertEquals(expectedList, counter1.getSbList());
		assertEquals(expectedList, counter0.getSbList());
		
		expectedList.add(new StringBuilder());
		counter0.appendTo(expectedList.getLast());
		assertEquals("new FalseValidator()", expectedList.getLast().toString());
		assertEquals(1, counter1.getAppendToCount());
		assertEquals(1, counter0.getAppendToCount());
		assertEquals(expectedList, counter1.getSbList());
		assertEquals(expectedList, counter0.getSbList());

		expectedList.add(new StringBuilder());
		counter0.appendTo(expectedList.getLast());
		assertEquals("new FalseValidator()", expectedList.getLast().toString());
		assertEquals(2, counter1.getAppendToCount());
		assertEquals(2, counter0.getAppendToCount());
		assertEquals(expectedList, counter1.getSbList());
		assertEquals(expectedList, counter0.getSbList());

		assertEquals(0, counter1.getIntepretCount());
		assertEquals(0, counter0.getIntepretCount());
		assertEquals(0, counter1.getAddMessageToCount());
		assertEquals(0, counter0.getAddMessageToCount());
	}

	public void testAppendToZero() throws IOException{
		final CountValidator counter0 = new CountValidator(0, ValidatorConstants.FALSE);
		final LinkedList<Appendable> expectedList = new LinkedList<Appendable>();
		assertEquals(0, counter0.getSbList().size());
		
		expectedList.add(new StringBuilder());
		counter0.appendTo(expectedList.getLast());
		assertEquals("new FalseValidator()", expectedList.getLast().toString());
		assertEquals(0, counter0.getSbList().size());

		expectedList.add(new StringBuilder());
		counter0.appendTo(expectedList.getLast());
		assertEquals("new FalseValidator()", expectedList.getLast().toString());
		assertEquals(0, counter0.getSbList().size());
	}

	public void testAppendToMax() throws IOException{
		final CountValidator counter0 = new CountValidator(4, ValidatorConstants.FALSE);
		final LinkedList<Appendable> expectedList = new LinkedList<Appendable>();

		expectedList.add(new StringBuilder());
		counter0.appendTo(expectedList.getLast());
		expectedList.add(new StringBuilder());
		counter0.appendTo(expectedList.getLast());
		expectedList.add(new StringBuilder());
		counter0.appendTo(expectedList.getLast());
		expectedList.add(new StringBuilder());
		counter0.appendTo(expectedList.getLast());
		assertEquals(expectedList, counter0.getSbList());

		expectedList.add(new StringBuilder());
		counter0.appendTo(expectedList.getLast());
		expectedList.removeFirst();
		assertEquals(expectedList, counter0.getSbList());
	}
	
	public void testAddMessageToCount(){
		final CountValidator counter1 = new CountValidator(ValidatorConstants.FALSE);
		final CountValidator counter0 = new CountValidator(counter1);
		final LinkedList<Collection<String>> expectedList = new LinkedList<Collection<String>>();
		assertEquals(0, counter1.getAddMessageToCount());
		assertEquals(0, counter0.getAddMessageToCount());
		assertEquals(expectedList, counter1.getCollectionList());
		assertEquals(expectedList, counter0.getCollectionList());
		
		expectedList.add(new ArrayList<String>());
		counter0.addMessageTo(expectedList.getLast());
		assertEquals(1, counter1.getAddMessageToCount());
		assertEquals(1, counter0.getAddMessageToCount());
		assertEquals(expectedList, counter1.getCollectionList());
		assertEquals(expectedList, counter0.getCollectionList());

		expectedList.add(new ArrayList<String>());
		counter0.addMessageTo(expectedList.getLast());
		assertEquals(2, counter1.getAddMessageToCount());
		assertEquals(2, counter0.getAddMessageToCount());
		assertEquals(expectedList, counter1.getCollectionList());
		assertEquals(expectedList, counter0.getCollectionList());

		assertEquals(0, counter1.getIntepretCount());
		assertEquals(0, counter0.getIntepretCount());
		assertEquals(0, counter1.getAppendToCount());
		assertEquals(0, counter0.getAppendToCount());
	}
	
	public void testAddMessageToZero(){
		final CountValidator counter0 = new CountValidator(0, ValidatorConstants.FALSE);
		final LinkedList<Collection<String>> expectedList = new LinkedList<Collection<String>>();
		assertEquals(0, counter0.getCollectionList().size());
		
		expectedList.add(new ArrayList<String>());
		counter0.addMessageTo(expectedList.getLast());
		assertEquals(0, counter0.getCollectionList().size());

		expectedList.add(new ArrayList<String>());
		counter0.addMessageTo(expectedList.getLast());
		assertEquals(0, counter0.getCollectionList().size());
	}
	
	public void testAddMessageToMax(){
		final CountValidator counter0 = new CountValidator(4, ValidatorConstants.FALSE);
		final LinkedList<Collection<String>> expectedList = new LinkedList<Collection<String>>();

		expectedList.add(new ArrayList<String>());
		counter0.addMessageTo(expectedList.getLast());
		expectedList.add(new ArrayList<String>());
		counter0.addMessageTo(expectedList.getLast());
		expectedList.add(new ArrayList<String>());
		counter0.addMessageTo(expectedList.getLast());
		expectedList.add(new ArrayList<String>());
		counter0.addMessageTo(expectedList.getLast());
		assertEquals(expectedList, counter0.getCollectionList());

		expectedList.add(new ArrayList<String>());
		counter0.addMessageTo(expectedList.getLast());
		expectedList.removeFirst();
		assertEquals(expectedList, counter0.getCollectionList());
	}
}
