package map;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.print.PrinterException;
import java.io.File;
import java.io.IOException;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.WindowConstants;
import javax.swing.filechooser.FileFilter;

/**
 * 地図を表示するフレームです。
 * @author Kumano Tatsuo
 * Created on 2003/11/01, 16:00
 */
public class MainFrame extends JFrame {
    /**
     * 地図を表示するパネル
     */
    final MapPanel panel;

    /**
     * 出力するファイル
     */
    File file;

    /**
     * 出力するファイルの種類を表す定数です。
     * @author Kumano Tatsuo
     * Created on 2005/02/28 17:35:07
     */
    enum FileType {
        /**
         * PSファイル
         */
        PS
    }

    /**
     * ファイルの種類
     */
    FileType fileType;

    /**
     * 上書き保存メニュー
     */
    JMenuItem saveItem;

    /**
     * 新しくフレームを初期化します。
     * @param panel 地図を表示するパネル
     */
    public MainFrame(final MapPanel panel) {
        this.panel = panel;
        this.setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
        this.setSize(800, 600);
        this.setTitle("Map");
        this.setLayout(new BorderLayout());
        this.add(panel, BorderLayout.CENTER);
        JMenuBar menuBar = new JMenuBar();
        this.add(menuBar, BorderLayout.NORTH);
        JMenu fileMenu = new JMenu("ファイル(F)");
        fileMenu.setMnemonic('F');
        menuBar.add(fileMenu);
        this.saveItem = new JMenuItem("上書き保存(S)");
        this.saveItem.setMnemonic('S');
        this.saveItem.addActionListener(new SaveListener());
        this.saveItem.setEnabled(false);
        fileMenu.add(this.saveItem);
        JMenuItem saveAsItem = new JMenuItem("名前を付けて保存(A)...");
        saveAsItem.setMnemonic('A');
        saveAsItem.addActionListener(new SaveAsListener());
        fileMenu.add(saveAsItem);
        fileMenu.addSeparator();
        JMenuItem exitItem = new JMenuItem("終了(X)");
        exitItem.setMnemonic('X');
        exitItem.addActionListener(new ExitListener());
        fileMenu.add(exitItem);
    }

    /**
     * 上書き保存の実装です。
     * @author Kumano Tatsuo
     * Created on 2005/02/28 17:37:25
     */
    private class SaveListener implements ActionListener {
        public void actionPerformed(ActionEvent e) {
            try {
                if (MainFrame.this.fileType == FileType.PS) {
                    MainFrame.this.panel.printPS(MainFrame.this.file);
                }
            } catch (PrinterException e1) {
                e1.printStackTrace();
            } catch (IOException e1) {
                e1.printStackTrace();
            }
        }
    }

    /**
     * 名前を付けて保存の実装です。
     * @author Kumano Tatsuo
     * Created on 2005/02/28 15:39:42
     */
    private class SaveAsListener implements ActionListener {
        public void actionPerformed(ActionEvent e) {
            try {
                JFileChooser chooser = new JFileChooser(new File("."));
                chooser.addChoosableFileFilter(new FileFilter() {
                    public boolean accept(File file) {
                        return file.getName().endsWith(".ps");
                    }

                    public String getDescription() {
                        return "PSファイル（*.ps）";
                    }
                });
                if (MainFrame.this.file != null) {
                    chooser.setSelectedFile(MainFrame.this.file);
                }
                chooser.showSaveDialog(MainFrame.this);
                MainFrame.this.fileType = FileType.PS;
                final File selectedFile = chooser.getSelectedFile();
                if (selectedFile != null) {
                    final String fileName = selectedFile.getName();
                    if (fileName != null) {
                        if (fileName.contains(".")) {
                            MainFrame.this.file = new File(fileName);
                        } else {
                            MainFrame.this.file = new File(fileName + ".ps");
                        }
                        MainFrame.this.panel.printPS(MainFrame.this.file);
                        MainFrame.this.saveItem.setEnabled(true);
                        MainFrame.this.setTitle(MainFrame.this.file.getName() + " - Map");
                    }
                }
            } catch (PrinterException e1) {
                e1.printStackTrace();
            } catch (IOException e1) {
                e1.printStackTrace();
            }
        }
    }
    
    /**
     * 終了の実装です。
     * @author Kumano Tatsuo
     * Created on 2005/02/28 17:49:02
     */
    private class ExitListener implements ActionListener {

        public void actionPerformed(ActionEvent e) {
            System.exit(0);
        }
        
    }
}
