package car;

import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.GeneralPath;
import java.awt.geom.Line2D;
import java.awt.geom.PathIterator;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Collection;

import javax.media.j3d.Appearance;
import javax.media.j3d.BranchGroup;
import javax.media.j3d.Transform3D;
import javax.media.j3d.TransformGroup;
import javax.vecmath.Point3f;
import javax.vecmath.Vector3d;

import com.sun.j3d.utils.geometry.Box;

/**
 * ユーティリティクラスです。
 * @author Kumano Tatsuo
 * Created on 2005/08/24
 */
public class CarUtil {
	/**
	 * 2点を指定して障壁を情景に追加します。
	 * @param x1
	 * @param y1
	 * @param x2
	 * @param y2
	 * @param thickness 障壁の厚み
	 * @param height 障壁の高さ
	 * @param zOffset 高さ方向のオフセット
	 * @param appearance 外見
	 * @param scene 情景
	 */
	public static void addBarrier(final double x1, final double y1,
			final double x2, final double y2, final float thickness,
			final float height, final float zOffset, final Appearance appearance,
			final BranchGroup scene) {
		final double width = Point2D.distance(x1, y1, x2, y2);
		final double centerX = (x1 + x2) / 2;
		final double centerY = (y1 + y2) / 2;
		final double angle = Math.atan2(y2 - y1, x2 - x1);
		final Box box = new Box((float) (width + thickness) / 2, thickness / 2,
				height / 2, appearance);
		final Transform3D transform = new Transform3D();
		transform.set(new Vector3d(centerX, centerY, height / 2 + zOffset));
		final Transform3D rotateTransform = new Transform3D();
		rotateTransform.rotZ(angle);
		transform.mul(rotateTransform);
		final TransformGroup transformGroup = new TransformGroup(transform);
		transformGroup.addChild(box);
		scene.addChild(transformGroup);
	}

	/**
	 * Shapeオブジェクトを3次元の点の一覧に変換します。
	 * @param shape Shapeオブジェクト
	 * @param height 高さ
	 * @return 点の一覧
	 */
	public static Point3f[] toPolygonArray(final Shape shape, final float height) {
		final Collection<Point3f> ret = new ArrayList<Point3f>();
		Point2D firstPoint = null;
		final PathIterator iterator = shape
				.getPathIterator(new AffineTransform());
		float lastX = Float.NaN;
		float lastY = Float.NaN;
		while (!iterator.isDone()) {
			final float[] coords = new float[6];
			final int type = iterator.currentSegment(coords);
			if (type == PathIterator.SEG_CLOSE) {
				if (firstPoint != null) {
					ret.add(new Point3f((float) firstPoint.getX(),
							(float) firstPoint.getY(), height));
				}
			} else {
				final float x = coords[0];
				final float y = coords[1];
				if (firstPoint == null) {
					firstPoint = new Point2D.Float(x, y);
				} else {
					final double distance = Point2D
							.distance(x, y, lastX, lastY);
					ret.add(new Point3f(x, y, height));
				}
				lastX = x;
				lastY = y;
			}
			iterator.next();
		}
		return ret.toArray(new Point3f[] {});
	}

	/**
	 * Shapeオブジェクトを線分の一覧に分解します。
	 * @param shape Shapeオブジェクト
	 * @return 線分の一覧
	 */
	public static ArrayList<Line2D> toLines(final Shape shape) {
		final ArrayList<Line2D> ret = new ArrayList<Line2D>();
		Point2D lastPoint = null;
		Point2D firstPoint = null;
		final PathIterator iterator = shape
				.getPathIterator(new AffineTransform());
		while (!iterator.isDone()) {
			final double[] coords = new double[6];
			final int type = iterator.currentSegment(coords);
			if (type == PathIterator.SEG_CLOSE) {
				if (lastPoint != null && firstPoint != null) {
					ret.add(new Line2D.Double(lastPoint.getX(), lastPoint
							.getY(), firstPoint.getX(), firstPoint.getY()));
				}
			} else {
				final double x = coords[0];
				final double y = coords[1];
				if (lastPoint == null) {
					firstPoint = new Point2D.Double(x, y);
				} else {
					ret.add(new Line2D.Double(lastPoint.getX(), lastPoint
							.getY(), x, y));
				}
				lastPoint = new Point2D.Double(x, y);
			}
			iterator.next();
		}
		return ret;
	}

	/**
	 * 点列から領域を作成します。
	 * @param points 点列
	 * @return 領域
	 */
	public static Shape toShape(final Point2D[] points) {
		GeneralPath ret = null;
		for (final Point2D point : points) {
			if (ret == null) {
				ret = new GeneralPath();
				ret.moveTo((float) point.getX(), (float) point.getY());
			} else {
				ret.lineTo((float) point.getX(), (float) point.getY());
			}
		}
		ret.closePath();
		return ret;
	}

	/**
	 * 点列を折れ線とみなし、ある幅で描画したときの3次元点列を求めます。
	 * @param points 点列
	 * @param width 線幅
	 * @param height 高さ
	 * @return 3次元点列
	 */
	public static Point3f[] createStrokedPolygonArray(final Point2D[] points,
			final double width, final float height) {
		final Collection<Point3f> ret = new ArrayList<Point3f>();
		for (int i = 0; i < points.length; i++) {
			double angle;
			if (i == 0) {
				angle = Math.atan2(points[i + 1].getY() - points[i].getY(),
						points[i + 1].getX() - points[i].getX());
			} else if (i == points.length - 1) {
				angle = Math.atan2(points[i].getY() - points[i - 1].getY(),
						points[i].getX() - points[i - 1].getX());
			} else {
				angle = Math.atan2(points[i + 1].getY() - points[i - 1].getY(),
						points[i + 1].getX() - points[i - 1].getX());
			}
			final double dx = Math.cos(angle + Math.PI / 2) * width / 2;
			final double dy = Math.sin(angle + Math.PI / 2) * width / 2;
			ret.add(new Point3f((float) (points[i].getX() + dx),
					(float) (points[i].getY() + dy), height));
			ret.add(new Point3f((float) (points[i].getX() - dx),
					(float) (points[i].getY() - dy), height));
		}
		return ret.toArray(new Point3f[] {});
	}
	/**
	 * 点列を折れ線とみなし、ある幅で描画したときの領域を取得します。
	 * @param points 点列
	 * @param width 線幅
	 * @return 領域
	 */
	public static Shape createStrokedShape(final Point2D[] points, final double width) {
		GeneralPath ret = null;
		final ArrayList<Point2D> list = new ArrayList<Point2D>();
		for (int i = 0; i < points.length; i++) {
			double angle;
			if (i == 0) {
				angle = Math.atan2(points[i + 1].getY() - points[i].getY(),
						points[i + 1].getX() - points[i].getX());
			} else if (i == points.length - 1) {
				angle = Math.atan2(points[i].getY() - points[i - 1].getY(),
						points[i].getX() - points[i - 1].getX());
			} else {
				angle = Math.atan2(points[i + 1].getY() - points[i - 1].getY(),
						points[i + 1].getX() - points[i - 1].getX());
			}
			final double dx = Math.cos(angle + Math.PI / 2) * width / 2;
			final double dy = Math.sin(angle + Math.PI / 2) * width / 2;
			list.add(new Point2D.Double(points[i].getX() + dx, points[i].getY()
					+ dy));
			list.add(new Point2D.Double(points[i].getX() - dx, points[i].getY()
					- dy));
		}
		for (int i = 0; i < list.size() - 1; i += 2) {
			if (ret == null) {
				ret = new GeneralPath();
				ret.moveTo((float) list.get(i).getX(), (float) list.get(i)
						.getY());
			} else {
				ret.lineTo((float) list.get(i).getX(), (float) list.get(i)
						.getY());
			}
		}
		for (int i = list.size() - 1; i >= 1; i -= 2) {
			ret.lineTo((float) list.get(i).getX(), (float) list.get(i).getY());
		}
		if (ret != null) {
			ret.closePath();
		}
		return ret;
	}
	
	/**
	 * @param accel 加速
	 * @param decel 減速
	 * @return 最高速度
	 */
	public static double getMaxSpeed(final double accel, final double decel) {
		return (accel * decel) / (1 - decel);
	}
	
	/**
	 * @param speed 最高速度
	 * @param accel 加速
	 * @return 減速
	 */
	public static double getDecel(final double speed, final double accel) {
		return speed / (speed + accel);
	}

}
