package car;

import java.awt.Shape;
import java.awt.event.KeyEvent;
import java.awt.geom.AffineTransform;
import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.Queue;

import car.character.Character;
import car.map.RoadMap;

/**
 * @author Kumano Tatsuo
 * Created on 2004/12/11
 */
public class MainLoop extends Thread {

	/**
	 * プレイヤ1のキャラクタ
	 */
	private int character1;

	/**
	 * プレイヤ2のキャラクタ
	 */
	private int character2;

	/**
	 * キャラクタの一覧
	 */
	final private Character[] characters;

	/**
	 * プレイヤ1用のチェックポイントの一覧
	 */
	private final Queue<Line2D> checkPoints1;

	/**
	 * プレイヤ2用のチェックポイントの一覧
	 */
	private final Queue<Line2D> checkPoints2;

	/**
	 * ゴールするのにかかった時間
	 */
	private long finishTime;

	/**
	 * ゲームの状態
	 */
	final private Game game;

	/**
	 * キーハンドラ
	 */
	private final KeyHandler handler;

	/**
	 * プレイヤ1がゴールしたかどうか
	 */
	private boolean isFinished1;

	/**
	 * プレイヤ2がゴールしたかどうか
	 */
	private boolean isFinished2;

	/**
	 * プレイヤ1がキャラクタを確定したかどうか
	 */
	private boolean isOK1;

	/**
	 * プレイヤ2がキャラクタを確定したかどうか
	 */
	private boolean isOK2;

	/**
	 * 1Pが逆走しているかどうか
	 */
	private boolean isWrongWay1;

	/**
	 * 2Pが逆走しているかどうか
	 */
	private boolean isWrongWay2;

	/**
	 * プレイヤ1の周回
	 */
	private int lap1;

	/**
	 * プレイヤ2の周回
	 */
	private int lap2;

	/**
	 * 最後にループが実行された時刻
	 */
	private long lastTime;

	/**
	 * ループの待ち時間
	 */
	private long waitTime;

	/**
	 * 地図
	 */
	private final RoadMap[] maps;

	/**
	 * パネル1
	 */
	private final PlayerPanel panel1;

	/**
	 * パネル2
	 */
	private final PlayerPanel panel2;

	/**
	 * プレイヤ1
	 */
	private final Player player1;

	/**
	 * プレイヤ2
	 */
	private final Player player2;

	/**
	 * 選択されている地図
	 */
	private RoadMap selectedMap;

	/**
	 * 地図の番号
	 */
	private int selectedMapIndex;

	/**
	 * ゲームが開始された時刻
	 */
	private long startTime;

	/**
	 * プレイヤ1の勝ち数
	 */
	private int win1;

	/**
	 * プレイヤ2の勝ち数
	 */
	private int win2;

	/**
	 * コンストラクタです。
	 * @param player1 プレイヤ1
	 * @param player2 プレイヤ2
	 * @param panel1 パネル1
	 * @param panel2 パネル2
	 * @param maps 地図の一覧
	 * @param handler キーハンドラ
	 * @param characters キャラクタの一覧
	 * @param game ゲームの状態
	 */
	public MainLoop(final Player player1, final Player player2, final PlayerPanel panel1,
			final PlayerPanel panel2, final RoadMap[] maps, final KeyHandler handler,
			final Character[] characters, final Game game) {
		this.player1 = player1;
		this.player2 = player2;
		this.panel1 = panel1;
		this.panel2 = panel2;
		this.maps = maps;
		this.checkPoints1 = new LinkedList<Line2D>();
		this.checkPoints2 = new LinkedList<Line2D>();
		switchMap(0);
		this.selectedMapIndex = 0;
		this.handler = handler;
		this.character1 = 0;
		this.character2 = 0;
		this.characters = characters;
		this.game = game;
		this.isOK1 = false;
		this.isOK2 = false;
		this.waitTime = 50;
		this.panel1.switchMyCar(false);
		this.panel2.switchMyCar(false);
		this.panel1.switchLap(Const.Lap.PRESS_ACCEL);
		this.panel2.switchLap(Const.Lap.PRESS_ACCEL);
	}

	/**
	 * 順位を調べます。
	 */
	private void checkRank() {
		if (this.checkPoints1.size() < this.checkPoints2.size()) {
			this.panel1.switchRank(0);
			this.panel2.switchRank(1);
		} else if (this.checkPoints2.size() < this.checkPoints1.size()) {
			this.panel2.switchRank(0);
			this.panel1.switchRank(1);
		} else {
			if (this.checkPoints1.size() > 0 && this.checkPoints2.size() > 0) {
				final Line2D line1 = this.checkPoints1.peek();
				final double distanceSq1 = Point2D.distanceSq(this.player1.getX(), this.player1
						.getY(), (line1.getX1() + line1.getX2()) / 2, (line1.getY1() + line1
						.getY2()) / 2);
				final Line2D line2 = this.checkPoints1.peek();
				final double distanceSq2 = Point2D.distanceSq(this.player2.getX(), this.player2
						.getY(), (line2.getX1() + line2.getX2()) / 2, (line2.getY1() + line2
						.getY2()) / 2);
				if (distanceSq1 < distanceSq2) {
					this.panel1.switchRank(0);
					this.panel2.switchRank(1);
				} else if (distanceSq2 < distanceSq1) {
					this.panel2.switchRank(0);
					this.panel1.switchRank(1);
				}
			}
		}
	}

	/**
	 * プレイヤを動かします。
	 * @param player プレイヤ
	 */
	private void movePlayer(final Player player) {
		boolean isRoad = this.selectedMap.getRoadShape().contains(player.getX(), player.getY());
		if (!isRoad) {
			for (final Shape shape : this.selectedMap.getStripes()) {
				if (shape.contains(player.getX(), player.getY())) {
					isRoad = true;
					break;
				}
			}
		}
		final double forwardAccel = isRoad ? player.getCharacter().getFowardRoadAccel() : player
				.getCharacter().getFowardDirtAccel();
		final double rearAccel = player.getCharacter().getRearAccel();
		final double normalDecel = player.getCharacter().getNormalDecel();
		final double handleDecel = player.getCharacter().getHandleDecel();
		final double handleDecelThreshold = player.getCharacter().getHandleDecelThreshold();
		final double stopThreshold = rearAccel * 0.99;
		final double forwardHandle = isRoad ? player.getCharacter().getFowardRoadHandle() : player
				.getCharacter().getFowardDirtHandle();
		final double rearHandle = player.getCharacter().getRearHandle();
		final double normalHandleDecel = player.getCharacter().getNormalHandleDecel();
		final double slideDecel = player.getCharacter().getSlideDecel();
		if (this.game.getStatus() == Const.Status.PLAYING) {
			if (player.getSpeed() > 0) {
				if (player.isDown()) {
					player.setSpeed(player.getSpeed() * player.getCharacter().getBrakeDecel());
				} else if (player.isUp()) {
					player.setSpeed(player.getSpeed() + forwardAccel);
				} else if (!player.isDown() && !player.isUp()) {
					player.setSpeed(player.getSpeed() * player.getCharacter().getEngineDecel());
				}
				player.setSpeed(player.getSpeed() * normalDecel);
				if (player.getSpeed() < forwardAccel) {
					player.setSpeed(0);
				}
				if (player.isLeft()) {
					player.setHandle(player.getHandle() + forwardHandle);
					player.setHandle(player.getHandle() * normalHandleDecel);
					if (player.getHandle() > handleDecelThreshold) {
						player.setSpeed(player.getSpeed() * handleDecel);
					}
				} else if (player.isRight()) {
					player.setHandle(player.getHandle() - forwardHandle);
					player.setHandle(player.getHandle() * normalHandleDecel);
					if (player.getHandle() < -handleDecelThreshold) {
						player.setSpeed(player.getSpeed() * handleDecel);
					}
				} else if (!player.isLeft() && !player.isRight()) {
					player.setHandle(player.getHandle() / 2);
				}
				player.setDirection(player.getDirection() + player.getHandle());
			} else if (player.getSpeed() < 0) {
				if (player.isDown()) {
					player.setSpeed(player.getSpeed() - rearAccel);
				} else if (player.isUp()) {
					player.setSpeed(player.getSpeed() * player.getCharacter().getBrakeDecel());
				} else if (!player.isDown() && !player.isUp()) {
					player.setSpeed(player.getSpeed() * player.getCharacter().getEngineDecel());
				}
				player.setSpeed(player.getSpeed() * normalDecel);
				if (player.getSpeed() > -rearAccel) {
					player.setSpeed(0);
				}
				if (player.isLeft()) {
					player.setHandle(player.getHandle() + forwardHandle);
					player.setHandle(player.getHandle() * normalHandleDecel);
				} else if (player.isRight()) {
					player.setHandle(player.getHandle() - forwardHandle);
					player.setHandle(player.getHandle() * normalHandleDecel);
				} else if (!player.isLeft() && !player.isRight()) {
					player.setHandle(player.getHandle() / 2);
				}
				player.setDirection(player.getDirection() - player.getHandle());
			} else {
				if (player.isDown()) {
					player.setSpeed(player.getSpeed() - rearAccel);
				} else if (player.isUp()) {
					player.setSpeed(player.getSpeed() + forwardAccel);
				}
				player.setHandle(player.getHandle() / 2);
			}
		} else if (this.game.getStatus() == Const.Status.GAME_OVER) {
			player.setSpeed(player.getSpeed() * normalDecel);
			player.setSpeed(player.getSpeed() * player.getCharacter().getEngineDecel());
		}

		// ドリフト
		final double slideForce = player.getSpeed() * Math.sin(player.getHandle());
		if (Math.abs(slideForce) > player.getCharacter().getDriftThreshold()) {
			player.setSlideSpeed(player.getSlideSpeed() - slideForce
					* player.getCharacter().getDriftAmount()
					* (Math.abs(slideForce) - player.getCharacter().getDriftThreshold()));
		}

		player.setX(player.getX() + player.getSpeed() * Math.cos(player.getDirection())
				+ player.getSlideSpeed() * Math.cos(player.getDirection() + Math.PI / 2)
				+ player.getDx());
		player.setY(player.getY() + player.getSpeed() * Math.sin(player.getDirection())
				+ player.getSlideSpeed() * Math.sin(player.getDirection() + Math.PI / 2)
				+ player.getDy());
		player.setSlideSpeed(player.getSlideSpeed() * slideDecel);
		player.setDx(player.getDx() * slideDecel);
		player.setDy(player.getDy() * slideDecel);
		// 壁との衝突判定
		if (!isRoad) {
			final Rectangle2D rectangle = new Rectangle2D.Double(
					-player.getCharacter().getWidth() / 2, -player.getCharacter().getLength() / 2,
					player.getCharacter().getWidth(), player.getCharacter().getLength());
			for (final Line2D line : this.selectedMap.getBarriers()) {
				final AffineTransform transform = new AffineTransform();
				transform.rotate(-player.getDirection());
				transform.translate(-player.getX(), -player.getY());
				if (transform.createTransformedShape(line).intersects(rectangle)) {
					double angle = Math.atan2(line.getY2() - line.getY1(), line.getX2()
							- line.getX1())
							- player.getDirection()
							+ Math.PI
							/ 2
							* 3
							- Math.atan2(player.getSlideSpeed(), player.getSpeed());
					angle %= Math.PI;
					if (angle > Math.PI / 2) {
						angle -= Math.PI;
					}
					final double rebound = Math.max(Math.cos(angle), 0.5);
					if (angle > 0) {
						angle = Math.min(angle, Math.PI / 180);
					} else {
						angle = Math.max(angle, -Math.PI / 180);
					}
					player.setHandle(player.getHandle() - angle);
					double lineDirection = Math.atan2(line.getY2() - line.getY1(), line.getX2()
							- line.getX1())
							+ Math.PI / 2;
					final double dx = Math.cos(lineDirection)
							* Math.abs(player.getSpeed() * rebound);
					final double dy = Math.sin(lineDirection)
							* Math.abs(player.getSpeed() * rebound);
					player.setDx(dx);
					player.setDy(dy);
					player.setSpeed(player.getSpeed() / 2);
					player.setSlideSpeed(0);
					break;
				}
			}
		}
		// プレイヤ同士の当たり判定
		if (Point2D.distanceSq(this.player1.getX(), this.player1.getY(), this.player2.getX(),
				this.player2.getY()) < (this.player1.getCharacter().getWidth() / 2 + this.player2
				.getCharacter().getWidth() / 2)
				* (this.player1.getCharacter().getLength() / 2 + this.player2.getCharacter()
						.getLength() / 2)) {
			final double distance = (Point2D.distance(this.player1.getX(), this.player1.getY(),
					this.player2.getX(), this.player2.getY()));
			final double angle = Math.atan2(this.player2.getY() - this.player1.getY(), this.player2
					.getX()
					- this.player1.getX());
			this.player2.setSpeed(this.player2.getSpeed()
					+ Math.cos(angle - this.player2.getDirection()) / distance
					* Math.min(1, Math.abs(this.player1.getSpeed())));
			this.player2.setSlideSpeed(this.player2.getSlideSpeed()
					+ Math.sin(angle - this.player2.getDirection()) / distance
					* Math.min(1, Math.abs(this.player1.getSpeed())));
			this.player1.setSpeed(this.player1.getSpeed()
					+ Math.cos(angle - this.player1.getDirection() + Math.PI) / distance
					* Math.min(1, Math.abs(this.player2.getSpeed())));
			this.player1.setSlideSpeed(this.player1.getSlideSpeed()
					+ Math.sin(angle - this.player1.getDirection() + Math.PI) / distance
					* Math.min(1, Math.abs(this.player2.getSpeed())));
		}
		// チェックポイントを調べる
		if (!this.checkPoints1.isEmpty()) {
			final Line2D line = this.checkPoints1.peek();
			if (line.intersects(this.player1.getX() - this.player1.getCharacter().getLength(),
					this.player1.getY() - this.player1.getCharacter().getLength(), Math.max(
							this.player1.getCharacter().getLength() * 2,
							this.player1.getSpeed() * 2), Math.max(this.player1.getCharacter()
							.getLength() * 2, this.player1.getSpeed() * 2))) {
				this.checkPoints1.poll();
				if (this.checkPoints1.size() % this.selectedMap.getCheckPoints().length == 0) {
					this.lap1++;
				}
				this.panel1.switchLap(this.lap1 < this.selectedMap.getLap() - 1 ? this.lap1
						: Const.Lap.FINAL_LAP);
				if (this.checkPoints1.isEmpty()) {
					this.finishTime = (System.currentTimeMillis() - this.startTime);
					this.panel1.switchLap(Const.Lap.HIDE);
					this.isFinished1 = true;
				}
			}
			final double lastDistanceSq = Point2D.distanceSq(this.player1.getLastX(), this.player1
					.getLastY(), (line.getX1() + line.getX2()) / 2,
					(line.getY1() + line.getY2()) / 2);
			final double distanceSq = Point2D.distanceSq(this.player1.getX(), this.player1.getY(),
					(line.getX1() + line.getX2()) / 2, (line.getY1() + line.getY2()) / 2);
			if (lastDistanceSq < distanceSq) {
				if (!this.isWrongWay1) {
					this.panel1.switchLap(Const.Lap.WRONG_WAY);
				}
				this.isWrongWay1 = true;
			} else {
				if (this.isWrongWay1) {
					this.panel1.switchLap(this.lap1 < this.selectedMap.getLap() - 1 ? this.lap1
							: Const.Lap.FINAL_LAP);
				}
				this.isWrongWay1 = false;
			}
		}
		if (!this.checkPoints2.isEmpty()) {
			final Line2D line = this.checkPoints2.peek();
			if (line.intersects(this.player2.getX() - this.player2.getCharacter().getLength(),
					this.player2.getY() - this.player2.getCharacter().getLength(), Math.max(
							this.player2.getCharacter().getLength() * 2,
							this.player2.getSpeed() * 2), Math.max(this.player2.getCharacter()
							.getLength() * 2, this.player2.getSpeed() * 2))) {
				this.checkPoints2.poll();
				if (this.checkPoints2.size() % this.selectedMap.getCheckPoints().length == 0) {
					this.lap2++;
				}
				this.panel2.switchLap(this.lap2 < this.selectedMap.getLap() - 1 ? this.lap2
						: Const.Lap.FINAL_LAP);
				if (this.checkPoints2.isEmpty()) {
					this.finishTime = (System.currentTimeMillis() - this.startTime);
					this.panel2.switchLap(Const.Lap.HIDE);
					this.isFinished2 = true;
				}
			}
			final double lastDistanceSq = Point2D.distanceSq(this.player2.getLastX(), this.player2
					.getLastY(), (line.getX1() + line.getX2()) / 2,
					(line.getY1() + line.getY2()) / 2);
			final double distanceSq = Point2D.distanceSq(this.player2.getX(), this.player2.getY(),
					(line.getX1() + line.getX2()) / 2, (line.getY1() + line.getY2()) / 2);
			if (lastDistanceSq < distanceSq) {
				if (!this.isWrongWay2) {
					this.panel2.switchLap(Const.Lap.WRONG_WAY);
				}
				this.isWrongWay2 = true;
			} else {
				if (this.isWrongWay2) {
					this.panel2.switchLap(this.lap2 < this.selectedMap.getLap() - 1 ? this.lap2
							: Const.Lap.FINAL_LAP);
				}
				this.isWrongWay2 = false;
			}
		}
	}

	/**
	 * キーを取得してプレイヤに設定します。
	 */
	private void pickKeys() {
		if (this.handler.isPressed(Const.Key.PLAYER1_ACCEL)
				&& !this.handler.isPressed(Const.Key.PLAYER1_BRAKE)) {
			this.player1.setUp(true);
			this.player1.setDown(false);
		} else if (this.handler.isPressed(Const.Key.PLAYER1_BRAKE)
				&& !this.handler.isPressed(Const.Key.PLAYER1_ACCEL)) {
			this.player1.setDown(true);
			this.player1.setUp(false);
		} else if (!this.handler.isPressed(Const.Key.PLAYER1_ACCEL)
				&& !this.handler.isPressed(Const.Key.PLAYER1_BRAKE)) {
			this.player1.setUp(false);
			this.player1.setDown(false);
		} else {
			this.player1.setUp(false);
			this.player1.setDown(false);
		}
		if (this.handler.isPressed(Const.Key.PLAYER1_LEFT)
				&& !this.handler.isPressed(Const.Key.PLAYER1_RIGHT)) {
			this.player1.setLeft(true);
			this.player1.setRight(false);
		} else if (this.handler.isPressed(Const.Key.PLAYER1_RIGHT)
				&& !this.handler.isPressed(Const.Key.PLAYER1_LEFT)) {
			this.player1.setRight(true);
			this.player1.setLeft(false);
		} else if (!this.handler.isPressed(Const.Key.PLAYER1_RIGHT)
				&& !this.handler.isPressed(Const.Key.PLAYER1_LEFT)) {
			this.player1.setLeft(false);
			this.player1.setRight(false);
		} else {
			this.player1.setLeft(false);
			this.player1.setRight(false);
		}
		if (this.handler.isPressed(Const.Key.PLAYER2_ACCEL)
				&& !this.handler.isPressed(Const.Key.PLAYER2_BRAKE)) {
			this.player2.setUp(true);
			this.player2.setDown(false);
		} else if (this.handler.isPressed(Const.Key.PLAYER2_BRAKE)
				&& !this.handler.isPressed(Const.Key.PLAYER2_ACCEL)) {
			this.player2.setDown(true);
			this.player2.setUp(false);
		} else if (!this.handler.isPressed(Const.Key.PLAYER2_ACCEL)
				&& !this.handler.isPressed(Const.Key.PLAYER2_BRAKE)) {
			this.player2.setUp(false);
			this.player2.setDown(false);
		} else {
			this.player2.setUp(false);
			this.player2.setDown(false);
		}
		if (this.handler.isPressed(Const.Key.PLAYER2_LEFT)
				&& !this.handler.isPressed(Const.Key.PLAYER2_RIGHT)) {
			this.player2.setLeft(true);
			this.player2.setRight(false);
		} else if (this.handler.isPressed(Const.Key.PLAYER2_RIGHT)
				&& !this.handler.isPressed(Const.Key.PLAYER2_LEFT)) {
			this.player2.setRight(true);
			this.player2.setLeft(false);
		} else if (!this.handler.isPressed(Const.Key.PLAYER2_RIGHT)
				&& !this.handler.isPressed(Const.Key.PLAYER2_LEFT)) {
			this.player2.setLeft(false);
			this.player2.setRight(false);
		} else {
			this.player2.setLeft(false);
			this.player2.setRight(false);
		}
	}

	public void run() {
		while (!this.handler.isPressed(KeyEvent.VK_ESCAPE)) {
			try {
				pickKeys();
				switch (this.game.getStatus()) {
				case TITLE:
					if (this.handler.wasPressed(Const.Key.PLAYER1_ACCEL)) {
						this.panel1.switchLap(Const.Lap.HIDE);
						this.isOK1 = true;
					}
					if (this.handler.wasPressed(Const.Key.PLAYER2_ACCEL)) {
						this.panel2.switchLap(Const.Lap.HIDE);
						this.isOK2 = true;
					}
					if (this.isOK1 && this.isOK2) {
						this.isOK1 = false;
						this.isOK2 = false;
						this.selectedMapIndex = 1;
						this.panel1.switchLap(Const.Lap.SELECT_CHARACTER);
						this.panel2.switchLap(Const.Lap.SELECT_CHARACTER);
						this.handler.clearKeys();
						this.game.setStatus(Const.Status.SELECT_CHARACTER);
					}
					this.panel1.hideCourseMap();
					this.panel2.hideCourseMap();
					this.panel1.update(-1, -1, -1);
					this.panel2.update(-1, -1, -1);
					break;
				case SELECT_CHARACTER:
					if (this.handler.wasPressed(Const.Key.PLAYER1_RIGHT) && !this.isOK1) {
						this.character1 = (this.character1 + 1) % this.characters.length;
						this.player1.setCharacter(this.characters[this.character1]);
						this.panel1.switchCharacter(this.character1);
						this.panel2.switchCharacter(this.character1);
					} else if (this.handler.wasPressed(Const.Key.PLAYER1_LEFT) && !this.isOK1) {
						this.character1 = (this.character1 - 1 + this.characters.length)
								% this.characters.length;
						this.player1.setCharacter(this.characters[this.character1]);
						this.panel1.switchCharacter(this.character1);
						this.panel2.switchCharacter(this.character1);
					} else if (this.handler.wasPressed(Const.Key.PLAYER1_ACCEL)) {
						this.isOK1 = true;
						this.panel1.switchLap(Const.Lap.HIDE);
					} else if (this.handler.wasPressed(Const.Key.PLAYER1_BRAKE)) {
						this.isOK1 = false;
						this.panel1.switchLap(Const.Lap.SELECT_CHARACTER);
					}
					if (this.handler.wasPressed(Const.Key.PLAYER2_RIGHT) && !this.isOK2) {
						this.character2 = (this.character2 + 1) % this.characters.length;
						this.player2.setCharacter(this.characters[this.character2]);
						this.panel1.switchCharacter2(this.character2);
						this.panel2.switchCharacter2(this.character2);
					} else if (this.handler.wasPressed(Const.Key.PLAYER2_LEFT) && !this.isOK2) {
						this.character2 = (this.character2 - 1 + this.characters.length)
								% this.characters.length;
						this.player2.setCharacter(this.characters[this.character2]);
						this.panel1.switchCharacter2(this.character2);
						this.panel2.switchCharacter2(this.character2);
					} else if (this.handler.wasPressed(Const.Key.PLAYER2_ACCEL)) {
						this.isOK2 = true;
						this.panel2.switchLap(Const.Lap.HIDE);
					} else if (this.handler.wasPressed(Const.Key.PLAYER2_BRAKE)) {
						this.isOK2 = false;
						this.panel2.switchLap(Const.Lap.SELECT_CHARACTER);
					}
					if (this.isOK1) {
						this.player1.setDirection(this.selectedMap.getInitialDirection());
					} else {
						this.player1.setDirection(this.player1.getDirection() + 0.03);
					}
					if (this.isOK2) {
						this.player2.setDirection(this.selectedMap.getInitialDirection());
					} else {
						this.player2.setDirection(this.player2.getDirection() + 0.03);
					}
					if (this.isOK1 && this.isOK2) {
						this.handler.clearKeys();
						this.isOK1 = false;
						this.isOK2 = false;
						this.game.setStatus(Const.Status.SELECT_COURSE);
						this.panel1.switchLap(Const.Lap.SELECT_COURSE);
						this.panel2.switchLap(Const.Lap.SELECT_COURSE);
						switchMap(this.selectedMapIndex);
						this.panel1.switchMap(this.selectedMapIndex);
						this.panel2.switchMap(this.selectedMapIndex);
						this.player1.setX(this.selectedMap.getInitialLocation1().getX());
						this.player1.setX(this.selectedMap.getInitialLocation1().getX());
						this.player1.setY(this.selectedMap.getInitialLocation1().getY());
						this.player1.setY(this.selectedMap.getInitialLocation1().getY());
						this.player1.setDirection(this.selectedMap.getInitialDirection());
						this.player2.setX(this.selectedMap.getInitialLocation2().getX());
						this.player2.setX(this.selectedMap.getInitialLocation2().getX());
						this.player2.setY(this.selectedMap.getInitialLocation2().getY());
						this.player2.setY(this.selectedMap.getInitialLocation2().getY());
						this.player2.setDirection(this.selectedMap.getInitialDirection());
					}
					this.panel1.update(-1, -1, -1);
					this.panel2.update(-1, -1, -1);
					break;
				case SELECT_COURSE:
					if (!this.isOK1 && !this.isOK2) {
						if (this.handler.wasPressed(Const.Key.PLAYER1_RIGHT)
								|| this.handler.wasPressed(Const.Key.PLAYER2_RIGHT)) {
							this.selectedMapIndex = this.selectedMapIndex % (this.maps.length - 1)
									+ 1;
							switchMap(this.selectedMapIndex);
							this.panel1.switchMap(this.selectedMapIndex);
							this.panel2.switchMap(this.selectedMapIndex);
							this.player1.setX(this.selectedMap.getInitialLocation1().getX());
							this.player1.setX(this.selectedMap.getInitialLocation1().getX());
							this.player1.setY(this.selectedMap.getInitialLocation1().getY());
							this.player1.setY(this.selectedMap.getInitialLocation1().getY());
							this.player1.setDirection(this.selectedMap.getInitialDirection());
							this.player2.setX(this.selectedMap.getInitialLocation2().getX());
							this.player2.setX(this.selectedMap.getInitialLocation2().getX());
							this.player2.setY(this.selectedMap.getInitialLocation2().getY());
							this.player2.setY(this.selectedMap.getInitialLocation2().getY());
							this.player2.setDirection(this.selectedMap.getInitialDirection());
						}
						if (this.handler.wasPressed(Const.Key.PLAYER1_LEFT)
								|| this.handler.wasPressed(Const.Key.PLAYER2_LEFT)) {
							this.selectedMapIndex--;
							if (this.selectedMapIndex == 0) {
								this.selectedMapIndex = this.maps.length - 1;
							}
							switchMap(this.selectedMapIndex);
							this.panel1.switchMap(this.selectedMapIndex);
							this.panel2.switchMap(this.selectedMapIndex);
							this.player1.setX(this.selectedMap.getInitialLocation1().getX());
							this.player1.setX(this.selectedMap.getInitialLocation1().getX());
							this.player1.setY(this.selectedMap.getInitialLocation1().getY());
							this.player1.setY(this.selectedMap.getInitialLocation1().getY());
							this.player1.setDirection(this.selectedMap.getInitialDirection());
							this.player2.setX(this.selectedMap.getInitialLocation2().getX());
							this.player2.setX(this.selectedMap.getInitialLocation2().getX());
							this.player2.setY(this.selectedMap.getInitialLocation2().getY());
							this.player2.setY(this.selectedMap.getInitialLocation2().getY());
							this.player2.setDirection(this.selectedMap.getInitialDirection());
						}
					}
					if (this.handler.wasPressed(Const.Key.PLAYER1_ACCEL)) {
						this.panel1.switchLap(Const.Lap.HIDE);
						if (!this.isOK2) {
							this.panel2.switchLap(Const.Lap.PRESS_ACCEL);
						}
						this.isOK1 = true;
					}
					if (this.handler.wasPressed(Const.Key.PLAYER2_ACCEL)) {
						this.panel2.switchLap(Const.Lap.HIDE);
						if (!this.isOK1) {
							this.panel1.switchLap(Const.Lap.PRESS_ACCEL);
						}
						this.isOK2 = true;
					}
					if (this.handler.wasPressed(Const.Key.PLAYER1_BRAKE)
							|| this.handler.wasPressed(Const.Key.PLAYER2_BRAKE)) {
						this.panel1.switchLap(Const.Lap.SELECT_COURSE);
						this.panel2.switchLap(Const.Lap.SELECT_COURSE);
						this.isOK1 = false;
						this.isOK2 = false;
					}
					if (this.isOK1 && this.isOK2) {
						this.startTime = System.currentTimeMillis() + 3000;
						this.lap1 = 0;
						this.lap2 = 0;
						this.isWrongWay1 = false;
						this.isWrongWay2 = false;
						this.isFinished1 = false;
						this.isFinished2 = false;
						this.isOK1 = false;
						this.isOK2 = false;
						this.panel1.switchMyCar(true);
						this.panel2.switchMyCar(true);
						this.panel1.switchLap(Const.Lap.READY);
						this.panel2.switchLap(Const.Lap.READY);
						this.panel1.switchRank(0);
						this.panel2.switchRank(1);
						this.panel1.switchSpeedBar(true);
						this.panel2.switchSpeedBar(true);
						this.panel1.switchHelp(false);
						this.panel2.switchHelp(false);
						this.handler.clearKeys();
						this.player1.setSpeed(0);
						this.player2.setSpeed(0);
						this.player1.setHandle(0);
						this.player2.setHandle(0);
						this.game.setStatus(Const.Status.PLAYING);
					}
					this.panel1.update(-1, -1, -1);
					this.panel2.update(-1, -1, -1);
					break;
				case PLAYING:
					if (System.currentTimeMillis() > this.startTime) {
						if (this.lastTime <= this.startTime) {
							this.panel1.switchLap(Const.Lap.GO);
							this.panel2.switchLap(Const.Lap.GO);
						}
						for (final Player player : new Player[] { this.player1, this.player2 }) {
							movePlayer(player);
						}
						checkRank();
						if (this.isFinished1 && !this.isFinished2) {
							this.win1++;
							this.panel1.showWin("win(" + this.win1 + "-" + this.win2 + ")");
							this.panel2.showWin("lose(" + this.win2 + "-" + this.win1 + ")");
							this.panel1.switchLap(Const.Lap.PRESS_ACCEL);
							this.panel2.switchLap(Const.Lap.PRESS_ACCEL);
							this.handler.clearKeys();
							this.game.setStatus(Const.Status.GAME_OVER);
						} else if (this.isFinished2 && !this.isFinished1) {
							this.win2++;
							this.panel1.showWin("lose(" + this.win1 + "-" + this.win2 + ")");
							this.panel2.showWin("win(" + this.win2 + "-" + this.win1 + ")");
							this.panel1.switchLap(Const.Lap.PRESS_ACCEL);
							this.panel2.switchLap(Const.Lap.PRESS_ACCEL);
							this.handler.clearKeys();
							this.game.setStatus(Const.Status.GAME_OVER);
						} else if (this.isFinished1 && this.isFinished2) {
							this.panel1.showWin("draw(" + this.win1 + "-" + this.win2 + ")");
							this.panel2.showWin("draw(" + this.win2 + "-" + this.win1 + ")");
							this.panel1.switchLap(Const.Lap.PRESS_ACCEL);
							this.panel2.switchLap(Const.Lap.PRESS_ACCEL);
							this.handler.clearKeys();
							this.game.setStatus(Const.Status.GAME_OVER);
						}
						for (PlayerPanel panel : new PlayerPanel[] { this.panel1, this.panel2 }) {
							final int time = (int) ((panel == this.panel1 && !this.isFinished1 || panel == this.panel2
									&& !this.isFinished2) ? System.currentTimeMillis()
									- this.startTime : this.finishTime);
							final int minutes = time / 1000 / 60 % 60;
							final int seconds = (time / 1000) % 60;
							final int millis = (time % 1000) / 10;
							panel.update(minutes, seconds, millis);
						}
					} else {
						this.panel1.update(0, 0, 0);
						this.panel2.update(0, 0, 0);
					}
					break;
				case GAME_OVER:
					if (this.handler.wasPressed(Const.Key.PLAYER1_ACCEL)) {
						this.panel1.switchLap(Const.Lap.HIDE);
						this.isOK1 = true;
					}
					if (this.handler.wasPressed(Const.Key.PLAYER2_ACCEL)) {
						this.panel2.switchLap(Const.Lap.HIDE);
						this.isOK2 = true;
					}
					if (this.isOK1 && this.isOK2) {
						this.lap1 = 0;
						this.lap2 = 0;
						this.isOK1 = false;
						this.isOK2 = false;
						switchMap(this.selectedMapIndex);
						this.panel1.switchMap(this.selectedMapIndex);
						this.panel2.switchMap(this.selectedMapIndex);
						this.panel1.hideCourseMap();
						this.panel2.hideCourseMap();
						this.player1.setX(this.selectedMap.getInitialLocation1().getX());
						this.player1.setX(this.selectedMap.getInitialLocation1().getX());
						this.player1.setY(this.selectedMap.getInitialLocation1().getY());
						this.player1.setY(this.selectedMap.getInitialLocation1().getY());
						this.player1.setDirection(this.selectedMap.getInitialDirection());
						this.player2.setX(this.selectedMap.getInitialLocation2().getX());
						this.player2.setX(this.selectedMap.getInitialLocation2().getX());
						this.player2.setY(this.selectedMap.getInitialLocation2().getY());
						this.player2.setY(this.selectedMap.getInitialLocation2().getY());
						this.player2.setDirection(this.selectedMap.getInitialDirection());
						this.panel1.switchMyCar(false);
						this.panel2.switchMyCar(false);
						this.panel1.switchSpeedBar(false);
						this.panel2.switchSpeedBar(false);
						this.panel1.switchLap(Const.Lap.SELECT_CHARACTER);
						this.panel2.switchLap(Const.Lap.SELECT_CHARACTER);
						this.panel1.switchHelp(true);
						this.panel2.switchHelp(true);
						this.panel1.switchRank(-1);
						this.panel2.switchRank(-1);
						this.panel1.showWin("");
						this.panel2.showWin("");
						this.handler.clearKeys();
						this.game.setStatus(Const.Status.SELECT_CHARACTER);
					} else {
						for (final Player player : new Player[] { this.player1, this.player2 }) {
							movePlayer(player);
							final int time = (int) this.finishTime;
							final int minutes = time / 1000 / 60 % 60;
							final int seconds = (time / 1000) % 60;
							final int millis = (time % 1000) / 10;
							if (player == this.player1) {
								this.panel1.update(minutes, seconds, millis);
							} else {
								this.panel2.update(minutes, seconds, millis);
							}
						}
					}
					break;
				}
				final long loopTime = System.currentTimeMillis() - this.lastTime;
				if (loopTime > 50) {
					this.waitTime--;
				} else if (loopTime < 50) {
					this.waitTime++;
				}
				this.lastTime = System.currentTimeMillis();
				Thread.sleep(this.waitTime);
			} catch (InterruptedException e) {
				e.printStackTrace();
			}
		}
		System.exit(0);
	}

	/**
	 * ステージを切り替えます。
	 * @param index ステージ番号
	 */
	public void switchMap(int index) {
		this.selectedMap = this.maps[index];
		this.checkPoints1.clear();
		this.checkPoints2.clear();
		for (int i = 0; i < this.selectedMap.getLap(); i++) {
			this.checkPoints1.addAll(Arrays.asList(this.selectedMap.getCheckPoints()));
			this.checkPoints2.addAll(Arrays.asList(this.selectedMap.getCheckPoints()));
		}
	}

}
