package car;

import java.awt.Font;
import java.awt.GraphicsConfiguration;
import java.awt.GridLayout;
import java.awt.event.KeyListener;
import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.Formatter;

import javax.media.j3d.Appearance;
import javax.media.j3d.BranchGroup;
import javax.media.j3d.Canvas3D;
import javax.media.j3d.GeometryArray;
import javax.media.j3d.Group;
import javax.media.j3d.QuadArray;
import javax.media.j3d.Shape3D;
import javax.media.j3d.Switch;
import javax.media.j3d.Transform3D;
import javax.media.j3d.TransformGroup;
import javax.media.j3d.TransparencyAttributes;
import javax.media.j3d.TriangleArray;
import javax.swing.JPanel;
import javax.vecmath.Color3f;
import javax.vecmath.Point3d;
import javax.vecmath.Point3f;
import javax.vecmath.TexCoord2f;
import javax.vecmath.Vector3d;

import car.character.Character;
import car.map.RoadMap;

import com.sun.j3d.utils.geometry.Text2D;
import com.sun.j3d.utils.image.TextureLoader;
import com.sun.j3d.utils.universe.PlatformGeometry;
import com.sun.j3d.utils.universe.SimpleUniverse;

/**
 * プレイヤの車を表示するパネルです。
 * @author Kumano Tatsuo
 * Created on 2004/12/11
 */
public class PlayerPanel extends JPanel {
	/**
	 * カメラの座標
	 */
	private Point3d cameraLocation;

	/**
	 * チェックポイントの一覧
	 */
	private Line2D[] checkPoints;

	/**
	 * 次のチェックポイント
	 */
	private int nextCheckPoint;

	/**
	 * 目標値の位置
	 */
	private Point3d lookAtLocation;

	/**
	 * 目標値の速度
	 */
	private Vector3d lookAtSpeed;

	/**
	 * カメラを動かすための変換組です。
	 */
	private final TransformGroup cameraTransformGroup;

	/**
	 * プレイヤ1の車のグラフィックを切り替えるスイッチ
	 */
	private final Switch carCharacterSwitch;

	/**
	 * プレイヤ2の車のグラフィックを切り替えるスイッチ
	 */
	private final Switch carCharacterSwitch2;

	/**
	 * プレイヤ1の車を動かすための変換組です。
	 */
	private final TransformGroup carTransformGroup;

	/**
	 * プレイヤ2の車を動かすための変換組です。
	 */
	private final TransformGroup carTransformGroup2;

	/**
	 * コースマップを切り替えるスイッチ
	 */
	private final Switch courseMapSwitch;

	/**
	 * ゲームの状態
	 */
	final private Game game;

	/**
	 * プレイヤ1を注視するかどうか
	 */
	private final boolean isPlayer1;

	/**
	 * 周回を表示するスイッチ
	 */
	private final Switch lapSwitch;

	/**
	 * 地図
	 */
	final private RoadMap[] maps;

	/**
	 * ミリ秒を表示するスイッチ
	 */
	private final Switch millisSwitch;

	/**
	 * 分を表示するスイッチ
	 */
	private final Switch minutesSwitch;

	/**
	 * 自車のグラフィックを切り替えるスイッチ
	 */
	private final Switch myCarCharacterSwitch;

	/**
	 * 自車を回転させるための変換組
	 */
	private final TransformGroup myCarTransformGroup;

	/**
	 * プレイヤ
	 */
	private final Player player1;

	/**
	 * コースマップにプレイヤ1の位置を表示するための変換組
	 */
	private final TransformGroup[] player1LocationTransformGroups;

	/**
	 * プレイヤ2
	 */
	private final Player player2;

	/**
	 * コースマップにプレイヤ2の位置を表示するための変換組
	 */
	private final TransformGroup[] player2LocationTransformGroups;

	/**
	 * 順位を表示するスイッチ
	 */
	private final Switch rankSwitch;

	/**
	 * 秒を表示するスイッチ
	 */
	private final Switch secondsSwitch;

	/**
	 * 選択されている地図
	 */
	private int selectedMap;

	/**
	 * 速度計の針を動かすための変換組
	 */
	private final TransformGroup speedBarTransformGroup;

	/**
	 * 速度計の針のスイッチ
	 */
	private final Switch speedBarSwitch;

	/**
	 * マップの情景
	 */
	private final BranchGroup[] scenes;

	/**
	 * 現在表示されているマップの情景
	 */
	private BranchGroup nowScene;

	/**
	 * コンストラクタです。
	 * @param player1 プレイヤ1
	 * @param player2 プレイヤ2
	 * @param isPlayer1 プレイヤ1を注視するかどうか
	 * @param listener キーリスナ
	 * @param maps ステージの一覧
	 * @param characters プレイヤのグラフィックの一覧
	 * @param game ゲームの状態
	 */
	public PlayerPanel(final Player player1, final Player player2, final boolean isPlayer1,
			final KeyListener listener, final RoadMap[] maps, final Character[] characters,
			final Game game) {
		this.player1 = player1;
		this.player2 = player2;
		this.isPlayer1 = isPlayer1;
		this.game = game;
		this.maps = maps;
		this.scenes = new BranchGroup[maps.length];
		for (int i = 0; i < maps.length; i++) {
			this.scenes[i] = maps[i].getScene();
		}
		// 3Dシーンを作る。
		final GraphicsConfiguration config = SimpleUniverse.getPreferredConfiguration();
		final Canvas3D canvas = new Canvas3D(config);
		canvas.addKeyListener(listener);
		this.setLayout(new GridLayout());
		this.add(canvas);
		this.scene = new BranchGroup();
		this.scene.setCapability(Group.ALLOW_CHILDREN_EXTEND);
		this.scene.setCapability(Group.ALLOW_CHILDREN_WRITE);
		// 敵車を作る
		final Color3f bodyColor = new Color3f(1, 0.2f, 0.2f);
		final Color3f bodyColor2 = new Color3f(1, 0.8f, 0.2f);
		final Color3f rubberColor = new Color3f(0.2f, 0.2f, 0.2f);
		final Color3f personColor = new Color3f(0.2f, 0.5f, 0.1f);
		final Color3f headColor = new Color3f(0.5f, 0.5f, 0.5f);
		this.carCharacterSwitch = new Switch();
		this.carCharacterSwitch.setCapability(Switch.ALLOW_SWITCH_WRITE);
		this.carCharacterSwitch.setCapability(Switch.ALLOW_SWITCH_READ);
		this.carCharacterSwitch2 = new Switch();
		this.carCharacterSwitch2.setCapability(Switch.ALLOW_SWITCH_WRITE);
		this.carCharacterSwitch2.setCapability(Switch.ALLOW_SWITCH_READ);
		for (final Character character : characters) {
			this.carCharacterSwitch.addChild(character.getCarTransformGroup(this.player1,
					bodyColor, rubberColor, personColor, headColor));
			this.carCharacterSwitch2.addChild(character.getCarTransformGroup(this.player2,
					bodyColor2, rubberColor, personColor, headColor));
		}
		this.carTransformGroup2 = new TransformGroup();
		this.carTransformGroup2.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
		this.carTransformGroup2.addChild(this.carCharacterSwitch2);
		this.scene.addChild(this.carTransformGroup2);
		this.carTransformGroup = new TransformGroup();
		this.carTransformGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
		this.carTransformGroup.addChild(this.carCharacterSwitch);
		this.scene.addChild(this.carTransformGroup);
		// 自車を作る
		final PlatformGeometry platform = new PlatformGeometry();
		{
			this.myCarCharacterSwitch = new Switch();
			this.myCarCharacterSwitch.setCapability(Switch.ALLOW_SWITCH_WRITE);
			this.myCarCharacterSwitch.setCapability(Switch.ALLOW_SWITCH_READ);
			for (final Character character : characters) {
				if (this.isPlayer1) {
					this.myCarCharacterSwitch.addChild(character.getCarTransformGroup(this.player1,
							bodyColor, rubberColor, personColor, headColor));
				} else {
					this.myCarCharacterSwitch.addChild(character.getCarTransformGroup(this.player2,
							bodyColor2, rubberColor, personColor, headColor));
				}
			}
			this.myCarTransformGroup = new TransformGroup();
			this.myCarTransformGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
			this.myCarTransformGroup.addChild(this.myCarCharacterSwitch);
			platform.addChild(this.myCarTransformGroup);
		}
		// タイムを表示してみる。
		{
			final Transform3D minutesTransform = new Transform3D();
			minutesTransform.setTranslation(new Vector3d(0.65, 0.35, -4.0));
			final TransformGroup minutesTransformGroup = new TransformGroup(minutesTransform);
			this.minutesSwitch = new Switch();
			this.minutesSwitch.setCapability(Switch.ALLOW_SWITCH_WRITE);
			final Color3f whiteColor = new Color3f(1, 1, 1);
			for (int i = 0; i < 60; i++) {
				this.minutesSwitch.addChild(new Text2D(new Formatter().format("%02d'", i)
						.toString(), whiteColor, "Arial", 50, Font.BOLD));
			}
			minutesTransformGroup.addChild(this.minutesSwitch);
			platform.addChild(minutesTransformGroup);
			final Transform3D secondsTransform = new Transform3D();
			secondsTransform.setTranslation(new Vector3d(1, 0.35, -4.0));
			final TransformGroup secondsTransformGroup = new TransformGroup(secondsTransform);
			this.secondsSwitch = new Switch();
			this.secondsSwitch.setCapability(Switch.ALLOW_SWITCH_WRITE);
			for (int i = 0; i < 60; i++) {
				this.secondsSwitch.addChild(new Text2D(new Formatter().format("%02d\"", i)
						.toString(), whiteColor, "Arial", 50, Font.BOLD));
			}
			secondsTransformGroup.addChild(this.secondsSwitch);
			platform.addChild(secondsTransformGroup);
			final Transform3D millisTransform = new Transform3D();
			millisTransform.setTranslation(new Vector3d(1.35, 0.35, -4.0));
			final TransformGroup millisTransformGroup = new TransformGroup(millisTransform);
			this.millisSwitch = new Switch();
			this.millisSwitch.setCapability(Switch.ALLOW_SWITCH_WRITE);
			for (int i = 0; i < 100; i++) {
				this.millisSwitch.addChild(new Text2D(new Formatter().format("%02d", i).toString(),
						whiteColor, "Arial", 50, Font.BOLD));
			}
			millisTransformGroup.addChild(this.millisSwitch);
			platform.addChild(millisTransformGroup);
		}
		// 順位を表示してみる。
		{
			final Transform3D rankTransform = new Transform3D();
			rankTransform.setTranslation(new Vector3d(1.05, -0.55, -4));
			final TransformGroup rankTransformGroup = new TransformGroup(rankTransform);
			this.rankSwitch = new Switch();
			this.rankSwitch.setCapability(Switch.ALLOW_SWITCH_WRITE);
			this.rankSwitch
					.addChild(new Text2D("1st", new Color3f(1, 1, 0), "Arial", 80, Font.BOLD));
			this.rankSwitch.addChild(new Text2D("    2nd", new Color3f(0.9f, 0.9f, 0.9f), "Arial",
					40, Font.BOLD));
			rankTransformGroup.addChild(this.rankSwitch);
			platform.addChild(rankTransformGroup);
		}
		// 周回を表示してみる。
		{
			final Transform3D lapTransform = new Transform3D();
			lapTransform.setTranslation(new Vector3d(-1.6, 0.43, -4));
			final TransformGroup lapTransformGroup = new TransformGroup(lapTransform);
			this.lapSwitch = new Switch();
			this.lapSwitch.setCapability(Switch.ALLOW_SWITCH_WRITE);
			for (int i = 1; i <= 4; i++) {
				this.lapSwitch.addChild(new Text2D("lap " + i, new Color3f(1, 1, 1), "Arial", 30,
						Font.BOLD));
			}
			this.lapSwitch.addChild(new Text2D("final lap", new Color3f(1, 1, 1), "Arial", 30,
					Font.BOLD));
			this.lapSwitch.addChild(new Text2D("finished", new Color3f(1, 1, 1), "Arial", 30,
					Font.BOLD));
			this.lapSwitch.addChild(new Text2D("press accel", new Color3f(1, 1, 1), "MS UI Gothic",
					30, Font.BOLD));
			this.lapSwitch.addChild(new Text2D("◀ select a car ▶", new Color3f(1, 1, 1),
					"MS UI Gothic", 30, Font.BOLD));
			this.lapSwitch.addChild(new Text2D("◀ select a course ▶", new Color3f(1, 1, 1),
					"MS UI Gothic", 30, Font.BOLD));
			this.lapSwitch.addChild(new Text2D("ready...", new Color3f(1, 1, 1), "MS UI Gothic",
					30, Font.BOLD));
			this.lapSwitch.addChild(new Text2D("go!", new Color3f(1, 1, 1), "MS UI Gothic", 30,
					Font.BOLD));
			this.lapSwitch.addChild(new Text2D("wrong way", new Color3f(1, 1, 1), "Arial", 30,
					Font.BOLD));
			lapTransformGroup.addChild(this.lapSwitch);
			platform.addChild(lapTransformGroup);
		}
		// コースマップを表示する。
		{
			this.courseMapSwitch = new Switch();
			this.player1LocationTransformGroups = new TransformGroup[maps.length];
			this.player2LocationTransformGroups = new TransformGroup[maps.length];
			for (int i = 0; i < maps.length; i++) {
				this.player1LocationTransformGroups[i] = new TransformGroup();
				this.player1LocationTransformGroups[i]
						.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
				final TriangleArray player1LocationBorderArray = new TriangleArray(3,
						GeometryArray.COORDINATES | GeometryArray.COLOR_3);
				player1LocationBorderArray.setCoordinates(0, new Point3f[] {
						new Point3f(20, -20, 0.1f), new Point3f(0, 20, 0.1f),
						new Point3f(-20, -20, 0.1f) });
				this.player1LocationTransformGroups[i].addChild(new Shape3D(
						player1LocationBorderArray));
				final TriangleArray player1LocationArray = new TriangleArray(3,
						GeometryArray.COORDINATES | GeometryArray.COLOR_3);
				player1LocationArray.setCoordinates(0, new Point3f[] { new Point3f(12, -15, 0.15f),
						new Point3f(0, 11, 0.15f), new Point3f(-12, -15, 0.15f) });
				final Color3f redColor = new Color3f(1, 0, 0);
				player1LocationArray.setColors(0, new Color3f[] { redColor, redColor, redColor });
				this.player1LocationTransformGroups[i].addChild(new Shape3D(player1LocationArray));
				this.player2LocationTransformGroups[i] = new TransformGroup();
				this.player2LocationTransformGroups[i]
						.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
				final TriangleArray player2LocationBorderArray = new TriangleArray(3,
						GeometryArray.COORDINATES | GeometryArray.COLOR_3);
				player2LocationBorderArray.setCoordinates(0, new Point3f[] {
						new Point3f(20, -20, 0), new Point3f(0, 20, 0), new Point3f(-20, -20, 0) });
				this.player2LocationTransformGroups[i].addChild(new Shape3D(
						player2LocationBorderArray));
				final TriangleArray player2LocationArray = new TriangleArray(3,
						GeometryArray.COORDINATES | GeometryArray.COLOR_3);
				player2LocationArray.setCoordinates(0, new Point3f[] { new Point3f(12, -15, 0.05f),
						new Point3f(0, 11, 0.05f), new Point3f(-12, -15, 0.05f) });
				final Color3f yellowColor = new Color3f(1, 1, 0f);
				player2LocationArray.setColors(0, new Color3f[] { yellowColor, yellowColor,
						yellowColor });
				this.player2LocationTransformGroups[i].addChild(new Shape3D(player2LocationArray));
				final TransformGroup roadTransformGroup = maps[i].getCourseMap();
				roadTransformGroup.addChild(this.player1LocationTransformGroups[i]);
				roadTransformGroup.addChild(this.player2LocationTransformGroups[i]);
				this.courseMapSwitch.addChild(roadTransformGroup);
			}
			this.courseMapSwitch.setCapability(Switch.ALLOW_SWITCH_WRITE);
			this.courseMapSwitch.setWhichChild(0);
			platform.addChild(this.courseMapSwitch);
		}
		// 速度計を表示してみる。
		{
			final Color3f redColor = new Color3f(1, 0, 0);
			this.speedBarSwitch = new Switch();
			this.speedBarSwitch.setCapability(Switch.ALLOW_SWITCH_WRITE);
			this.speedBarTransformGroup = new TransformGroup();
			this.speedBarTransformGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
			final QuadArray speedBarArray = new QuadArray(4, GeometryArray.COORDINATES
					| GeometryArray.COLOR_3);
			speedBarArray.setCoordinates(0, new Point3f[] { new Point3f(0.015f, 0, 0),
					new Point3f(0.005f, 0.3f, 0), new Point3f(-0.005f, 0.3f, 0),
					new Point3f(-0.015f, 0, 0) });
			speedBarArray.setColors(0, new Color3f[] { redColor, redColor, redColor, redColor });
			this.speedBarTransformGroup.addChild(new Shape3D(speedBarArray));
			this.speedBarSwitch.addChild(this.speedBarTransformGroup);
			platform.addChild(this.speedBarSwitch);
		}
		// 勝敗表示をしてみる。
		{
			final Transform3D winTransform = new Transform3D();
			winTransform.setTranslation(new Vector3d(-0.85, -0.2, -4.0));
			final TransformGroup winTransformGroup = new TransformGroup(winTransform);
			this.winText = new Text2D("             ", new Color3f(1, 1, 1), "Arial", 100,
					Font.BOLD);
			this.winText.setCapability(Shape3D.ALLOW_APPEARANCE_READ);
			this.winText.getAppearance().setCapability(Appearance.ALLOW_TEXTURE_READ);
			this.winText.getAppearance().setCapability(Appearance.ALLOW_TEXTURE_WRITE);
			winTransformGroup.addChild(this.winText);
			platform.addChild(winTransformGroup);
		}
		// 操作説明表示をしてみる。
		{
			final Color3f whiteColor = new Color3f(1, 1, 0);
			this.helpSwitch = new Switch();
			this.helpSwitch.setCapability(Switch.ALLOW_SWITCH_WRITE);
			final Transform3D helpTransform = new Transform3D();
			helpTransform.setTranslation(new Vector3d(1.3, -0.2, -4));
			final TransformGroup helpTransformGroup = new TransformGroup(helpTransform);
			final Appearance helpAppearance = new Appearance();
			helpAppearance.setTexture(new TextureLoader(PlayerPanel.class
					.getResource(this.isPlayer1 ? "help1.png" : "help2.png"), this).getTexture());
			helpAppearance.setTransparencyAttributes(new TransparencyAttributes(
					TransparencyAttributes.BLENDED, 0));
			final QuadArray helpArray = new QuadArray(4, GeometryArray.COORDINATES
					| GeometryArray.TEXTURE_COORDINATE_2);
			helpArray.setCoordinates(0, new Point3f[] { new Point3f(-0.35f, -0.35f, 0),
					new Point3f(0.35f, -0.35f, 0), new Point3f(0.35f, 0.35f, 0),
					new Point3f(-0.35f, 0.35f, 0) });
			helpArray.setTextureCoordinates(0, 0, new TexCoord2f[] { new TexCoord2f(0, 0),
					new TexCoord2f(1, 0), new TexCoord2f(1, 1), new TexCoord2f(0, 1) });
			helpTransformGroup.addChild(new Shape3D(helpArray, helpAppearance));
			this.helpSwitch.addChild(helpTransformGroup);
			this.helpSwitch.setWhichChild(0);
			platform.addChild(this.helpSwitch);
		}
		final SimpleUniverse universe = new SimpleUniverse(canvas);
		universe.getViewingPlatform().setPlatformGeometry(platform);
		// シーンをコンパイルする。
		this.scene.compile();
		// カメラを設定する。
		final Transform3D cameraTransForm = new Transform3D();
		this.cameraLocation = new Point3d();
		this.lookAtLocation = new Point3d();
		this.lookAtSpeed = new Vector3d();
		this.cameraTransformGroup = universe.getViewingPlatform().getViewPlatformTransform();
		this.cameraTransformGroup.setTransform(cameraTransForm);
		universe.addBranchGraph(this.scene);
		universe.getViewer().getView().setBackClipDistance(100);
	}

	/**
	 * @param lap 周回
	 */
	public void switchLap(int lap) {
		this.lapSwitch.setWhichChild(lap);
	}

	/**
	 * @param rank 順位
	 */
	public void switchRank(int rank) {
		this.rankSwitch.setWhichChild(rank);
	}

	/**
	 * プレイヤ1のグラフィックを切り替えます。
	 * @param index
	 */
	public void switchCharacter(int index) {
		this.carCharacterSwitch.setWhichChild(index);
		if (this.isPlayer1) {
			if (this.myCarCharacterSwitch.getWhichChild() != -1) {
				this.myCarCharacterSwitch.setWhichChild(index);
			}
		}
	}

	/**
	 * プレイヤ2のグラフィックを切り替えます。
	 * @param index
	 */
	public void switchCharacter2(int index) {
		this.carCharacterSwitch2.setWhichChild(index);
		if (!this.isPlayer1) {
			if (this.myCarCharacterSwitch.getWhichChild() != -1) {
				this.myCarCharacterSwitch.setWhichChild(index);
			}
		}
	}

	/**
	 * ステージを切り替えます。
	 * @param index ステージ番号
	 */
	public void switchMap(int index) {
		this.courseMapSwitch.setWhichChild(index);
		this.selectedMap = index;
		this.checkPoints = this.maps[this.selectedMap].getCheckPoints();
		this.nextCheckPoint = 0;
		this.cameraLocation.x = (this.maps[this.selectedMap].getInitialLocation1().getX() + this.maps[this.selectedMap]
				.getInitialLocation2().getX()) / 2;
		this.cameraLocation.y = (this.maps[this.selectedMap].getInitialLocation1().getY() + this.maps[this.selectedMap]
				.getInitialLocation2().getY()) / 2;
		this.lookAtLocation.x = (this.checkPoints[this.nextCheckPoint].getX1() + this.checkPoints[this.nextCheckPoint]
				.getX2()) / 2;
		this.lookAtLocation.y = (this.checkPoints[this.nextCheckPoint].getY1() + this.checkPoints[this.nextCheckPoint]
				.getY2()) / 2;
		if (this.nowScene == null) {
			this.scene.addChild(this.scenes[this.selectedMap]);
			this.nowScene = this.scenes[this.selectedMap];
		} else {
			this.scene.removeChild(this.nowScene);
			this.scene.addChild(this.scenes[this.selectedMap]);
			this.nowScene = this.scenes[this.selectedMap];
		}
	}

	/**
	 * コースマップを非表示にします。
	 */
	public void hideCourseMap() {
		this.courseMapSwitch.setWhichChild(-1);
	}

	/**
	 * 速度計の針を切り替えます。
	 * @param isVisible 表示するかどうか
	 */
	public void switchSpeedBar(final boolean isVisible) {
		this.speedBarSwitch.setWhichChild(isVisible ? 0 : -1);
	}

	/**
	 * @param isShow 自車を表示するかどうか
	 */
	public void switchMyCar(final boolean isShow) {
		if (isShow) {
			this.myCarCharacterSwitch.setWhichChild(this.isPlayer1 ? this.carCharacterSwitch
					.getWhichChild() : this.carCharacterSwitch2.getWhichChild());
			if (this.isPlayer1) {
				this.carCharacterSwitch.setWhichChild(-1);
			} else {
				this.carCharacterSwitch2.setWhichChild(-1);
			}
		} else {
			if (this.isPlayer1) {
				this.carCharacterSwitch.setWhichChild(this.myCarCharacterSwitch.getWhichChild());
			} else {
				this.carCharacterSwitch2.setWhichChild(this.myCarCharacterSwitch.getWhichChild());
			}
			this.myCarCharacterSwitch.setWhichChild(-1);
		}
	}

	/**
	 * 描画を更新します。
	 * @param minutes 分
	 * @param seconds 秒
	 * @param millis ミリ秒
	 */
	public void update(final int minutes, final int seconds, final int millis) {
		// プレイヤ2を動かす
		final Transform3D carRotateTransform2 = new Transform3D();
		carRotateTransform2.rotZ(this.player2.getDirection() + this.player2.getHandle() * 4
				* Math.signum(this.player2.getSpeed()) - Math.PI / 2);
		final Transform3D carMoveTransform2 = new Transform3D();
		carMoveTransform2.set(new Vector3d(this.player2.getX(), this.player2.getY(), 0));
		carMoveTransform2.mul(carRotateTransform2);
		this.carTransformGroup2.setTransform(carMoveTransform2);
		// プレイヤ1を動かす
		final Transform3D carRotateTransform = new Transform3D();
		carRotateTransform.rotZ(this.player1.getDirection() + this.player1.getHandle() * 4
				* Math.signum(this.player1.getSpeed()) - Math.PI / 2);
		final Transform3D carMoveTransform = new Transform3D();
		carMoveTransform.set(new Vector3d(this.player1.getX(), this.player1.getY(), 0));
		carMoveTransform.mul(carRotateTransform);
		this.carTransformGroup.setTransform(carMoveTransform);
		// カメラを動かす
		final Player focalPlayer = this.isPlayer1 ? this.player1 : this.player2;
		final Transform3D cameraTransform = new Transform3D();
		switch (this.game.getStatus()) {
		case SELECT_CHARACTER: {
			final double cameraDistance = -10;
			this.cameraLocation.x = focalPlayer.getX() - cameraDistance
					* Math.cos(this.maps[this.selectedMap].getInitialDirection());
			this.cameraLocation.y = focalPlayer.getY() - cameraDistance
					* Math.sin(this.maps[this.selectedMap].getInitialDirection());
			this.cameraLocation.z = 3.5;
			cameraTransform.lookAt(this.cameraLocation, new Point3d(focalPlayer.getX(), focalPlayer
					.getY(), 0.5), new Vector3d(0, 0, 1));
			break;
		}
		case TITLE:
		case SELECT_COURSE: {
			if (this.isPlayer1) {
				final Rectangle2D bounds = this.maps[this.selectedMap].getRoadShape().getBounds2D();
				final double centerX = bounds.getCenterX();
				final double centerY = bounds.getCenterY();
				final double angle = Math.atan2(this.cameraLocation.y - centerY,
						this.cameraLocation.x - centerX) + 0.01;
				double radius = Point2D.distance(this.cameraLocation.x, this.cameraLocation.y,
						centerX, centerY);
				final double idealRadius = this.game.getStatus() == Const.Status.SELECT_COURSE ? Math
						.min(bounds.getWidth(), bounds.getHeight())
						: Math.min(bounds.getWidth(), bounds.getHeight()) * 2;
				if (radius < idealRadius - 1) {
					radius += 1;
				} else if (radius > idealRadius + 1) {
					radius -= 1;
				}
				this.cameraLocation.x = radius * Math.cos(angle) + centerX;
				this.cameraLocation.y = radius * Math.sin(angle) + centerY;
				final double idealHeight = this.game.getStatus() == Const.Status.SELECT_COURSE ? idealRadius / 3
						: idealRadius / 3;
				if (this.cameraLocation.z < idealHeight - 1) {
					this.cameraLocation.z += 1;
				} else if (this.cameraLocation.z > idealHeight + 1) {
					this.cameraLocation.z -= 1;
				}
				cameraTransform.lookAt(this.cameraLocation, new Point3d(centerX, centerY, 0),
						new Vector3d(0, 0, 1));
			} else {
				// コースを走る。
				final double cameraDistance = 15;
				final double nextX = (this.checkPoints[this.nextCheckPoint].getX1() + this.checkPoints[this.nextCheckPoint]
						.getX2()) / 2;
				final double nextY = (this.checkPoints[this.nextCheckPoint].getY1() + this.checkPoints[this.nextCheckPoint]
						.getY2()) / 2;
				if (Point2D.distance(this.lookAtLocation.x, this.lookAtLocation.y, nextX, nextY) < 10) {
					this.nextCheckPoint = (this.nextCheckPoint + 1) % this.checkPoints.length;
				}
				if (this.lookAtLocation.x < nextX - 1) {
					this.lookAtSpeed.x += 0.1;
				} else if (this.lookAtLocation.x > nextX + 1) {
					this.lookAtSpeed.x -= 0.1;
				}
				if (this.lookAtLocation.y < nextY - 1) {
					this.lookAtSpeed.y += 0.1;
				} else if (this.lookAtLocation.y > nextY + 1) {
					this.lookAtSpeed.y -= 0.1;
				}
				this.lookAtSpeed.x *= 0.9;
				this.lookAtSpeed.y *= 0.9;
				this.lookAtLocation.x += this.lookAtSpeed.x;
				this.lookAtLocation.y += this.lookAtSpeed.y;
				this.lookAtLocation.z = 2;
				final double angle = Math.atan2(this.lookAtLocation.y - this.cameraLocation.y,
						this.lookAtLocation.x - this.cameraLocation.x);
				this.cameraLocation.x = -cameraDistance * Math.cos(angle) + this.lookAtLocation.x;
				this.cameraLocation.y = -cameraDistance * Math.sin(angle) + this.lookAtLocation.y;
				this.cameraLocation.z = 3.5;
				cameraTransform.lookAt(this.cameraLocation, this.lookAtLocation, new Vector3d(0, 0,
						1));
			}
			break;
		}
		case PLAYING:
		case GAME_OVER: {
			final double cameraDistance = 15;
			this.cameraLocation.x = focalPlayer.getX() - cameraDistance
					* Math.cos(focalPlayer.getDirection()) - focalPlayer.getSlideSpeed()
					* Math.cos(focalPlayer.getDirection() + focalPlayer.getHandle() + Math.PI / 2);
			this.cameraLocation.y = focalPlayer.getY() - cameraDistance
					* Math.sin(focalPlayer.getDirection()) - focalPlayer.getSlideSpeed()
					* Math.sin(focalPlayer.getDirection() + focalPlayer.getHandle() + Math.PI / 2);
			this.cameraLocation.z = 3.5;
			cameraTransform.lookAt(this.cameraLocation, new Point3d(focalPlayer.getX()
					- focalPlayer.getSlideSpeed()
					* Math.cos(focalPlayer.getDirection() + focalPlayer.getHandle() + Math.PI / 2),
					focalPlayer.getY()
							- focalPlayer.getSlideSpeed()
							* Math.sin(focalPlayer.getDirection() + focalPlayer.getHandle()
									+ Math.PI / 2), 2), new Vector3d(0, 0, 1));
			break;
		}
		}
		cameraTransform.invert();
		this.cameraTransformGroup.setTransform(cameraTransform);
		// 自車を動かす
		final Transform3D myCarTransform = new Transform3D();
		myCarTransform.setTranslation(new Vector3d(-focalPlayer.getSlideSpeed(), -2, -15.0));
		final Transform3D myCarRotateXTransform = new Transform3D();
		myCarRotateXTransform.rotX(-Math.PI / 2);
		myCarTransform.mul(myCarRotateXTransform);
		final Transform3D myCarRotateZTransform = new Transform3D();
		myCarRotateZTransform.rotZ(focalPlayer.getHandle() * 4);
		myCarTransform.mul(myCarRotateZTransform);
		this.myCarTransformGroup.setTransform(myCarTransform);
		// HUDを動かす
		this.minutesSwitch.setWhichChild(minutes);
		this.secondsSwitch.setWhichChild(seconds);
		this.millisSwitch.setWhichChild(millis);
		final Transform3D player1LocationTransform = new Transform3D();
		player1LocationTransform.setTranslation(new Vector3d(this.player1.getX(), this.player1
				.getY(), 0));
		final Transform3D player1LocationRotateTransform = new Transform3D();
		player1LocationRotateTransform.rotZ(this.player1.getDirection() - Math.PI / 2);
		player1LocationTransform.mul(player1LocationRotateTransform);
		for (final TransformGroup transformGroup : this.player1LocationTransformGroups) {
			transformGroup.setTransform(player1LocationTransform);
		}
		final Transform3D player2LocationTransform = new Transform3D();
		player2LocationTransform.setTranslation(new Vector3d(this.player2.getX(), this.player2
				.getY(), 0));
		final Transform3D player2LocationRotateTransform = new Transform3D();
		player2LocationRotateTransform.rotZ(this.player2.getDirection() - Math.PI / 2);
		player2LocationTransform.mul(player2LocationRotateTransform);
		for (final TransformGroup transformGroup : this.player2LocationTransformGroups) {
			transformGroup.setTransform(player2LocationTransform);
		}
		final Transform3D speedBarTransform = new Transform3D();
		speedBarTransform.setTranslation(new Vector3d(-1.3, -0.45, -4));
		final Transform3D speedBarRotateTransform = new Transform3D();
		speedBarRotateTransform.rotZ(Math.PI / 2 - focalPlayer.getSpeed() * 0.79);
		speedBarTransform.mul(speedBarRotateTransform);
		this.speedBarTransformGroup.setTransform(speedBarTransform);
	}

	/**
	 * 勝敗を表示するテキスト
	 */
	private final Text2D winText;

	/**
	 * 操作説明を表示するスイッチ
	 */
	private Switch helpSwitch;

	/**
	 * 3Dシーン
	 */
	private BranchGroup scene;

	/**
	 * 操作説明を切り替えます。
	 * @param isVisible 表示するかどうか
	 */
	public void switchHelp(final boolean isVisible) {
		this.helpSwitch.setWhichChild(isVisible ? 0 : -1);
	}

	/**
	 * 勝ち負けを表示します。
	 * @param string 文字列
	 */
	public void showWin(final String string) {
		this.winText.setString(string);
	}

}
