/*
 * Copyright (c) 2007, 2018 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0, which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the
 * Eclipse Public License v. 2.0 are satisfied: GNU General Public License,
 * version 2 with the GNU Classpath Exception, which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 */

/*
 * $Id$
 */

package com.sun.ts.tests.jaxws.wsi.j2w.rpc.literal.R2401;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Properties;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.sun.javatest.Status;
import com.sun.ts.tests.jaxws.sharedclients.ClientFactory;
import com.sun.ts.tests.jaxws.sharedclients.SOAPClient;
import com.sun.ts.tests.jaxws.sharedclients.rpclitclient.*;
import com.sun.ts.tests.jaxws.wsi.constants.DescriptionConstants;
import com.sun.ts.tests.jaxws.wsi.constants.SOAPConstants;
import com.sun.ts.tests.jaxws.wsi.utils.DescriptionUtils;

import com.sun.ts.lib.harness.*;

public class Client extends ServiceEETest
    implements DescriptionConstants, SOAPConstants {
  /**
   * The client.
   */
  private SOAPClient client;

  static J2WRLShared service = null;

  /**
   * Test entry point.
   * 
   * @param args
   *          the command-line arguments.
   */
  public static void main(String[] args) {
    Client test = new Client();
    Status status = test.run(args, System.out, System.err);
    status.exit();
  }

  /**
   * @class.testArgs: -ap jaxws-url-props.dat
   * @class.setup_props: webServerHost; webServerPort; platform.mode;
   *
   * @param args
   * @param properties
   *
   * @throws Fault
   */
  public void setup(String[] args, Properties properties) throws Fault {
    client = ClientFactory.getClient(J2WRLSharedClient.class, properties, this,
        service);
    logMsg("setup ok");
  }

  public void cleanup() {
    logMsg("cleanup");
  }

  /**
   * @testName: testBindingConformance
   *
   * @assertion_ids: WSI:SPEC:R2401
   *
   * @test_Strategy: Retrieve the WSDL, generated by the Java-to-WSDL tool, and
   *                 examine the wsdl:binding elements to ensure that they
   *                 conform to the WSDL 1.1 section 3 specifics.
   *
   * @throws Fault
   */
  public void testBindingConformance() throws Fault {
    Document document = client.getDocument();
    Element[] bindings = DescriptionUtils.getBindings(document);
    for (int i = 0; i < bindings.length; i++) {
      verifyBinding(bindings[i]);
    }
  }

  protected void verifyBinding(Element element) throws Fault {
    Element[] children;
    children = DescriptionUtils.getChildElements(element, SOAP_NAMESPACE_URI,
        SOAP_BINDING_LOCAL_NAME);
    if (children.length == 0) {
      throw new Fault("Required soap:binding element not present (BP-R2401)");
    }
    verifySOAPBinding(children[0]);
    children = DescriptionUtils.getChildElements(element, WSDL_NAMESPACE_URI,
        WSDL_OPERATION_LOCAL_NAME);
    for (int i = 0; i < children.length; i++) {
      verifyOperation(children[i]);
    }
  }

  protected void verifySOAPBinding(Element element) throws Fault {
    String transport = element.getAttribute(SOAP_TRANSPORT_ATTR);
    if (transport.length() > 0) {
      verifyURI(transport);
    } else {
      throw new Fault("Required soap:binding attribute not present (BP-R2401)");
    }
    String style = element.getAttribute(SOAP_STYLE_ATTR);
    if (style.length() > 0) {
      verifyStyle(style);
    }
  }

  protected void verifyOperation(Element element) throws Fault {
    Element[] children = DescriptionUtils.getChildElements(element,
        SOAP_NAMESPACE_URI, SOAP_OPERATION_LOCAL_NAME);
    if (children.length == 0) {
      throw new Fault("Required soap:operation element not found (BP-R2401)");
    }
  }

  protected void verifySOAPOperation(Element element) throws Fault {
    String soapAction = element.getAttribute(SOAP_SOAPACTION_ATTR);
    if (soapAction.length() > 0) {
      verifyURI(soapAction);
    }
    String style = element.getAttribute(DescriptionUtils.SOAP_STYLE_ATTR);
    if (style.length() > 0) {
      verifyStyle(style);
    }
    Element[] children = DescriptionUtils.getChildElements(element,
        WSDL_NAMESPACE_URI, null);
    for (int i = 0; i < children.length; i++) {
      String localName = children[i].getLocalName();
      if ((localName.equals(WSDL_INPUT_LOCAL_NAME))
          || (localName.equals(WSDL_OUTPUT_LOCAL_NAME))) {
        verifyInputOutput(children[i]);
      }
    }
  }

  protected void verifyInputOutput(Element element) throws Fault {
    Element[] children = DescriptionUtils.getChildElements(element,
        SOAP_NAMESPACE_URI, SOAP_BODY_LOCAL_NAME);
    if (children.length == 0) {
      throw new Fault("Required soap:body element not found (BP-R2401)");
    }
    verifySOAPBody(children[0]);
  }

  protected void verifySOAPBody(Element element) throws Fault {
    String use = element.getAttribute(SOAP_USE_ATTR);
    if (use.length() > 0) {
      verifyUse(use);
    }
    String encodingStyle = element.getAttribute(SOAP_ENCODINGSTYLE_ATTR);
    if (encodingStyle.length() > 0) {
      verifyURI(encodingStyle);
    }
    String namespace = element.getAttribute(SOAP_NAMESPACE_ATTR);
    if (namespace.length() > 0) {
      verifyURI(namespace);
    }
  }

  protected void verifyStyle(String style) throws Fault {
    if (style.equals(SOAP_DOCUMENT)) {
      return;
    }
    if (style.equals(SOAP_RPC)) {
      return;
    }
    throw new Fault("The SOAP style '" + style + "' is invalid (BP-R2401)");
  }

  protected void verifyUse(String use) throws Fault {
    if (use.equals(SOAP_ENCODED)) {
      return;
    }
    if (use.equals(SOAP_LITERAL)) {
      return;
    }
    throw new Fault("The SOAP use '" + use + "' is invalid (BP-R2401)");
  }

  protected void verifyURI(String uri) throws Fault {
    try {
      new URL(uri);
    } catch (MalformedURLException e) {
      throw new Fault("The URI '" + uri + "' is not valid (BP-R2401)", e);
    }
  }
}
