/* wptSigList.cpp - Listview for key signatures
 *	Copyright (C) 2001-2004 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation; either version 2 
 * of the License, or (at your option) any later version.
 *  
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */

#include <windows.h>
#include <time.h>

#include "wptGPG.h"
#include "wptCommonCtl.h"
#include "wptKeylist.h"
#include "wptNLS.h"
#include "wptUTF8.h"
#include "wptErrors.h"
#include "wptTypes.h"
#include "wptW32API.h"


#define IS_UID_CERT(_class) ((_class) >= 0x10 && (_class) <= 0x13)

static int
siglist_build( listview_ctrl_t * lv, HWND ctrl )
{
    struct listview_ctrl_s *c;
    struct listview_column_s implist[] = {
	{0, 240, (char *)_("User ID")},
	{1,  50, (char *)_("Valid")},
	{2,  40, (char *)_("Class")},
	{3,  68, (char *)_("Creation")},
	{4,  80, (char *)_("Key ID")},
	{5,  68, (char *)_("Expiration")},
	{6,  56, (char *)_("Algorithm")},
	{0,   0, NULL}
    };	
    int j, rc = 0;
    
    rc = listview_new( &c );
    if( rc )
	return rc;
    c->ctrl = ctrl;
    for ( j=0; implist[j].fieldname != NULL; j++ )
	listview_add_column( c, &implist[j] );
    listview_set_ext_style( c );
    *lv = c;
    return 0;
} /* siglist_build */


void 
siglist_delete( listview_ctrl_t lv )
{
    if( lv ) {
	listview_release( lv );	
    }
} /* siglist_delete */


static char *
indent_string( const char * old, int ilvl )
{
    char * p;
    int i;

    p = new char[strlen( old ) + 1 + ilvl];
    if( !p )
	BUG( NULL );
    for( i = 0; i < ilvl; i++ )
	p[i] = ' ';
    strcpy( p+i, old );
    return p;
} /* indent_string */


static int
siglist_add_key( listview_ctrl_t lv, gpgme_key_t key, int idx )
{
    char t[128];
    const char *attr;
    const char *s;
    int sig_class;
    int rc = 0, key_attr, root = 0;
    
    s = gpgme_key_get_string_attr( key, GPGME_ATTR_SIG_KEYID, NULL, idx );
    root = s && strlen( s )? 0 : 1;

    sig_class = gpgme_key_get_ulong_attr( key, GPGME_ATTR_SIG_CLASS, NULL, idx );
    if( !IS_UID_CERT( sig_class ) && !root )
	return 0;

    rc = listview_add_item( lv, " " );
    if( rc )	
	return rc;

    attr = gpgme_key_get_string_attr( key, GPGME_ATTR_SIG_ISSUER, NULL, idx );
    if( attr && strlen( attr ) ) {
	char * uid = utf8_to_wincp (attr, strlen (attr)), * p;
	p = root? m_strdup( uid ) : indent_string( uid, 2 );
	listview_add_sub_item( lv, 0, 0, p );
	free( uid );
	free_if_alloc( p );
    }
    else
	listview_add_sub_item( lv, 0, 0, _("  user ID not found") );

    if( root )
	return 0;

    key_attr = gpgme_key_get_ulong_attr( key, GPGME_ATTR_SIG_STAT, NULL, idx );
    if( key_attr == GPGME_SIG_STAT_NONE )	
	return WPTERR_GENERAL;
    switch (key_attr) {
    case GPGME_SIG_STAT_GOOD: s = "YES";   break;
    case GPGME_SIG_STAT_NOKEY:s = "NOKEY"; break;
    case GPGME_SIG_STAT_BAD:  s = "NO";    break;
    case GPGME_SIG_STAT_NOSIG:s = "NOSIG"; break;
    case GPGME_SIG_STAT_ERROR:
    default:                  s = "ERROR"; break;
    }
    listview_add_sub_item( lv, 0, 1, s );

    switch (sig_class) {
    case 0x10: strcpy (t, " "); break;
    case 0x11: strcpy (t, "1"); break;
    case 0x12: strcpy (t, "2"); break;
    case 0x13: strcpy (t, "3"); break;
    }
    key_attr = gpgme_key_get_ulong_attr (key, GPGME_ATTR_SIG_FLAGS, NULL, idx);
    if (key_attr)
	strcat (t, " ");
    if (key_attr & GPGME_SIGF_LOCAL)
	strcat (t, "L");
    if (key_attr & GPGME_SIGF_NREV)
	strcat (t, "R");
    listview_add_sub_item (lv, 0, 2, t);

    key_attr = gpgme_key_get_ulong_attr( key, GPGME_ATTR_SIG_CREATED, NULL, idx );
    if( key_attr ) {
	s = get_key_created( key_attr );
	listview_add_sub_item( lv, 0, 3, s );
    }
    
    attr = gpgme_key_get_string_attr( key, GPGME_ATTR_SIG_KEYID, NULL, idx );
    if( attr && strlen( attr ) == 16 ) {
	_snprintf( t, sizeof t -1, "0x%s", attr + 8 );
	listview_add_sub_item( lv, 0, 4, t );
    }

    key_attr = gpgme_key_get_ulong_attr (key, GPGME_ATTR_SIG_EXPIRES, NULL, idx);
    if (key_attr)
    {
	s = get_key_created (key_attr);
	listview_add_sub_item (lv, 0, 5, s);
    }

    key_attr = gpgme_key_get_ulong_attr( key, GPGME_ATTR_SIG_ALGO, NULL, idx );
    attr = gpgme_key_expand_attr( GPGME_ATTR_ALGO, key_attr );
    if( attr )
	listview_add_sub_item( lv, 0, 6, (char *)attr );
    
    return 0;
} /* siglist_add_key */


listview_ctrl_t
siglist_load( HWND ctrl, const char *keyid )
{    
    gpgme_key_t key;
    listview_ctrl_t lv;
    int rc = 0, i;

    if( siglist_build( &lv, ctrl ) )
	BUG( NULL );
    rc = get_pubkey( keyid, &key );
    if( rc )
	BUG( NULL );
    for( i = 0; i < gpgme_key_count_items( key, GPGME_ATTR_SIG_KEYID ); i++ ) {
	rc = siglist_add_key( lv, key, i );
	if( rc )
	    break;
    }
    if( rc ) {
	siglist_delete( lv );
	lv = NULL;
    }
    return lv;
} /* siglist_load */
