/* $Id: handle.cc,v 1.2 2003/04/30 18:25:18 murrayc Exp $ */               

/* Copyright 2003 gnome-vfsmm Development Team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <libgnomevfsmm/handle.h>
#include <libgnomevfs/gnome-vfs-ops.h>

namespace Gnome
{

namespace Vfs
{

GnomeVFSHandle* Handle::gobj()
{
  return gobj_;
}

const GnomeVFSHandle* Handle::gobj() const
{
  return gobj_;
}

Handle::Handle()
: gobj_(0)
{
}

Handle::~Handle()
{
  //TODO: close() it if it's open? murrayc.
}

void Handle::open(const Glib::ustring& text_uri, GnomeVFSOpenMode open_mode) throw(exception)
{
  GnomeVFSResult result = gnome_vfs_open(&gobj_, text_uri.c_str(), open_mode);
  handle_result(result);
}

void Handle::open(const Glib::RefPtr<const Uri>& uri, GnomeVFSOpenMode open_mode) throw(exception)
{
  GnomeVFSResult result = gnome_vfs_open_uri(&gobj_, const_cast<GnomeVFSURI*>(uri->gobj()), open_mode);
  handle_result(result);
}

void Handle::create(const Glib::ustring& text_uri, GnomeVFSOpenMode open_mode, bool exclusive, guint perm) throw(exception)
{
  GnomeVFSResult result = gnome_vfs_create(&gobj_, text_uri.c_str(), open_mode, exclusive, perm);
  handle_result(result);
}

void Handle::create(const Glib::RefPtr<const Uri>& uri, GnomeVFSOpenMode open_mode, bool exclusive, guint perm) throw(exception)
{
  GnomeVFSResult result = gnome_vfs_create_uri(&gobj_, const_cast<GnomeVFSURI*>(uri->gobj()), open_mode, exclusive, perm);
  handle_result(result);
}

void Handle::close() throw(exception)
{
  GnomeVFSResult result = gnome_vfs_close(gobj());
  handle_result(result);
}
  
GnomeVFSFileSize Handle::read(gpointer buffer, GnomeVFSFileSize bytes) throw(exception)
{
  GnomeVFSFileSize bytes_read = 0;
  GnomeVFSResult result = gnome_vfs_read(gobj(), buffer, bytes, &bytes_read);
  handle_result(result);
  return bytes_read;
}

GnomeVFSFileSize Handle::write(gconstpointer buffer, GnomeVFSFileSize bytes) throw(exception)
{
  GnomeVFSFileSize bytes_written = 0;
  GnomeVFSResult result = gnome_vfs_write(gobj(), buffer, bytes, &bytes_written);
  handle_result(result);
  return bytes_written;
}

void Handle::seek(GnomeVFSSeekPosition whence, GnomeVFSFileOffset offset) throw(exception)
{
  GnomeVFSResult result = gnome_vfs_seek(gobj(), whence, offset);
  handle_result(result);
}

void Handle::tell(GnomeVFSFileSize& offset_return) throw(exception)
{
  GnomeVFSResult result = gnome_vfs_tell(gobj(), &offset_return);
  handle_result(result);
}

//static:
Glib::RefPtr<FileInfo> Handle::get_file_info(const Glib::ustring& text_uri, FileInfoOptions options) throw(exception)
{
  GnomeVFSFileInfo* file_info = gnome_vfs_file_info_new();
  GnomeVFSResult result = gnome_vfs_get_file_info(text_uri.c_str(), file_info, static_cast<GnomeVFSFileInfoOptions>(options));
  handle_result(result);
  return Glib::wrap(file_info);
}

Glib::RefPtr<FileInfo> Handle::get_file_info(FileInfoOptions options) const throw(exception)
{
  GnomeVFSFileInfo* file_info = gnome_vfs_file_info_new();
  GnomeVFSResult result = gnome_vfs_get_file_info_from_handle(const_cast<GnomeVFSHandle*>(gobj()), file_info, static_cast<GnomeVFSFileInfoOptions>(options));
  handle_result(result);
  return Glib::wrap(file_info);
}

//static:
void Handle::truncate(const Glib::ustring& text_uri, GnomeVFSFileSize length) throw(exception)
{
  GnomeVFSResult result = gnome_vfs_truncate(text_uri.c_str(), length);
  handle_result(result);
}

void Handle::truncate(GnomeVFSFileSize length) throw(exception)
{
  GnomeVFSResult result = gnome_vfs_truncate_handle(gobj(), length);
  handle_result(result);
}

//static:
void Handle::make_directory(const Glib::ustring& text_uri, guint perm) throw(exception)
{
  GnomeVFSResult result = gnome_vfs_make_directory(text_uri.c_str(), perm);
  handle_result(result);
}

//static:
void Handle::remove_directory(const Glib::ustring& text_uri) throw(exception)
{
  GnomeVFSResult result = gnome_vfs_remove_directory(text_uri.c_str());
  handle_result(result);
}

//static:
void Handle::unlink(const Glib::ustring& text_uri) throw(exception)
{
  GnomeVFSResult result = gnome_vfs_unlink(text_uri.c_str());
  handle_result(result);
}

//static:
void Handle::move(const Glib::ustring& old_text_uri, const Glib::ustring& new_text_uri, bool force_replace) throw(exception)
{
  GnomeVFSResult result = gnome_vfs_move(old_text_uri.c_str(), new_text_uri.c_str(), force_replace);
  handle_result(result);
}

//static:
bool Handle::check_same_fs(const Glib::ustring& source, const Glib::ustring& target) throw(exception)
{
  gboolean bResult = FALSE;
  GnomeVFSResult result = gnome_vfs_check_same_fs(source.c_str(), target.c_str(), &bResult);
  return bResult;
}


//static:
void Handle::set_file_info(const Glib::ustring& text_uri, const Glib::RefPtr<const FileInfo>& info, GnomeVFSSetFileInfoMask mask) throw(exception)
{
  GnomeVFSResult result = gnome_vfs_set_file_info(text_uri.c_str(), const_cast<GnomeVFSFileInfo*>(info->gobj()), mask);
  handle_result(result);
}

void Handle::monitor_add(GnomeVFSMonitorHandle **handle, const Glib::ustring& text_uri, GnomeVFSMonitorType monitor_type, GnomeVFSMonitorCallback callback, gpointer user_data) throw(exception)
{
  GnomeVFSResult result = gnome_vfs_monitor_add(handle, text_uri.c_str(), monitor_type, callback, user_data);
  handle_result(result);
}

//static: TODO: Wrap MonitorHandle?
void Handle::monitor_cancel(GnomeVFSMonitorHandle *handle) throw(exception)
{
  GnomeVFSResult result = gnome_vfs_monitor_cancel(handle);
  handle_result(result);
}

void Handle::file_control(const Glib::ustring& operation, gpointer operation_data) throw(exception)
{
  GnomeVFSResult result = gnome_vfs_file_control(gobj(), operation.c_str(), operation_data);
  handle_result(result);
}

void Handle::handle_result(GnomeVFSResult result) throw(exception)
{
  if(result != GNOME_VFS_OK)
    throw exception(result);
}
  
} //namespace Vfs

} //namespace Gnome



