/* -*- mode: C; c-file-style: "gnu" -*- */
/*
 * Copyright (C) 2003 Richard Hult <richard@imendio.com>
 * Copyright (C) 2003 Johan Dahlin <johan@gnome.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <glib.h>
#include "playlist-xml.h"
#include "playlist.h"

enum {
  NO_ERROR,
  PARSE_ERROR,
};

typedef enum {
  STATE_START,
  STATE_PLAYLISTS,
  STATE_PLAYLIST,
  STATE_QUERY,
  STATE_EXPR_BINARY,
  STATE_EXPR_UNARY,
  STATE_VARIABLE,
  STATE_CONSTANT,
  STATE_LIMIT
} State;

typedef struct {
  GMarkupParseContext *context;
  GMarkupParser       *parser;
  Query               *query;
  Playlist            *playlist;
  
  GList               *playlists;
  GList               *expr_stack;
  GList               *state_stack;
} ParserData;

static GQuark error_quark; 


static void start_element_handler (GMarkupParseContext  *context,
				   const char           *node_name,
				   const char          **attribute_names,
				   const char          **attribute_values,
				   gpointer              user_data,
				   GError              **error);
static void end_element_handler   (GMarkupParseContext  *context,
				   const char           *node_name,
				   gpointer              user_data,
				   GError              **error);
static void error_handler         (GMarkupParseContext  *context,
				   GError               *error,
				   gpointer              user_data);


static GList * playlists_from_xml (const char *str);

static void
push_expr (ParserData *data, Expr *expr)
{
  data->expr_stack = g_list_prepend (data->expr_stack, expr);
}      

static Expr *
pop_expr (ParserData *data)
{
  Expr *expr;

  if (!data->expr_stack)
    return NULL;
  
  expr = data->expr_stack->data;
  data->expr_stack = g_list_delete_link (data->expr_stack, data->expr_stack);

  return expr;
}      

static void
push_state (ParserData *data, State state)
{
  data->state_stack = g_list_prepend (data->state_stack, GINT_TO_POINTER (state));
}      

static State
pop_state (ParserData *data)
{
  State state;
  
  state = GPOINTER_TO_INT (data->state_stack->data);
  data->state_stack = g_list_delete_link (data->state_stack, data->state_stack);

  return state;
}      

static State
peek_state (ParserData *data)
{
  State state;
  
  state = GPOINTER_TO_INT (data->state_stack->data);

  return state;
}      

static void
set_error (GError              **err,
           GMarkupParseContext  *context,
           int                   error_code,
           const char           *format,
           ...)
{
  int line, ch;
  va_list args;
  char *str;
  
  g_markup_parse_context_get_position (context, &line, &ch);
  
  va_start (args, format);
  str = g_strdup_vprintf (format, args);
  va_end (args);

  g_set_error (err, error_quark, error_code, "Line %d character %d: %s", line, ch, str);
  
  g_free (str);
}

static const char *
get_attribute_value (const char  *name,
		     const char **names,
		     const char **values)
{
  int i = 0;

  while (names[i])
    {
      if (strcmp (name, names[i]) == 0)
	return values[i];
      i++;
    }

  return NULL;
}

static void
handle_expr (GMarkupParseContext  *context,
	     const char           *element,
	     const char          **attribute_names,
	     const char          **attribute_values,
	     gpointer              user_data,
	     GError              **error)
{
  ParserData *data = user_data;
  const char *tmp;
  ExprOp op;
  Expr *expr;
  
  tmp = get_attribute_value ("type", attribute_names, attribute_values);
  if (!tmp)
    {
      set_error (error, context, PARSE_ERROR,
		 "<expr> tag must have \"type\" attribute");
      return;
    }
  
  op = operator_from_string (tmp);
  if (op == EXPR_OP_NONE)
    {
      set_error (error, context, PARSE_ERROR,
		 "invalid \"type\" attribute \"%s\"",
		 tmp);
      return;
    }

  expr = expr_new (op);
  push_expr (data, expr);
  
  switch (op)
    {
    case EXPR_BINARY_OPS:
      push_state (data, STATE_EXPR_BINARY);
      break;
    case EXPR_UNARY_OPS:
      push_state (data, STATE_EXPR_UNARY);
      break;
    default:
      g_assert_not_reached ();
    }
}

static void
start_element_handler (GMarkupParseContext  *context,
                       const char           *element,
                       const char          **attribute_names,
                       const char          **attribute_values,
                       gpointer              user_data,
                       GError              **error)
{
  ParserData *data = user_data;
  const char *tmp, *tmp2;
  Expr *expr;
  ConstantType constant_type;
  Constant *constant;
  Variable variable;
  State state;

  state = peek_state (data);
  
  switch (state)
    {
    case STATE_START:
      if (strcmp (element, "playlists") != 0)
	{
	  set_error (error, context, PARSE_ERROR,
		     "Outermost element must be a <playlists> not <%s>",
		     element);
	  break;
	}

      push_state (data, STATE_PLAYLISTS);
      break;

    case STATE_PLAYLISTS:
      {
	const char *name;
	int id;
	
	if (strcmp (element, "playlist") != 0)
	  {
	    set_error (error, context, PARSE_ERROR,
		       "<playlist> tag expected, not <%s>",
		       element);
	    break;
	  }
	
	name = get_attribute_value ("name", attribute_names, attribute_values);
	tmp = get_attribute_value ("type", attribute_names, attribute_values);
	
	if (!strcmp (tmp, "regular"))
	  {
	    data->playlist = playlist_new (PLAYLIST_TYPE_REGULAR, name);

	    tmp = get_attribute_value ("id", attribute_names, attribute_values);
	    if (tmp)
	      {
		id = atoi (tmp);
		playlist_set_id (data->playlist, id);
	      }
	  }
	else if (!strcmp (tmp, "smart"))
	  data->playlist = playlist_new (PLAYLIST_TYPE_SMART, name);

	push_state (data, STATE_PLAYLIST);
	
	break;
      }
      
    case STATE_PLAYLIST:
      if (data->playlist->type == PLAYLIST_TYPE_SMART)
	{
	  if (strcmp (element, "query") != 0)
	    {
	      set_error (error, context, PARSE_ERROR,
			 "<query> tag expected, not <%s>",
			 element);
	      break;
	    }

	  data->query = query_new (NULL);
	  
	  push_state (data, STATE_QUERY);
	}
      else if (data->playlist->type == PLAYLIST_TYPE_REGULAR)
	{
	  data->playlists = g_list_append (data->playlists,
					   data->playlist);
	  data->playlist = NULL;

	  push_state (data, STATE_PLAYLISTS);
	}
      
      break;

    case STATE_QUERY:
      if (strcmp (element, "expr") == 0)
	  handle_expr (context,
		       element,
		       attribute_names,
		       attribute_values,
		       user_data,
		       error);
      else if (strcmp (element, "limit") == 0)
	{
	  tmp = get_attribute_value ("type", attribute_names, attribute_values);
	  if (!tmp)
	    {
	      set_error (error, context, PARSE_ERROR,
			 "<limit> tag must have \"type\" attribute");
	      break;
	    }
	  query_set_limit_type (data->query, limit_type_from_string (tmp));
	  
	  tmp = get_attribute_value ("value", attribute_names, attribute_values);
	  if (!tmp)
	    {
	      set_error (error, context, PARSE_ERROR,
			 "<limit> tag must have \"value\" attribute");
	      break;
	    }
	  query_set_limit (data->query, atoi (tmp));

	  tmp = get_attribute_value ("selection", attribute_names, attribute_values);
	  if (!tmp)
	    {
	      set_error (error, context, PARSE_ERROR,
			 "<limit> tag must have \"selection\" attribute");
	      break;
	    }
	  query_set_selection_type (data->query, selection_type_from_string (tmp));
	  
	  push_state (data, STATE_LIMIT);
	}
      else
	{
	  set_error (error, context, PARSE_ERROR,
		     "Element <%s> is not allowed inside a <query> element",
		     element);
	}

      break;
      
    case STATE_EXPR_UNARY:
    case STATE_EXPR_BINARY:
      if (strcmp (element, "constant") == 0)
	{
	  tmp = get_attribute_value ("type", attribute_names, attribute_values);
	  if (!tmp)
	    {
	      set_error (error, context, PARSE_ERROR,
			 "<constant> tag must have \"type\" attribute");
	      break;
	    }
	  
	  constant_type = constant_type_from_string (tmp);
	  if (constant_type == CONSTANT_TYPE_NONE)
	      g_warning ("unknown type");

	  if (constant_type == CONSTANT_TYPE_DATE)
	    {
	      tmp = get_attribute_value ("value", attribute_names, attribute_values);
	      if (!tmp)
		{
		  set_error (error, context, PARSE_ERROR,
			     "<constant> tag of type date must have \"value\" attribute");
		  break;
		}
	      
	      tmp2 = get_attribute_value ("unit", attribute_names, attribute_values);
	      if (!tmp2)
		{
		  set_error (error, context, PARSE_ERROR,
			     "<constant> tag of type date must have \"unit\" attribute");
		  break;
		}
	      
	    }
	  else if (constant_type == CONSTANT_TYPE_RANGE)
	    {
	      tmp = get_attribute_value ("start", attribute_names, attribute_values);
	      if (!tmp)
		{
		  set_error (error, context, PARSE_ERROR,
			     "<constant> tag of type range must have \"start\" attribute");
		  break;
		}
	      
	      tmp2 = get_attribute_value ("end", attribute_names, attribute_values);
	      if (!tmp2)
		{
		  set_error (error, context, PARSE_ERROR,
			     "<constant> tag of type range must have \"end\" attribute");
		  break;
		}
	    }
	  else
	    {
	      tmp = get_attribute_value ("value", attribute_names, attribute_values);
	      if (!tmp)
		{
		  set_error (error, context, PARSE_ERROR,
			     "<constant> tag must have \"value\" attribute");
		  break;
		}
	      tmp2 = NULL;
	    }

	  switch (constant_type)
	    {
	    case CONSTANT_TYPE_STRING:
	      constant = constant_string_new (tmp);
	      break;
	    case CONSTANT_TYPE_INT:
	    case CONSTANT_TYPE_SIZE:
	      constant = constant_int_new (atoi (tmp));
	      break;	      
	    case CONSTANT_TYPE_DATE:
	      constant = constant_date_new (atol (tmp), unit_from_string (tmp2));
	      break;
	    case CONSTANT_TYPE_RANGE:
	      constant = constant_range_new (atoi (tmp), atoi (tmp2));
	      break;
	    default:
	      constant = NULL;
	      g_assert_not_reached ();
	    }
	  
	  expr = expr_constant_new (constant);
	  push_expr (data, expr);
	  push_state (data, STATE_CONSTANT);
	}
      else if (strcmp (element, "variable") == 0)
	{
	  tmp = get_attribute_value ("type", attribute_names, attribute_values);
	  if (!tmp)
	    {
	      set_error (error, context, PARSE_ERROR,
			 "<variable> tag must have \"type\" attribute");
	      break;
	    }
	  
	  variable = variable_from_string (tmp);
	  if (variable == VARIABLE_NONE)
	    g_warning ("unknown type");

	  expr = expr_variable_new (variable);
	  push_expr (data, expr);
	  push_state (data, STATE_VARIABLE);
	}
      else if (strcmp (element, "expr") == 0)
	{
	  handle_expr (context,
		       element,
		       attribute_names,
		       attribute_values,
		       user_data,
		       error);
	}
      else
	{
	  set_error (error, context, PARSE_ERROR,
		     "Element <%s> is not allowed inside a <expr> element",
		     element);
	}
      
      break;
      
    case STATE_CONSTANT:
      break;
       
    case STATE_VARIABLE:
      break;

    case STATE_LIMIT:
      break;
    }
}

static void
end_element_handler (GMarkupParseContext  *context,
                     const char           *element_name,
                     gpointer              user_data,
                     GError              **error)
{
  ParserData *data = user_data;
  Expr *expr, *e1, *e2;
  State state;

  state = pop_state (data);

  switch (state)
    {
    case STATE_START:
      break;

    case STATE_PLAYLISTS:
      break;

    case STATE_PLAYLIST:
      data->playlists = g_list_append (data->playlists, data->playlist);
      break;
      
    case STATE_QUERY:
      if (data->query)
	{
	  expr = pop_expr (data);
	  query_set_expr (data->query, expr);
	}
      playlist_set_query (data->playlist, data->query);

      query_unref (data->query);
      data->query = NULL;
      
      break;

    case STATE_EXPR_BINARY:
      e1 = pop_expr (data);
      e2 = pop_expr (data);
      
      expr = pop_expr (data);
      expr_set_binary_op (expr, expr->op, e1, e2);

      push_expr (data, expr);
      break;

    case STATE_EXPR_UNARY:
      e1 = pop_expr (data);

      expr = pop_expr (data);
      expr_set_unary_op (expr, expr->op, e1);

      push_expr (data, expr);
      break;

    case STATE_CONSTANT:
      break;

    case STATE_VARIABLE:
      break;

    case STATE_LIMIT:
      break;
    }

}

static void
error_handler (GMarkupParseContext *context,
	       GError              *error,
	       gpointer             user_data)
{
  g_warning ("%s", error->message);
}

static GList *
playlists_from_xml (const char *str)
{
  ParserData data;
  static gboolean inited = FALSE;
  
  g_return_val_if_fail (str != NULL, NULL);

  if (!inited)
    {
      error_quark = g_quark_from_static_string ("query-xml-error-quark");
      inited = TRUE;
    }
  
  memset (&data, 0, sizeof (data));
  
  push_state (&data, STATE_START);
  
  data.parser = g_new0 (GMarkupParser, 1);

  data.parser->start_element = start_element_handler;
  data.parser->end_element = end_element_handler;
  data.parser->error = error_handler;

  data.context = g_markup_parse_context_new (data.parser, 0, &data, NULL);
  
  if (!g_markup_parse_context_parse (data.context, str, -1, NULL))
    data.playlists = NULL;
  
  g_markup_parse_context_free (data.context);
  g_free (data.parser);

  return data.playlists;
}

/*
 * Output
 */ 

static void
string_append_padding (GString *str, int depth)
{
  char *tmp;
  int len;

  len = depth * 2;
  
  tmp = g_new (char, len + 1);
  memset (tmp, ' ', len);
  tmp[len] = '\0';

  g_string_append (str, tmp);

  g_free (tmp);
}

static void
output_constant (GString *str, Constant *constant)
{
  g_string_append_printf (str, "<constant type=\"%s\" ", constant_type_to_string (constant->type));

  switch (constant->type)
    {
    case CONSTANT_TYPE_INT:
      g_string_append_printf (str, "value=\"%d\"/>\n", constant_get_int_value (constant));
      break;

    case CONSTANT_TYPE_STRING:
      g_string_append_printf (str, "value=\"%s\"/>\n",  constant_get_string_value (constant));
      break;

    case CONSTANT_TYPE_DATE:
      g_string_append_printf (str, "value=\"%d\" unit=\"%s\"/>\n",
			      constant_get_int_value (constant),
			      unit_to_string (constant_get_unit (constant)));
      break;

    case CONSTANT_TYPE_RANGE:
      {
	int start, end;
	constant_get_range_values (constant, &start, &end);
	
	g_string_append_printf (str, "start=\"%d\" end=\"%d\"/>\n", start, end);
	break;
      }
	
    default:
      g_warning ("Type not implemented.");
      break;
    }
}

static void
output_expr (GString *str, Expr *expr, int depth)
{
  string_append_padding (str, depth);
  
  switch (expr->op)
    {
    case EXPR_UNARY_OPS:
      g_string_append (str, "<unary>");
      output_expr (str, expr->v.unary.expr, depth + 1);
      g_string_append (str, "</unary>");
      break;
      
    case EXPR_BINARY_OPS:
      g_string_append_printf (str, "<expr type=\"%s\">\n",
			      operator_to_string (expr->op));
      
      output_expr (str, expr->v.binary.expr_1, depth + 1);
      output_expr (str, expr->v.binary.expr_2, depth + 1);
      
      string_append_padding (str, depth);
      g_string_append (str, "</expr>\n");
      break;

    case EXPR_OP_CONSTANT:
      output_constant (str, expr->v.constant);
      break;

    case EXPR_OP_VARIABLE:
      g_string_append_printf (str, "<variable type=\"%s\"/>\n",
			      variable_to_string (expr->v.variable));
      break;

    case EXPR_OP_NONE:
      g_string_append (str, "<no-op>");
      break;
      
    default:
      g_assert_not_reached ();
      break;
    }

}

static void
output_limit (GString *str, Query *query, int depth)
{
  string_append_padding (str, depth);
  g_string_append_printf (str, "<limit type=\"%s\" value=\"%d\" selection=\"%s\"/>\n",
			  limit_type_to_string (query_get_limit_type (query)),
			  query_get_limit (query),
			  selection_type_to_string (query_get_selection_type (query)));
}

char *
query_to_xml (Query *query, int *length)
{
  GString *str;

  g_return_val_if_fail (query != NULL, NULL);

  str = g_string_new ("");
  g_string_append_printf (str, "    <query type=\"%s\">\n",
			  match_type_to_string (query_get_match_type (query)));

  if (query_get_expr (query)) 
    output_expr (str, query_get_expr (query), 3);

  if (query_get_limit_type (query) != LIMIT_TYPE_NONE)
    output_limit (str, query, 3);

  g_string_append (str, "    </query>\n");
  
  if (length)
    *length = str->len;
  
  return g_string_free (str, FALSE);
}

void
playlist_xml_save_playlists (GList *playlists)
{
  char *filename;
  char *str;
  int len;
  FILE *file;
  GList *list;
  
  g_return_if_fail (playlists != NULL);

  filename = g_build_filename (g_get_home_dir (),
			       ".gnome2",
			       "jamboree",
			       "playlists.xml",
			       NULL);
  
  file = fopen (filename, "w");

  /* FIXME: Should handle when we don't write it all... */
  
  fprintf (file, "<playlists>\n");
  
  for (list = playlists; list; list = list->next)
    {
      Playlist * playlist = list->data;

      if (playlist->type == PLAYLIST_TYPE_REGULAR)
	{
	  fprintf (file, "  <playlist type=\"regular\" id=\"%d\" name=\"%s\"/>\n",
		   playlist->id, playlist_get_name (playlist));
	}
      else if (playlist->type == PLAYLIST_TYPE_SMART)
	{
	  fprintf (file, "  <playlist type=\"smart\" id=\"%d\" name=\"%s\">\n",
		   playlist->id, playlist_get_name (playlist));
	  str = query_to_xml (playlist_get_query (playlist), &len);
	  fwrite (str, len, 1,  file);
	  g_free (str);
	  fprintf (file, "  </playlist>\n");
	}
      else
	{
	  g_assert_not_reached ();
	  fclose (file);
	  goto fail;
	}
    }
  
  fprintf (file, "</playlists>\n");
  fclose (file);
  
 fail:
  g_free (filename);
}

GList *
playlist_xml_load_playlists (void)
{
  char *filename;
  char *str;
  static GList *playlists = NULL;

  if (playlists)
    return playlists;
  
  filename = g_build_filename (g_get_home_dir (),
			       ".gnome2",
			       "jamboree",
			       "playlists.xml",
			       NULL);
  
  if (g_file_get_contents (filename, &str, NULL, NULL))
    playlists = playlists_from_xml (str);

  g_free (str);
  g_free (filename);
  
  return playlists;
}

