/* -*- mode: C; c-file-style: "gnu" -*- */
/*
 * Copyright (C) 2003-2004 Richard Hult <richard@imendio.com>
 * Copyright (C) 2003      Anders Carlsson <andersca@gnome.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include "song.h"

static GHashTable *string_entries = NULL;

struct _StringEntry {
  int   refcount;
  char *str;
  char *collated;
  char *folded;
};

void
string_entry_init (void)
{
  if (string_entries == NULL)
    string_entries = g_hash_table_new (g_str_hash, g_str_equal);
}

static void
foreach_dump (gpointer key, gpointer data, gpointer user_data)
{
  g_print ("%s\n", (gchar *) key);
}

/* For memleak finding purposes. */
void
string_entry_shutdown (void)
{
  int n_entries;
  
  if (string_entries != NULL)
    {
      n_entries = g_hash_table_size (string_entries);
      if (n_entries != 0)
	{
	  g_warning ("%d string entries left.", n_entries);
	  g_hash_table_foreach (string_entries, foreach_dump, NULL);
	}
	  
      g_hash_table_destroy (string_entries);

      string_entries = NULL;
    }
}

void
song_free (Song *song)
{
  g_free (song->filename);

  string_entry_unref (song->title);
  string_entry_unref (song->album);
  string_entry_unref (song->artist);
  song->title = NULL;
  song->album = NULL;
  song->artist = NULL;
  
  g_list_free (song->playlists);
  song->playlists = NULL;
  
  g_free (song);
}

static char *
create_collate_key (const char *p)
{
  GString *string;
  gunichar c;
  gboolean is_space = FALSE, last_was_space = FALSE;
  char *ret;

  string = g_string_new (NULL);

  if (g_ascii_strncasecmp (p, "the", 3) == 0)
    p += 3;
  
  while (*p)
    {
      c = g_utf8_get_char (p);
      if (g_unichar_isspace (c))
	{
	  if (last_was_space)
	    {
	      p = g_utf8_next_char (p);
	      continue;
	    }
	  
	  is_space = TRUE;
	}
      
      if (c == '(' || c == ')' || c == '"' || c == '\'' || c == '-' ||
	  c == '.' || c == '[' || c == ']' || c == '{' || c == '}')
	{
	  if (!last_was_space)
	    g_string_append_c (string, ' ');

	  last_was_space = TRUE;
	}
      else
	{
	  g_string_append_unichar (string, c);
	  last_was_space = is_space;
	}
      
      p = g_utf8_next_char (p);
    }

  ret = g_utf8_collate_key (string->str, -1);
  g_string_free (string, TRUE);
  
  return ret;
}

static void
string_entry_destroy (StringEntry *entry)
{
  g_assert (entry->refcount == 0);

  g_hash_table_remove (string_entries, entry->str);
  
  g_free (entry->str);
  g_free (entry->folded);
  g_free (entry->collated);
  g_free (entry);
}

StringEntry *
string_entry_ref (StringEntry *entry)
{
  entry->refcount++;

  return entry;
}

void
string_entry_unref (StringEntry *entry)
{
  entry->refcount--;

  if (entry->refcount == 0)
    string_entry_destroy (entry);
}

/* Note: The string entry owns the string after this call. It might be freed, so
 * don't touch it.
 */
StringEntry *
string_entry_add (gchar *str)
{
  StringEntry *entry;
  
  entry = g_hash_table_lookup (string_entries, str ? str : "");
  if (entry)
    {
      g_free (str);
      return string_entry_ref (entry);
    }
  
  entry = g_new0 (StringEntry, 1);
  entry->str = str ? str : g_strdup ("");
  entry->refcount = 1;

  g_hash_table_insert (string_entries, entry->str, entry);

  return entry;
}

StringEntry *
string_entry_add_const (const gchar *str)
{
  return string_entry_add (g_strdup (str));
}

const gchar *
string_entry_get_str (StringEntry *entry)
{
  return entry->str;
}

const gchar *
string_entry_get_collated (StringEntry *entry)
{
  if (G_UNLIKELY (entry->collated == NULL))
    entry->collated = create_collate_key (entry->str);
  
  return entry->collated;
}

const gchar *
string_entry_get_folded (StringEntry *entry)
{
  if (G_UNLIKELY (entry->folded == NULL))
    entry->folded = g_utf8_casefold (entry->str, -1);
  
  return entry->folded;
}
