/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <libgda/gda-config.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <libgnomeui/gnome-druid-page-edge.h>
#include <libgnomeui/gnome-druid-page-standard.h>
#include <libgnomedb/gnome-db-dsn-config-druid.h>
#include <libgnomedb/gnome-db-provider-selector.h>
#include <libgnomedb/gnome-db-util.h>
#include <bonobo/bonobo-i18n.h>

#define PARENT_TYPE GNOME_TYPE_DRUID

struct _GnomeDbDsnConfigDruidPrivate {
	GdaDataSourceInfo *dsn_info;

	GdkPixbuf *logo;

	/* widgets */
	GtkWidget *start_page;
	GtkWidget *general_page;
	GtkWidget *general_name;
	GtkWidget *general_provider;
	GtkWidget *general_description;
	GtkWidget *general_username;
	GtkWidget *provider_page;
	GtkWidget *provider_container;
	GtkWidget *provider_detail;
	GHashTable *provider_params;
	GtkWidget *end_page;
};

static void gnome_db_dsn_config_druid_class_init (GnomeDbDsnConfigDruidClass *klass);
static void gnome_db_dsn_config_druid_init       (GnomeDbDsnConfigDruid *druid,
						  GnomeDbDsnConfigDruidClass *klass);
static void gnome_db_dsn_config_druid_finalize   (GObject *object);

enum {
	FINISHED,
	LAST_SIGNAL
};

static gint config_druid_signals[LAST_SIGNAL] = { 0, };
static GObjectClass *parent_class = NULL;

/*
 * Private functions
 */

static void
add_to_string (gpointer key, gpointer value, gpointer user_data)
{
	const gchar *contents;
	GString **str = (GString **) user_data;


	if (GTK_IS_ENTRY (value)) {
		contents = gtk_entry_get_text (GTK_ENTRY (value));
		if (!contents || !contents[0])
			return;

		if (*str)
			*str = g_string_append (*str, ";");
		else
			*str = g_string_new ("");

		*str = g_string_append (*str, (const gchar *) key);
		*str = g_string_append (*str, "=");
		*str = g_string_append (*str, contents);
	}
}

static gchar *
params_to_string (GnomeDbDsnConfigDruid *druid)
{
	GString *str = NULL;
	gchar *retval;

	g_return_val_if_fail (GNOME_DB_IS_DSN_CONFIG_DRUID (druid), NULL);

	g_hash_table_foreach (druid->priv->provider_params, (GHFunc) add_to_string, &str);
	if (str) {
		retval = str->str;
		g_string_free (str, FALSE);
	}
	else
		retval = g_strdup ("");

	return retval;
}

static gboolean
remove_hash_func (gpointer key, gpointer value, gpointer user_data)
{
	g_free (key);
	return TRUE;
}

/*
 * Callbacks
 */

static void
druid_cancelled_cb (GnomeDruidPage *druid_page, GtkWidget *druid_widget, gpointer user_data)
{
	GnomeDbDsnConfigDruid *druid = (GnomeDbDsnConfigDruid *) user_data;

	g_return_if_fail (GNOME_DB_IS_DSN_CONFIG_DRUID (druid));
	g_signal_emit (G_OBJECT (druid), config_druid_signals[FINISHED], 0, TRUE);
}

static void
druid_finished_cb (GnomeDruidPage *druid_page, GtkWidget *druid_widget, gpointer user_data)
{
	GnomeDbDsnConfigDruid *druid = (GnomeDbDsnConfigDruid *) user_data;

	g_return_if_fail (GNOME_DB_IS_DSN_CONFIG_DRUID (druid));

	/* set the internal dsn_info with the values of the widgets */
	if (druid->priv->dsn_info)
		gda_config_free_data_source_info (druid->priv->dsn_info);

	druid->priv->dsn_info = g_new0 (GdaDataSourceInfo, 1);
	druid->priv->dsn_info->name =
		g_strdup (gtk_entry_get_text (GTK_ENTRY (druid->priv->general_name)));
	druid->priv->dsn_info->provider =
		g_strdup (gnome_db_option_menu_get_selection (
				  GTK_OPTION_MENU (druid->priv->general_provider)));
	druid->priv->dsn_info->cnc_string = params_to_string (druid);
	druid->priv->dsn_info->description =
		g_strdup (gtk_entry_get_text (GTK_ENTRY (druid->priv->general_description)));
	druid->priv->dsn_info->username =
		g_strdup (gtk_entry_get_text (GTK_ENTRY (druid->priv->general_username)));

	/* notify listeners */
	g_signal_emit (G_OBJECT (druid), config_druid_signals[FINISHED], 0, FALSE);
}

static gboolean
general_next_pressed_cb (GnomeDruidPage *druid_page,
			 GtkWidget *druid_widget,
			 gpointer user_data)
{
	GdaProviderInfo *prov_info;
	GList *l;
	const gchar *name;
	const gchar *provider;
	GnomeDbDsnConfigDruid *druid = (GnomeDbDsnConfigDruid *) user_data;

	g_return_val_if_fail (GNOME_DB_IS_DSN_CONFIG_DRUID (druid), TRUE);

	/* check required fields have values */
	name = gtk_entry_get_text (GTK_ENTRY (druid->priv->general_name));
	if (!name || strlen (name) < 1) {
		gnome_db_show_error (_("You must enter a name for the new data source"));
		gtk_widget_grab_focus (druid->priv->general_name);
		return TRUE;
	}

	provider = gnome_db_option_menu_get_selection (GTK_OPTION_MENU (druid->priv->general_provider));
	if (!provider) {
		gnome_db_show_error (_("You must select a provider for the new data source"));
		gtk_widget_grab_focus (druid->priv->general_provider);
		return TRUE;
	}

	/* set up widgets for this specific provider */
	prov_info = gda_config_get_provider_by_name (provider);
	if (!prov_info) {
		gnome_db_show_error (_("Provider not found in your configuration"));
		gtk_widget_grab_focus (druid->priv->general_provider);
		return TRUE;
	}

	g_hash_table_foreach_remove (druid->priv->provider_params,
				     (GHRFunc) remove_hash_func,
				     NULL);
	if (GTK_IS_WIDGET (druid->priv->provider_detail))
		gtk_widget_destroy (druid->priv->provider_detail);

	druid->priv->provider_detail = gnome_db_new_vbox_widget (TRUE, 0);
	for (l = prov_info->gda_params; l != NULL; l = l->next) {
		const gchar *param;
		GtkWidget *box;
		GtkWidget *label;
		GtkWidget *widget;

		param = (const gchar *) l->data;
		if (param) {
			box = gnome_db_new_hbox_widget (TRUE, 0);
			label = gnome_db_new_label_widget (param);
			gtk_box_pack_start (GTK_BOX (box), label, TRUE, TRUE, 0);
			widget = gnome_db_new_entry_widget (0, TRUE);
			gtk_box_pack_start (GTK_BOX (box), widget, TRUE, TRUE, 0);
			gtk_box_pack_start (GTK_BOX (druid->priv->provider_detail),
					    box, TRUE, TRUE, 0);

			g_hash_table_insert (druid->priv->provider_params,
					     g_strdup (param),
					     widget);
		}
	}

	gtk_box_pack_start (GTK_BOX (druid->priv->provider_container),
			    druid->priv->provider_detail, TRUE, TRUE, 0);

	gda_config_free_provider_info (prov_info);

	return FALSE;
}

/*
 * GnomeDbDsnConfigDruid class implementation
 */

static void
gnome_db_dsn_config_druid_class_init (GnomeDbDsnConfigDruidClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	config_druid_signals[FINISHED] =
		g_signal_new ("finished",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GnomeDbDsnConfigDruidClass, finished),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__INT,
			      G_TYPE_NONE, 1, G_TYPE_INT);

	object_class->finalize = gnome_db_dsn_config_druid_finalize;
}

static void
gnome_db_dsn_config_druid_init (GnomeDbDsnConfigDruid *druid,
				GnomeDbDsnConfigDruidClass *klass)
{
	GtkWidget *label;

	g_return_if_fail (GNOME_DB_IS_DSN_CONFIG_DRUID (druid));

	/* create private structure */
	druid->priv = g_new0 (GnomeDbDsnConfigDruidPrivate, 1);
	druid->priv->dsn_info = g_new0 (GdaDataSourceInfo, 1);
	druid->priv->provider_detail = NULL;
	druid->priv->provider_params = g_hash_table_new (g_str_hash, g_str_equal);

	/* load icons */
	druid->priv->logo = gdk_pixbuf_new_from_file (LIBGNOMEDB_ICONSDIR "/gnome-db.png", NULL);

	/* set up widgets */
	gnome_druid_set_show_help (GNOME_DRUID (druid), FALSE);

	/* start page */
	druid->priv->start_page = gnome_druid_page_edge_new_with_vals (
		GNOME_EDGE_START, TRUE, _("Add a new data source..."),
		_("This druid will guide you through the process of\n"
		  "creating a new data source for your GNOME-DB\n"
		  "installation. Just follow the steps!"),
		druid->priv->logo, NULL, NULL);
	gtk_widget_show (druid->priv->start_page);
	gnome_druid_append_page (GNOME_DRUID (druid),
				 GNOME_DRUID_PAGE (druid->priv->start_page));

	g_signal_connect (G_OBJECT (druid->priv->start_page), "cancel", G_CALLBACK (druid_cancelled_cb), druid);

	/* general info page */
	druid->priv->general_page = gnome_druid_page_standard_new_with_vals (
		_("General Information"), druid->priv->logo, NULL);

	label = gnome_db_new_label_widget (NULL);
	gtk_label_set_markup (GTK_LABEL (label),
		_("The following fields represent the basic information\n"
		  "items for your new data source. Some of them (description,\n"
		  "username) are optional, whereas the others are mandatory.\n"
		  "If you don't know which provider to use, just select\n"
		  "the default one"));
	gnome_druid_page_standard_append_item (
		GNOME_DRUID_PAGE_STANDARD (druid->priv->general_page), NULL, label, NULL);

	druid->priv->general_name = gnome_db_new_entry_widget (0, TRUE);
	gnome_druid_page_standard_append_item (
		GNOME_DRUID_PAGE_STANDARD (druid->priv->general_page),
		_("Data source name"), druid->priv->general_name, NULL);

	druid->priv->general_provider = gnome_db_provider_selector_new ();
	gtk_widget_show (druid->priv->general_provider);
	gnome_druid_page_standard_append_item (
		GNOME_DRUID_PAGE_STANDARD (druid->priv->general_page),
		_("Provider"), druid->priv->general_provider, NULL);

	druid->priv->general_description = gnome_db_new_entry_widget (0, TRUE);
	gnome_druid_page_standard_append_item (
		GNOME_DRUID_PAGE_STANDARD (druid->priv->general_page),
		_("Description"), druid->priv->general_description, NULL);

	druid->priv->general_username = gnome_db_new_entry_widget (0, TRUE);
	gnome_druid_page_standard_append_item (
		GNOME_DRUID_PAGE_STANDARD (druid->priv->general_page),
		_("Username"), druid->priv->general_username, NULL);

	gtk_widget_show (druid->priv->general_page);
	gnome_druid_append_page (GNOME_DRUID (druid),
				 GNOME_DRUID_PAGE (druid->priv->general_page));

	g_signal_connect (G_OBJECT (druid->priv->general_page), "cancel",
			  G_CALLBACK (druid_cancelled_cb), druid);
	g_signal_connect (G_OBJECT (druid->priv->general_page), "next",
			  G_CALLBACK (general_next_pressed_cb), druid);

	/* provider parameters page */
	druid->priv->provider_page = gnome_druid_page_standard_new_with_vals (
		_("Provider Parameters"), druid->priv->logo, NULL);

	label = gnome_db_new_label_widget (NULL);
	gtk_label_set_markup (GTK_LABEL (label),
		_("The following fields represent the information needed by the\n"
		  "provider you selected in the previous page to open a given\n"
		  "connection. This information is provider-specific, so check\n"
		  "the manual for more information"));
	gnome_druid_page_standard_append_item (
		GNOME_DRUID_PAGE_STANDARD (druid->priv->provider_page), NULL, label, NULL);

	druid->priv->provider_container = gnome_db_new_vbox_widget (FALSE, 0);
	gnome_druid_page_standard_append_item (
		GNOME_DRUID_PAGE_STANDARD (druid->priv->provider_page), NULL,
		druid->priv->provider_container, NULL);

	gtk_widget_show (druid->priv->provider_page);
	gnome_druid_append_page (GNOME_DRUID (druid),
				 GNOME_DRUID_PAGE (druid->priv->provider_page));

	g_signal_connect (G_OBJECT (druid->priv->provider_page), "cancel",
			  G_CALLBACK (druid_cancelled_cb), druid);

	/* end page */
	druid->priv->end_page = gnome_druid_page_edge_new_with_vals (
		GNOME_EDGE_FINISH, TRUE, _("All information retrieved"),
		_("All information needed to create a new data source\n"
		  "has been retrieved. Now, press 'Apply' to close\n"
		  "this dialog. To open your newly created data source,\n"
		  "just select it when asked for a data source"),
		  druid->priv->logo, NULL, NULL);
	gtk_widget_show (druid->priv->end_page);
	gnome_druid_append_page (GNOME_DRUID (druid),
				 GNOME_DRUID_PAGE (druid->priv->end_page));

	g_signal_connect (G_OBJECT (druid->priv->end_page), "cancel", G_CALLBACK (druid_cancelled_cb), druid);
	g_signal_connect (G_OBJECT (druid->priv->end_page), "finish", G_CALLBACK (druid_finished_cb), druid);
}

static void
gnome_db_dsn_config_druid_finalize (GObject *object)
{
	GnomeDbDsnConfigDruid *druid = (GnomeDbDsnConfigDruid *) object;

	g_return_if_fail (GNOME_DB_IS_DSN_CONFIG_DRUID (druid));

	/* free memory */
	gdk_pixbuf_unref (druid->priv->logo);
	gda_config_free_data_source_info (druid->priv->dsn_info);

	g_hash_table_foreach_remove (druid->priv->provider_params,
				     (GHRFunc) remove_hash_func,
				     NULL);
	g_hash_table_destroy (druid->priv->provider_params);
	druid->priv->provider_params = NULL;

	g_free (druid->priv);
	druid->priv = NULL;

	parent_class->finalize (object);
}

GType
gnome_db_dsn_config_druid_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbDsnConfigDruidClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_dsn_config_druid_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbDsnConfigDruid),
			0,
			(GInstanceInitFunc) gnome_db_dsn_config_druid_init
		};
		type = g_type_register_static (PARENT_TYPE, "GnomeDbDsnConfigDruid",
					       &info, 0);
	}
	return type;
}

/**
 * gnome_db_dsn_config_druid_new
 */
GtkWidget *
gnome_db_dsn_config_druid_new (void)
{
	GnomeDbDsnConfigDruid *druid;

	druid = g_object_new (GNOME_DB_TYPE_DSN_CONFIG_DRUID, NULL);
	return GTK_WIDGET (druid);
}

/**
 * gnome_db_dsn_config_druid_get_dsn
 */
const GdaDataSourceInfo *
gnome_db_dsn_config_druid_get_dsn (GnomeDbDsnConfigDruid *druid)
{
	g_return_val_if_fail (GNOME_DB_IS_DSN_CONFIG_DRUID (druid), NULL);
	return (const GdaDataSourceInfo *) druid->priv->dsn_info;
}
