/* gnome-db-data-handler.c
 *
 * Copyright (C) 2003 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-data-handler.h"
#include "handlers/gnome-db-data-cell-renderer-textual.h"
#include "marshal.h"


/* signals */
enum
{
	DUMMY,
	LAST_SIGNAL
};

static gint gnome_db_data_handler_signals[LAST_SIGNAL] = { 0 };
static void gnome_db_data_handler_iface_init (gpointer g_class);

guint
gnome_db_data_handler_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbDataHandlerIface),
			(GBaseInitFunc) gnome_db_data_handler_iface_init,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) NULL,
			NULL,
			NULL,
			0,
			0,
			(GInstanceInitFunc) NULL
		};
		
		type = g_type_register_static (G_TYPE_INTERFACE, "GnomeDbDataHandler", &info, 0);
	}
	return type;
}


static void
gnome_db_data_handler_iface_init (gpointer g_class)
{
	static gboolean initialized = FALSE;

	if (! initialized) {
		initialized = TRUE;
	}
}

/**
 * gnome_db_data_handler_get_entry_from_value
 * @dh: an object which implements the #GnomeDbDataHandler interface
 * @value: the original value to display or %NULL
 * @type: the requested data type (if @value is not %NULL or of type NULL, then this parameter is ignored)
 *
 * Create a new GnomeDbDataEntry widget to edit the given value. If the value is NULL or of
 * type GDA_VALUE_TYPE_NULL, then the type argument is used and determines the real requested
 * type (it is otherwise ignored).
 *
 * Also, if the value is NULL or of type GDA_VALUE_TYPE_NULL, then the initial edited value in the
 * widget will be the sane initial value provided by the GnomeDbDataHandler object.
 *
 * Returns: the new widget
 */
GnomeDbDataEntry *
gnome_db_data_handler_get_entry_from_value (GnomeDbDataHandler *dh, const GdaValue *value, GdaValueType type)
{
	g_return_val_if_fail (dh && IS_GNOME_DB_DATA_HANDLER (dh), NULL);

	if (!value || gda_value_is_null (value))
		g_return_val_if_fail (gnome_db_data_handler_accepts_gda_type (GNOME_DB_DATA_HANDLER (dh), type), NULL);

	if (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_entry_from_value)
		return (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_entry_from_value) (dh, value, type);
	
	return NULL;
}

/**
 * gnome_db_data_handler_get_cell_renderer
 * @dh: an object which implements the #GnomeDbDataHandler interface
 * @type: the requested data type
 *
 * Creates a new #GtkCellRenderer object to handle @type of data.
 *
 * Returns: the new cell renderer
 */
GtkCellRenderer *
gnome_db_data_handler_get_cell_renderer (GnomeDbDataHandler *dh, GdaValueType type)
{
	g_return_val_if_fail (dh && IS_GNOME_DB_DATA_HANDLER (dh), NULL);

	if (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_cell_renderer)
		return (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_cell_renderer) (dh, type);
	else 
		return gnome_db_data_cell_renderer_textual_new (dh, type);
}

/**
 * gnome_db_data_handler_get_sql_from_value
 * @dh: an object which implements the #GnomeDbDataHandler interface
 * @value: the value to be converted to a string
 *
 * Creates a new string which is an SQL representation of the given value. If the value is NULL or
 * is of type GDA_VALUE_TYPE_NULL, the returned string is NULL.
 *
 * Returns: the new string.
 */
gchar *
gnome_db_data_handler_get_sql_from_value (GnomeDbDataHandler *dh, const GdaValue *value)
{
	g_return_val_if_fail (dh && IS_GNOME_DB_DATA_HANDLER (dh), NULL);
	
	if (! value || gda_value_is_null (value))
		return NULL;

	/* Calling the real function with value != NULL and not of type GDA_VALUE_TYPE_NULL */
	if (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_sql_from_value)
		return (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_sql_from_value) (dh, value);
	
	return NULL;
}

/**
 * gnome_db_data_handler_get_str_from_value
 * @dh: an object which implements the #GnomeDbDataHandler interface
 * @value: the value to be converted to a string
 *
 * Creates a new string which is a "user friendly" representation of the given value (usually
 * it will be in the users's locale, specially for the dates). If the value is 
 * NULL or is of type GDA_VALUE_TYPE_NULL, the returned string is a copy of "" (empty string).
 *
 * Returns: the new string.
 */
gchar *
gnome_db_data_handler_get_str_from_value (GnomeDbDataHandler *dh, const GdaValue *value)
{
	g_return_val_if_fail (dh && IS_GNOME_DB_DATA_HANDLER (dh), NULL);

	if (! value || gda_value_is_null (value))
		return g_strdup ("");

	/* Calling the real function with value != NULL and not of type GDA_VALUE_TYPE_NULL */
	if (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_str_from_value)
		return (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_str_from_value) (dh, value);
	
	return NULL;
}

/**
 * gnome_db_data_handler_get_value_from_sql
 * @dh: an object which implements the #GnomeDbDataHandler interface
 * @sql:
 * @type: 
 *
 * Creates a new GdaValue which represents the SQL value given as argument. This is
 * the opposite of the function gnome_db_data_handler_get_sql_from_value(). The type argument
 * is used to determine the real data type requested for the returned value.
 *
 * If the sql string is NULL, then the returned GdaValue is of type GDA_VALUE_TYPE_NULL;
 * if the sql string does not correspond to a valid SQL string for the requested type, then
 * NULL is returned.
 *
 * Returns: the new GdaValue or NULL on error
 */
GdaValue *
gnome_db_data_handler_get_value_from_sql (GnomeDbDataHandler *dh, const gchar *sql, GdaValueType type)
{
	g_return_val_if_fail (dh && IS_GNOME_DB_DATA_HANDLER (dh), NULL);
	g_return_val_if_fail (gnome_db_data_handler_accepts_gda_type (GNOME_DB_DATA_HANDLER (dh), type), NULL);

	if (!sql)
		return gda_value_new_null ();

	if (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_value_from_sql)
		return (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_value_from_sql) (dh, sql, type);
	
	return NULL;
}


/**
 * gnome_db_data_handler_get_value_from_str
 * @dh: an object which implements the #GnomeDbDataHandler interface
 * @str:
 * @type: 
 *
 * Creates a new GdaValue which represents the STR value given as argument. This is
 * the opposite of the function gnome_db_data_handler_get_str_from_value(). The type argument
 * is used to determine the real data type requested for the returned value.
 *
 * If the str string is NULL, then the returned GdaValue is of type GDA_VALUE_TYPE_NULL;
 * if the str string does not correspond to a valid STR string for the requested type, then
 * NULL is returned.
 *
 * Returns: the new GdaValue or NULL on error
 */
GdaValue *
gnome_db_data_handler_get_value_from_str (GnomeDbDataHandler *dh, const gchar *str, GdaValueType type)
{
	g_return_val_if_fail (dh && IS_GNOME_DB_DATA_HANDLER (dh), NULL);
	g_return_val_if_fail (gnome_db_data_handler_accepts_gda_type (GNOME_DB_DATA_HANDLER (dh), type), NULL);

	if (!str)
		return gda_value_new_null ();

	if (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_value_from_str)
		return (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_value_from_str) (dh, str, type);
	else {
		/* if the get_value_from_str() method is not implemented, then we try the
		   get_value_from_sql() method */
		if (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_value_from_sql)
			return (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_value_from_sql) (dh, str, type);
	}
	
	return NULL;
}


/**
 * gnome_db_data_handler_get_sane_init_value
 * @dh: an object which implements the #GnomeDbDataHandler interface
 * @type: 
 *
 * Creates a new GdaValue which holds a sane initial value to be used if no value is specifically
 * provided. For example for a simple string, this would return gda_value_new_string ("").
 *
 * Returns: the new GdaValue.
 */
GdaValue *
gnome_db_data_handler_get_sane_init_value (GnomeDbDataHandler *dh, GdaValueType type)
{
	g_return_val_if_fail (dh && IS_GNOME_DB_DATA_HANDLER (dh), NULL);
	g_return_val_if_fail (gnome_db_data_handler_accepts_gda_type (GNOME_DB_DATA_HANDLER (dh), type), NULL);

	if (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_sane_init_value)
		return (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_sane_init_value) (dh, type);
	
	return NULL;
}

/**
 * gnome_db_data_handler_get_nb_gda_types
 * @dh: an object which implements the #GnomeDbDataHandler interface
 *
 * Get the number of GdaValueType types the GnomeDbDataHandler can handle correctly
 *
 * Returns: the number.
 */
guint
gnome_db_data_handler_get_nb_gda_types (GnomeDbDataHandler *dh)
{
	g_return_val_if_fail (dh && IS_GNOME_DB_DATA_HANDLER (dh), 0);

	if (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_nb_gda_types)
		return (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_nb_gda_types) (dh);
	
	return 0;
}

/**
 * gnome_db_data_handler_accepts_gda_type
 * @dh: an object which implements the #GnomeDbDataHandler interface
 * @type:
 *
 * Checks wether the GnomeDbDataHandler is able to handle the gda type given as argument.
 *
 * Returns: TRUE if the gda type can be handled
 */
gboolean
gnome_db_data_handler_accepts_gda_type (GnomeDbDataHandler *dh, GdaValueType type)
{
	g_return_val_if_fail (dh && IS_GNOME_DB_DATA_HANDLER (dh), FALSE);

	if (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->accepts_gda_type)
		return (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->accepts_gda_type) (dh, type);
	
	return FALSE;
}

/**
 * gnome_db_data_handler_get_gda_type_index
 * @dh: an object which implements the #GnomeDbDataHandler interface
 * @index: 
 *
 * Get the GdaValueType handled by the GnomeDbDataHandler, at the given position (starting at zero).
 *
 * Returns: the GdaValueType
 */
GdaValueType
gnome_db_data_handler_get_gda_type_index (GnomeDbDataHandler *dh, guint index)
{
	g_return_val_if_fail (dh && IS_GNOME_DB_DATA_HANDLER (dh), GDA_VALUE_TYPE_UNKNOWN);
	g_return_val_if_fail (index < gnome_db_data_handler_get_nb_gda_types (dh),
			      GDA_VALUE_TYPE_UNKNOWN);

	if (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_gda_type_index)
		return (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_gda_type_index) (dh, index);
	
	return GDA_VALUE_TYPE_UNKNOWN;
}

/**
 * gnome_db_data_handler_get_descr
 * @dh: an object which implements the #GnomeDbDataHandler interface
 *
 * Get a short description of the GnomeDbDataHandler
 *
 * Returns: the description
 */
const gchar *
gnome_db_data_handler_get_descr (GnomeDbDataHandler *dh)
{
	g_return_val_if_fail (dh && IS_GNOME_DB_DATA_HANDLER (dh), NULL);

	if (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_descr)
		return (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_descr) (dh);
	
	return NULL;
}

/**
 * gnome_db_data_handler_get_descr_detail
 * @dh: an object which implements the #GnomeDbDataHandler interface
 *
 * Get a detailled description of the GnomeDbDataHandler
 *
 * Returns: the description
 */
const gchar *
gnome_db_data_handler_get_descr_detail (GnomeDbDataHandler *dh)
{
	g_return_val_if_fail (dh && IS_GNOME_DB_DATA_HANDLER (dh), NULL);

	if (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_descr_detail)
		return (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_descr_detail) (dh);
	
	return NULL;
}

/**
 * gnome_db_data_handler_get_version
 * @dh: an object which implements the #GnomeDbDataHandler interface
 *
 * Get the GnomeDbDataHandler's version
 *
 * Returns: the version
 */
const gchar *
gnome_db_data_handler_get_version (GnomeDbDataHandler *dh)
{
	g_return_val_if_fail (dh && IS_GNOME_DB_DATA_HANDLER (dh), NULL);

	if (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_version)
		return (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_version) (dh);
	
	return NULL;
}

/**
 * gnome_db_data_handler_is_plugin
 * @dh: an object which implements the #GnomeDbDataHandler interface
 *
 * Get wether the GnomeDbDataHandler is a plugin or not
 *
 * Returns: TRUE if the GnomeDbDataHandler is a plugin
 */
gboolean
gnome_db_data_handler_is_plugin (GnomeDbDataHandler *dh)
{
	g_return_val_if_fail (dh && IS_GNOME_DB_DATA_HANDLER (dh), FALSE);

	if (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->is_plugin)
		return (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->is_plugin) (dh);
	
	return FALSE;
}

/**
 * gnome_db_data_handler_get_plugin_name
 * @dh: an object which implements the #GnomeDbDataHandler interface
 *
 * Get the name of the GnomeDbDataHandler if it is a plugin
 *
 * Returns: the name of the GnomeDbDataHandler if it is a plugin or NULL
 */
const gchar *gnome_db_data_handler_get_plugin_name (GnomeDbDataHandler *dh)
{
	g_return_val_if_fail (dh && IS_GNOME_DB_DATA_HANDLER (dh), NULL);

	if (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_plugin_name)
		return (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_plugin_name) (dh);
	
	return NULL;
}

/**
 * gnome_db_data_handler_get_plugin_file
 * @dh: an object which implements the #GnomeDbDataHandler interface
 *
 * Get the file name (.so on Unix) of the GnomeDbDataHandler if it is a plugin
 *
 * Returns: the file name of the GnomeDbDataHandler if it is a plugin or NULL
 */
const gchar *
gnome_db_data_handler_get_plugin_file (GnomeDbDataHandler *dh)
{
	g_return_val_if_fail (dh && IS_GNOME_DB_DATA_HANDLER (dh), NULL);

	if (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_plugin_file)
		return (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_plugin_file) (dh);
	
	return NULL;
}

/**
 * gnome_db_data_handler_get_key
 * @dh: an object which implements the #GnomeDbDataHandler interface
 *
 * Get a unique identifier for the GnomeDbDataHandler
 *
 * Returns: the identifier
 */
gchar *
gnome_db_data_handler_get_key (GnomeDbDataHandler *dh)
{
	g_return_val_if_fail (dh && IS_GNOME_DB_DATA_HANDLER (dh), NULL);

	if (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_key)
		return (GNOME_DB_DATA_HANDLER_GET_IFACE (dh)->get_key) (dh);
	
	return NULL;
}
