# (c) Copyright 2022. CodeWeavers, Inc.

import cxobjc
import cxproduct
import cxurlget
import cxutils

import os
import json
import shutil
import tempfile


class IconUtils(cxobjc.Proxy):
    url = 'https://www.codeweavers.com/bin/rest/app/img_icon'

    def __init__(self):
        cxobjc.Proxy.__init__(self)
        self.icon_list = {}
        self.failed = False

    def init(self):
        self = cxobjc.Proxy.nsobject_init(self)
        if self is not None:
            self.__init__()
        return self

    @cxobjc.namedSelector(b'getIconPath:')
    def get_icon_path(self, appid):
        sha1 = self.icon_list.get(appid, {}).get('sha1')
        if not sha1:
            return None

        return os.path.join(cxproduct.get_icons_dir(), sha1)

    @cxobjc.namedSelector(b'fetchIconList')
    def fetch_icon_list(self):
        try:
            content = cxurlget.fetch_content(self.url, timeout=5)
        except Exception: # pylint: disable=W0703
            return False

        data = json.loads(cxutils.string_to_unicode(content))

        for app in data:
            self.icon_list['com.codeweavers.c4.%s' % app.get('app_id', '')] = app

        return True

    @cxobjc.python_method
    def urlgetter_failed(self, _urlgetter, _exception):
        self.failed = True

    @cxobjc.namedSelector(b'fetchIcon:')
    def fetch_icon(self, appid):
        if appid not in self.icon_list:
            return False

        icons_dir = cxproduct.get_icons_dir()
        cxutils.mkdirs(icons_dir)

        name = self.get_icon_path(appid)
        if name and not os.path.exists(name):
            self.failed = False

            tmpfileno, tmppath = tempfile.mkstemp(prefix='local.tmp.', dir=icons_dir)
            # fdopen() acquires the fd as its own
            tmpfile = os.fdopen(tmpfileno, "wb")

            url = self.icon_list[appid].get('img_icon')
            getter = cxurlget.UrlGetter(url, tmpfile, notify_failed=self.urlgetter_failed)
            getter.fetch() # closes tmpfile for us

            if self.failed:
                os.unlink(tmppath)
                return False

            shutil.move(tmppath, name)

        return True
