from datetime import datetime
from typing import Optional, List

from pydantic import BaseModel
from pydantic_partial import PartialModelMixin
from sqlalchemy import Column, Integer, String, func, TIMESTAMP
from sqlalchemy.orm import relationship

from .base import Base
from .hotspot_analysis.hotspot_analysis import HotspotAnalysis, HotspotAnalysisShortInDB
from .threading_analysis.application_stats import ApplicationStats
from .threading_analysis.cpu_info import CpuInfo
from .threading_analysis.health_check import HealthCheck
from .threading_analysis.process_utilization import ProcessUtilisation
from .threading_analysis.thread_concurrency import ThreadConcurrency
from .threading_analysis.thread_time import ThreadTime


class Report(Base):
    __tablename__ = 'report'
    id = Column(Integer, primary_key=True, index=True)
    created_at = Column(TIMESTAMP(timezone=True), server_default=func.now())
    updated_at = Column(TIMESTAMP(timezone=True), onupdate=func.now())
    name = Column(String, default='')

    gtl_dq_job_id = Column(Integer, nullable=True)
    gtl_application_id = Column(Integer, nullable=True)
    gtl_application_name = Column(String, nullable=True)
    fc_job_id = Column(String, nullable=True)
    fc_task_id = Column(String, nullable=True)
    nsys_source_resolved_gtlfs_uuid = Column(String, nullable=True)
    nsys_source_resolved_gtlfs_path = Column(String, nullable=True)
    nsys_source_gtlfs_uuid = Column(String, nullable=True)
    nsys_source_gtlfs_path = Column(String, nullable=True)
    nsys_source_sqlite_gtlfs_uuid = Column(String, nullable=True)
    nsys_source_sqlite_gtlfs_path = Column(String, nullable=True)

    application_stats = relationship(ApplicationStats, uselist=False, back_populates='report')
    cpu_info = relationship(CpuInfo, uselist=False, back_populates='report')
    health_check = relationship(HealthCheck, back_populates='report')
    process_utilisation = relationship(ProcessUtilisation, back_populates='report')
    thread_concurrency = relationship(ThreadConcurrency, back_populates='report')
    thread_time = relationship(ThreadTime, back_populates='report')
    hotspot_analysis = relationship(HotspotAnalysis, back_populates="report")


class ReportInDB(BaseModel):
    id: int
    created_at: datetime
    updated_at: Optional[datetime]
    name: str
    gtl_dq_job_id: Optional[int]
    gtl_application_id: Optional[int]
    gtl_application_name: Optional[str]
    fc_job_id: Optional[str]
    fc_task_id: Optional[str]
    nsys_source_resolved_gtlfs_uuid: Optional[str]
    nsys_source_resolved_gtlfs_path: Optional[str]
    nsys_source_gtlfs_uuid: Optional[str]
    nsys_source_gtlfs_path: Optional[str]
    nsys_source_sqlite_gtlfs_uuid: Optional[str]
    nsys_source_sqlite_gtlfs_path: Optional[str]

    hotspot_analysis: List[HotspotAnalysisShortInDB]

    class Config:
        from_attributes = True


class ReportCreate(PartialModelMixin, BaseModel):
    name: Optional[str]
    gtl_dq_job_id: Optional[int]
    gtl_application_id: Optional[int]
    gtl_application_name: Optional[str]
    fc_job_id: Optional[str]
    fc_task_id: Optional[str]
    nsys_source_resolved_gtlfs_uuid: Optional[str]
    nsys_source_resolved_gtlfs_path: Optional[str]
    nsys_source_gtlfs_uuid: Optional[str]
    nsys_source_gtlfs_path: Optional[str]
    nsys_source_sqlite_gtlfs_uuid: Optional[str]
    nsys_source_sqlite_gtlfs_path: Optional[str]
