/* md.c - message digest wrapper
 *        Copyright (C) 2003, 2004 Timo Schulz
 *
 * This file is part of GPGLIB.
 *
 * GPGLIB is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GPGLIB is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPGLIB; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <windows.h>
#include <stdio.h>
#include <malloc.h>
#include <sys/types.h>

#include "openpgp.h"
#include "md.h"

struct gpgm_md_s {
    void * ctx;
    int algo;
    void (*write)( void *, unsigned char *inbuf, size_t inlen );
};


gpg_md_t
gpg_md_open( int algo )
{
    gpg_md_t md;

    if (algo != MD_SHA1 && algo != MD_MD5 && algo != MD_RMD160 
	&& algo != MD_SHA256 && algo != MD_SHA384 && algo != MD_SHA512)
	return NULL;
    md = calloc (1, sizeof * md);
    if( !md )
	return NULL;
    switch( algo ) {
    case MD_MD5:
	md->ctx = calloc( 1, sizeof (MD5_CONTEXT) );
	if( !md->ctx ) {
	    free( md );
	    return NULL;
	}
	md5_init( (MD5_CONTEXT *)md->ctx );
	md->write = md5_write;
	break;

    case MD_SHA1:
	md->ctx = calloc( 1, sizeof (SHA1_CONTEXT) );
	if( !md->ctx ) {
	    free( md );
	    return NULL;
	}
	sha1_init( (SHA1_CONTEXT *)md->ctx );
	md->write = sha1_write;
	break;

    case MD_RMD160:
	md->ctx = calloc( 1, sizeof (RMD160_CONTEXT) );
	if( !md->ctx ) {
	    free( md );
	    return NULL;
	}
	rmd160_init( (RMD160_CONTEXT *)md->ctx );
	md->write = rmd160_write;
	break;

    case MD_SHA256:
	md->ctx = calloc (1, sizeof (sha256_context));
	if (!md->ctx) {
	    free (md);
	    return NULL;
	}
	sha256_init ((sha256_context *)md->ctx);
	md->write = sha256_write;
	break;

    case MD_SHA384:
    case MD_SHA512:
	md->ctx = calloc (1, sizeof (sha512_context));
	if (!md->ctx) {
	    free (md);
	    return NULL;
	}
	sha512_init ((sha512_context *)md->ctx);
	md->write = sha512_write;
	break;

    default:
	free( md );
	return NULL;
    }
    md->algo = algo;
    return md;
}


void
gpg_md_close( gpg_md_t md )
{
    if( md ) {
	md->algo = 0;
	md->write = NULL;
	free( md->ctx ); md->ctx = NULL;
	free( md );
    }
}


void
gpg_md_final( gpg_md_t md )
{
    switch( md->algo ) {
    case MD_MD5   : md5_final( (MD5_CONTEXT *)md->ctx ); break;
    case MD_SHA1  : sha1_final( (SHA1_CONTEXT *) md->ctx ); break;
    case MD_RMD160: rmd160_final( (RMD160_CONTEXT *) md->ctx ); break;
    case MD_SHA256: sha256_final ((sha256_context *)md->ctx); break;
    case MD_SHA384:
    case MD_SHA512: sha512_final ((sha512_context *)md->ctx); break;
    }
}


const unsigned char *
gpg_md_read (gpg_md_t md)
{
    switch( md->algo ) {
    case MD_MD5   : return md5_read( (MD5_CONTEXT *)md->ctx ); break;
    case MD_SHA1  : return sha1_read( (SHA1_CONTEXT *)md->ctx ); break;
    case MD_RMD160: return rmd160_read( (RMD160_CONTEXT *)md->ctx ); break;
    case MD_SHA256: return sha256_read ((sha256_context *)md->ctx); break;
    case MD_SHA384:
    case MD_SHA512: return sha512_read ((sha512_context *)md->ctx); break;
    }
    return NULL;
}


void
gpg_md_putc( gpg_md_t md, int c )
{
    unsigned char buf[1];
    buf[0] = c & 0xff;
    md->write( md->ctx, buf, 1 );
}


size_t
_md_get_digest_len (gpg_md_t md)
{
    switch (md->algo) {
    case MD_MD5:
	return 16;
    case MD_SHA1:
    case MD_RMD160:
	return 20;
    case MD_SHA256:
	return 32;
    case MD_SHA384:
	return 48;
    case MD_SHA512:
	return 64;
    }
    return 0;
}


void
gpg_md_write( gpg_md_t md, unsigned char *inbuf, size_t len )
{
    if( md )
	md->write( md->ctx, inbuf, len );
}


int
gpg_md_hash_file( int mdalgo, const char *file, unsigned char *digest, size_t *nlen )
{
    gpg_md_t md;
    FILE * fp;
    char buf[8192];
    int n;

    md = gpg_md_open( mdalgo );
    if( !md )
	return -1;
    fp = fopen( file, "rb" );
    if( !fp ) {
	gpg_md_close( md );
	return -2;
    }
    while( !feof( fp ) ) {
	n = fread( buf, 1, sizeof buf-1, fp );
	if( !n )
	    break;
	md->write( md->ctx, buf, n );
    }
    *nlen = _md_get_digest_len( md );
    gpg_md_final( md );
    memcpy( digest, gpg_md_read( md ), *nlen );
    fclose( fp );
    gpg_md_close( md );
    return 0;
}


static int
do_check_md (int algo, char * msg, const unsigned char * digest, size_t nlen)
{
    const unsigned char * tmp;
    int check=0;
    gpg_md_t md;

    md = gpg_md_open (algo);
    if (!md)
	return -1;
    gpg_md_write (md, msg, strlen (msg));
    gpg_md_final (md);
    tmp = gpg_md_read (md);
    check = memcmp (digest, tmp, nlen);
    gpg_md_close (md);

    return check;
}

struct {
    int algo;
    int dlen;
    char * msg;
    const unsigned char * md;
} md_test [] = {
    {MD_RMD160, 20, "abc", "\x8e\xb2\x08\xf7\xe0\x5d\x98\x7a\x9b\x04\x4a\x8e\x98\xc6\xb0\x87\xf1\x5a\x0b\xfc"},
    {0, 0, NULL, NULL}
};


int
gpg_md_selftest (void)
{
    int i, rc=0;

    for (i=0; md_test[i].algo; i++) {
	rc = do_check_md (md_test[i].algo, md_test[i].msg, md_test[i].md,
			  md_test[i].dlen);
	if (rc)
	    return rc;
    }
    return 0;
}