/* w32-clip.c - W32 API clipboard functions
 *	Copyright (C) 2001, 2002, 2003 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>

#include "util.h"
#include "ops.h"
#include "context.h"


static char*
get_w32_clip_text( int *r_size )
{
    int rc, size;
    char *private_clip = NULL;
    char *clip = NULL;
    HANDLE cb;
    
    rc = OpenClipboard( NULL );
    if( !rc )
        return NULL;
    cb = GetClipboardData( CF_TEXT );
    if( !cb )
        goto leave;    
    private_clip = GlobalLock( cb );
    if( !private_clip )
        goto leave;
    size = strlen( private_clip );
    clip = malloc( size + 1 );
    if( !clip ) {
        GlobalUnlock( cb );
        goto leave;
    }
    memcpy( clip, private_clip, size );
    clip[size] = '\0';
    *r_size = size;
    GlobalUnlock( cb );
    
leave:
    CloseClipboard();
    return clip;
} /* get_w32_clip_text */


static void
set_w32_clip_text( const char *data, int size )
{
    int rc;
    HANDLE cb;
    char *private_data;
    
    rc = OpenClipboard( NULL );
    if( !rc )
        return;
    EmptyClipboard();
    cb = GlobalAlloc( GHND, size+1 );
    if( !cb )
        goto leave;
    private_data = GlobalLock( cb );
    if( !private_data )
        goto leave;
    memcpy( private_data, data, size );
    private_data[size] = '\0';
    SetClipboardData( CF_TEXT, cb );
    GlobalUnlock( cb );
    
leave:    
    CloseClipboard( );
    GlobalFree( cb );
} /* set_w32_clip_text */


gpgme_error_t
gpgme_data_new_from_clipboard (gpgme_data_t *r_dh)
{
    gpgme_error_t err = 0;
    gpgme_data_t dh;
    char *clip_text;
    int size;

    if (!r_dh)
	return mk_error (Invalid_Value);
    *r_dh = NULL;
    clip_text = get_w32_clip_text (&size);
    if (!clip_text)
        return mk_error (No_Data);
    err = gpgme_data_new_from_mem (&dh, clip_text, size, 1);
    safe_free (clip_text);
    *r_dh = dh;
    return err;
} /* gpgme_data_new_from_clipboard */


void
gpgme_data_release_and_set_clipboard (gpgme_data_t dh)
{	
    char *clip_text;
    int size = 0;
    
    if (!dh)
	return;
    clip_text = _gpgme_data_get_as_string (dh);
    if (clip_text && (size = strlen (clip_text))) {
        set_w32_clip_text (clip_text, size);
	safe_free (clip_text);
    }
    gpgme_data_release (dh);
} /* gpgme_data_release_and_set_clipboard */


gpgme_error_t
gpgme_data_fix_clipboard( gpgme_data_t dh )
{
    const char *packets[] = {	
	"-----BEGIN PGP MESSAGE-----"	    
	"-----BEGIN PGP PRIVATE KEY-----",
	"-----BEGIN PGP PUBLIC KEY-----",
	"-----BEGIN PGP SIGNED MESSAGE-----"
    };
    const char *headers[] = {
	"Comment"
	"Hash",
	"Version"
    };
    char *p;
    int idx;

    p = _gpgme_data_get_as_string( dh );
    if( p == NULL )
	return mk_error( Invalid_Value );
    for( idx = 0; idx < 4; idx++ ) {
	if( strstr( p, packets[idx] ) )
	    break;
    }
    safe_free( p );

    return 1;
} /* gpgme_data_fix_clipboard */
