/* wptKeyserverDlg.cpp - Keyserver dialog
 *	Copyright (C) 2000-2005 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>
#include <commctrl.h>
#include <malloc.h>

#include "../resource.h"
#include "wptKeyserver.h"
#include "wptErrors.h"
#include "wptTypes.h"
#include "wptCommonCtl.h"
#include "wptNLS.h"
#include "wptW32API.h"
#include "wptVersion.h"
#include "wptGPG.h"
#include "wptKeyManager.h"
#include "wptContext.h" /* for passphrase_s */
#include "wptDlgs.h"


#define MAX_KEYSIZE 70000

char * get_reg_entry_keyserver (const char *);
int set_reg_entry_keyserver (const char *, const char *);


static void
hkp_err_box (HWND dlg, const char * host, u16 port, int rc)
{
    const char *err = kserver_strerror();

    log_box( _("Keyserver"), MB_ERR, "%s:%d: %s", host, port, winpt_strerror (rc));
    if (err)
        msg_box (dlg, err, wsock_strerror (), MB_ERR);
} /* hkp_err_box */


int
hkp_send_key (HWND dlg, const char *kserver, u16 port, const char *pattern)
{
    gpgme_ctx_t c;	
    gpgme_recipients_t rset;
    gpgme_data_t keydata;
    gpgme_error_t ec;
    char *rawkey = NULL, msg[1024];
    int rc;
    
    ec = gpgme_recipients_new( &rset );
    if( ec )
	BUG( NULL );
    gpgme_recipients_add_name( rset, pattern );
    ec = gpgme_new( &c );
    if( ec )
	BUG( NULL );
    gpgme_control( c, GPGME_CTRL_ARMOR, 1 );
    ec = gpgme_data_new( &keydata );
    if( ec )
	BUG( NULL );
    rc = (int) gpgme_op_export( c, rset, keydata );
    if( rc ) {
        msg_box( dlg, gpgme_strerror( (gpgme_error_t)rc ), _("Export"), MB_ERR );
        goto leave;
    }
    rawkey = gpgme_data_release_and_return_string (keydata);
    rc = kserver_sendkey (kserver, port, rawkey, strlen (rawkey));
    if (rc) {
        hkp_err_box (dlg, kserver, port, rc);
        goto leave;
    }
    
    _snprintf (msg, sizeof (msg) -1, _("Key '%s' successfully sent"), pattern);
    status_box (dlg, msg, _("GnuPG status"));
    
leave:
    gpgme_release (c);
    gpgme_recipients_release (rset);
    safe_free (rawkey);
    return rc;
} /* hkp_send_key */


int
hkp_recv_key (HWND dlg, const char *kserver, u16 port, const char *pattern, int proto, int flags)
{
    gpgme_ctx_t ctx;
    gpgme_data_t keydata;
    gpgme_error_t ec;
    int rc, import_res[14];
    char * rawkey = NULL, msg[384];
    
    rawkey = new char[MAX_KEYSIZE];
    if (!rawkey)
	BUG (0);
    memset (rawkey, 0, MAX_KEYSIZE);
    if( proto == KSPROTO_LDAP ) {
        rc = ldap_recvkey( kserver, pattern, rawkey, MAX_KEYSIZE-1 );
        if( rc ) {
	    msg_box( dlg, _("LDAP key import failed.\n"
			    "Please make sure you have an online connection"
			    " and gpgkeys_ldap.exe is installed"),
			    _("Keyserver"), MB_ERR );
	    free_if_alloc( rawkey );
            return rc;
	}
    }
    else if (proto == KSPROTO_FINGER) {
	rc = finger_recvkey (kserver, pattern, rawkey, MAX_KEYSIZE-1);
	if (rc) {
	    log_box (_("Keyserver"), MB_ERR, _("Finger key import failed: %s\n"),
		     winpt_strerror (rc));
	    free_if_alloc( rawkey );
	    return rc;
	}
    }
    else if( ( rc = kserver_recvkey( kserver, port, kserver_check_keyid( pattern ),
				     rawkey, MAX_KEYSIZE-1 ) ) ) {
        hkp_err_box (dlg, kserver, port, rc);
	free_if_alloc (rawkey);
        return rc;
    }
    else {
        if( !strstr( rawkey, "BEGIN PGP PUBLIC KEY BLOCK" ) ) {
            msg_box( dlg, _("This is not a valid OpenPGP key."), _("Keyserver"), MB_ERR );
            goto leave;
        }
        ec = gpgme_new( &ctx );
	if( ec )
	    BUG( NULL );
        gpgme_data_new_from_mem( &keydata, rawkey, strlen( rawkey ), 1 );
        rc = gpgme_op_import( ctx, NULL, keydata );
        if( rc ) {
            msg_box( dlg, gpgme_strerror( (gpgme_error_t)rc ), _("Import"), MB_ERR );
            goto leave;
        }
	gpgme_get_import_status( ctx, import_res, NULL );
    }
    
    /* if we use the refresh mode, a lot of keys will be fetched and thus only
       a summarize at the end is presented and not for each key. */
    if( !(flags & KM_KS_REFRESH) ) {
	if( import_res[GPGME_IMPSTAT_NPKEYS] == 0 ) {    
	    _snprintf( msg, sizeof (msg) - 1, 
		       _("Key '%s' successfully received but nothing was changed."), pattern );
	    status_box( dlg, msg, _("GnuPG Status") );
	    rc = WPTERR_GENERAL;
	    goto leave;
	}
	_snprintf( msg, sizeof (msg) - 1, _("Key '%s' sucessfully received and imported."), pattern );
	status_box( dlg, msg, _("GnuPG Status") );
    }
    
leave:
    free_if_alloc (rawkey);
    gpgme_release (ctx);
    gpgme_data_release (keydata);

    return rc;
} /* hkp_recv_key */


#define my_iskeychar(a) ( ( (a) >='0' && (a) <= '9' ) || ( (a) >= 'A' && (a) <= 'F' ) )

static int
check_pattern( const char *pattern )
{
    int rc = 1;
    
    /* Whitespace are not allowed! */
    if( strchr( pattern, ' ') ) {
        rc = WPTERR_GENERAL;
        goto leave;
    }
    
    if( (( strstr( pattern, "0x" ) ) && ( strlen( pattern ) == 10 ) )
         || (strstr(pattern, "0x")) && ( strlen( pattern )  == 18 ) ) {
        rc = 0;
        goto leave;
    }
    
    if( (( my_iskeychar( pattern[0] )) && ( strlen( pattern )  == 8 ) )
         || (my_iskeychar(pattern[0])) && ( strlen( pattern ) == 16) ) {
        rc = 0;
        goto leave;
    }
    
    if( ( strchr( pattern, '@' ) ) && ( strlen( pattern ) >= 3 ) ) {
        rc = 0;
        goto leave;
    }
    
leave:
    return rc;
} /* check_pattern */


static void
set_proxy (HWND dlg)
{
    char proxy[384];
    int port = 0;
    
    strcpy (proxy, "HTTP proxy: ");
    if (kserver_get_proxy (&port)) {
        char t[256];
        const char *http = kserver_get_proxy (&port);
        _snprintf (t, sizeof (t) - 1, "\"%s:%d\"", http, port);
        strcat (proxy, t);
    }
    else
        strcat( proxy, "none" );
    SetDlgItemText( dlg, IDC_KEYSERVER_PROXY, proxy );
} /* set_proxy */


static int inline
kserver_get_pos (listview_ctrl_t lv)
{
    if (listview_count_items (lv, 0) == 1)
	return 0;
    return listview_get_curr_pos (lv);
}


static u16 inline
kserver_get_port (listview_ctrl_t lv)
{
    char buf[16];

    listview_get_item_text (lv, kserver_get_pos (lv), 3, buf, 15);
    return (u16)strtoul (buf, NULL, 10);
}

static void
load_default_ks (listview_ctrl_t lv)
{
    char * p, buf[192];
    int i;

    p = get_reg_entry_keyserver ("Default");
    if (!p)
	return;
    for( i = 0; i < listview_count_items( lv, 0 ); i++ ) {
	listview_get_item_text( lv, i, 0, buf, sizeof (buf)-1 );
	if( !strncmp( p, buf, strlen( p ) ) ) {
	    listview_add_sub_item( lv, i, 2, "x" );
	    break;
	}
    }
    free_if_alloc (p);
} /* load_default_ks */


static int
save_default_ks (listview_ctrl_t lv)
{
    char buf[192], port[32];
    int idx, i;   

    idx = listview_get_curr_pos( lv );
    if( idx == -1 ) {
	msg_box( NULL, _("Please select one of the servers."), _("Keyserver"), MB_ERR );
	return -1;
    }
    listview_get_item_text( lv, idx, 1, buf, sizeof (buf)-1 );
    if( *buf != 'H' ) {
	msg_box( NULL, _("Only HTTP keyserver can be used."), _("Keyserver"), MB_ERR );
	return -1;
    }
    for (i = 0; i < listview_count_items( lv, 0 ); i++)
	listview_add_sub_item (lv, i, 2, "");
    listview_add_sub_item (lv, idx, 2, "x");
    listview_get_item_text (lv, idx, 0, buf, sizeof (buf)-1);
    set_reg_entry_keyserver ("Default", buf);
    i = kserver_get_port (lv);
    sprintf (port, "%d", i);
    set_reg_entry_keyserver ("Default_Port", port);
    keyserver_set_default (buf, (u16)i);
    return 0;
} /* save_default_ks */


int
keyserver_list_build (listview_ctrl_t *r_lv, HWND hwnd)
{
    struct listview_column_s keyserver[] = {
        {0, 160, (char *)_("DNS Name")},        
        {1,  52, (char *)_("Protocol")},
	{2,  46, (char *)_("Default")},
	{3,  46, (char *)_("Port")},
        {0,   0, NULL}
    };
    listview_ctrl_t lv;
    char buf[32];
    int j;

    listview_new (&lv);    
    lv->ctrl = hwnd;       
    for (j=0; keyserver[j].fieldname; j++)
        listview_add_column (lv, &keyserver[j]);        
    for (j = 0; j<MAX_KEYSERVERS; j++) {
	if (!server[j].used)
	    continue;
	listview_add_item (lv, " ");
	listview_add_sub_item (lv, 0, 0, server[j].name);
	switch (server[j].proto) {
	case KSPROTO_HTTP:
	    listview_add_sub_item( lv, 0, 1, "H" ); break;
	case KSPROTO_LDAP:
	    listview_add_sub_item( lv, 0, 1, "L" ); break;
	case KSPROTO_FINGER:
	    listview_add_sub_item( lv, 0, 1, "F" ); break;
	}
	sprintf (buf, "%d", server[j].port);
	listview_add_sub_item (lv, 0, 3, buf);
    }
    load_default_ks (lv);
    if (listview_count_items (lv, 0) == 0) {
	listview_add_item (lv, "");
	listview_add_sub_item (lv, 0, 0, DEF_HKP_KEYSERVER);
	listview_add_sub_item (lv, 0, 1, "H");
    }
    listview_set_ext_style (lv);
    *r_lv = lv;
    return 0;
}


BOOL CALLBACK
keyserver_dlg_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    static listview_ctrl_t lv = NULL;
    static int lv_idx = 0;
    int rc, proto_nr = 0;
    char kserver[128], pattern[128];
    char proto[16];    
    keyserver_ctx ksc;
    
    switch ( msg ) {
    case WM_INITDIALOG:
#ifndef LANG_DE
        SetWindowText( dlg, _("Keyserver Access") );
        SetDlgItemText( dlg, IDC_KEYSERVER_SEND, 
                        _("Send key (default is receiving)") );
        SetDlgItemText( dlg, IDC_KEYSERVER_INFO,
                        _("Please enter the key ID or email address that belongs to the key"));                        
        SetDlgItemText( dlg, IDC_KEYSERVER_INDEX, _("&Search") );
#endif
        set_proxy (dlg);
        keyserver_list_build (&lv, GetDlgItem (dlg, IDC_KEYSERVER_LIST));
	center_window (dlg);
        SetForegroundWindow (dlg);
        return TRUE;
        
    case WM_NOTIFY:
        NMHDR *notify;
        notify = (NMHDR *)lparam;
        if( notify && notify->code == NM_CLICK
            && notify->idFrom == IDC_KEYSERVER_LIST )
            lv_idx = listview_get_curr_pos( lv );
        return TRUE;
        
    case WM_DESTROY:
        if( lv ) {
            listview_release( lv );
            lv = NULL;
        }
        lv_idx = 0;
        return FALSE;
        
    case WM_SYSCOMMAND:
        if( LOWORD( wparam ) == SC_CLOSE )
            EndDialog( dlg, TRUE );
        return FALSE;
        
    case WM_COMMAND:
        switch( LOWORD( wparam ) ) {
        case IDC_KEYSERVER_PROXSETT:
            dialog_box_param( glob_hinst, (LPCTSTR)IDD_WINPT_KEYSERVER_PROXY, glob_hwnd,
                              keyserver_proxy_dlg_proc, NULL, 
                              _("Proxy Settings"), IDS_WINPT_KEYSERVER_PROXY );
            set_proxy( dlg );
            return TRUE;
            
        case IDC_KEYSERVER_INDEX:
            if (!lv_idx) {
                lv_idx = kserver_get_pos (lv);
                if (lv_idx == -1) {
                    msg_box (dlg, _("Please select one of the keyservers."), _("Keyserver"), MB_INFO);
                    return FALSE;
                }
            }
            listview_get_item_text (lv, lv_idx, 1, proto, sizeof (proto)-1);
            if (*proto == 'L') {
                msg_box( dlg, _("This is not implemented yet!"), _("Keyserver"), MB_ERR );
                return FALSE;
            }
            listview_get_item_text (lv, lv_idx, 0, kserver, sizeof (kserver)-1);
            if (!GetDlgItemText (dlg, IDC_KEYSERVER_SEARCH, pattern, sizeof (pattern)-1)) {
                msg_box (dlg, _("Please enter the search pattern."), _("Keyserver"), MB_INFO);
                return FALSE;
            }
            ksc.name = kserver;
            ksc.pattern = pattern;
	    ksc.port = kserver_get_port (lv);
            DialogBoxParam( glob_hinst, (LPCSTR)IDD_WINPT_HKPSEARCH, dlg,
                            hkpsearch_dlg_proc, (LPARAM) &ksc );
            return TRUE;
            
        case IDC_KEYSERVER_RECV:
            memset (&kserver, 0, sizeof (kserver));
            if (!lv_idx) {
                lv_idx = kserver_get_pos (lv);
                if (lv_idx == -1) {
                    msg_box( dlg, _("Please select one of the keyservers."), _("Keyserver"), MB_INFO );
                    return FALSE;
                }
            }
            listview_get_item_text( lv, lv_idx, 1, proto, sizeof (proto)-1 );
	    proto_nr = KSPROTO_HTTP;
            if( *proto == 'L' )
		proto_nr = KSPROTO_LDAP;
	    else if( *proto == 'F' )
		proto_nr = KSPROTO_FINGER;
            listview_get_item_text( lv, lv_idx, 0, kserver, sizeof (kserver)-1 );
            if( !GetDlgItemText( dlg, IDC_KEYSERVER_SEARCH, pattern, sizeof (pattern)-1 ) ) {
                msg_box( dlg, _("Please enter the search pattern."), _("Keyserver"), MB_INFO );
                return FALSE;
            }
            if( proto_nr == KSPROTO_LDAP && strchr( pattern, '@' ) ) {
                msg_box( dlg, _("Only keyids are allowed."), _("LDAP Keyserver"), MB_INFO );
                return FALSE;
            }
	    else if( proto_nr == KSPROTO_FINGER && 
		     ( strchr( pattern, '@' ) || strchr( pattern, ' ' ) ) ) {
		msg_box( dlg, _("Only enter the name of the user."), _("FINGER Keyserver"), MB_INFO );
		return FALSE;
	    }
            else if( check_pattern( pattern ) ) {
                msg_box( dlg, _("Only email addresses or keyids are allowed."), _("Keyserver"), MB_INFO );
                return FALSE;
            }
	    rc = hkp_recv_key (dlg, kserver, kserver_get_port (lv), pattern, proto_nr, 0);
	    if (!rc)
		keycache_set_reload (1);
            return TRUE;

	case IDC_KEYSERVER_DEFAULT:
	    save_default_ks( lv );
	    break;
            
        case IDCANCEL:
            EndDialog( dlg, FALSE );
            return FALSE;
        }
        break;
    }
    
    return FALSE;
} /* keyserver_dlg_proc */
