/* wptVerifyList.cpp - Listview for verifying signatures
 *	Copyright (C) 2001, 2002, 2003 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation; either version 2 
 * of the License, or (at your option) any later version.
 *  
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */
/* x-todo-status: OK */

#include <windows.h>
#include <time.h>

#include "wptTypes.h"
#include "wptGPG.h"
#include "wptCommonCtl.h"
#include "wptKeylist.h"
#include "wptNLS.h"
#include "wptErrors.h"
#include "wptUTF8.h"
#include "wptW32API.h"


static char *
extract_filename( const char * path )
{
    char * fname, * p;
    int n, len = 0;

    p = strrchr( path, '\\' );
    if( !p )
	return m_strdup( path );
    n = p - path;
    len = strlen( path ) - n;
    fname = new char[len+1];
    if( !fname )
	BUG( NULL );    
    memcpy( fname, path+n+1, len );
    fname[len] = '\0';
    return fname;
} /* extract_filename */


const char *
strtimestamp( long timestamp )
{
    static char timebuf[64] = {0};
    struct tm * warp;
    
    warp = localtime( &timestamp );
    _snprintf( timebuf, sizeof timebuf - 1, "%04d-%02d-%02d %02d:%02d:%02d",
               warp->tm_year+1900, warp->tm_mon+1, warp->tm_mday,	
               warp->tm_hour, warp->tm_min, warp->tm_sec );
    return timebuf;
} /* strtimestamp */


int
verlist_build( listview_ctrl_t * lv, HWND ctrl, int fm_mode )
{
    int j, rc = 0;
    struct listview_ctrl_s * c;
    struct listview_column_s fm_verlist[] = {	
	{0, 100, (char *)_("Name") },
	{1, 120, (char *)_("Status") },
	{2, 115, (char *)_("Signed") },
	{3,  58, (char *)_("Trust") },		
	{4, 160, (char *)_("User ID") },
	{5, 0, NULL }
	
    };
    struct listview_column_s verlist[] = {	
	{0, 140, (char *)_("Status") },
	{1, 120, (char *)_("Signed") },
	{2,  58, (char *)_("Trust") },
	{3,  76, (char *)_("Key ID" )},
	{4, 160, (char *)_("User ID") },	
	{5, 0, NULL }	
    };
    
    rc = listview_new( &c );
    if( rc )
	return rc;
    
    c->ctrl = ctrl;
    if( fm_mode ) {
	for( j = 0; fm_verlist[j].fieldname; j++ )	
	    listview_add_column( c, &fm_verlist[j] );
    }
    else {
	for( j = 0; verlist[j].fieldname; j++ )
	    listview_add_column( c, &verlist[ j ] );	
    }
    listview_set_ext_style( c );
    *lv = c;
    return 0;
} /* verlist_build */


void
verlist_delete( listview_ctrl_t lv )
{
    if( lv ) {
	listview_release( lv );	
    }
} /* verlist_delete */


int
verlist_add_sig( listview_ctrl_t lv, gpgme_sig_t sig )
{
    gpgme_key_t key;
    const char * attr;
    u32 key_attr, t;
    char keyid[32+1];
    char * uid = NULL;
    
    if( listview_add_item( lv, " " ) )
	return WPTERR_GENERAL;
    
    if( !gpgme_sig_get_ulong_attr( sig, 0, GPGME_ATTR_TYPE ) )
	return WPTERR_GENERAL; /* No key was stored */
    
    t = gpgme_sig_get_ulong_attr( sig, 0, GPGME_ATTR_VALIDITY );
    attr = gpg_sigstat[t % SIGSTAT_MASK];
    if( attr )
	listview_add_sub_item( lv, 0, 0, (char *)attr );
    
    t = gpgme_sig_get_ulong_attr( sig, 0, GPGME_ATTR_CREATED );
    attr = strtimestamp( t );
    if( attr )
	listview_add_sub_item( lv, 0, 1, (char *)attr );
    
    t = gpgme_sig_get_ulong_attr( sig, 0, GPGME_ATTR_TYPE );
    key = (gpgme_key_t)t;
    key_attr = gpgme_key_get_ulong_attr( key, GPGME_ATTR_VALIDITY, NULL, 0 );
    attr = gpgme_key_expand_attr( GPGME_ATTR_VALIDITY, key_attr );
    if( attr )
	listview_add_sub_item( lv, 0, 2, (char *)attr );
    
    attr = gpgme_key_get_string_attr( key, GPGME_ATTR_KEYID, NULL, 0 );
    if ( !attr || strlen( attr ) < 8 )
	attr = "DEADBEEFDEADBEEF";
    _snprintf( keyid, sizeof keyid -1, "0x%s", attr + 8 );
    listview_add_sub_item( lv, 0, 3, keyid );
    
    attr = gpgme_key_get_string_attr( key, GPGME_ATTR_NAME, NULL, 0 );
    if( !attr ) {
	attr = _("Invalid User ID");
	listview_add_sub_item( lv, 0, 4, (char *)attr );
    }	
    else {
	uid = utf8_to_wincp (attr, strlen (attr));
	if ( uid ) {
	    listview_add_sub_item( lv, 0, 4, (char *)uid );
	    free( uid );
	}
    }
    
    return 0;
} /* verlist_add_sig */


int
verlist_add_sig_log( listview_ctrl_t lv, siglog_context_t log )
{
    gpgme_sig_t sig = log->sig;
    const char * attr;    
    char t[64], * name;
    u32 i_attr;

    if( listview_add_item( lv, "" ) )
	return WPTERR_GENERAL;
    
    name = extract_filename( log->file );
    if( name )
	listview_add_sub_item( lv, 0, 0, name );
    free_if_alloc( name );
    
    i_attr = gpgme_sig_get_ulong_attr( sig, 0, GPGME_ATTR_VALIDITY );
    attr = gpg_sigstat[i_attr % SIGSTAT_MASK];
    if( attr )
	listview_add_sub_item( lv, 0, 1, attr );

    attr = strtimestamp( gpgme_sig_get_ulong_attr( sig, 0, GPGME_ATTR_CREATED ) );
    if( attr )
	listview_add_sub_item( lv, 0, 2, attr );
    
    attr = gpgme_key_expand_attr( GPGME_ATTR_VALIDITY,
				  gpgme_sig_get_ulong_attr( sig, 0, GPGME_ATTR_OTRUST ) );
    if( attr )
	listview_add_sub_item( lv, 0, 3, attr );
    
    attr = gpgme_sig_get_string_attr( sig, GPGME_ATTR_KEYID );
    if( !log->use_uid && strlen( attr ) == 16 ) {
	_snprintf( t, sizeof t-1, "0x%s", attr + 8 );
	listview_add_sub_item( lv, 0, 4, t );
    }
    else if( !log->use_uid && strlen( attr ) == 8 ) {
	_snprintf( t, sizeof t-1, "0x%s", attr );
	listview_add_sub_item( lv, 0, 4, t );
    }
    else if( log->user_id ) {
	char * p = utf8_to_wincp (log->user_id, strlen (log->user_id));
	if( p ) {
	    listview_add_sub_item( lv, 0, 4, p );
	    free( p );
	}
    }
    return 0;
} /* verlist_add_sig_log */
