/* revoke.c - Revocation certificates
 *	Copyright (C) 2001, 2002, 2003 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <string.h>
#include <sys/types.h>

#include "ops.h"
#include "context.h"
#include "util.h"
#include "openpgp.h"

struct revoke_result_s {
    int bad_passphrase;
};


static void
revoke_set_reason( gpgme_editkey_t ctx, int id )
{
    switch (id) {
    case GPGME_REVOKE_INSECURE:   ctx->u.revoke.reason_id = "1"; break;
    case GPGME_REVOKE_SUPERSEDED: ctx->u.revoke.reason_id = "2"; break;
    case GPGME_REVOKE_NOUSED:     ctx->u.revoke.reason_id = "3"; break;
    default:                      ctx->u.revoke.reason_id = "0"; break;
    }
} /* revoke_set_reason */


gpgme_error_t
gpgme_revoke_set( gpgme_editkey_t ctx, const char * uid, const char * text,
		 int reason, int pgpmode, const char *pass )
{
    char * p;

    if( !ctx )
	return mk_error( Invalid_Value );
    revoke_set_reason( ctx, reason );
    ctx->type = GPGME_EDITKEY_REVOKE;
    ctx->u.revoke.uid = uid;
    if( text ) {
	p = ctx->u.revoke.reason_text = strdup( text );
	if (!p)
	    return mk_error (Out_Of_Core);
	ctx->u.revoke.reason_sent = 0;
    }
    ctx->u.revoke.passwd = pass;
    ctx->u.revoke.pgp = pgpmode;
    return 0;
} /* gpgme_revoke_set */


static void
revoke_status_handler (gpgme_ctx_t ctx, gpg_statcode_t code, char * args)
{
    struct revoke_result_s * c = ctx->tmp_res;

    if( ctx->out_of_core )
        return;
    
    switch( code ) {
    case STATUS_EOF: 
	break;

    case STATUS_GOOD_PASSPHRASE:
	c->bad_passphrase = 0;
        break;
        
    default: /* ignore all other codes */
        break;
    }
} /* revoke_status_handler */


static const char *
revoke_command_handler( void *opaque, gpg_statcode_t code, const char *key )
{
    gpgme_editkey_t c = opaque;
    
    if( !code ) {
        /* cleanup */
        return NULL;
    }
    
    if( !key )
        return NULL;
    
    if( code == STATUS_GET_BOOL && !strcmp( key, "gen_revoke.okay" ) ) {
        c->u.revoke.reason_sent = 0;
        return "Y";
    }

    if( code == STATUS_GET_LINE && !strcmp( key, "ask_revocation_reason.code" ) )
        return c->u.revoke.reason_id;

    if( code == STATUS_GET_LINE && !strcmp( key, "ask_revocation_reason.text" ) ) {
        if( c->u.revoke.reason_sent == 0 && c->u.revoke.reason_text ) {
            c->u.revoke.reason_sent++;
            return c->u.revoke.reason_text;
        }
        return "";
    }
    
    if( code == STATUS_GET_BOOL && !strcmp( key, "ask_revocation_reason.okay" ) )
        return "Y";
    
    if( code == STATUS_GET_HIDDEN && !strcmp( key, "passphrase.enter" ) )
        return c->u.revoke.passwd;
    
    return NULL;
} /* command_handler */


static gpgme_error_t
revoke_start( gpgme_ctx_t ctx, gpgme_editkey_t rev, gpgme_data_t cert )
{
    gpgme_error_t rc;
    
    fail_on_pending_request( ctx );
    ctx->pending = 1;
    
    _gpgme_gpg_release( &ctx->gpg ); 
    rc = _gpgme_gpg_new( &ctx->gpg );
    if( rc )
        goto leave;

    if( ctx->use_logging )
	_gpgme_gpg_set_logging_handler( ctx->gpg, ctx );
    _gpgme_gpg_set_status_handler( ctx->gpg, revoke_status_handler, ctx );
    _gpgme_gpg_set_command_handler( ctx->gpg, revoke_command_handler, rev );
    
    _gpgme_data_set_mode( cert, GPGME_DATA_MODE_IN );
    
    if( rev->u.revoke.pgp )
	_gpgme_gpg_add_arg( ctx->gpg, "--pgp7" );
    _gpgme_gpg_add_arg( ctx->gpg, "--output");
    _gpgme_gpg_add_arg( ctx->gpg, "-" );
    _gpgme_gpg_add_data( ctx->gpg, cert, 1 );
    
    _gpgme_gpg_add_arg( ctx->gpg, "--gen-revoke" );
    _gpgme_gpg_add_arg( ctx->gpg, rev->u.revoke.uid );
    
    rc = _gpgme_gpg_spawn( ctx->gpg, ctx );
    
leave:
    if( rc ) {
        _gpgme_gpg_release( &ctx->gpg );
        ctx->pending = 0;
    }
    
    return rc;
} /* gpgme_op_revoke_start */


gpgme_error_t
gpgme_op_revoke (gpgme_ctx_t ctx, gpgme_editkey_t rev, gpgme_data_t cert)
{
    struct revoke_result_s res;
    gpgme_error_t rc;
    
    res.bad_passphrase = 1;
    ctx->tmp_res = &res;

    rc = revoke_start (ctx, rev, cert);
    if( !rc ) {
        gpgme_wait (ctx, 1);
        if (res.bad_passphrase)
            rc = mk_error (Bad_Passphrase);
        ctx->pending = 0;
    }
    
    return rc;
} /* gpgme_op_revoke */


gpgme_error_t
gpgme_revcert_new( gpgme_revcert_t *r_cert )
{
    gpgme_revcert_t ctx;

    if( !r_cert )
	return mk_error( Invalid_Value );
    ctx = calloc( 1, sizeof *ctx );
    if( !ctx )
	return mk_error( Out_Of_Core );
    *r_cert = ctx;
    return 0;
} /* gpgme_revcert_new */


void
gpgme_revcert_release( gpgme_revcert_t cert )
{
    if( cert ) {
	safe_free( cert->reason );
	safe_free( cert );	
    }
} /* gpgme_revcert_release */

gpgme_error_t
gpgme_revcert_parse( const char *data, gpgme_revcert_t cert, char * keyid )
{
    gpg_iobuf_t tmp;
    armor_filter_context_t afx;
    PACKET *pkt = calloc( 1, sizeof *pkt );
    PKT_signature * sig;
    int rc = 0, len = 0, seq = 0;
    const byte *p;
    FILE * fp;
    
    if( !data || !cert )
	return mk_error( Invalid_Value );

    fp = fopen( "temp.winpt.xxx", "wb" );
    if( fp ) {
	fwrite( data, 1, strlen( data ), fp );
	fclose( fp );
    }
    tmp = gpg_iobuf_open( "temp.winpt.xxx" );
    if( tmp == NULL )
	return mk_error( General_Error );
    gpg_iobuf_ioctl( tmp, 3, 1, NULL ); /* disable cache */
    if( gpg_use_armor_filter( tmp ) ) {
	memset( &afx, 0, sizeof afx );
	gpg_iobuf_push_filter( tmp, gpg_armor_filter, &afx );
    }

    gpg_iobuf_seek( tmp, 0 );
    gpg_init_packet( pkt );
    while( ( rc = gpg_parse_packet( tmp, pkt ) ) != -1 ) {
	if( pkt->pkttype == PKT_SIGNATURE )
	    break;
	gpg_free_packet( pkt );
    }

    if( pkt->pkttype != PKT_SIGNATURE )
	goto fail;
    sig = pkt->pkt.signature;

    p = gpg_parse_sig_subpkt2( sig, SIGSUBPKT_REVOC_REASON, &len );
    if( p ) {
	cert->code = p[0];
	if( len > 1 ) {
	    cert->reason = calloc( 1, len );
	    if( !cert->reason )
		goto fail;
	    memcpy( cert->reason , p + 1, len - 1 );
	    cert->reason[len-1] = '\0';
	}
	else
	    cert->reason = NULL;
    }
    sprintf( cert->keyid, "%08lX%08lX", sig->keyid[0], sig->keyid[1] );
    if( keyid )
	strcpy( keyid, cert->keyid );
    cert->timestamp = 0;
    p = gpg_parse_sig_subpkt2( sig, SIGSUBPKT_SIG_CREATED, &len );
    if( p  && len == 4 )
	cert->timestamp = p[0] << 24 | p[1] << 16 | p[2] << 8 | p[3];    

fail:
    gpg_iobuf_close( tmp );
    unlink( "temp.winpt.xxx" );
    gpg_free_packet( pkt );
    safe_free( pkt );    
    return 0;
} /* gpgme_revcert_parse */