/* wptKeysigDlg.cpp - Key signature listing
 *	Copyright (C) 2001-2005 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>
#include <commctrl.h>
#include <time.h>

#include "../resource.h"
#include "wptGPG.h"
#include "wptCommonCtl.h"
#include "wptContext.h" /* for passphrase_s */
#include "wptDlgs.h"
#include "wptW32API.h"
#include "wptNLS.h"
#include "wptKeyList.h"
#include "wptKeyserver.h"
#include "wptUTF8.h"
#include "wptTypes.h"
#include "wptVersion.h"
#include "wptErrors.h"

static subclass_s siglist_proc;


static int inline
is_sig (listview_ctrl_t lv, int pos)
{
    char tmpbuf[256];

    if (pos == -1)
	pos = listview_get_curr_pos (lv);
    listview_get_item_text (lv, pos, 0, tmpbuf, 200);
    if (*tmpbuf == ' ')
	return -1;
    return 0;
}


static int
do_delsig (HWND dlg, listview_ctrl_t lv, winpt_key_t key)
{
    int pos, npos, id;
    int signo=0, uidno=0;
    gpgme_ctx_t ctx;
    gpgme_editkey_t ek;
    gpgme_error_t rc;

    npos = pos = listview_get_curr_pos (lv);
    if (!is_sig (lv, -1))
	return -1;
    while (pos > 0 && is_sig (lv, pos))
    {
	signo++;
	pos--;
    }
    pos = npos;
    while (npos > 0)
    {
	if (!is_sig (lv, npos))
	    uidno++;
	npos--;
    }
    uidno++;
    id = log_box(_("Key Manager"), MB_YESNO, 
		  _("Are you really sure you want to delete this signature from\n"
		    "  \"%s\""), key->uid);
    if (id == IDNO)
	return 0;
    rc = gpgme_new (&ctx);
    if (rc)
	BUG (0);
    rc = gpgme_editkey_new (&ek);
    if (rc)
	BUG (0);
    gpgme_editkey_delsig_set (ek, uidno, signo);
    gpgme_set_edit_ctx (ctx, ek, GPGME_EDITKEY_DELSIG);
    rc = gpgme_op_editkey (ctx, key->keyid);
    if (rc)
	msg_box (dlg, gpgme_strerror (rc), _("Key Manager"), MB_ERR);
    else
	listview_del_item (lv, pos);
    gpgme_release (ctx);
    gpgme_editkey_release (ek);
    return 0;
}


static BOOL CALLBACK
sigprops_dlg_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    static listview_ctrl_t lv;
    char tmpbuf[256];
    int n;
    struct {
	unsigned int exportable:1;
	unsigned int expired:1;
	unsigned int nrev:1;
	unsigned int rev:1;
	const char * alg;
	int _class;
    } ctx;
    const char * fmt_templ = "%s %s signature";

    switch (msg)
    {
    case WM_SYSCOMMAND:
	if (LOWORD (wparam) == SC_CLOSE)
            EndDialog (dlg, TRUE);
        return FALSE;

    case WM_INITDIALOG:
	lv = (listview_ctrl_t)lparam;
	if (!lv)
	    dlg_fatal_error (dlg, "could not get dialog param");
	memset (&ctx, 0, sizeof ctx);
	n = listview_get_curr_pos (lv);
	listview_get_item_text (lv, n, 2, tmpbuf, DIM (tmpbuf)-1);
	if (!strstr (tmpbuf, "L"))
	    ctx.exportable = 1;
	ctx._class = atoi (tmpbuf);
	if (ctx._class == 0)
	    ctx._class = 10;
	else if (ctx._class < 10)
	    ctx._class += 10;
	listview_get_item_text (lv, n, 6, tmpbuf, DIM (tmpbuf)-1);
	if (strstr (tmpbuf, "DSA"))	    
	    ctx.alg = "DSA";
	else if (strstr (tmpbuf, "RSA"))
	    ctx.alg = "RSA";
	else
	    ctx.alg = "ELG";
	_snprintf (tmpbuf, DIM (tmpbuf)-1, fmt_templ, 
		   ctx.exportable? "Exportable" : "Non-exportable", ctx.alg);
	SetDlgItemText (dlg, IDC_SIGPROPS_INFO, tmpbuf);
	listview_get_item_text (lv, n, 4, tmpbuf, DIM (tmpbuf)-1);
	SetDlgItemText (dlg, IDC_SIGPROPS_KEYID, tmpbuf);
	SetDlgItemInt (dlg, IDC_SIGPROPS_CLASS, ctx._class, FALSE);
	if (ctx.exportable)
	    CheckDlgButton (dlg, IDC_SIGPROPS_EXP, BST_CHECKED);
	listview_get_item_text (lv, n, 0, tmpbuf, DIM (tmpbuf)-1);
	SetDlgItemText (dlg, IDC_SIGPROPS_ISSUER, tmpbuf+1);
	tmpbuf[0] = 0;
	listview_get_item_text (lv, n, 5, tmpbuf, DIM (tmpbuf)-1);
	if (strlen (tmpbuf) == 0) {
	    ShowWindow (GetDlgItem (dlg, IDC_SIGPROPS_EXPSTR), SW_HIDE);
	    ShowWindow (GetDlgItem (dlg, IDC_SIGPROPS_EXPDATE), SW_HIDE);
	}
	else {
	    SYSTEMTIME st;
	    struct tm * tm;
	    time_t t = time (NULL);

	    memset (&st, 0, sizeof st);
	    st.wYear = atoi (tmpbuf);
	    st.wMonth = atoi (tmpbuf+5);
	    st.wDay = atoi (tmpbuf+8);
	    DateTime_SetSystemtime (GetDlgItem (dlg, IDC_SIGPROPS_EXPDATE),
				    GDT_VALID, &st);

	    tm = localtime (&t);
	    tm->tm_mon++;
	    tm->tm_year += 1900;
	    if (tm->tm_year > st.wYear)
		ctx.expired = 1;
	    else if (tm->tm_mon > st.wMonth)
		ctx.expired = 1;
	    if (ctx.expired)
		CheckDlgButton (dlg, IDC_SIGPROPS_EXPIRED, BST_CHECKED);
	}
	SetDlgItemText (dlg, IDC_SIGPROPS_EXP, _("Exportable"));
	SetDlgItemText (dlg, IDC_SIGPROPS_NREV, _("Non-revocably"));
	SetDlgItemText (dlg, IDC_SIGPROPS_REV, _("Revoked"));
	SetDlgItemText (dlg, IDC_SIGPROPS_EXPIRED, _("Expired"));
	SetForegroundWindow (dlg);
	center_window (dlg);
	return TRUE;

    case WM_COMMAND:
	switch (LOWORD (wparam)) {
	case IDOK:
	    EndDialog (dlg, TRUE);
	    break;
	}
    }

    return FALSE;
}


static BOOL CALLBACK
subclass_dlg_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    listview_ctrl_t lv;
    winpt_key_t key;

    switch (msg)
    {
    case WM_KEYUP:
        int virt_key = (int)wparam;
	key = (winpt_key_t)siglist_proc.opaque;
	lv = key->callback.ctl;
        if (virt_key == VK_SPACE)
	{
	    if (is_sig (lv, -1))
		DialogBoxParam (glob_hinst, (LPCTSTR)IDD_WINPT_SIGPROPS, dlg,
				sigprops_dlg_proc, (LPARAM)lv);
	}
	else if (virt_key == VK_DELETE)
	    do_delsig (dlg, lv, key);
	break;
    }
    return CallWindowProc (siglist_proc.old, dlg, msg, wparam, lparam);
}


static int
check_for_missing_keys (listview_ctrl_t lv)
{
    int i, n;
    char id[128];
    
    n = listview_count_items( lv, 0 );
    for( i = 0; i < n; i++ ) {
        listview_get_item_text( lv, i, 1, id, sizeof id - 1 );
        if( !strncmp( id, "NOKEY", 5 ) )
            return 1;
    }
    
    return 0;
} /* check_for_missing_keys */


static int
recv_missing_keys (HWND dlg, listview_ctrl_t lv)
{
    int i, n, rc;
    char id[128], keyid[18+1];
    
    n = listview_count_items (lv, 0);
    for( i = 0; i < n; i++ ) {
        listview_get_item_text( lv, i, 1, id, sizeof id - 1 );
        if( !strncmp( id, "NOKEY", 5 ) ) {
            listview_get_item_text( lv, i, 4, keyid, sizeof keyid -1 );
            rc = hkp_recv_key( dlg, default_keyserver, default_keyserver_port, keyid, 0, 0 );
            if( rc )               
		break;
        }
    }
    
    return rc;
} /* recv_missing_keys */


static void
do_create_popup (HWND dlg)
{
    HMENU hm, sm;
    POINT p;

    GetCursorPos (&p);
    hm = LoadMenu (glob_hinst, MAKEINTRESOURCE (IDR_WINPT_KEYSIG_CTX));
    sm = GetSubMenu (hm, 0);

    TrackPopupMenu (sm, TPM_RIGHTALIGN, p.x, p.y, 0, dlg, NULL);

    DestroyMenu (hm);
    DestroyMenu (sm);
}


static void
do_load_keyprops (HWND dlg, listview_ctrl_t lv)
{
    winpt_key_s k;
    gpgme_key_t key;
    char keyid[32] = {0};
    char status[64] = {0}, creation[64] = {0};
    int n = listview_get_curr_pos (lv);

    listview_get_item_text (lv, n, 1, status, DIM (status)-1);
    listview_get_item_text (lv, n, 3, creation, DIM (creation)-1);
    listview_get_item_text (lv, n, 4, keyid, DIM (keyid)-1);
    if (!strcmp (status, "NOKEY")) {
	msg_box (dlg, _("Key not found in keyring, please get it from the keyserver first."),
		 _("Key Manager"), MB_INFO);
	return;
    }
    
    if ((strlen (keyid) < 3 ||get_pubkey (keyid, &key))) {
	if (strlen (creation) > 0)
	    msg_box (dlg, _("Key not found in keyring."), _("Key Manager"), MB_INFO);
	return;
    }
    memset (&k, 0, sizeof k);
    k.keyid = keyid;
    DialogBoxParam (glob_hinst, (LPCTSTR)IDD_WINPT_KEYPROPS, dlg,
		    keyprops_dlg_proc, (LPARAM)&k);
}


BOOL CALLBACK
keysig_dlg_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    static listview_ctrl_t lv = NULL;
    struct winpt_key_s *k;
    char inf[384], keyid[18+1];    
    int idx = 0, id, rc;
    HWND sl;
    
    switch( msg ) {
    case WM_INITDIALOG:
        k = (winpt_key_t) lparam;
	if (!k)
	    BUG (0);
        if( k->uid )
            _snprintf( inf, sizeof inf - 1, _("Signature List for \"%s\""), k->uid );
        SetWindowText( dlg, inf );
	#ifndef LANG_DE
        SetDlgItemText( dlg, IDC_KEYSIG_RECVKEY, _("&Receive Key") );
	SetDlgItemText (dlg, IDC_KEYSIG_SIGPROPS, _("&Properties"));
	#endif
	lv = siglist_load( GetDlgItem( dlg, IDC_KEYSIG_LIST ), k->keyid );
	if( !check_for_missing_keys( lv ) )
	    EnableWindow( GetDlgItem( dlg, IDC_KEYSIG_RECVKEY ), FALSE );
	k->callback.ctl = lv;
	sl = GetDlgItem (dlg, IDC_KEYSIG_LIST);
	siglist_proc.dlg = dlg;
	siglist_proc.opaque = k;
	siglist_proc.current = (WNDPROC)subclass_dlg_proc;
	siglist_proc.old = (WNDPROC)GetWindowLong (sl, GWL_WNDPROC);
	if (siglist_proc.old)
	{
	    if (!SetWindowLong (sl, GWL_WNDPROC, (LONG)siglist_proc.current)) 
	    {
		msg_box (dlg, _("Could not set keylist window procedure."), _("Key Manager"), MB_ERR);
		BUG (0);
	    }
	}
        SetForegroundWindow (dlg);
	center_window (dlg);
        return TRUE;
        
    case WM_DESTROY:
        if ( lv ) {
            siglist_delete( lv );
            lv = NULL;
        }
        return FALSE;
        
    case WM_SYSCOMMAND:
        if( LOWORD ( wparam ) == SC_CLOSE )
            EndDialog( dlg, TRUE );
        return FALSE;

    case WM_NOTIFY:
	NMHDR * notify;
        
        notify = (NMHDR *)lparam;
        if( notify && notify->code == NM_DBLCLK 
	    && notify->idFrom == IDC_KEYSIG_LIST )
	    do_load_keyprops (dlg, lv);
	if (notify && notify->code == NM_RCLICK &&
	    notify->idFrom == IDC_KEYSIG_LIST)
	    do_create_popup (dlg);
	break;
        
    case WM_COMMAND:
        switch ( LOWORD( wparam ) ) {

	case ID_SIGCTX_KEYPROPS:
	    do_load_keyprops (dlg, lv);
	    break;

	case ID_SIGCTX_PROPS:
	    if (is_sig (lv, -1))
		DialogBoxParam (glob_hinst, (LPCTSTR)IDD_WINPT_SIGPROPS, dlg,
				sigprops_dlg_proc, (LPARAM)lv);
	    break;

        case IDC_KEYSIG_RECVKEY:
	    idx = listview_get_curr_pos( lv );
            if( idx == -1 ) {
                id = msg_box( dlg, _("Really receive all missing keys?"),
				_("Key Manager"), MB_YESNO|MB_INFO );
                if( id == IDYES ) {
                    rc = recv_missing_keys( dlg, lv );
                    if( !rc )
                        keycache_set_reload( 1 );
                    return TRUE;
                }
                return FALSE;
            }
            listview_get_item_text (lv, idx, 4, keyid, sizeof keyid - 1);
            rc = hkp_recv_key (dlg, default_keyserver, default_keyserver_port, keyid, 0, 0);
            if (!rc)
                keycache_set_reload (1);
            return TRUE;

	case IDC_KEYSIG_SIGPROPS:
	    if (is_sig (lv, -1))
		DialogBoxParam (glob_hinst, (LPCTSTR)IDD_WINPT_SIGPROPS, dlg,
				sigprops_dlg_proc, (LPARAM)lv);
	    return TRUE;
            
        case IDOK:
            EndDialog (dlg, TRUE);
            return TRUE;
        }
        break;
    }
    return FALSE;
} /* keysig_dlg_proc */
