/*
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with the program; if not, see <http://www.gnu.org/licenses/>
 *
 *
 * Authors:
 *		Michael Zucchi <notzed@ximian.com>
 *		Srinivasa Ragavan <sragavan@novell.com>
 *
 * Copyright (C) 1999-2008 Novell, Inc. (www.novell.com)
 *
 */

/*
  Concrete class for formatting mails to html
*/

#ifndef EM_FORMAT_MAIL_H
#define EM_FORMAT_MAIL_H

#include <mail/em-format.h>
#include <mail/mail-config.h>
#include <camel/camel-medium.h>
#include <camel/camel-mime-part.h>
#include <camel/camel-stream.h>
#include <camel/camel-url.h>

#include "mail-message-view.h"

/* Standard GObject macros */
#define EM_TYPE_FORMAT_MAIL \
	(em_format_mail_get_type ())
#define EM_FORMAT_MAIL(obj) \
	(G_TYPE_CHECK_INSTANCE_CAST \
	((obj), EM_TYPE_FORMAT_MAIL, EMFormatMail))
#define EM_FORMAT_MAIL_CLASS(cls) \
	(G_TYPE_CHECK_CLASS_CAST \
	((cls), EM_TYPE_FORMAT_MAIL, EMFormatMailClass))
#define EM_IS_FORMAT_MAIL(obj) \
	(G_TYPE_CHECK_INSTANCE_TYPE \
	((obj), EM_TYPE_FORMAT_MAIL))
#define EM_IS_FORMAT_MAIL_CLASS(cls) \
	(G_TYPE_CHECK_CLASS_TYPE \
	((cls), EM_TYPE_FORMAT_MAIL))
#define EM_FORMAT_MAIL_GET_CLASS(obj) \
	(G_TYPE_INSTANCE_GET_CLASS \
	((obj), EM_TYPE_FORMAT_MAIL, EMFormatMailClass))

G_BEGIN_DECLS

typedef struct _EMFormatMail EMFormatMail;
typedef struct _EMFormatMailClass EMFormatMailClass;
typedef struct _EMFormatMailPrivate EMFormatMailPrivate;

typedef enum {
	MAIL_BEGIN_KEEP_SCROLL = 1 << 0,
	MAIL_BEGIN_KEEP_IMAGES = 1 << 1,
	MAIL_BEGIN_BLOCK_UPDATES = 1 << 2,
	MAIL_BEGIN_BLOCK_IMAGES = 1 << 3,
	/*enable autochange content_type*/
	MAIL_BEGIN_CHANGECONTENTTYPE = 1 << 4
} MailBeginFlags;

enum _em_format_mail_header_flags {
	EM_FORMAT_MAIL_HEADER_TO = 1<<0,
	EM_FORMAT_MAIL_HEADER_CC = 1<<1,
	EM_FORMAT_MAIL_HEADER_BCC = 1<<2
};

typedef enum {
	EM_FORMAT_MAIL_STATE_NONE = 0,
	EM_FORMAT_MAIL_STATE_RENDERING
} EMFormatMailState;

/* A MailJob will be executed in another thread, in sequence.
   It's job is to write to its stream, close it if successful,
   then exit. */

typedef struct _EMFormatMailJob EMFormatMailJob;

/**
 * struct _EMFormatMailJob - A formatting job.
 *
 * @next: Double linked list header.
 * @prev: Double linked list header.
 * @format: Set by allocation function.
 * @stream: Free for use by caller.
 * @puri_level: Set by allocation function.
 * @base: Set by allocation function, used to save state.
 * @callback: This callback will always be invoked, only once, even if the user
 * cancelled the display.  So the callback should free any extra data
 * it allocated every time it is called.
 * @u: Union data, free for caller to use.
 *
 * This object is used to queue a long-running-task which cannot be
 * processed in the primary thread.  When its turn comes, the job will
 * be de-queued and the @callback invoked to perform its processing,
 * restoring various state to match the original state.  This is used
 * for image loading and other internal tasks.
 *
 * This object is struct-subclassable.  Only em_format_mail_job_new()
 * may be used to allocate these.
 **/
struct _EMFormatMailJob {
	EMFormatMailJob *next;
	EMFormatMailJob *prev;

	EMFormatMail *format;
	CamelStream *stream;

	/* We need to track the state of the visibility tree at
	   the point this uri was generated */
	struct _EMFormatPURITree *puri_level;
	CamelURL *base;

	void (*callback)(EMFormatMailJob *job, gint cancelled);
	union {
		gchar *uri;
		CamelMedium *msg;
		EMFormatPURI *puri;
		struct _EMFormatPURITree *puri_level;
		gpointer data;
	} u;
};

/* Pending object (classid: url) */
typedef struct _EMFormatMailPObject EMFormatMailPObject;

typedef GtkWidget * (*EMFormatMailPObjectFunc)(EMFormatMail *md, GtkWidget *eb, EMFormatMailPObject *pobject);

/**
 * struct _EMFormatMailPObject - Pending object.
 *
 * @next: Double linked list header.
 * @prev: Double linked list header.
 * @free: Invoked when the object is no longer needed.
 * @format: The parent formatter.
 * @classid: The assigned class id as passed to add_pobject().
 * @func: Callback function.
 * @part: The part as passed to add_pobject().
 *
 * This structure is used to track OBJECT tags which have been
 * inserted into the Mail stream.  When GtkHTML requests them the
 * @func will be invoked to create the embedded widget.
 *
 * This object is struct-subclassable.  Only
 * em_format_mail_add_pobject() may be used to allocate these.
 **/
struct _EMFormatMailPObject {
	EMFormatMailPObject *next;
	EMFormatMailPObject *prev;

	void (*free)(EMFormatMailPObject *);
	EMFormatMail *format;

	gchar *classid;

	EMFormatMailPObjectFunc func;
	CamelMimePart *part;
	/* Private */
	GtkWidget *body;
	GtkWidget *view;
	struct _CamelStream *stream;	
};

#define EM_FORMAT_MAIL_HEADER_NOCOLUMNS (EM_FORMAT_HEADER_LAST)
#define EM_FORMAT_MAIL_HEADER_MAIL (EM_FORMAT_HEADER_LAST<<1) /* header already in html format */
#define EM_FORMAT_MAIL_HEADER_NODEC (EM_FORMAT_HEADER_LAST<<2) /* header already in html format */
#define EM_FORMAT_MAIL_HEADER_LAST (EM_FORMAT_HEADER_LAST<<8)

#define EM_FORMAT_MAIL_VPAD "<table cellspacing=0 cellpadding=3><tr><td><a name=\"padding\"></a></td></tr></table>\n"

/**
 * struct _EMFormatMail - Mail formatter object.
 *
 * @format:
 * @priv:
 * @msg_view:
 * @body:
 * @pending_object_list:
 * @headers:
 * @text_mail_flags:
 * @body_colour:
 * @header_colour:
 * @text_colour:
 * @frame_colour:
 * @content_colour:
 * @citation_colour:
 * @load_http:2:
 * @load_http_now:1:
 * @mark_citations:1:
 * @simple_headers:1:
 * @hide_headers:1:
 * @show_icon:1:
 *
 * Most of these fields are private or read-only.
 *
 * The base HTML formatter object.  This object drives HTML generation
 * into a GtkHTML parser.  It also handles text to HTML conversion,
 * multipart/related objects and inline images.
 **/
struct _EMFormatMail {
	EMFormat format;

	EMFormatMailPrivate *priv;

	GtkWidget *msg_view;
	GtkWidget *body;
	GtkWidget *wheaders;
	CamelStream *header_stream;

	EDList pending_object_list;

	GSList *headers;

	guint32 text_html_flags; /* default flags for text to html conversion */
	guint32 body_colour;	/* header box colour */
	guint32 header_colour;
	guint32 text_colour;
	guint32 frame_colour;
	guint32 content_colour;
	guint32 citation_colour;
	guint load_http:2;
	guint load_http_now:1;
	guint mark_citations:1;
	guint simple_headers:1; /* simple header format, no box/table */
	guint hide_headers:1; /* no headers at all */
	guint show_icon:1; /* show an icon when the sender used Evo */
	guint32 header_wrap_flags;

	EMFormatMailState state; /* actual state of the object */
};

struct _EMFormatMailClass {
	EMFormatClass format_class;
};

GType		em_format_mail_get_type		(void);
EMFormatMail *	em_format_mail_new		(void);
void		em_format_mail_load_http	(EMFormatMail *efh);

void		em_format_mail_set_load_http	(EMFormatMail *efh,
						 gint style);
void		em_format_mail_set_mark_citations
						(EMFormatMail *efh,
						 gint state,
						 guint32 citation_colour);

/* retrieves a pseudo-part icon wrapper for a file */
CamelMimePart *	em_format_mail_file_part	(EMFormatMail *efh,
						 const gchar *mime_type,
						 const gchar *filename);

/* for implementers */
EMFormatMailPObject *
		em_format_mail_add_pobject	(EMFormatMail *efh,
						 gsize size,
						 const gchar *classid,
						 CamelMimePart *part,
						 EMFormatMailPObjectFunc func);
EMFormatMailPObject *
		em_format_mail_find_pobject	(EMFormatMail *efh,
						 const gchar *classid);
EMFormatMailPObject *
		em_format_mail_find_pobject_func(EMFormatMail *efh,
						 CamelMimePart *part,
						 EMFormatMailPObjectFunc func);
void		em_format_mail_remove_pobject	(EMFormatMail *efh,
						 EMFormatMailPObject *pobject);
void		em_format_mail_clear_pobject	(EMFormatMail *efh);

EMFormatMailJob *
		em_format_mail_job_new		(EMFormatMail *efh,
						 void (*callback)(EMFormatMailJob *job, gint cancelled),
						 gpointer data);
void		em_format_mail_job_queue	(EMFormatMail *efh,
						 EMFormatMailJob *job);
void 		em_format_mail_set_message_view (EMFormatMail *efw, 
						 struct _MailMessageView *mmv, 
						 GtkWidget *body,
						 GtkWidget *headers);

G_END_DECLS

#endif /* EM_FORMAT_MAIL_H */
