/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * This file is part of capuchin-glib. 
 *
 * Copyright (C) Sebastian Pölsterl 2008 <marduk@k-d-w.org>
 * 
 * capuchin-glib is free software.
 * 
 * You may redistribute it and/or modify it under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option)
 * any later version.
 * 
 * capuchin-glib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with capuchin-glib.
 * If not, see <http://www.gnu.org/licenses/>.
 */

#include <capuchin-glib/capuchin-glib.h>

#define REPO_URL "http://www.k-d-w.org/clipboard/deskbar/deskbar.xml" 

static GMainLoop *mainloop;

static void
_test_get_application_name (CapuchinGAppObject *appobject)
{
	GError *error = NULL;
	gchar *name;
	
	name = capuchin_g_app_object_get_application_name (appobject, &error);
	
	if (error != NULL) {
		g_critical ("Error while retrieving application name: %s", error->message);
		g_error_free (error);
	} else {
		g_printf ("APPLICATION-NAME: %s\n", name);
		g_free (name);
	}
}

static void
_test_get_available_plugins (CapuchinGAppObject *appobject)
{
	GError *error = NULL;
	gchar **plugins;
	
	plugins = capuchin_g_app_object_get_available_plugins (appobject, &error);
	
	if (error != NULL) {
		g_critical ("Error while retrieving plugins: %s", error->message);
		g_error_free (error);
	} else {
		gchar *pid;
		gint i=0;
		for (pid = plugins[0]; pid != NULL; pid = plugins[i])
		{
			_print_plugin_info (appobject, pid);
			i++;
		}
	
		g_strfreev (plugins);
	}
}

static void
_test_get_available_updates (CapuchinGAppObject *appobject)
{
	GError *error = NULL;
	gchar *changes;
	CapuchinGPluginInfo **info;
	gchar **plugins;
	gchar *p;
	
	info = g_new (CapuchinGPluginInfo*, 4);
	info[0] = capuchin_g_plugin_info_new ("ssh.py", "0.0.9");
	info[1] = capuchin_g_plugin_info_new ("ekiga.py", "0.1.0");
	info[2] = capuchin_g_plugin_info_new ("ssh.py", "0.1.0");
	info[3] = NULL;
	
	plugins = capuchin_g_app_object_get_available_updates (appobject, info, &error);
	
	if (error != NULL) {
		g_critical ("Error while retrieving list of updates: %s", error->message);
		g_error_free (error);
	} else {
		g_printf ("UPDATES:\n");
	
		gint i=0;
		for (p = plugins[0]; p != NULL; p = plugins[i])
		{
			g_printf ("ID: %s\n", p);
		
			changes = capuchin_g_app_object_get_plugin_changes (appobject,
																"ekiga.py", "1.1.0.0",
																&error);
			if (error != NULL) {
				g_critical ("Error while retrieving changes: %s", error->message);
				g_error_free (error);
			} else {
				g_printf ("Changes: %s\n", changes);
				g_free (changes);
			}
		
			i++;
		}
	
		g_strfreev (plugins);
	}
	
	g_object_unref (info[0]);
	g_object_unref (info[1]);
	g_free (info);
}

static void
_test_get_tags (CapuchinGAppObject *appobject)
{
	GError *error = NULL;
	gchar **tags;
	
	tags = capuchin_g_app_object_get_tags (appobject, &error);
	
	if (error != NULL) {
		g_critical ("Error while retrieving tags: %s", error->message);
		g_error_free (error);
	} else {
		g_printf ("TAGS:\n");
	
		gchar *tag;
		gint i=0;
		for (tag = tags[0]; tag != NULL; tag = tags[i])
		{
			g_printf("%s\n", tag);
			i++;
		}
	
		g_strfreev (tags);
	}
}

static void
_test_get_plugins_with_tag (CapuchinGAppObject *appobject)
{
	GError *error = NULL;
	gchar **plugins;
	
	plugins = capuchin_g_app_object_get_plugins_with_tag (appobject, "phone", &error);
	
	if (error != NULL) {
		g_critical ("Error while retrieving plugins with tag: %s", error->message);
		g_error_free (error);
	} else {
		g_printf ("PLUGINS WITH TAG:\n");
	
		gchar *pid;
		gint i=0;
		for (pid = plugins[0]; pid != NULL; pid = plugins[i])
		{
			_print_plugin_info (appobject, pid);
			i++;
		}
	
		g_strfreev (plugins);
	}
}

void
_print_plugin_info (CapuchinGAppObject *appobject, const gchar *plugin_id)
{
	GError *error = NULL;
	gchar *name;
	gchar *desc;
	gchar *version;
	gchar **tags;
	CapuchinGPluginAuthor *author;
	gchar *author_name;
	gchar *author_email;
	
	name = capuchin_g_app_object_get_plugin_name (appobject, plugin_id, &error);
	if (error != NULL) {
		g_critical ("Error while retrieving name: %s", error->message);
		g_error_free (error);
		return;
	}
	desc = capuchin_g_app_object_get_plugin_description (appobject, plugin_id, &error);
	if (error != NULL) {
		g_critical ("Error while retrieving description: %s", error->message);
		g_error_free (error);
		return;
	}
	version = capuchin_g_app_object_get_plugin_version (appobject, plugin_id, &error);
	if (error != NULL) {
		g_critical ("Error while retrieving version: %s", error->message);
		g_error_free (error);
		return;
	}
	tags = capuchin_g_app_object_get_plugin_tags (appobject, plugin_id, &error);
	if (error != NULL) {
		g_critical ("Error while retrieving tags: %s", error->message);
		g_error_free (error);
		return;
	}
	author = capuchin_g_app_object_get_plugin_author (appobject, plugin_id, &error);
	if (error != NULL) {
		g_critical ("Error while retrieving author: %s", error->message);
		g_error_free (error);
		return;
	}
	
	g_object_get (G_OBJECT (author),
				  "name", &author_name,
				  "email", &author_email,
				  NULL);
	
	g_printf ("ID: %s\n", plugin_id);
	g_printf ("Name: %s\n", name);
	g_printf ("Description: %s\n", desc);
	g_printf ("Version: %s\n", version);
	g_printf ("Author: %s <%s>\n", author_name, author_email);
	
	g_printf ("TAGS for %s:\n", plugin_id);
	gchar *tag;
	gint i=0;
	for (tag = tags[0]; tag != NULL; tag = tags[i])
	{
		g_printf ("%s\n", tag);
		i++;
	}
	
	g_printf ("\n");
	
	g_free (name);
	g_free (desc);
	g_free (version);
	g_strfreev (tags);
	g_free (author_name);
	g_free (author_email);
	g_object_unref (author);
}

static void
_test_install (CapuchinGAppObject *appobject)
{
	GError *error = NULL;

	capuchin_g_app_object_install (appobject, "ekiga.py", &error);
	
	if (error != NULL) {
		g_critical ("Error while installing: %s", error->message);
		g_error_free (error);
	}
}

static void
_update_finished_cb (CapuchinGAppObject *appobject, gpointer data)
{
    g_printf ("Update finished\n");
	
	_test_get_application_name (appobject);
	_test_get_tags (appobject);
	_test_get_available_plugins (appobject);
	_test_get_available_updates (appobject);
	_test_get_plugins_with_tag (appobject);
	
	_test_install (appobject);
}

static void
_status_cb (CapuchinGAppObject *appobject, CapuchinGAppObjectActionType type,
				   const gchar *plugin_id, gdouble progress, gint speed)
{
	if (type == CAPUCHIN_G_APP_OBJECT_ACTION_DOWNLOADING_PLUGIN)
		g_printf ("Downloading\n");
	else if (type == CAPUCHIN_G_APP_OBJECT_ACTION_EXTRACTING_PLUGIN)
		g_printf ("Extracting\n");
	g_printf ("Status: %s %d %i\n", plugin_id, progress, speed);
}

static void
_install_finished_cb (CapuchinGAppObject *appobject, const gchar *plugin_id)
{
	GError *error = NULL;
	g_printf ("Plugin %s installed\n", plugin_id);
	
	capuchin_g_app_object_close (appobject, &error);
	
	if (error != NULL) {
		g_critical ("Error while closing: %s", error->message);
		g_error_free (error);
	}
	
	g_object_unref (appobject);
	g_main_loop_quit (mainloop);
}

int
main (int argc, char **argv)
{
    CapuchinGAppObjectManager *manager;
    CapuchinGAppObject *appobject;
	GError *error = NULL;
    
    g_type_init ();
    mainloop = g_main_loop_new (NULL, FALSE);
    
    manager = capuchin_g_app_object_manager_new ();
	
	if (manager == NULL) g_critical ("Could not create AppObjectManager: %s", error->message);
    
    appobject = capuchin_g_app_object_manager_get_appobject (manager, REPO_URL,
															 &error);
	if (error != NULL) g_critical ("Could not create AppObject: %s", error->message);
		
	g_object_unref (manager);
	
    g_signal_connect (appobject,
                      "update-finished",
                      G_CALLBACK (_update_finished_cb),
                      NULL);
	g_signal_connect (appobject,
					  "status",
					  G_CALLBACK (_status_cb),
					  NULL);
	g_signal_connect (appobject,
					  "install-finished",
					  G_CALLBACK (_install_finished_cb),
					  NULL);
    
    capuchin_g_app_object_update_async (appobject, FALSE);
    
	// FIXME: unfref manager and appobject when done
    
    g_main_loop_run (mainloop);

    return 0;
}

