/* GAIL - The GNOME Accessibility Implementation Library
 * Copyright 2001 Sun Microsystems Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include "gailcontainer.h"
#include "gailcontainerfactory.h"

static void         gail_container_class_init          (GailContainerClass *klass);
static void         gail_container_object_init         (GailContainer      *container);

static gint         gail_container_get_n_children      (AtkObject          *obj);
static AtkObject*   gail_container_ref_child           (AtkObject          *obj,
                                                        gint               i);
static gint         gail_container_add_gtk             (GtkContainer       *container,
                                                        GtkWidget          *widget);
static gint         gail_container_remove_gtk          (GtkContainer       *container,
                                                        GtkWidget          *widget);

static void          gail_container_real_init          (GailWidget         *widget,
                                                        GtkWidget          *gtk_widget);

static void          gail_container_finalize           (GObject            *object);

static GailWidgetClass *parent_class = NULL;

GType
gail_container_get_type (void)
{
  static GType type = 0;

  if (!type)
  {
    static const GTypeInfo tinfo =
    {
      sizeof (GailContainerClass),
      (GBaseInitFunc) NULL, /* base init */
      (GBaseFinalizeFunc) NULL, /* base finalize */
      (GClassInitFunc) gail_container_class_init, /* class init */
      (GClassFinalizeFunc) NULL, /* class finalize */
      NULL, /* class data */
      sizeof (GailContainer), /* instance size */
      0, /* nb preallocs */
      (GInstanceInitFunc) gail_container_object_init, /* instance init */
      NULL /* value table */
    };

    type = g_type_register_static (GAIL_TYPE_WIDGET,
                                    "GailContainer", &tinfo, 0);
  }

  return type;
}

static void
gail_container_class_init (GailContainerClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
  AtkObjectClass *class = ATK_OBJECT_CLASS (klass);
  GailWidgetClass *widget_class;

  widget_class = (GailWidgetClass*)klass;

  parent_class = g_type_class_ref (GAIL_TYPE_WIDGET);

  gobject_class->finalize = gail_container_finalize;

  class->get_n_children = gail_container_get_n_children;
  class->ref_child = gail_container_ref_child;

  widget_class->init = gail_container_real_init;
}

static void
gail_container_object_init (GailContainer      *container)
{
  container->children = NULL;
}

GtkAccessible* 
gail_container_new (GtkWidget *widget)
{
  GObject *object;
  GtkAccessible *accessible;

  g_return_val_if_fail (GTK_IS_CONTAINER (widget), NULL);

  object = g_object_new (GAIL_TYPE_CONTAINER, NULL);

  g_return_val_if_fail (GTK_IS_ACCESSIBLE (object), NULL);

  gail_widget_init (GAIL_WIDGET (object), widget);

  accessible = GTK_ACCESSIBLE (object);
  if (GTK_IS_TOOLBAR (widget))
  {
    ATK_OBJECT(accessible)->role = ATK_ROLE_TOOL_BAR;
  }
  else if (GTK_IS_VIEWPORT (widget))
  {
    ATK_OBJECT(accessible)->role = ATK_ROLE_VIEWPORT;
  }
  else
  {
    ATK_OBJECT(accessible)->role = ATK_ROLE_PANEL;
  }
  return accessible;
}

static gint 
gail_container_get_n_children (AtkObject* obj)
{
  GtkWidget *widget;
  GList *children;
  gint count = 0;

  g_return_val_if_fail (GAIL_IS_CONTAINER (obj), count);

  widget = GTK_ACCESSIBLE (obj)->widget;
  if (widget == NULL)
  {
    return count;
  }

  children = gtk_container_children (GTK_CONTAINER(widget));
  count = g_list_length (children);
  g_list_free (children);

  return count; 
}

static AtkObject* 
gail_container_ref_child (AtkObject *obj,
                          gint       i)
{
  GList *children, *tmp_list;
  AtkObject  *accessible;
  GtkWidget *widget;

  g_return_val_if_fail (GAIL_IS_CONTAINER (obj), NULL);
  g_return_val_if_fail ((i >= 0), NULL);
  widget = GTK_ACCESSIBLE (obj)->widget;
  if (widget == NULL)
  {
    return NULL;
  }

  children = gtk_container_children (GTK_CONTAINER(widget));
  tmp_list = g_list_nth (children, i);
  accessible = gtk_widget_get_accessible (GTK_WIDGET (tmp_list->data));

  g_list_free (children);
  g_object_ref (accessible);
  return accessible; 
}

static gint
gail_container_add_gtk (GtkContainer       *container,
                        GtkWidget          *widget)
{
  AtkObject* atk_parent = gtk_widget_get_accessible (GTK_WIDGET (container));
  AtkObject* atk_child = gtk_widget_get_accessible (widget);
  GailContainer *gail_container = GAIL_CONTAINER (atk_parent);
  gint       index;

  g_object_notify (G_OBJECT (atk_child), "accessible_parent");

  g_list_free (gail_container->children);
  gail_container->children = gtk_container_children (container);
  index = g_list_index (gail_container->children, widget);
  g_signal_emit_by_name (atk_parent, "children_changed::add", 
                         index, atk_child, NULL);

  return 1;
}

static gint
gail_container_remove_gtk (GtkContainer       *container,
                           GtkWidget          *widget) 
{
  AtkPropertyValues values = { 0, };
  AtkObject *atk_parent = gtk_widget_get_accessible (GTK_WIDGET (container));
  AtkObject *atk_child = gtk_widget_get_accessible (widget);
  GailContainer *gail_container = GAIL_CONTAINER (atk_parent);
  gint       index;

  g_value_init (&values.old_value, G_TYPE_POINTER);
  g_value_set_pointer (&values.old_value, atk_parent);
    
  values.property_name = "accessible_parent";
  g_signal_emit_by_name (atk_child,
                         "property_change::accessible_parent", &values, NULL);

  index = g_list_index (gail_container->children, widget);
  g_list_free (gail_container->children);
  gail_container->children = gtk_container_children (container);
  g_signal_emit_by_name (atk_parent, "children_changed::remove", 
                         index, atk_child, NULL);

  return 1;
}

static void
gail_container_real_init (GailWidget         *widget,
                          GtkWidget          *gtk_widget)
{
  GailContainer *container = GAIL_CONTAINER (widget);
  parent_class->init (widget, gtk_widget);

  container->children = gtk_container_children (GTK_CONTAINER (gtk_widget));

  /*
   * We store the handler ids for these signals as some obejcts, e.g.
   * GailButton need to remove these handlers.
   */
  container->add_handler =
      g_signal_connect (G_OBJECT (gtk_widget),
                        "add",
                        G_CALLBACK (gail_container_add_gtk),
                        NULL);
  container->remove_handler =
      g_signal_connect (G_OBJECT (gtk_widget),
                        "remove",
                        G_CALLBACK (gail_container_remove_gtk),
                        NULL);
}

static void
gail_container_finalize (GObject            *object)
{
  GailContainer *container = GAIL_CONTAINER (object);

  g_list_free (container->children);
  G_OBJECT_CLASS (parent_class)->finalize (object);
}
