/* GAIL - The GNOME Accessibility Implementation Library
 * Copyright 2001 Sun Microsystems Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include "gailwidget.h"
#include "gailnotebookpage.h"

static void gail_widget_class_init (GailWidgetClass *klass);

static void gail_widget_connect_widget_destroyed (GtkAccessible    *accessible);
static void gail_widget_destroyed                (GtkWidget        *widget,
                                                  GtkAccessible    *accessible);

static G_CONST_RETURN gchar* gail_widget_get_name (AtkObject *accessible);
static G_CONST_RETURN gchar* gail_widget_get_description (AtkObject *accessible);
static AtkObject* gail_widget_get_parent (AtkObject *accessible);
static AtkRole gail_widget_get_role (AtkObject *accessible);
static AtkStateSet* gail_widget_ref_state_set (AtkObject *accessible);
static gint gail_widget_get_index_in_parent (AtkObject *accessible);

static void atk_component_interface_init (AtkComponentIface *iface);

static guint    gail_widget_add_focus_handler
                                           (AtkComponent    *component,
                                            AtkFocusHandler handler);

static void     gail_widget_get_extents    (AtkComponent    *component,
                                            gint            *x,
                                            gint            *y,
                                            gint            *width,
                                            gint            *height,
                                            AtkCoordType    coord_type);

static void     gail_widget_get_size       (AtkComponent    *component,
                                            gint            *width,
                                            gint            *height);

static gboolean gail_widget_grab_focus     (AtkComponent    *component);


static void     gail_widget_remove_focus_handler 
                                           (AtkComponent    *component,
                                            guint           handler_id);

static gboolean gail_widget_set_extents    (AtkComponent    *component,
                                            gint            x,
                                            gint            y,
                                            gint            width,
                                            gint            height,
                                            AtkCoordType    coord_type);

static gboolean gail_widget_set_position   (AtkComponent    *component,
                                            gint            x,
                                            gint            y,
                                            AtkCoordType    coord_type);

static gboolean gail_widget_set_size       (AtkComponent    *component,
                                            gint            width,
                                            gint            height);

#if 0
/*
 * We will get the parent from the widget structure rather than
 * use this function
 */
static GtkWidget* _gail_widget_get_parent        (GtkWidget     *widget);
#endif

static gint       gail_widget_map_gtk            (GtkWidget     *widget);
static void       gail_widget_real_notify_gtk    (GObject       *obj,
                                                  GParamSpec    *pspec);
static void       gail_widget_notify_gtk         (GObject       *obj,
                                                  GParamSpec    *pspec);
static gboolean   gail_widget_focus_gtk          (GtkWidget     *widget,
                                                  GdkEventFocus *event);
static gboolean   gail_widget_real_focus_gtk     (GtkWidget     *widget,
                                                  GdkEventFocus *event);

static void       gail_widget_focus_event        (AtkObject     *obj,
                                                  gboolean      focus_in);

static void       gail_widget_real_init          (GailWidget    *widget,
                                                  GtkWidget     *gtk_widget);
static GtkWidget* gail_widget_find_viewport      (GtkWidget     *widget);
static gboolean   gail_widget_on_screen          (GtkWidget     *widget);

static gpointer parent_class = NULL;

GType
gail_widget_get_type (void)
{
  static GType type = 0;

  if (!type)
  {
    static const GTypeInfo tinfo =
    {
      sizeof (GailWidgetClass),
      (GBaseInitFunc) NULL, /* base init */
      (GBaseFinalizeFunc) NULL, /* base finalize */
      (GClassInitFunc) gail_widget_class_init, /* class init */
      (GClassFinalizeFunc) NULL, /* class finalize */
      NULL, /* class data */
      sizeof (GailWidget), /* instance size */
      0, /* nb preallocs */
      (GInstanceInitFunc) NULL, /* instance init */
      NULL /* value table */
    };

    static const GInterfaceInfo atk_component_info =
    {
        (GInterfaceInitFunc) atk_component_interface_init,
        (GInterfaceFinalizeFunc) NULL,
        NULL
    };


    type = g_type_register_static (GTK_TYPE_ACCESSIBLE,
                                    "GailWidget", &tinfo, 0);
    g_type_add_interface_static (type, ATK_TYPE_COMPONENT,
                                 &atk_component_info);
  }

  return type;
}

static void
gail_widget_class_init (GailWidgetClass *klass)
{
  AtkObjectClass *class = ATK_OBJECT_CLASS (klass);
  GtkAccessibleClass *accessible_class = GTK_ACCESSIBLE_CLASS (klass);

  parent_class = g_type_class_ref (GTK_TYPE_ACCESSIBLE);

  klass->init = gail_widget_real_init;
  klass->notify_gtk = gail_widget_real_notify_gtk;
  klass->focus_gtk = gail_widget_real_focus_gtk;

  accessible_class->connect_widget_destroyed = gail_widget_connect_widget_destroyed;

  class->get_name = gail_widget_get_name;
  class->get_description = gail_widget_get_description;
  class->get_parent = gail_widget_get_parent;
  class->get_role = gail_widget_get_role;
  class->ref_state_set = gail_widget_ref_state_set;
  class->get_index_in_parent = gail_widget_get_index_in_parent;
}

/**
 * gail_widget_init:
 * @widget: a #GailWidget
 * @gtk_widget: a #GtkWidget for which the GailWidget was created.
 *
 * This function is called when implementing subclasses of #GailWidget.
 * It does initialization required for the new object. It is intended 
 * that this function is called only in the gail_..._new() methods used 
 * to create an instance of a subclass of #GailWidget
 **/
void 
gail_widget_init (GailWidget *widget,
                  GtkWidget  *gtk_widget)
{
  GailWidgetClass *klass;
  g_return_if_fail (GAIL_IS_WIDGET (widget));

  klass = GAIL_WIDGET_GET_CLASS (widget);
  if (klass->init)
    klass->init (widget, gtk_widget); 
}

/**
 * This function  specifies the GtkWidget for which the GailWidget was created 
 * and specifies a handler to be called when the GtkWidget is destroyed.
 **/
static void 
gail_widget_real_init  (GailWidget *widget,
                        GtkWidget  *gtk_widget)
{
  GtkAccessible *accessible;

  accessible = GTK_ACCESSIBLE (widget);
  accessible->widget = gtk_widget;
  gtk_accessible_connect_widget_destroyed (accessible);
  g_signal_connect_after (GTK_OBJECT (gtk_widget),
                          "focus-in-event",
                          G_CALLBACK (gail_widget_focus_gtk),
                          NULL);
  g_signal_connect_after (G_OBJECT (gtk_widget),
                          "focus-out-event",
                          G_CALLBACK (gail_widget_focus_gtk),
                          NULL);
  g_signal_connect (G_OBJECT (gtk_widget),
                    "notify",
                    G_CALLBACK (gail_widget_notify_gtk),
                    widget);
  /*
   * Support state property change for focus events
   */
  atk_component_add_focus_handler (ATK_COMPONENT (accessible),
                                   gail_widget_focus_event);
  /*
   * Add signal handlers for GTK signals required to support property changes
   */
  g_signal_connect (G_OBJECT (gtk_widget),
                    "map",
                    G_CALLBACK (gail_widget_map_gtk),
                    NULL);
  g_signal_connect (G_OBJECT (gtk_widget),
                    "unmap",
                    G_CALLBACK (gail_widget_map_gtk),
                    NULL);
}

GtkAccessible* 
gail_widget_new (GtkWidget *widget)
{
  GObject *object;
  GtkAccessible *accessible;

  g_return_val_if_fail (GTK_IS_WIDGET (widget), NULL);

  object = g_object_new (GAIL_TYPE_WIDGET, NULL);

  g_return_val_if_fail (GTK_IS_ACCESSIBLE (object), NULL);

  gail_widget_init (GAIL_WIDGET (object), widget);

  accessible = GTK_ACCESSIBLE (object);
  ATK_OBJECT(accessible)->role = ATK_ROLE_UNKNOWN;

  return accessible;
}

/*
 * This function specifies the function to be called when the widget
 * is destroyed
 */
static void
gail_widget_connect_widget_destroyed (GtkAccessible *accessible)
{
  if (accessible->widget)
  {
    g_signal_connect (G_OBJECT (accessible->widget),
                      "destroy",
                      G_CALLBACK (gail_widget_destroyed),
                      accessible);
  }
}

/*
 * This function is called when the widget is destroyed.
 * It sets the widget field in the GtkAccessible structure to NULL
 * and signals a property change event for "accessible-state" with
 * the new state being ATK_STATE_DEFUNCT
 */
static void 
gail_widget_destroyed (GtkWidget     *widget,
                       GtkAccessible *accessible)
{
  atk_object_notify_state_change (ATK_OBJECT (accessible), ATK_STATE_DEFUNCT,
                                  TRUE); 
}

static G_CONST_RETURN gchar*
gail_widget_get_name (AtkObject *accessible)
{
  if (accessible->name)
  {
    return accessible->name;
  }
  else
  {
    /*
     * Get the widget name is it exists
     */
    GtkWidget* widget = GTK_ACCESSIBLE (accessible)->widget;

    if (widget == NULL)
    {
      /*
       * State is defunct
       */
      return NULL;
    }
    g_return_val_if_fail (GTK_IS_WIDGET (widget), NULL);

    return widget->name;
  }
}

static G_CONST_RETURN gchar*
gail_widget_get_description (AtkObject *accessible)
{
  if (accessible->description)
  {
    return accessible->description;
  }
  else
  {
    /* Get the tooltip from the widget */
    GtkAccessible *obj = GTK_ACCESSIBLE (accessible);
    GtkTooltipsData *data;

    g_return_val_if_fail (obj, NULL);

    if (obj->widget == NULL)
    {
      /*
       * Object is defunct
       */
      return NULL;
    }
    g_return_val_if_fail (GTK_WIDGET (obj->widget), NULL);
    
    data = gtk_tooltips_data_get(obj->widget);
    if (data == NULL)
    {
        return NULL;
    }

    return data->tip_text;
  }
}

static AtkObject* 
gail_widget_get_parent (AtkObject *accessible)
{
  AtkObject *parent;

  parent = accessible->accessible_parent;

  if (parent != NULL)
  {
    g_return_val_if_fail (ATK_IS_OBJECT (parent), NULL);
  }
  else
  {
    GtkWidget *widget, *parent_widget;

    widget = GTK_ACCESSIBLE (accessible)->widget;
    if (widget == NULL)
    {
      /*
       * State is defunct
       */
      return NULL;
    }
    parent_widget = widget->parent;
    if (parent_widget == NULL)
      return NULL;

    parent = gtk_widget_get_accessible (parent_widget);
  }
  return parent;
}

static AtkRole 
gail_widget_get_role (AtkObject *accessible)
{
  return accessible->role;
}

static AtkStateSet*
gail_widget_ref_state_set (AtkObject *accessible)
{
  GtkWidget *widget = GTK_ACCESSIBLE (accessible)->widget;
  AtkStateSet *state_set;

  state_set = ATK_OBJECT_CLASS (parent_class)->ref_state_set (accessible);

  if (widget == NULL)
  {
    atk_state_set_add_state (state_set, ATK_STATE_DEFUNCT);
  }
  else
  {
    if (GTK_WIDGET_IS_SENSITIVE (widget))
    {
      atk_state_set_add_state (state_set, ATK_STATE_SENSITIVE);
    }
  
    if (GTK_WIDGET_CAN_FOCUS (widget))
    {
      atk_state_set_add_state (state_set, ATK_STATE_FOCUSABLE);
    }
    /*
     * We do not currently generate notifications when an ATK object 
     * corresponding to a GtkWidget changes visibility by being scrolled 
     * on or off the screen.  The testcase for this is the main window 
     * of the testgtk application in which a set of buttons in a GtkVBox 
     * is in a scrooled window with a viewport.
     *
     * To generate the notifications we would need to do the following: 
     * 1) Find the GtkViewPort among the antecendents of the objects
     * 2) Create an accesible for the GtkViewPort
     * 3) Connect to the value-changed signal on the viewport
     * 4) When the signal is received we need to traverse the children 
     * of the viewport and check whether the children are visible or not 
     * visible; we may want to restrict this to the widgets for which 
     * accessible objects have been created.
     * 5) We probably need to store a variable on_screen in the 
     * GailWidget data structure so we can determine whether the value has 
     * changed.
     */
    if (gail_widget_on_screen (widget))
    {
      if (GTK_WIDGET_VISIBLE (widget))
      {
        atk_state_set_add_state (state_set, ATK_STATE_VISIBLE);

        if (GTK_WIDGET_MAPPED (widget))
        {
          atk_state_set_add_state (state_set, ATK_STATE_SHOWING);
        }
      }
    }

    if (GTK_WIDGET_HAS_FOCUS (widget))
    {
      atk_state_set_add_state (state_set, ATK_STATE_FOCUSED);
    }
    atk_state_set_add_state (state_set, ATK_STATE_ENABLED);
  }
  return state_set;
}

static gint
gail_widget_get_index_in_parent (AtkObject *accessible)
{
  GtkWidget *widget, *parent_widget;
  gint index = -1;
  GList *children, *tmp_list;

  if (accessible->accessible_parent)
  {
    if (GAIL_IS_NOTEBOOK_PAGE (accessible->accessible_parent))
      return 0;
  }

  widget = GTK_ACCESSIBLE (accessible)->widget;

  if (widget == NULL)
  {
    /*
     * State is defunct
     */
    return -1;
  }
  parent_widget = widget->parent;
  g_return_val_if_fail (GTK_IS_CONTAINER (parent_widget), -1);

  tmp_list = children = gtk_container_children (GTK_CONTAINER (parent_widget));

  while (tmp_list)
  {
    if (GTK_WIDGET (tmp_list->data) == widget)
    {
      index++;
      g_list_free (children);
      return index;
    }
    tmp_list = g_list_next (tmp_list);
    index++;
  }
  g_list_free (children);
  return -1;  
}

static void 
atk_component_interface_init (AtkComponentIface *iface)
{
  g_return_if_fail (iface != NULL);

  /*
   * Use default implementation for contains and get_position
   */
  iface->add_focus_handler = gail_widget_add_focus_handler;
  iface->get_extents = gail_widget_get_extents;
  iface->get_size = gail_widget_get_size;
  iface->grab_focus = gail_widget_grab_focus;
  iface->remove_focus_handler = gail_widget_remove_focus_handler;
  iface->set_extents = gail_widget_set_extents;
  iface->set_position = gail_widget_set_position;
  iface->set_size = gail_widget_set_size;
}

static guint 
gail_widget_add_focus_handler (AtkComponent    *component,
                               AtkFocusHandler handler)
{
  return g_signal_connect_closure (component, 
                                   "focus-event",
                                   g_cclosure_new (
                                   G_CALLBACK (handler), NULL,
                                   (GClosureNotify) NULL),
                                   FALSE);
}

static void 
gail_widget_get_extents (AtkComponent   *component,
                         gint           *x,
                         gint           *y,
                         gint           *width,
                         gint           *height,
                         AtkCoordType   coord_type)
{
  GdkWindow *window;
  gint x_parent, y_parent, x_toplevel, y_toplevel;
  GtkWidget *widget = GTK_ACCESSIBLE (component)->widget;

  if (widget == NULL)
  {
    /*
     * Object is defunct
     */
    return;
  }

  *width = widget->allocation.width;
  *height = widget->allocation.height;
  if (!gail_widget_on_screen (widget) || (!GTK_WIDGET_DRAWABLE (widget)))
  {
    *x = G_MININT;
    *y = G_MININT;
    return;
  }

  if (widget->parent)
  {
    window = gtk_widget_get_parent_window(widget);
  }
  else
  {
    window = widget->window;
  }
  {
    int x, y, width, height, depth;
    gdk_window_get_position (window, &x, &y);
    gdk_window_get_geometry (window, &x, &y, &width, &height, &depth);
  }
  gdk_window_get_origin(window, &x_parent, &y_parent);
  window = gdk_window_get_toplevel(widget->window);
  gdk_window_get_origin(window, &x_toplevel, &y_toplevel);

  if (coord_type == ATK_XY_SCREEN) 
  { 
    *x = widget->allocation.x + x_parent;
    *y = widget->allocation.y + y_parent;
  }
  else if (coord_type == ATK_XY_WINDOW) 
  { 
    *x = widget->allocation.x + (x_parent - x_toplevel);
    *y = widget->allocation.y + (y_parent - y_toplevel);
  }
}

static void 
gail_widget_get_size    (AtkComponent   *component,
                         gint           *width,
                         gint           *height)
{
  GtkWidget *widget = GTK_ACCESSIBLE (component)->widget;

  if (widget == NULL)
  {
    /*
     * Object is defunct
     */
    return;
  }
  *width = widget->allocation.width;
  *height = widget->allocation.height;
}

static gboolean 
gail_widget_grab_focus(AtkComponent   *component)
{
  GtkWidget *widget = GTK_ACCESSIBLE (component)->widget;

  g_return_val_if_fail (widget != NULL, FALSE);
  if (GTK_WIDGET_CAN_FOCUS(widget))
  {
    gtk_widget_grab_focus (widget);
    return TRUE;
  }
  else
    return FALSE;
}

static void 
gail_widget_remove_focus_handler (AtkComponent   *component,
                                  guint          handler_id)
{
  g_signal_handler_disconnect (ATK_OBJECT (component), handler_id);
}

static gboolean 
gail_widget_set_extents (AtkComponent   *component,
                         gint           x,
                         gint           y,
                         gint           width,
                         gint           height,
                         AtkCoordType   coord_type)
{
  GtkWidget *widget = GTK_ACCESSIBLE (component)->widget;

  if (widget == NULL)
  {
    /*gtkwindow.c
     * Object is defunct
     */
    return FALSE;
  }
  if (GTK_WIDGET_TOPLEVEL(widget))
  {
    if (coord_type == ATK_XY_WINDOW)
    {
      gint x_current, y_current;
      GdkWindow *window = widget->window;
      gdk_window_get_origin(window, &x_current, &y_current);
      x_current += x;
      y_current += y;
      if (x_current < 0 || y_current < 0)
        return FALSE;
      else
      {
        gtk_widget_set_uposition (widget, x_current, y_current);
        gtk_widget_set_usize (widget, width, height);
        return TRUE;
      }
    }
    else if (coord_type == ATK_XY_SCREEN)
    {  
      gtk_widget_set_uposition (widget, x, y);
      gtk_widget_set_usize (widget, width, height);
      return TRUE;
    }
  }
  return FALSE;
}

static gboolean
gail_widget_set_position(AtkComponent   *component,
                         gint           x,
                         gint           y,
                         AtkCoordType   coord_type)
{
  GtkWidget *widget = GTK_ACCESSIBLE (component)->widget;

  if (widget == NULL)
  {
    /*
     * Object is defunct
     */
    return FALSE;
  }
  if (GTK_WIDGET_TOPLEVEL(widget))
  {
    if (coord_type == ATK_XY_WINDOW)
    {
      gint x_current, y_current;
      GdkWindow *window = widget->window;
      gdk_window_get_origin(window, &x_current, &y_current);
      x_current += x;
      y_current += y;
      if (x_current < 0 || y_current < 0)
        return FALSE;
      else
      {
        gtk_widget_set_uposition (widget, x_current, y_current);
        return TRUE;
      }
    }
    else if (coord_type == ATK_XY_SCREEN)
    {  
      gtk_widget_set_uposition (widget, x, y);
      return TRUE;
    }
  }
  return FALSE;
}

static gboolean 
gail_widget_set_size    (AtkComponent   *component,
                         gint           width,
                         gint           height)
{
  GtkWidget *widget = GTK_ACCESSIBLE (component)->widget;

  if (widget == NULL)
  {
    /*
     * Object is defunct
     */
    return FALSE;
  }
  if (GTK_WIDGET_TOPLEVEL(widget))
  {
    gtk_widget_set_usize (widget, width, height);
    return TRUE;
  }
  else
   return FALSE;
}

#if 0
static GtkWidget*
_gail_widget_get_parent (GtkWidget* widget)
{
  GtkWidget *parent_widget;
  gpointer pointer;
  GValue value = { 0, };

  g_return_val_if_fail (GTK_WIDGET (widget), NULL);

  g_value_init (&value, G_TYPE_OBJECT);
  g_object_get_property (G_OBJECT (widget), "parent", &value);
  pointer = g_value_get_object (&value);
 
  g_return_val_if_fail (GTK_IS_WIDGET (pointer), NULL);

  return GTK_WIDGET (pointer);
}
#endif


/*
 * This function is a signal handler for notify_in_event and focus_out_event
 * signal which gets emitted on a GtkWidget.
 */
static gboolean
gail_widget_focus_gtk (GtkWidget     *widget,
                       GdkEventFocus *event)
{
  GailWidget *gail_widget;
  GailWidgetClass *klass;

  gail_widget = GAIL_WIDGET (gtk_widget_get_accessible (widget));
  klass = GAIL_WIDGET_GET_CLASS (gail_widget);
  if (klass->focus_gtk)
    return klass->focus_gtk (widget, event);
  else
    return FALSE;
}

/*
 * This function is the signal handler defined for focus_in_event and
 * focus_out_event got GailWidget.
 *
 * It emits a focus-event signal on the GailWidget.
 */
static gboolean
gail_widget_real_focus_gtk (GtkWidget     *widget,
                            GdkEventFocus *event)
{
  AtkObject* accessible;
  gboolean return_val;
  return_val = FALSE;

  accessible = gtk_widget_get_accessible (widget);
  g_signal_emit_by_name (accessible, "focus_event", event->in, &return_val);
  return FALSE;
}

/*
 * This function is the signal handler defined for map and unmap signals.
 */
static gint
gail_widget_map_gtk (GtkWidget     *widget)
{
  AtkObject* accessible;

  accessible = gtk_widget_get_accessible (widget);
  atk_object_notify_state_change (accessible, ATK_STATE_SHOWING,
                                  GTK_WIDGET_MAPPED (widget));
  return 1;
}

/*
 * This function is a signal handler for notify signal which gets emitted 
 * when a property changes value on the GtkWidget associated with the object.
 *
 * It calls a function for the GailWidget type
 */
static void 
gail_widget_notify_gtk (GObject     *obj,
                        GParamSpec  *pspec)
{
  GailWidget *widget;
  GailWidgetClass *klass;

  widget = GAIL_WIDGET (gtk_widget_get_accessible (GTK_WIDGET (obj)));
  klass = GAIL_WIDGET_GET_CLASS (widget);
  if (klass->notify_gtk)
    klass->notify_gtk (obj, pspec);
}

/*
 * This function is a signal handler for notify signal which gets emitted 
 * when a property changes value on the GtkWidget associated with a GailWidget.
 *
 * It constructs an AtkPropertyValues structure and emits a "property_changed"
 * signal which causes the user specified AtkPropertyChangeHandler
 * to be called.
 */
static void 
gail_widget_real_notify_gtk (GObject     *obj,
                             GParamSpec  *pspec)
{
  GtkWidget* widget = GTK_WIDGET (obj);
  AtkObject* atk_obj = gtk_widget_get_accessible (widget);
  AtkState state;
  gboolean value;

  if (strcmp (pspec->name, "visible") == 0)
    {
      state = ATK_STATE_VISIBLE;
      value = GTK_WIDGET_VISIBLE (widget);
    }
  else if (strcmp (pspec->name, "has-focus") == 0)
    /*
     * We use focus-in-event and focus-out-event signals to catch
     * focus changes so we ignore this.
     */
    return;
  else if (strcmp (pspec->name, "sensitive") == 0)
    {
      state = ATK_STATE_SENSITIVE;
      value = GTK_WIDGET_SENSITIVE (widget);
    }
  else
    return;

  atk_object_notify_state_change (atk_obj, state, value);
}

/*
 * This function is the signal handler defined for focus event on
 * AtkObject by the property change mechanism so that we get a
 * property change for accessible-state when focus changes.
 */
static void 
gail_widget_focus_event (AtkObject   *obj,
                         gboolean    focus_in)
{
  atk_object_notify_state_change (obj, ATK_STATE_FOCUSED, focus_in);
}

/*
 * The following function is general purpose, so it might make sense to move
 * it to a general purpose utility file at some point.
 */

/**
 * gail_widget_get_origins:
 * @widget: a #GtkWidget
 * @x_widget: the x-origin of the widget->window
 * @y_widget: the y-origin of the widget->window
 * @x_window: the top-level x-origin of the widget->window
 * @y_window: the top-level y-origin of the widget->window
 *
 * Gets the origin of the widget window, and the origin of the
 * widgets top-level window.
 **/
void
gail_widget_get_origins(GtkWidget	*widget,
                             gint	*x_widget,
                             gint	*y_widget,
                             gint	*x_window,
                             gint	*y_window)
{
  GdkWindow *window;

  gdk_window_get_origin(GDK_WINDOW(widget->window),
    x_widget, y_widget);
  window = gdk_window_get_toplevel(GDK_WINDOW(widget->window));
  gdk_window_get_origin(window, x_window, y_window);
}

static GtkWidget*
gail_widget_find_viewport (GtkWidget *widget)
{
  /*
   * Find an antecedent which is a GtkViewPort
   */
  GtkWidget *parent;

  parent = widget->parent;
  while (parent != NULL)
  {
    if (GTK_IS_VIEWPORT (parent))
      break;
    parent = parent->parent;
  }
  return parent;
}

/*
 * This function checks whether the widget has an antecedent which is 
 * a GtkViewport and, if so, whether any part of the widget intersects
 * the visible rectangle of the GtkViewport.
 */ 
static gboolean gail_widget_on_screen (GtkWidget *widget)
{
  GtkWidget *viewport;
  gboolean return_value;

  viewport = gail_widget_find_viewport (widget);
  if (viewport)
  {
    GtkAdjustment *adjustment;
    GdkRectangle visible_rect;

    adjustment = gtk_viewport_get_vadjustment (GTK_VIEWPORT (viewport));
    visible_rect.y = adjustment->value;
    adjustment = gtk_viewport_get_hadjustment (GTK_VIEWPORT (viewport));
    visible_rect.x = adjustment->value;
    visible_rect.width = viewport->allocation.width;
    visible_rect.height = viewport->allocation.height;
             
    if (((widget->allocation.x + widget->allocation.width) < visible_rect.x) ||
       ((widget->allocation.y + widget->allocation.height) < visible_rect.y) ||
       (widget->allocation.x > (visible_rect.x + visible_rect.width)) ||
       (widget->allocation.y > (visible_rect.y + visible_rect.height)))
    {
      return_value = FALSE;
    }
    else
    {
      return_value = TRUE;
    }
  }
  else
  {
    return_value = TRUE;
  }
  return return_value;
}
