/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2003 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include <pthread.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <unistd.h>
#include "gnome-cmd-includes.h"
#include "gnome-cmd-cmdline.h"
#include "gnome-cmd-file-selector-types.h"
#include "gnome-cmd-file-selector-funcs.h"
#include "useractions.h"
#include "gnome-cmd-main-win-funcs.h"
#include "gnome-cmd-main-menu.h"
#include "gnome-cmd-data.h"
#include "gnome-cmd-style.h"
#include "gnome-cmd-combo.h"
#include "gnome-cmd-dir-funcs.h"
#include "gnome-cmd-connection-funcs.h"
#include "gnome-cmd-bookmark-dialog.h"
#include "dir_pool.h"
#include "utils.h"

#include "../pixmaps/ftp_connect.xpm"


enum {
  SWITCH_FS,
  LAST_SIGNAL
};

typedef enum {
	LEFT,
	RIGHT
} FileSelectorID;


struct _GnomeCmdMainWinPrivate
{
	FileSelectorID current_fs;
	
	GtkWidget *main_win;
	GtkWidget *vbox;
	GtkWidget *hpaned;
	GtkWidget *file_selector[2];
	GtkWidget *cmdline;
	GtkAccelGroup *accel_group;

	GtkWidget *back_btn;
	GtkWidget *forward_btn;
	
	GtkWidget *con_drop_btn;

	GtkWidget *view_btn;
	GtkWidget *edit_btn;
	GtkWidget *copy_btn;
	GtkWidget *move_btn;
	GtkWidget *mkdir_btn;
	GtkWidget *delete_btn;
	GtkWidget *find_btn;
	GtkWidget *quit_btn;

	GtkWidget *menubar;
	GtkWidget *toolbar;
	GtkWidget *toolbar_sep;
	GtkWidget *buttonbar;
	GtkWidget *buttonbar_sep;
};

static GnomeAppClass *parent_class = NULL;

static guint main_win_signals[LAST_SIGNAL] = { 0 };
static GtkTooltips *toolbar_tooltips = NULL;
extern gchar *start_dir; //main.c

static void
gnome_cmd_main_win_real_switch_fs (GnomeCmdMainWin *mw, GnomeCmdFileSelector *fs);



static GtkWidget *
add_buttonbar_button (char *label,
					  GnomeCmdMainWin *mw,
					  char *data_label,
					  GtkAccelGroup *accel_group,
					  gint accel_signal_id)
{
	GtkWidget *button;

	button = create_styled_button (label);
	gtk_object_set_data_full (GTK_OBJECT (main_win),
							  data_label,
							  button,
							  (GtkDestroyNotify) gtk_widget_unref);
	GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	gtk_widget_modify_style (GTK_BIN (button)->child, main_rc_style);

	gtk_box_pack_start (GTK_BOX (mw->priv->buttonbar), button, TRUE, TRUE, 0);
	gtk_widget_add_accelerator (button, "clicked",
								accel_group, accel_signal_id,
								0, GTK_ACCEL_VISIBLE);
	return button;
}


static GtkWidget *
create_separator (gboolean vertical)
{
	GtkWidget *sep;
	GtkWidget *box;

	if (vertical) {
		sep = gtk_vseparator_new ();
		box = gtk_vbox_new (FALSE, 0);
	}
	else {
		sep = gtk_hseparator_new ();
		box = gtk_hbox_new (FALSE, 0);
	}

	gtk_widget_ref (sep);
	gtk_widget_show (sep);

	gtk_widget_ref (box);
	gtk_widget_show (box);

	gtk_box_pack_start (GTK_BOX (box), sep, TRUE, TRUE, 3);

	return box;
}


static void
create_toolbar (GnomeCmdMainWin *mw, GnomeUIInfo *uiinfo)
{
	gint i;

	mw->priv->toolbar = gtk_hbox_new (FALSE, 0);
	gtk_widget_ref (mw->priv->toolbar);
	gtk_object_set_data_full (GTK_OBJECT (mw), "toolbar", mw->priv->toolbar,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->toolbar);
	
	if (!toolbar_tooltips)
		toolbar_tooltips = gtk_tooltips_new ();
	
	for ( i=0 ; uiinfo[i].type != GNOME_APP_UI_ENDOFINFO ; i++ ) {
		GtkWidget *w;
		
		if (uiinfo[i].type == GNOME_APP_UI_SEPARATOR) {
			w = create_separator (TRUE);
		}
		else {
			GtkWidget *pixmap;
			
			w = create_styled_button (NULL);
			gtk_signal_connect (GTK_OBJECT (w), "clicked",
								uiinfo[i].moreinfo, uiinfo[i].user_data);
			gtk_tooltips_set_tip (
				toolbar_tooltips, w, uiinfo[i].hint, NULL);
			GTK_WIDGET_UNSET_FLAGS (w, GTK_CAN_FOCUS);			
			
			pixmap = create_ui_pixmap (GTK_WIDGET (mw), uiinfo[i].pixmap_type, uiinfo[i].pixmap_info);
			if (pixmap) {
				gtk_widget_ref (pixmap);
				gtk_widget_show (pixmap);			
				gtk_container_add (GTK_CONTAINER (w), pixmap);
			}
			
			switch (i)
			{
			  case  1:  mw->priv->back_btn = w;  break;
			  case  2:  mw->priv->forward_btn = w;  break;
			  case 11:  mw->priv->con_drop_btn = w;  break;
			}
		}

		gtk_widget_show (w);
		gtk_box_pack_start (GTK_BOX (mw->priv->toolbar), w, FALSE, TRUE, 2);
	}

	mw->priv->toolbar_sep = create_separator (FALSE);
}


static void
slide_set_100_0 (GtkMenu *menu, gpointer user_data)
{
	gtk_paned_set_position (GTK_PANED (main_win->priv->hpaned),
							GTK_WIDGET (main_win)->allocation.width);	
}


static void
slide_set_80_20 (GtkMenu *menu, gpointer user_data)
{
	gtk_paned_set_position (GTK_PANED (main_win->priv->hpaned),
							(int)(GTK_WIDGET (main_win)->allocation.width*0.8f));	
}


static void
slide_set_60_40 (GtkMenu *menu, gpointer user_data)
{
	gtk_paned_set_position (GTK_PANED (main_win->priv->hpaned),
							(int)(GTK_WIDGET (main_win)->allocation.width*0.6f));	
}


static void
slide_set_50_50 (GtkMenu *menu, gpointer user_data)
{
	gtk_paned_set_position (GTK_PANED (main_win->priv->hpaned),
							(int)(GTK_WIDGET (main_win)->allocation.width*0.5f));	
}


static void
slide_set_40_60 (GtkMenu *menu, gpointer user_data)
{
	gtk_paned_set_position (GTK_PANED (main_win->priv->hpaned),
							(int)(GTK_WIDGET (main_win)->allocation.width*0.4f));	
}


static void
slide_set_20_80 (GtkMenu *menu, gpointer user_data)
{
	gtk_paned_set_position (GTK_PANED (main_win->priv->hpaned),
							(int)(GTK_WIDGET (main_win)->allocation.width*0.2f));	
}


static void
slide_set_0_100 (GtkMenu *menu, gpointer user_data)
{
	gtk_paned_set_position (GTK_PANED (main_win->priv->hpaned), 0);	
}


static GtkWidget *
create_slide_popup ()
{
	gint i;
	GtkWidget *menu;
	
	static GnomeUIInfo popmenu_uiinfo[] =
	{
		{
			GNOME_APP_UI_ITEM, "100 - 0",
			NULL,
			slide_set_100_0, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, "80 - 20",
			NULL,
			slide_set_80_20, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, "60 - 40",
			NULL,
			slide_set_60_40, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, "50 - 50",
			NULL,
			slide_set_50_50, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, "40 - 60",
			NULL,
			slide_set_40_60, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, "20 - 80",
			NULL,
			slide_set_20_80, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, "0 - 100",
			NULL,
			slide_set_0_100, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		}
	};

	/* Set default callback data 
	 */
	i = 0;
	while (popmenu_uiinfo[i].type != GNOME_APP_UI_ENDOFINFO) {
		if (popmenu_uiinfo[i].type == GNOME_APP_UI_ITEM) {
			popmenu_uiinfo[i].user_data = main_win;
		}
		i++;
	}

	menu = gtk_menu_new ();
	gtk_widget_ref (menu);
	gtk_object_set_data_full (GTK_OBJECT (main_win),
							  "slide-popup", menu,
							  (GtkDestroyNotify)gtk_widget_unref);
	
	/* Fill the menu
	 */
	gnome_app_fill_menu (GTK_MENU_SHELL (menu), popmenu_uiinfo,
						 NULL, FALSE, 0);

	return GTK_WIDGET (menu);	
}


/*****************************************************************************
	Buttonbar callbacks
*****************************************************************************/

static void
on_help_clicked                        (GtkButton       *button,
                                        GnomeCmdMainWin *mw)
{
	help_help (NULL, mw);
}


static void
on_refresh_clicked                        (GtkButton       *button,
									   GnomeCmdMainWin *mw)
{
	view_refresh (NULL, mw);
}


static void
on_view_clicked                        (GtkButton       *button,
                                        GnomeCmdMainWin *mw)
{
	file_view (NULL, mw);
}


static void
on_edit_clicked                        (GtkButton       *button,
                                        GnomeCmdMainWin *mw)
{
	file_edit (NULL, mw);
}


static void
on_copy_clicked                        (GtkButton       *button,
                                        GnomeCmdMainWin *mw)
{
	file_copy (NULL, mw);
}


static void
on_move_clicked                        (GtkButton       *button,
                                        GnomeCmdMainWin *mw)
{
	file_move (NULL, mw);
}


static void
on_mkdir_clicked                       (GtkButton       *button,
                                        GnomeCmdMainWin *mw)
{
	file_mkdir (NULL, mw);
}


static void
on_delete_clicked                      (GtkButton       *button,
                                        GnomeCmdMainWin *mw)
{
	file_delete (NULL, mw);
}


static void
on_search_clicked                      (GtkButton       *button,
                                        GnomeCmdMainWin *mw)
{
	edit_search (NULL, mw);
}


static void
on_quit_clicked                        (GtkButton       *button,
                                        GnomeCmdMainWin *mw)
{
    gtk_main_quit ();
}


static void
create_buttonbar (GnomeCmdMainWin *mw)
{
	mw->priv->buttonbar_sep = create_separator (FALSE);
	
	mw->priv->buttonbar = gtk_hbox_new (FALSE, 0);
	gtk_widget_ref (mw->priv->buttonbar);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "buttonbar", mw->priv->buttonbar,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->buttonbar);

	mw->priv->view_btn = add_buttonbar_button(
		_("F3 View"), main_win, "view_btn", mw->priv->accel_group, 0);
	gtk_box_pack_start (GTK_BOX (mw->priv->buttonbar), create_separator (TRUE), FALSE, TRUE, 0);
	mw->priv->edit_btn = add_buttonbar_button(
		_("F4 Edit"), main_win, "edit_btn", mw->priv->accel_group, 0);
	gtk_box_pack_start (GTK_BOX (mw->priv->buttonbar), create_separator (TRUE), FALSE, TRUE, 0);
	mw->priv->copy_btn = add_buttonbar_button(
		_("F5 Copy"), main_win, "copy_btn", mw->priv->accel_group, 0);
	gtk_box_pack_start (GTK_BOX (mw->priv->buttonbar), create_separator (TRUE), FALSE, TRUE, 0);
	mw->priv->move_btn = add_buttonbar_button(
		_("F6 Move"), main_win, "move_btn", mw->priv->accel_group, 0);
	gtk_box_pack_start (GTK_BOX (mw->priv->buttonbar), create_separator (TRUE), FALSE, TRUE, 0);
	mw->priv->mkdir_btn = add_buttonbar_button(
		_("F7 Mkdir"), main_win, "mkdir_btn", mw->priv->accel_group, 0);
	gtk_box_pack_start (GTK_BOX (mw->priv->buttonbar), create_separator (TRUE), FALSE, TRUE, 0);
	mw->priv->delete_btn = add_buttonbar_button(
		_("F8 Delete"), main_win, "delete_btn", mw->priv->accel_group, 0);
	gtk_box_pack_start (GTK_BOX (mw->priv->buttonbar), create_separator (TRUE), FALSE, TRUE, 0);
	mw->priv->find_btn = add_buttonbar_button(
		_("F9 Search"), main_win, "find_btn", mw->priv->accel_group, 0);
	gtk_box_pack_start (GTK_BOX (mw->priv->buttonbar), create_separator (TRUE), FALSE, TRUE, 0);
	mw->priv->quit_btn = add_buttonbar_button(
		_("F10 Quit"), main_win, "quit_btn", mw->priv->accel_group, 0);

	gtk_signal_connect (GTK_OBJECT (mw->priv->view_btn), "clicked",
						GTK_SIGNAL_FUNC (on_view_clicked), mw);
	gtk_signal_connect (GTK_OBJECT (mw->priv->edit_btn), "clicked",
						GTK_SIGNAL_FUNC (on_edit_clicked), mw);
	gtk_signal_connect (GTK_OBJECT (mw->priv->copy_btn), "clicked",
						GTK_SIGNAL_FUNC (on_copy_clicked), mw);
	gtk_signal_connect (GTK_OBJECT (mw->priv->move_btn), "clicked",
						GTK_SIGNAL_FUNC (on_move_clicked), mw);
	gtk_signal_connect (GTK_OBJECT (mw->priv->mkdir_btn), "clicked",
						GTK_SIGNAL_FUNC (on_mkdir_clicked), mw);
	gtk_signal_connect (GTK_OBJECT (mw->priv->delete_btn), "clicked",
						GTK_SIGNAL_FUNC (on_delete_clicked), mw);
	gtk_signal_connect (GTK_OBJECT (mw->priv->find_btn), "clicked",
						GTK_SIGNAL_FUNC (on_search_clicked), mw);
	gtk_signal_connect (GTK_OBJECT (mw->priv->quit_btn), "clicked",
						GTK_SIGNAL_FUNC (on_quit_clicked), mw);
}


/*****************************************************************************
	Misc widgets callbacks
*****************************************************************************/

static void
on_slide_button_press (GtkWidget *widget, GdkEventButton *event, GnomeCmdMainWin *mw)
{
	if (event->type == GDK_BUTTON_PRESS && event->button == 3) {
		GtkPaned *paned = GTK_PANED (mw->priv->hpaned);

		/* Check that the handle was clicked and not one of the children */
		if (paned->handle == event->window)
			gtk_menu_popup (GTK_MENU(create_slide_popup ()), NULL, NULL, NULL, NULL, 
							event->button, event->time);
	}
}

static void
on_main_win_realize                    (GtkWidget       *widget,
                                        GnomeCmdMainWin *mw)
{
	gnome_cmd_file_selector_set_active (
		gnome_cmd_main_win_get_left_fs (mw), TRUE);
	
	gnome_cmd_cmdline_set_dir (
		GNOME_CMD_CMDLINE (mw->priv->cmdline),
		gnome_cmd_dir_get_path (
			gnome_cmd_file_selector_get_directory (
				gnome_cmd_main_win_get_left_fs (mw))));
	
	gdk_window_set_icon (GTK_WIDGET (main_win)->window, NULL,
						 IMAGE_get_pixmap (PIXMAP_LOGO),
						 IMAGE_get_mask (PIXMAP_LOGO));
}


static void
on_main_win_size_allocate              (GtkWidget       *widget,
                                        GtkAllocation   *allocation,
                                        GnomeCmdMainWin *mw)
{
	gtk_paned_set_position (GTK_PANED (mw->priv->hpaned), allocation->width/2 - 5);
}


static void
on_left_fs_select                      (GtkCList *list,
										GdkEventButton *event,
                                        GnomeCmdMainWin *mw)
{
		mw->priv->current_fs = LEFT;
		
		gnome_cmd_file_selector_set_active (
			GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[LEFT]), TRUE);
		gnome_cmd_file_selector_set_active (
			GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[RIGHT]), FALSE);
}



static void
on_right_fs_select                     (GtkCList *list,
										GdkEventButton *event,
                                        GnomeCmdMainWin *mw)
{
		mw->priv->current_fs = RIGHT;
		
		gnome_cmd_file_selector_set_active (
			GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[RIGHT]), TRUE);
		gnome_cmd_file_selector_set_active (
			GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[LEFT]), FALSE);
}


static void
on_fs_list_resize_column            (GtkCList        *clist,
                                     gint             column,
                                     gint             width,
                                     GtkCList        *other_clist)
{
	static gboolean column_resize_lock = FALSE;

    /* the lock is used so that we dont get into the situation where
       the left list triggers the right witch triggers the left ... */
    if (!column_resize_lock)
    {
		column_resize_lock = TRUE;
		gnome_cmd_data_set_fs_col_width ((guint)column, width);
		gtk_clist_set_column_width (other_clist, column, width);
		column_resize_lock = FALSE;
    }
}


static void
on_size_allocate                  (GtkWidget *widget,
								   GtkAllocation *allocation,
								   gpointer user_data)
{
	gnome_cmd_data_set_position (allocation->width, allocation->height);
}


static void
update_browse_buttons             (GnomeCmdMainWin *mw,
								   GnomeCmdFileSelector *fs)
{
	g_return_if_fail (GNOME_CMD_IS_MAIN_WIN (mw));
	g_return_if_fail (GNOME_CMD_IS_FILE_SELECTOR (fs));
	
	if (fs == gnome_cmd_main_win_get_active_fs (mw)) {
		if (gnome_cmd_data_get_toolbar_visibility ()) {
			gtk_widget_set_sensitive (mw->priv->back_btn,
									  gnome_cmd_file_selector_can_back (fs));
			gtk_widget_set_sensitive (mw->priv->forward_btn,
									  gnome_cmd_file_selector_can_forward (fs));
		}

		gnome_cmd_main_menu_update_sens (GNOME_CMD_MAIN_MENU (mw->priv->menubar));
	}
}


static void
update_drop_con_button            (GnomeCmdMainWin *mw,
								   GnomeCmdFileSelector *fs)
{
	GtkWidget *btn;
	GnomeCmdPixmap *pm;
	GnomeCmdConnection *con;
	static GtkWidget *prev_pixmap = NULL;

	g_return_if_fail (GNOME_CMD_IS_MAIN_WIN (mw));
	g_return_if_fail (GNOME_CMD_IS_FILE_SELECTOR (fs));

	con = gnome_cmd_file_selector_get_connection (fs);
	g_return_if_fail (con != NULL);

	if (!gnome_cmd_data_get_toolbar_visibility ()
		|| (gnome_cmd_data_get_skip_mounting ()
			&& con->type == CON_TYPE_DEV))
		return;
	
	btn = mw->priv->con_drop_btn;
	g_return_if_fail (GTK_IS_BUTTON (btn));
	
	if (prev_pixmap) {
		gtk_object_destroy (GTK_OBJECT (prev_pixmap));
		prev_pixmap = NULL;
	}
	
	if (con->type == CON_TYPE_DEV) {
		pm = gnome_cmd_device_get_umount_pixmap (con->device);
	}
	else if (con->type == CON_TYPE_FTP) {
		pm = IMAGE_get_gnome_cmd_pixmap (PIXMAP_FTP_DISCONNECT);		
	}
	else {
		gtk_widget_set_sensitive (btn, FALSE);
		return;
	}

	g_return_if_fail (toolbar_tooltips != NULL);
	gtk_tooltips_set_tip (toolbar_tooltips, btn, con->con_drop_tooltip, NULL);
	gtk_widget_set_sensitive (btn, TRUE);

	if (pm) {
		GtkWidget *pixmap = gtk_pixmap_new (pm->pixmap, pm->mask);		
		if (pixmap) {
			gtk_widget_ref (pixmap);
			gtk_widget_show (pixmap);
			gtk_container_add (GTK_CONTAINER (btn), pixmap);
			prev_pixmap = pixmap;
		}
	}
	else {
		GtkWidget *label = gtk_label_new (con->con_drop_text);
		if (label) {
			gtk_widget_ref (label);
			gtk_widget_show (label);
			gtk_container_add (GTK_CONTAINER (btn), label);
			prev_pixmap = label;
		}
	}
}


static void
on_fs_dir_change                  (GnomeCmdFileSelector *fs,
								   const gchar dir,
								   GnomeCmdMainWin *mw)
{
	update_browse_buttons (mw, fs);
	update_drop_con_button (mw, fs);
}


/*******************************
 * Gtk class implementation
 *******************************/

static void
destroy (GtkObject *object)
{
	gtk_main_quit ();
}


static void
map (GtkWidget *widget)
{
	if (GTK_WIDGET_CLASS (parent_class)->map != NULL)
		GTK_WIDGET_CLASS (parent_class)->map (widget);
}


static void
class_init (GnomeCmdMainWinClass *class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = GTK_OBJECT_CLASS (class);
	widget_class = GTK_WIDGET_CLASS (class);
	parent_class = gtk_type_class (gnome_app_get_type ());

	main_win_signals[SWITCH_FS] =
		gtk_signal_new ("switch_fs",
			GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GnomeCmdMainWinClass, switch_fs),
		    gtk_marshal_NONE__POINTER,
		    GTK_TYPE_NONE,
			1, GTK_TYPE_POINTER);
  
	gtk_object_class_add_signals (object_class, main_win_signals, LAST_SIGNAL);
	object_class->destroy = destroy;
	widget_class->map = map;
	class->switch_fs = gnome_cmd_main_win_real_switch_fs;
}


static void
init (GnomeCmdMainWin *mw)
{
	gint main_win_width, main_win_height;
	
	/* It is very important that this global variable gets assigned here so that
	 * child widgets to this window can use that variable when initializing
	 */
	main_win = GNOME_CMD_MAIN_WIN (mw);

	mw->priv = g_new (GnomeCmdMainWinPrivate, 1);
	mw->priv->current_fs = LEFT;
	mw->priv->accel_group = gtk_accel_group_new ();
	mw->priv->buttonbar = NULL;
	mw->priv->buttonbar_sep = NULL;
	mw->priv->toolbar = NULL;
	mw->priv->toolbar_sep = NULL;

	gnome_cmd_data_get_position (&main_win_width, &main_win_height);

	gtk_object_set_data (GTK_OBJECT (main_win), "main_win", main_win);
	gtk_window_set_title (GTK_WINDOW (main_win), "GNOME Commander");
	gtk_window_set_position (GTK_WINDOW (main_win), GTK_WIN_POS_CENTER);
	gtk_window_set_policy (GTK_WINDOW (main_win), TRUE, TRUE, FALSE);
	gtk_widget_set_usize (GTK_WIDGET (main_win), main_win_width, main_win_height);

	mw->priv->vbox = gtk_vbox_new (FALSE, 0);
	gtk_widget_ref (mw->priv->vbox);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "vbox", mw->priv->vbox,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->vbox);

	
	mw->priv->menubar = gnome_cmd_main_menu_new ();
	gtk_widget_ref (mw->priv->menubar);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "vbox", mw->priv->menubar,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->menubar);
	gtk_box_pack_start (GTK_BOX (mw->priv->vbox), mw->priv->menubar, FALSE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (mw->priv->vbox), create_separator (FALSE), FALSE, TRUE, 0);
	
	gnome_app_set_contents (GNOME_APP (main_win), mw->priv->vbox);

	mw->priv->hpaned = gtk_hpaned_new ();
	gtk_widget_ref (mw->priv->hpaned);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "paned", mw->priv->hpaned,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->hpaned);
	gtk_box_pack_start (GTK_BOX (mw->priv->vbox), mw->priv->hpaned,
						TRUE, TRUE, 0);
	gtk_paned_set_handle_size (GTK_PANED (mw->priv->hpaned), 8);
	gtk_paned_set_gutter_size (GTK_PANED (mw->priv->hpaned), 10);

	create_buttonbar (mw);

	mw->priv->file_selector[LEFT] = gnome_cmd_file_selector_new ();
	gtk_widget_ref (mw->priv->file_selector[LEFT]);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "left_file_selector",
							  mw->priv->file_selector[LEFT],
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->file_selector[LEFT]);
	gtk_paned_pack1 (GTK_PANED (mw->priv->hpaned), mw->priv->file_selector[LEFT],
					 TRUE, TRUE);

	mw->priv->file_selector[RIGHT] = gnome_cmd_file_selector_new ();
	gtk_widget_ref (mw->priv->file_selector[RIGHT]);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "left_file_selector",
							  mw->priv->file_selector[RIGHT],
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->file_selector[RIGHT]);
	gtk_paned_pack2 (GTK_PANED (mw->priv->hpaned), mw->priv->file_selector[RIGHT],
					 TRUE, TRUE);

	gnome_cmd_connection_create_local ();
	gnome_cmd_connection_create_devices ();
	
	gnome_cmd_file_selector_update_connections (gnome_cmd_main_win_get_left_fs (mw));
	gnome_cmd_file_selector_update_connections (gnome_cmd_main_win_get_right_fs (mw));

	gnome_cmd_file_selector_set_connection (gnome_cmd_main_win_get_left_fs (mw),
											gnome_cmd_connection_get_home (), NULL);
	gnome_cmd_file_selector_set_connection (gnome_cmd_main_win_get_right_fs (mw),
											gnome_cmd_connection_get_home (), NULL);

	if (start_dir) {
		gchar *s = g_strdup_printf ("file://%s", start_dir);
		g_print ("start_dir: %s\n", s);
		gnome_cmd_file_selector_set_directory (
			gnome_cmd_main_win_get_left_fs (mw),
			gnome_cmd_dir_new_from_uri_str (s, TRUE));
	}

	mw->priv->cmdline = gnome_cmd_cmdline_new ();
	gtk_widget_ref (mw->priv->cmdline);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "cmdline", mw->priv->cmdline,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->cmdline);
	gtk_box_pack_start (GTK_BOX (mw->priv->vbox), create_separator (FALSE), FALSE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (mw->priv->vbox), mw->priv->cmdline, FALSE, TRUE, 1);

	gnome_cmd_main_win_set_toolbar_visibility (
		main_win, gnome_cmd_data_get_toolbar_visibility ());
	
	gnome_cmd_main_win_set_buttonbar_visibility (
		main_win, gnome_cmd_data_get_buttonbar_visibility ());
	
	gtk_signal_connect (GTK_OBJECT (main_win), "realize",
						GTK_SIGNAL_FUNC (on_main_win_realize), mw);
	gtk_signal_connect (GTK_OBJECT (main_win), "size_allocate",
						GTK_SIGNAL_FUNC (on_main_win_size_allocate), mw);
	gtk_signal_connect (GTK_OBJECT (mw->priv->file_selector[LEFT]), "changed-dir",
						GTK_SIGNAL_FUNC (on_fs_dir_change), mw);
	gtk_signal_connect (GTK_OBJECT (mw->priv->file_selector[RIGHT]), "changed-dir",
						GTK_SIGNAL_FUNC (on_fs_dir_change), mw);

	
	gtk_signal_connect (
		GTK_OBJECT (GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[LEFT])->list),
		"resize_column", GTK_SIGNAL_FUNC (on_fs_list_resize_column),
		GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[RIGHT])->list);
	gtk_signal_connect (
		GTK_OBJECT (GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[RIGHT])->list),
		"resize_column", GTK_SIGNAL_FUNC (on_fs_list_resize_column),
		GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[LEFT])->list);
	gtk_signal_connect (
		GTK_OBJECT (GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[LEFT])->list),
		"button_press_event", GTK_SIGNAL_FUNC (on_left_fs_select), mw);
	gtk_signal_connect (
		GTK_OBJECT (GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[RIGHT])->list),
		"button_press_event", GTK_SIGNAL_FUNC (on_right_fs_select), mw);
	gtk_signal_connect (
		GTK_OBJECT (mw), "size-allocate",
		GTK_SIGNAL_FUNC (on_size_allocate), mw);
	gtk_signal_connect (
		GTK_OBJECT (mw->priv->hpaned),
		"button_press_event", GTK_SIGNAL_FUNC (on_slide_button_press), mw);	

	gtk_window_add_accel_group (GTK_WINDOW (main_win), mw->priv->accel_group);
}


GtkType
gnome_cmd_main_win_get_type         (void)
{
	static GtkType mw_type = 0;

	if (mw_type == 0)
	{
		GtkTypeInfo mw_info =
		{
			"GnomeCmdMainWin",
			sizeof (GnomeCmdMainWin),
			sizeof (GnomeCmdMainWinClass),
			(GtkClassInitFunc) class_init,
			(GtkObjectInitFunc) init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		mw_type = gtk_type_unique (gnome_app_get_type (), &mw_info);
	}
	return mw_type;
}



GtkWidget*
gnome_cmd_main_win_new              ()
{
	GnomeCmdMainWin *mw;

	mw = gtk_type_new (gnome_cmd_main_win_get_type ());

	return GTK_WIDGET (mw);
}


GnomeCmdFileSelector*
gnome_cmd_main_win_get_active_fs         (GnomeCmdMainWin *mw)
{
	return GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[mw->priv->current_fs]);
}


GnomeCmdFileSelector*
gnome_cmd_main_win_get_inactive_fs         (GnomeCmdMainWin *mw)
{
	return GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[!mw->priv->current_fs]);
}


GnomeCmdFileSelector*
gnome_cmd_main_win_get_left_fs         (GnomeCmdMainWin *mw)
{
	return GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[LEFT]);
}


GnomeCmdFileSelector*
gnome_cmd_main_win_get_right_fs         (GnomeCmdMainWin *mw)
{
	return GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[RIGHT]);
}


void
gnome_cmd_main_win_update_style          (GnomeCmdMainWin *mw)
{
	g_return_if_fail (mw != NULL);
	g_return_if_fail (mw->priv != NULL);

	if (mw->priv->buttonbar) {
		gtk_widget_modify_style (GTK_BIN (mw->priv->view_btn)->child, main_rc_style);
		gtk_widget_modify_style (GTK_BIN (mw->priv->edit_btn)->child, main_rc_style);
		gtk_widget_modify_style (GTK_BIN (mw->priv->copy_btn)->child, main_rc_style);
		gtk_widget_modify_style (GTK_BIN (mw->priv->move_btn)->child, main_rc_style);
		gtk_widget_modify_style (GTK_BIN (mw->priv->mkdir_btn)->child, main_rc_style);
		gtk_widget_modify_style (GTK_BIN (mw->priv->delete_btn)->child, main_rc_style);
		gtk_widget_modify_style (GTK_BIN (mw->priv->find_btn)->child, main_rc_style);
		gtk_widget_modify_style (GTK_BIN (mw->priv->quit_btn)->child, main_rc_style);
	}

	IMAGE_clear_mime_cache ();

	gnome_cmd_file_selector_update_style (gnome_cmd_main_win_get_left_fs (mw));
	gnome_cmd_file_selector_update_style (gnome_cmd_main_win_get_right_fs (mw));

	gnome_cmd_cmdline_update_style (GNOME_CMD_CMDLINE (mw->priv->cmdline));
}


void
gnome_cmd_main_win_focus_cmdline         (GnomeCmdMainWin *mw)
{
	gnome_cmd_cmdline_focus (GNOME_CMD_CMDLINE (mw->priv->cmdline));
}


void
gnome_cmd_main_win_focus_file_lists      (GnomeCmdMainWin *mw)
{
	gnome_cmd_file_selector_set_active (
		gnome_cmd_main_win_get_active_fs (mw), TRUE);
	gnome_cmd_file_selector_set_active (
		gnome_cmd_main_win_get_inactive_fs (mw), FALSE);
}


gboolean
gnome_cmd_main_win_keypressed            (GnomeCmdMainWin *mw,
										  GdkEventKey *event)
{
	if (state_is_alt (event->state))
	{
		switch (event->keyval)
		{
			case GDK_1:
			{
				GnomeCmdFileSelector *fs = gnome_cmd_main_win_get_left_fs (mw);
				gnome_cmd_main_win_switch_fs (mw, fs);
 				gnome_cmd_combo_popup_list (
					GNOME_CMD_COMBO (fs->con_combo));
			}
			return TRUE;
			
			case GDK_2:
			{
				GnomeCmdFileSelector *fs = gnome_cmd_main_win_get_right_fs (mw);
				gnome_cmd_main_win_switch_fs (mw, fs);
 				gnome_cmd_combo_popup_list (
					GNOME_CMD_COMBO (fs->con_combo));
			}
			return TRUE;

			case GDK_F7:
				edit_search (NULL, NULL);
				return TRUE;
				
			case GDK_F8:
				gnome_cmd_cmdline_show_history (
					GNOME_CMD_CMDLINE (mw->priv->cmdline));				
				return TRUE;
		}
	}
	else if (state_is_ctrl_shift (event->state))
	{
		switch (event->keyval)
		{
			case GDK_F5:
				file_create_symlink (NULL, NULL);
				return TRUE;
				
			case GDK_F:
			case GDK_f:
				ftp_disconnect (NULL, NULL);
				return TRUE;

		}
	}
	else if (state_is_ctrl (event->state))
	{
		switch (event->keyval)
		{				
			case GDK_Down:
				gnome_cmd_cmdline_show_history (
					GNOME_CMD_CMDLINE (mw->priv->cmdline));				
				return TRUE;
				
			case GDK_u:
			case GDK_U:
			{
				GnomeCmdDir *dir1 = gnome_cmd_file_selector_get_directory (
					gnome_cmd_main_win_get_left_fs (mw));
				GnomeCmdDir *dir2 = gnome_cmd_file_selector_get_directory (
					gnome_cmd_main_win_get_right_fs (mw));

				gnome_cmd_dir_ref (dir1);
				gnome_cmd_dir_ref (dir2);

				gnome_cmd_file_selector_set_directory (
					gnome_cmd_main_win_get_left_fs (mw), dir2);
				gnome_cmd_file_selector_set_directory (
					gnome_cmd_main_win_get_right_fs (mw), dir1);

				gnome_cmd_dir_unref (dir1);
				gnome_cmd_dir_unref (dir2);

				clear_event_key (event);
			}
			return TRUE;

			case GDK_D:
			case GDK_d:
				bookmarks_edit (NULL, NULL);
				return TRUE;
				
			case GDK_Q:
			case GDK_q:
				gtk_widget_destroy (GTK_WIDGET (main_win));
				return TRUE;
				
			case GDK_O:
			case GDK_o:
				options_edit (NULL, NULL);
				return TRUE;

			case GDK_F:
			case GDK_f:
				ftp_connect (NULL, NULL);
				return TRUE;

			case GDK_G:
			case GDK_g:
				ftp_quick_connect (NULL, NULL);
				return TRUE;
		}
	}
	else if (state_is_alt_shift (event->state))
	{
		switch (event->keyval)
		{
			case GDK_S:
			case GDK_s:
				dir_pool_show_state ();
				break;
		}
	}
	else if (state_is_blank (event->state))
	{	
		switch (event->keyval)
		{
			case GDK_Tab:
				/* hack to avoid the deafult handling of the tab-key */
				clear_event_key (event);
				gnome_cmd_main_win_switch_fs (
					mw, gnome_cmd_main_win_get_inactive_fs (mw));
				return TRUE;
				
			case GDK_F1:
				on_help_clicked (NULL, mw);
				return TRUE;
			case GDK_F2:
				on_refresh_clicked (NULL, mw);
				return TRUE;
			case GDK_F3:
				on_view_clicked (NULL, mw);
				return TRUE;
			case GDK_F4:
				on_edit_clicked (NULL, mw);
				return TRUE;
			case GDK_F5:
				on_copy_clicked (NULL, mw);
				return TRUE;
			case GDK_F6:
				on_move_clicked (NULL, mw);
				return TRUE;
			case GDK_F7:
				on_mkdir_clicked (NULL, mw);
				return TRUE;
			case GDK_F8:
				on_delete_clicked (NULL, mw);
				return TRUE;
			case GDK_F9:
				on_search_clicked (NULL, mw);
				return TRUE;
			case GDK_F10:
				on_quit_clicked (NULL, mw);
				return TRUE;
		}
	}

	if (gnome_cmd_file_selector_keypressed (
			gnome_cmd_main_win_get_active_fs (mw), event))
		return TRUE;
	
	return FALSE;
}


void
gnome_cmd_main_win_switch_fs (GnomeCmdMainWin *mw, GnomeCmdFileSelector *fs)
{
	g_return_if_fail (GNOME_CMD_IS_MAIN_WIN (mw));
	g_return_if_fail (GNOME_CMD_IS_FILE_SELECTOR (fs));
	
	gtk_signal_emit (
		GTK_OBJECT (mw), main_win_signals[SWITCH_FS], fs);	
}


static void
gnome_cmd_main_win_real_switch_fs (GnomeCmdMainWin *mw, GnomeCmdFileSelector *fs)
{
	g_return_if_fail (GNOME_CMD_IS_MAIN_WIN (mw));
	g_return_if_fail (GNOME_CMD_IS_FILE_SELECTOR (fs));

	if (fs == gnome_cmd_main_win_get_active_fs (mw))
		return;
	
	mw->priv->current_fs = !mw->priv->current_fs;
	gnome_cmd_file_selector_set_active (
		gnome_cmd_main_win_get_active_fs (mw), TRUE);
	gnome_cmd_file_selector_set_active (
		gnome_cmd_main_win_get_inactive_fs (mw), FALSE);

	update_browse_buttons (mw, fs);
	update_drop_con_button (mw, fs);
}


GnomeCmdCmdline*
gnome_cmd_main_win_get_cmdline           (GnomeCmdMainWin *mw)
{
	g_return_val_if_fail (GNOME_CMD_IS_MAIN_WIN (mw), NULL);

	return GNOME_CMD_CMDLINE (mw->priv->cmdline);
}


void
gnome_cmd_main_win_update_bookmarks (GnomeCmdMainWin *mw)
{
	g_return_if_fail (GNOME_CMD_IS_MAIN_WIN (mw));
	
	gnome_cmd_main_menu_update_bookmarks (GNOME_CMD_MAIN_MENU (mw->priv->menubar));
}


void
gnome_cmd_main_win_set_toolbar_visibility (GnomeCmdMainWin *mw,
										   gboolean value)
{
	GnomeUIInfo toolbar_uiinfo[] =
	{
		{
			GNOME_APP_UI_ITEM, NULL,
			_("Refresh"),
			view_refresh, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_REFRESH,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, NULL,
			_("Back in history"),
			view_back, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_BACK,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, NULL,
			_("Forward in history"),
			view_forward, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_FORWARD,
			0, 0, NULL
		},
		GNOMEUIINFO_SEPARATOR,
		{
			GNOME_APP_UI_ITEM, NULL,
			_("Cut"),
			file_cap_cut, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_CUT,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, NULL,
			_("Copy"),
			file_cap_copy, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_COPY,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, NULL,
			_("Paste"),
			file_cap_paste, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_PASTE,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, NULL,
			_("Delete"),
			file_delete, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_TRASH,
			0, 0, NULL
		},
		GNOMEUIINFO_SEPARATOR,
		{
			GNOME_APP_UI_ITEM, NULL,
			_("FTP Connect"),
			ftp_connect, NULL, NULL,
			GNOME_APP_PIXMAP_DATA, ftp_connect_xpm,
			0, 0, NULL
		},
		GNOMEUIINFO_SEPARATOR,
		{
			GNOME_APP_UI_ITEM, NULL,
			_("Unmount Device"),
			file_umount, NULL, NULL,
			0, 0,
			0, 0, NULL
		},
		GNOMEUIINFO_END
	};

	gnome_cmd_data_set_toolbar_visibility (value);

	if (value) {
		create_toolbar (mw, toolbar_uiinfo);
		update_drop_con_button (mw, gnome_cmd_main_win_get_active_fs (mw));
		update_browse_buttons (mw, gnome_cmd_main_win_get_active_fs (mw));
		gtk_box_pack_start (GTK_BOX (mw->priv->vbox), mw->priv->toolbar, FALSE, TRUE, 0);
		gtk_box_reorder_child (GTK_BOX (mw->priv->vbox), mw->priv->toolbar, 2);
		gtk_box_pack_start (GTK_BOX (mw->priv->vbox), mw->priv->toolbar_sep, FALSE, TRUE, 0);
		gtk_box_reorder_child (GTK_BOX (mw->priv->vbox), mw->priv->toolbar_sep, 3);
	}
	else {
		if (mw->priv->toolbar)
			gtk_widget_destroy (mw->priv->toolbar);
		if (mw->priv->toolbar_sep)
			gtk_widget_destroy (mw->priv->toolbar_sep);
		mw->priv->toolbar = NULL;
		mw->priv->toolbar_sep = NULL;
	}
}


void
gnome_cmd_main_win_set_buttonbar_visibility (GnomeCmdMainWin *mw,
											 gboolean value)
{
	gnome_cmd_data_set_buttonbar_visibility (value);

	if (value) {
		create_buttonbar (mw);
		gtk_box_pack_start (GTK_BOX (mw->priv->vbox), mw->priv->buttonbar_sep, FALSE, TRUE, 0);
		gtk_box_pack_start (GTK_BOX (mw->priv->vbox), mw->priv->buttonbar, FALSE, TRUE, 0);
	}
	else {
		if (mw->priv->buttonbar)
			gtk_widget_destroy (mw->priv->buttonbar);
		if (mw->priv->buttonbar_sep)
			gtk_widget_destroy (mw->priv->buttonbar_sep);
		mw->priv->buttonbar = NULL;
		mw->priv->buttonbar_sep = NULL;
	}
}



