/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2003 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include <unistd.h>
#include "gnome-cmd-includes.h"
#include "gnome-cmd-xfer.h"
#include "gnome-cmd-file-selector-funcs.h"
#include "gnome-cmd-file-list.h"
#include "gnome-cmd-dir-funcs.h"
#include "gnome-cmd-file-funcs.h"
#include "gnome-cmd-xfer-progress-win.h"
#include "gnome-cmd-main-win-funcs.h"
#include "utils.h"

typedef struct
{
	// Source and target uri's. The first src_uri should be transfered to the
	// first dest_uri and so on...
	GList *src_uri_list;
	GList *dest_uri_list;

	GnomeCmdDir *to_dir;
	GnomeCmdFileList *src_fl;
	GList *src_files;

	// Used for showing the progress
	GnomeCmdXferProgressWin *win;
	GnomeVFSXferPhase prev_phase;
	gulong prev_file;
	gfloat prev_totalprog;
	gboolean first_time;

	GtkSignalFunc on_completed_func;
	gpointer on_completed_data;
	
} XferData;


static const char **
convert_varargs_to_name_array (va_list args)
{
	GPtrArray *resizeable_array;
	const char *name;
	const char **plain_ole_array;
	
	resizeable_array = g_ptr_array_new ();

	do {
		name = va_arg (args, const char *);
		g_ptr_array_add (resizeable_array, (gpointer) name);
	} while (name != NULL);

	plain_ole_array = (const char **) resizeable_array->pdata;
	
	g_ptr_array_free (resizeable_array, FALSE);

	return plain_ole_array;
}


static gboolean
delete_event_callback (gpointer data,
		       gpointer user_data)
{
	g_return_val_if_fail (GNOME_IS_DIALOG (data), FALSE);
	gtk_signal_emit_stop_by_name (GTK_OBJECT (data),
				      "delete_event");
	return TRUE;
}


static void
add_label_to_dialog (GnomeDialog *dialog, const char *message)
{
	GtkLabel *message_widget;

	if (message == NULL) {
		return;
	}
	
	message_widget = GTK_LABEL (gtk_label_new (message));
	gtk_label_set_line_wrap (message_widget, TRUE);
	gtk_label_set_justify (message_widget,
			       GTK_JUSTIFY_LEFT);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox),
			    GTK_WIDGET (message_widget),
			    TRUE, TRUE, 0);
}


static gint
eel_run_simple_dialog (GtkWidget *parent, gboolean ignore_close_box,
			const char *text, const char *title, ...)
{
	va_list button_title_args;
	const char **button_titles;
	GtkWidget *dialog;
	GtkWidget *top_widget;
	int result;

	
	/* Create the dialog. */
	va_start (button_title_args, title);
	button_titles = convert_varargs_to_name_array (button_title_args);
	va_end (button_title_args);
	dialog = gnome_dialog_newv (title, button_titles);
	g_free (button_titles);
	
	/* Allow close. */
	if (ignore_close_box) {
		gtk_signal_connect (GTK_OBJECT (dialog),
				    "delete_event",
				    GTK_SIGNAL_FUNC (delete_event_callback),
				    NULL);

	}
	
	gnome_dialog_close_hides (GNOME_DIALOG (dialog), TRUE);

	gtk_window_set_wmclass (GTK_WINDOW (dialog), "dialog", "Eel");

	/* Parent it if asked to. */
	if (parent != NULL) {
		top_widget = gtk_widget_get_toplevel (parent);
		if (GTK_IS_WINDOW (top_widget) && !GTK_OBJECT_DESTROYED (top_widget)) {
			gnome_dialog_set_parent (GNOME_DIALOG (dialog), GTK_WINDOW (top_widget));
		}
	}
	
	/* Title it if asked to. */
	add_label_to_dialog (GNOME_DIALOG (dialog), text);

	/* Run it. */
	gtk_widget_show_all (dialog);
	result = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
	while (result == -1 && ignore_close_box) {
		gtk_widget_show (GTK_WIDGET (dialog));
		result = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
	}
	gtk_widget_destroy (dialog);

	return result;
}


static void
free_xfer_data (XferData *data)
{
	GList *tmp_list;
	GnomeVFSURI *uri;

	if (data->on_completed_func)
		data->on_completed_func (data->on_completed_data);

	g_list_free (data->src_uri_list);

	/* free the list with target uris */
	tmp_list = data->dest_uri_list;
	while (tmp_list)
	{
		uri = (GnomeVFSURI*)tmp_list->data;
		gnome_vfs_uri_unref (uri);
		tmp_list = tmp_list->next;
	}
	g_list_free (data->dest_uri_list);

	g_free (data);
}


static gint
xfer_callback (GnomeVFSXferProgressInfo *info, gpointer user_data)
{
	return 1;
}


static gint
async_xfer_callback (GnomeVFSAsyncHandle *handle,
					 GnomeVFSXferProgressInfo *info,
					 XferData *data)
{
	gfloat total_diff=0;

	gdk_threads_enter ();
	if (data->win && data->win->cancel_pressed) {
		gnome_vfs_async_cancel (handle);
		gtk_widget_destroy (GTK_WIDGET (data->win));		
		gdk_threads_leave ();
		free_xfer_data (data);
		return 0;
	}

	if (info->status == GNOME_VFS_XFER_PROGRESS_STATUS_OVERWRITE) {
		gint ret;
		gchar *fn = gnome_vfs_get_local_path_from_uri (info->target_name);
		gchar *msg = g_strdup_printf (_("Do you want to overwrite the file:\n %s\n"), fn);
		g_free (fn);
		ret = eel_run_simple_dialog (GTK_WIDGET (main_win), TRUE, msg, _("Conflict while copying"),
			 _("Abort"), _("Replace"), _("Replace All"), _("Skip"), _("Skip All"), NULL);
		g_free (msg);
		gdk_threads_leave ();
		return ret;
	}

	if (info->phase == GNOME_VFS_XFER_PHASE_COPYING) {
		if (data->to_dir && !gnome_cmd_dir_uses_fam (data->to_dir))
			data->to_dir->content_changed = TRUE;
		
		if (data->prev_phase != GNOME_VFS_XFER_PHASE_COPYING) {
			gnome_cmd_xfer_progress_win_set_action (data->win, _("copying..."));
			data->prev_file = -1;
		}
		
		if (data->prev_file != info->file_index) {
			gchar *fn = str_uri_basename (info->source_name);
			gchar *msg = g_strdup_printf (
				_("[file %ld of %ld] \"%s\""),
				info->file_index, info->files_total, fn);
			
			gnome_cmd_xfer_progress_win_set_msg (data->win, msg);

			data->prev_file = info->file_index;
			
			if (msg) g_free (msg);
			if (fn) g_free (fn);
		}

		if (info->bytes_total > 0) {
			gfloat total_prog;
			
			total_prog = (gfloat)((gdouble)info->total_bytes_copied / (gdouble)info->bytes_total);
			total_diff = total_prog - data->prev_totalprog;

			if ((total_diff > (gfloat)0.01 && total_prog >= 0.0 && total_prog <= 1.0)
				|| data->first_time) {
				data->first_time = FALSE;
				gnome_cmd_xfer_progress_win_set_total_progress (
					data->win, info->total_bytes_copied, info->bytes_total);
				while (gtk_events_pending ())
					gtk_main_iteration_do (FALSE);
			}
		}
		
	}

	if (info->phase == GNOME_VFS_XFER_PHASE_MOVING) {
		if (data->to_dir && !gnome_cmd_dir_uses_fam (data->to_dir))
			data->to_dir->content_changed = TRUE;
#ifndef HAVE_LIBFAM
		if (data->src_fl && data->src_files)
			gnome_cmd_file_list_remove_files (data->src_fl, data->src_files);
#endif //HAVE_LIBFAM
	}

	if (info->phase == GNOME_VFS_XFER_PHASE_COMPLETED) {
		/* Only update the files if needed */
		if (data->to_dir && data->to_dir->ref_cnt > 1) {
			if (data->to_dir && !gnome_cmd_dir_uses_fam (data->to_dir)) {
				gnome_cmd_dir_list_files (data->to_dir, FALSE);
			}
		}

		if (data->to_dir)
			gnome_cmd_dir_unref (data->to_dir);
		data->to_dir = NULL;

		if (data->win) {
			gtk_widget_destroy (GTK_WIDGET (data->win));
			data->win = NULL;
		}
		
		free_xfer_data (data);
	}
	
	data->prev_phase = info->phase;
	
	gdk_threads_leave ();
	
	return 1;
}


static gboolean
uri_is_parent_to_dir (GnomeVFSURI *uri, GnomeCmdDir *dir)
{
	GnomeVFSURI *dir_uri;

	dir_uri = gnome_cmd_dir_get_uri (dir);

	return gnome_vfs_uri_is_parent (uri, dir_uri, TRUE);
}


static gchar *
remove_basename (gchar *in)
{
	gint i;
	gchar *out = g_strdup (in);

	for ( i=strlen(out)-1 ; i>0 ; i-- ) {
		if (out[i] == '/') {
			out[i] = '\0';
			return out;
		}
	}

	return NULL;
}


static gboolean
file_is_already_in_dir (GnomeVFSURI *uri, GnomeCmdDir *dir) {
	gchar *uri_str, *tmp;
	const gchar *dir_str;
	gboolean ret;

	tmp = gnome_vfs_uri_to_string (uri, 0);
	uri_str = remove_basename (tmp);
	dir_str = gnome_cmd_dir_get_uri_str (dir);

	ret = (strcmp (uri_str, dir_str) == 0);

	g_free (uri_str);
	g_free (tmp);

	return ret;
}


void
gnome_cmd_xfer_uris_start (GList *src_uri_list,
						   GnomeCmdDir *to_dir,
						   GnomeCmdFileList *src_fl,
						   GList *src_files,
						   gchar *dest_fn,
						   GnomeVFSXferOptions xferOptions,
						   GnomeVFSXferOverwriteMode xferOverwriteMode,
						   GtkSignalFunc on_completed_func,
						   gpointer on_completed_data)
{
	GnomeVFSURI *src_uri, *dest_uri;
	GnomeVFSResult result;
	gint num_files;
	GnomeVFSAsyncHandle *handle;
	XferData *data;
	GList *tmp;

	g_return_if_fail (src_uri_list != NULL);
	g_return_if_fail (to_dir != NULL);

	/*
	 * Sanity check
	 */
	tmp = src_uri_list;
	while (tmp) {
		src_uri = (GnomeVFSURI*)tmp->data;
		if (uri_is_parent_to_dir (src_uri, to_dir)) {
			gnome_error_dialog (_("Copying a directory into it self is a bad idea.\nThe whole operation was canceled."));
			return;
		}
		if (file_is_already_in_dir (src_uri, to_dir)) {
			if (dest_fn) {
				gchar *fn = gnome_vfs_escape_string (dest_fn);
				const gchar *bn = gnome_vfs_uri_get_basename (src_uri);
				
				g_printerr ("comparing: %s %s\n", fn, bn);				
				if (strcmp (fn, bn) == 0) {
					g_printerr (_("Copying a file to the same directory as it's already in, is not permitted\n"));
					g_free (fn);
					return;
				}
				g_free (fn);
			}
		}
		
		tmp = tmp->next;
	}	
	
	data = g_new (XferData, 1);

	data->src_uri_list = src_uri_list;
	data->dest_uri_list = NULL;
	data->to_dir = to_dir;
	data->src_fl = src_fl;
	data->src_files = src_files;
	data->win = NULL;
	data->prev_phase = -1;
	data->prev_file = -1;
	data->prev_totalprog = (gfloat)0.00;
	data->first_time = TRUE;
	data->on_completed_func = on_completed_func;
	data->on_completed_data = on_completed_data;

	num_files = g_list_length (src_uri_list);

	if (g_list_length (src_uri_list) == 1 && dest_fn != NULL) {
		dest_uri = gnome_cmd_dir_get_file_uri (to_dir, dest_fn);

		data->dest_uri_list = g_list_append (data->dest_uri_list, dest_uri);
	}
	else {
		while (src_uri_list) {
			gchar *basename;
			src_uri = (GnomeVFSURI*)src_uri_list->data;
			basename = gnome_vfs_unescape_string (
				gnome_vfs_uri_get_basename (src_uri), 0);

			dest_uri = gnome_cmd_dir_get_file_uri ( 
				to_dir, basename);
			g_free (basename);
		
			data->dest_uri_list = g_list_append (data->dest_uri_list, dest_uri);
				
			src_uri_list = src_uri_list->next;
		}
	}

	if (dest_fn != NULL)
		g_free (dest_fn);
	
	data->win = GNOME_CMD_XFER_PROGRESS_WIN (gnome_cmd_xfer_progress_win_new ());
	gtk_window_set_title (GTK_WINDOW (data->win), _("preparing..."));
	gtk_widget_show (GTK_WIDGET (data->win));
	
	/* start the transfer */
	result = gnome_vfs_async_xfer (
		&handle,
		data->src_uri_list,
		data->dest_uri_list,
		xferOptions,
		GNOME_VFS_XFER_ERROR_MODE_ABORT,
		xferOverwriteMode,
		(GnomeVFSAsyncXferProgressCallback)async_xfer_callback,
		data,
		xfer_callback,
		data);
}


void
gnome_cmd_xfer_start (GList *src_files,
					  GnomeCmdDir *to_dir,
					  GnomeCmdFileList *src_fl,
					  gchar *dest_fn,
					  GnomeVFSXferOptions xferOptions,
					  GnomeVFSXferOverwriteMode xferOverwriteMode,
					  GtkSignalFunc on_completed_func,
					  gpointer on_completed_data)
{
	GList *tmp = src_files;
	GList *src_uri_list = NULL;

	while (tmp) {
		GnomeCmdFile *finfo = (GnomeCmdFile*)tmp->data;
		src_uri_list = g_list_append (
			src_uri_list,
			gnome_cmd_file_get_uri (finfo));
		
		tmp = tmp->next;
	}

	gnome_cmd_xfer_uris_start (src_uri_list,
							   to_dir,
							   src_fl,
							   src_files,
							   dest_fn,
							   xferOptions,
							   xferOverwriteMode,
							   on_completed_func,
							   on_completed_data);
}


void
gnome_cmd_xfer_tmp_download (GnomeVFSURI *src_uri,
							 GnomeVFSURI *dest_uri,
							 GnomeVFSXferOptions xferOptions,
							 GnomeVFSXferOverwriteMode xferOverwriteMode,
							 GtkSignalFunc on_completed_func,
							 gpointer on_completed_data)
{
	gnome_cmd_xfer_tmp_download_multiple (
		g_list_append (NULL, src_uri),
		g_list_append (NULL, dest_uri),
		xferOptions,
		xferOverwriteMode,
		on_completed_func,
		on_completed_data);
		
}
							 

void
gnome_cmd_xfer_tmp_download_multiple (GList *src_uri_list,
									  GList *dest_uri_list,
									  GnomeVFSXferOptions xferOptions,
									  GnomeVFSXferOverwriteMode xferOverwriteMode,
									  GtkSignalFunc on_completed_func,
									  gpointer on_completed_data)
{
	GnomeVFSAsyncHandle *handle;
	GnomeVFSResult result;
	XferData *data;

	data = g_new (XferData, 1);

	data->src_uri_list = src_uri_list;
	data->dest_uri_list = dest_uri_list;
	data->to_dir = NULL;
	data->src_fl = NULL;
	data->src_files = NULL;
	data->win = NULL;
	data->prev_phase = -1;
	data->prev_file = -1;
	data->prev_totalprog = (gfloat)0.00;
	data->first_time = TRUE;
	data->on_completed_func = on_completed_func;
	data->on_completed_data = on_completed_data;
	
	data->win = GNOME_CMD_XFER_PROGRESS_WIN (gnome_cmd_xfer_progress_win_new ());
	gtk_window_set_title (GTK_WINDOW (data->win), _("downloading to /tmp"));
	gtk_widget_show (GTK_WIDGET (data->win));

	/* start the transfer */
	result = gnome_vfs_async_xfer (
		&handle,
		data->src_uri_list,
		data->dest_uri_list,
		xferOptions,
		GNOME_VFS_XFER_ERROR_MODE_ABORT,
		xferOverwriteMode,
		(GnomeVFSAsyncXferProgressCallback)async_xfer_callback,
		data,
		xfer_callback,
		data);
}

