#include <config.h>

#include "gnome-cups-manager.h"

#include <cups/cups.h>

#include <stdlib.h>
#include <bonobo/bonobo-exception.h>
#include <bonobo/bonobo-listener.h>
#include <bonobo/bonobo-main.h>
#include <glade/glade.h>
#include <gtk/gtkmain.h>
#include <gtk/gtkmessagedialog.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-client.h>
#include <libgnomeui/gnome-ui-init.h>

#include "view-queue.h"
#include "printer-properties.h"
#include "tray.h"

#include <libgnomecups/gnome-cups-util.h>
#include <libgnomecups/gnome-cups-ui-init.h>
#include <libgnomecups/gnome-cups-printer.h>
#include <libgnomecups/gnome-cups-request.h>
#include <libgnomecups/gnome-cups-ui-print.h>

void
gnome_cups_error_dialog (GtkWindow *window, 
			 const char *prefix,
			 GError *error)
{
	GtkWidget *dialog;
	char *msg;
	
	msg = error ? error->message : _("Unknown Error");

	dialog = gtk_message_dialog_new (window, 
					 GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL, 
					 GTK_MESSAGE_ERROR,
					 GTK_BUTTONS_OK,
					 "%s: %s", prefix, error->message);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog); 
}

static void
get_test_page (GnomeCupsPrinter *printer,
	       char **filename,
	       char **paper_name)
{
	char *paper_size;
	
	paper_size = gnome_cups_printer_get_option_value (printer, "PageSize");
	
	if (paper_size && !g_ascii_strcasecmp (paper_size, "a4")) {
		if (paper_name) {
			*paper_name = g_strdup_printf ("A4");
		}
		
		if (filename) {
			*filename = g_strdup (DATADIR "/gnome-cups-manager/xd2-testpage-a4.eps");
		}
	} else {
		if (paper_name) {
			*paper_name = g_strdup_printf ("Letter");
		}
		
		if (filename) {
			*filename = g_strdup (DATADIR "/gnome-cups-manager/xd2-testpage-letter.eps");
		}
	}
	
	g_free (paper_size);
}

static void
print_dialog_response_cb (GtkWidget *dialog, 
			  int response_code, 
			  gpointer user_data)
{
	gtk_widget_destroy (dialog);
}

void
gnome_cups_print_test_page (GnomeCupsPrinter *printer,
			    GtkWidget *parent_window)
{
	gboolean result;
	char *filename;
	char *paper_size;

	get_test_page (printer, &filename, &paper_size);

	result = gnome_cups_ui_print_file (printer, 
					   GTK_WINDOW (parent_window),
					   filename,
					   _("Test Page"), NULL);
	
	if (result) {
                GtkWidget *dialog;
 
                dialog = gtk_message_dialog_new
                        (GTK_WINDOW (parent_window),
                         GTK_DIALOG_DESTROY_WITH_PARENT,
                         GTK_MESSAGE_INFO, GTK_BUTTONS_OK,
                         _("%s test page has been sent to %s."),
			 paper_size,
                         gnome_cups_printer_get_name (printer),
                         -1);
		gtk_widget_show (dialog);
		g_signal_connect (dialog, "response",
				  G_CALLBACK (print_dialog_response_cb),
				  NULL);
	}

	g_free (paper_size);
	g_free (filename);
}

static void
event_cb (BonoboListener *listener, const char *event_name, 
	  const BonoboArg *arg, CORBA_Environment *ev,
	  gpointer user_data)
{
	char *printer_name;
	GnomeCupsPrinter *printer;
	
	printer_name = BONOBO_ARG_GET_STRING (arg);
	
	printer = gnome_cups_printer_get (printer_name);
	
	if (printer) {
		if (!strcmp (event_name, "show_printer")) {
			gnome_cups_manager_view_queue (printer);
		} else  if (!strcmp (event_name, "printer_properties")) {
			gnome_cups_manager_printer_properties (printer);
		}
		g_object_unref (printer);
	} else {
		GtkWidget *dialog;
		
		dialog = gtk_message_dialog_new (NULL, 
						 0,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_OK,
						 _("Printer not found: %s"), 
						 printer_name);
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog); 
	}
}

static Bonobo_Listener
get_listener (gboolean *is_new)
{
	Bonobo_Listener ret;
	CORBA_Environment ev;
	
	CORBA_exception_init (&ev);
	ret = bonobo_activation_activate_from_id ("OAFIID:gnome-cups-manager",
						  Bonobo_ACTIVATION_FLAG_EXISTING_ONLY,
						  NULL, &ev);
	if (!BONOBO_EX (&ev) && !CORBA_Object_is_nil (ret, &ev)) {
		*is_new = FALSE;
	} else {
		Bonobo_RegistrationResult reg_result;
		BonoboListener *listener;
		
		listener = bonobo_listener_new (event_cb, NULL);
		
		*is_new = TRUE;
		ret = BONOBO_OBJREF (listener);

		reg_result = bonobo_activation_active_server_register
			("OAFIID:gnome-cups-manager", ret);
		switch (reg_result) {
		case Bonobo_ACTIVATION_REG_SUCCESS :
			break;
		case Bonobo_ACTIVATION_REG_NOT_LISTED :
			g_print ("not listed\n");
			break;
		case Bonobo_ACTIVATION_REG_ALREADY_ACTIVE :
			g_print ("already active\n");
			break;
		case Bonobo_ACTIVATION_REG_ERROR :
			g_print ("error\n");
			break;
		}
	}

	CORBA_exception_free (&ev);

	return ret;
}

static void
show_printer (Bonobo_Listener listener, const char *printer)
{
	CORBA_Environment ev;
	BonoboArg *arg;
	
	arg = bonobo_arg_new (BONOBO_ARG_STRING);
	BONOBO_ARG_SET_STRING (arg, printer);

	CORBA_exception_init (&ev);
	Bonobo_Listener_event (listener, "show_printer", arg, &ev);
	CORBA_exception_free (&ev);

	CORBA_free (arg);
}	

static void
printer_properties (Bonobo_Listener listener, const char *printer)
{
	CORBA_Environment ev;
	BonoboArg *arg;
	
	arg = bonobo_arg_new (BONOBO_ARG_STRING);
	BONOBO_ARG_SET_STRING (arg, printer);

	CORBA_exception_init (&ev);
	Bonobo_Listener_event (listener, "printer_properties", arg, &ev);
	CORBA_exception_free (&ev);

	CORBA_free (arg);
}	

static void
die_cb (GnomeClient *client, gpointer data)
{
	gtk_main_quit ();
}

static void
add_to_session (void)
{
	GnomeClient *client;

	client = gnome_master_client ();
	
	gnome_client_set_priority (client, 50);
	
	gnome_client_set_restart_style (client, GNOME_RESTART_IMMEDIATELY);

	g_signal_connect (client, "die", 
			  G_CALLBACK (die_cb), NULL);
}

static void
remove_from_session (void)
{
	GnomeClient *client;

	client = gnome_master_client ();
	
	gnome_client_set_restart_style (client, GNOME_RESTART_NEVER);
}

static gboolean
check_cups_again (gpointer user_data)
{
	static int tries = 0;
	
	if (gnome_cups_check_daemon ()) {
		gnome_cups_manager_start_icon ();
		return FALSE;
	} else {		
		if (++tries == 3) {
			GtkWidget *dialog;
			/* FIXME: dialog text */
			dialog = gtk_message_dialog_new (NULL, 
							 GTK_DIALOG_DESTROY_WITH_PARENT, 
							 GTK_MESSAGE_ERROR,
							 GTK_BUTTONS_OK,
							 _("Could not start the printer tray icon, because the CUPS server could not be contacted."));
			gtk_dialog_run (GTK_DIALOG (dialog));
			gtk_widget_destroy (dialog); 
			
			remove_from_session ();
			
			gtk_main_quit ();
		}
		
	}

	return TRUE;	
}

static void
check_cups (void)
{
	if (gnome_cups_check_daemon ()) {
		gnome_cups_manager_start_icon ();
	} else {
		g_timeout_add (30000, check_cups_again, NULL);
	}
}

int 
main (int argc, char *argv[])
{
	GnomeProgram *program;
	poptContext ctx;
	char **args;
	gboolean is_new;
	int i;
	Bonobo_Listener listener;
	gboolean properties = FALSE;
	gboolean view = FALSE;

	struct poptOption options[] = {
		{ "properties", 'p', POPT_ARG_NONE, &properties, 0,
		  N_("Show printer properties for printers listed on the command line"), NULL
		},
		{ "view", 'v', POPT_ARG_NONE, &view, 0,
		  N_("View the queues of printers listed on the command line"), NULL
		},
		{ NULL }
	};	
	
	program = gnome_program_init ("gnome-cups-manager", 
				      VERSION,
				      LIBGNOMEUI_MODULE, argc, argv,
				      GNOME_PROGRAM_STANDARD_PROPERTIES,
				      GNOME_PARAM_POPT_TABLE, options,
				      GNOME_PARAM_HUMAN_READABLE_NAME, _("Gnome CUPS Manager"),
				      NULL);

	if (!properties && !view) {
		view = TRUE;
	}

	gnome_cups_ui_init ();

	bonobo_init (&argc, argv);
	bonobo_activate ();

	g_object_get (G_OBJECT (program), GNOME_PARAM_POPT_CONTEXT, &ctx, NULL);
	args = (char**)poptGetArgs (ctx);

	listener = get_listener (&is_new);

	if (is_new) {
		check_cups ();
		glade_init ();	
		add_to_session ();
	}

	if (args) {
		for (i = 0; args[i] != NULL; i++) {
			if (view) {
				show_printer (listener, args[i]);
			}
			if (properties) {
				printer_properties (listener, args[i]);
			}
		}
	}

	if (is_new) {
		gtk_main ();
	}

	return 0;
}
