
#include <src/Manager.h>
#include <gee/hashmap.h>
#include <gee/hashset.h>
#include <gee/map.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <gee/collection.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <dbus/dbus-glib.h>
#include <gio/gio.h>
#include "src/Scanner.h"
#include "src/Constants.h"
#include "src/Device.h"
#include "src/EPGScanner.h"
#include "src/Terrestrial/TerrestrialScanner.h"
#include "src/Satellite/SatelliteScanner.h"
#include "src/Cable/CableScanner.h"
#include "src/Recorder.h"
#include "src/ChannelList.h"
#include "src/Factory.h"
#include "src/ConfigStore.h"
#include "src/ChangeType.h"
#include "src/Channel.h"
#include "src/Schedule.h"

typedef struct _DBusObjectVTable _DBusObjectVTable;

struct _DBusObjectVTable {
	void (*register_object) (DBusConnection*, const char*, void*);
};



struct _DVBManagerPrivate {
	GeeHashMap* scanners;
	GeeHashMap* devices;
	GeeHashSet* schedules;
	GeeHashSet* recorders;
	GeeHashSet* channellists;
	GeeHashSet* epgscanners;
	guint device_group_counter;
};

#define DVB_MANAGER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DVB_TYPE_MANAGER, DVBManagerPrivate))
enum  {
	DVB_MANAGER_DUMMY_PROPERTY
};
static DVBManager* dvb_manager_instance = NULL;
static GStaticRecMutex dvb_manager_instance_mutex = {0};
static void _dvb_manager_on_scanner_destroyed_dvb_scanner_destroyed (DVBScanner* _sender, gpointer self);
static char** dvb_manager_real_GetScannerForDevice (DVBIDBusManager* base, guint adapter, guint frontend, int* result_length1);
static guint* dvb_manager_real_GetRegisteredDeviceGroups (DVBIDBusManager* base, int* result_length1);
static char* dvb_manager_real_GetTypeOfDeviceGroup (DVBIDBusManager* base, guint group_id);
static char* dvb_manager_real_GetRecorder (DVBIDBusManager* base, guint group_id);
static gboolean dvb_manager_real_AddDeviceToNewGroup (DVBIDBusManager* base, guint adapter, guint frontend, const char* channels_conf, const char* recordings_dir, const char* name);
static gboolean dvb_manager_real_AddDeviceToExistingGroup (DVBIDBusManager* base, guint adapter, guint frontend, guint group_id);
static gboolean dvb_manager_real_RemoveDeviceFromGroup (DVBIDBusManager* base, guint adapter, guint frontend, guint group_id);
static char* dvb_manager_real_GetDeviceGroupName (DVBIDBusManager* base, guint group_id);
static char* dvb_manager_real_GetChannelList (DVBIDBusManager* base, guint group_id);
static char** dvb_manager_real_GetDeviceGroupMembers (DVBIDBusManager* base, guint group_id, int* result_length1);
static char* dvb_manager_real_GetNameOfRegisteredDevice (DVBIDBusManager* base, guint adapter, guint frontend);
static char* dvb_manager_real_GetSchedule (DVBIDBusManager* base, guint group_id, guint channel_sid);
static DVBDevice* dvb_manager_create_device (guint adapter, guint frontend, const char* channels_conf, const char* recordings_dir);
static DBusGConnection* dvb_manager_get_dbus_connection (void);
static gboolean dvb_manager_device_is_in_any_group (DVBManager* self, DVBDevice* device);
static void dvb_manager_on_scanner_destroyed (DVBManager* self, DVBScanner* scanner);
static DVBDevice* dvb_manager_get_registered_device (DVBManager* self, guint adapter, guint frontend);
static DVBDeviceGroup* dvb_manager_get_device_group_of_device (DVBManager* self, DVBDevice* device);
static void dvb_manager_remove_group (DVBManager* self, DVBDeviceGroup* devgroup);
static GObject * dvb_manager_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static gpointer dvb_manager_parent_class = NULL;
void _dvb_manager_dbus_unregister (DBusConnection* connection, void* user_data);
static DBusMessage* _dbus_dvb_manager_introspect (DVBManager* self, DBusConnection* connection, DBusMessage* message);
static DVBIDBusManagerIface* dvb_manager_dvb_id_bus_manager_parent_iface = NULL;
static void dvb_manager_finalize (GObject* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static int _vala_strcmp0 (const char * str1, const char * str2);
static void _vala_dbus_register_object (DBusConnection* connection, const char* path, void* object);

static const DBusObjectPathVTable _dvb_manager_dbus_path_vtable = {_dvb_manager_dbus_unregister, dvb_manager_dbus_message};
static const _DBusObjectVTable _dvb_manager_dbus_vtable = {dvb_manager_dbus_register_object};


DVBManager* dvb_manager_get_instance (void) {
	g_static_rec_mutex_lock (&dvb_manager_instance_mutex);
	if (dvb_manager_instance == NULL) {
		DVBManager* _tmp0;
		_tmp0 = NULL;
		dvb_manager_instance = (_tmp0 = dvb_manager_new (), (dvb_manager_instance == NULL) ? NULL : (dvb_manager_instance = (g_object_unref (dvb_manager_instance), NULL)), _tmp0);
	}
	g_static_rec_mutex_unlock (&dvb_manager_instance_mutex);
	return dvb_manager_instance;
}


void dvb_manager_shutdown (void) {
	DVBManager* _tmp0;
	DVBManager* m;
	g_static_rec_mutex_lock (&dvb_manager_instance_mutex);
	_tmp0 = NULL;
	m = (_tmp0 = dvb_manager_instance, (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0));
	if (dvb_manager_instance != NULL) {
		DVBManager* _tmp5;
		{
			GeeCollection* _tmp1;
			GeeIterator* _tmp2;
			GeeIterator* _scanner_it;
			_tmp1 = NULL;
			_tmp2 = NULL;
			_scanner_it = (_tmp2 = gee_iterable_iterator ((GeeIterable*) (_tmp1 = gee_map_get_values ((GeeMap*) m->priv->scanners))), (_tmp1 == NULL) ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)), _tmp2);
			while (gee_iterator_next (_scanner_it)) {
				DVBScanner* scanner;
				scanner = (DVBScanner*) gee_iterator_get (_scanner_it);
				g_debug ("Manager.vala:54: Stopping scanner");
				dvb_scanner_Destroy (scanner);
				(scanner == NULL) ? NULL : (scanner = (g_object_unref (scanner), NULL));
			}
			(_scanner_it == NULL) ? NULL : (_scanner_it = (g_object_unref (_scanner_it), NULL));
		}
		gee_map_clear ((GeeMap*) m->priv->scanners);
		gee_collection_clear ((GeeCollection*) m->priv->schedules);
		gee_collection_clear ((GeeCollection*) m->priv->recorders);
		gee_collection_clear ((GeeCollection*) m->priv->channellists);
		gee_collection_clear ((GeeCollection*) m->priv->epgscanners);
		{
			GeeCollection* _tmp3;
			GeeIterator* _tmp4;
			GeeIterator* _devgrp_it;
			_tmp3 = NULL;
			_tmp4 = NULL;
			_devgrp_it = (_tmp4 = gee_iterable_iterator ((GeeIterable*) (_tmp3 = gee_map_get_values ((GeeMap*) m->priv->devices))), (_tmp3 == NULL) ? NULL : (_tmp3 = (g_object_unref (_tmp3), NULL)), _tmp4);
			while (gee_iterator_next (_devgrp_it)) {
				DVBDeviceGroup* devgrp;
				devgrp = (DVBDeviceGroup*) gee_iterator_get (_devgrp_it);
				dvb_device_group_destroy (devgrp);
				(devgrp == NULL) ? NULL : (devgrp = (g_object_unref (devgrp), NULL));
			}
			(_devgrp_it == NULL) ? NULL : (_devgrp_it = (g_object_unref (_devgrp_it), NULL));
		}
		gee_map_clear ((GeeMap*) m->priv->devices);
		_tmp5 = NULL;
		dvb_manager_instance = (_tmp5 = NULL, (dvb_manager_instance == NULL) ? NULL : (dvb_manager_instance = (g_object_unref (dvb_manager_instance), NULL)), _tmp5);
	}
	g_static_rec_mutex_unlock (&dvb_manager_instance_mutex);
	(m == NULL) ? NULL : (m = (g_object_unref (m), NULL));
}


static void _dvb_manager_on_scanner_destroyed_dvb_scanner_destroyed (DVBScanner* _sender, gpointer self) {
	dvb_manager_on_scanner_destroyed (self, _sender);
}


/**
         * @adapter: Number of the device's adapter
         * @frontend: Number of the device's frontend
         * @returns: Object path and interface to the scanner service
         *
         * Get the object path of the channel scanner for this device.
         */
static char** dvb_manager_real_GetScannerForDevice (DVBIDBusManager* base, guint adapter, guint frontend, int* result_length1) {
	DVBManager * self;
	char* path;
	DVBDevice* device;
	DVBDevice* reg_dev;
	char* dbusiface;
	char** _tmp23;
	const char* _tmp22;
	const char* _tmp21;
	char** _tmp24;
	char** _tmp25;
	self = (DVBManager*) base;
	path = g_strdup_printf (DVB_CONSTANTS_DBUS_SCANNER_PATH, adapter, frontend);
	device = NULL;
	reg_dev = dvb_manager_get_registered_device (self, adapter, frontend);
	if (reg_dev == NULL) {
		DVBDevice* _tmp0;
		/* Create new device*/
		_tmp0 = NULL;
		device = (_tmp0 = dvb_device_new (adapter, frontend, TRUE), (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), _tmp0);
	} else {
		DVBEPGScanner* _tmp2;
		DVBDeviceGroup* _tmp1;
		DVBEPGScanner* _tmp3;
		DVBEPGScanner* epgscanner;
		DVBDevice* _tmp5;
		DVBDevice* _tmp4;
		/* Stop epgscanner for device if there's any*/
		_tmp2 = NULL;
		_tmp1 = NULL;
		_tmp3 = NULL;
		epgscanner = (_tmp3 = (_tmp2 = dvb_device_group_get_epgscanner (_tmp1 = dvb_manager_get_device_group_of_device (self, reg_dev)), (_tmp2 == NULL) ? NULL : g_object_ref (_tmp2)), (_tmp1 == NULL) ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)), _tmp3);
		if (epgscanner != NULL) {
			dvb_epg_scanner_stop (epgscanner);
		}
		/* Assign existing device*/
		_tmp5 = NULL;
		_tmp4 = NULL;
		device = (_tmp5 = (_tmp4 = reg_dev, (_tmp4 == NULL) ? NULL : g_object_ref (_tmp4)), (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), _tmp5);
		(epgscanner == NULL) ? NULL : (epgscanner = (g_object_unref (epgscanner), NULL));
	}
	dbusiface = NULL;
	switch (dvb_device_get_Type (device)) {
		case DVB_ADAPTER_TYPE_DVB_T:
		{
			char* _tmp6;
			_tmp6 = NULL;
			dbusiface = (_tmp6 = g_strdup ("org.gnome.DVB.Scanner.Terrestrial"), dbusiface = (g_free (dbusiface), NULL), _tmp6);
			break;
		}
		case DVB_ADAPTER_TYPE_DVB_S:
		{
			char* _tmp7;
			_tmp7 = NULL;
			dbusiface = (_tmp7 = g_strdup ("org.gnome.DVB.Scanner.Satellite"), dbusiface = (g_free (dbusiface), NULL), _tmp7);
			break;
		}
		case DVB_ADAPTER_TYPE_DVB_C:
		{
			char* _tmp8;
			_tmp8 = NULL;
			dbusiface = (_tmp8 = g_strdup ("org.gnome.DVB.Scanner.Cable"), dbusiface = (g_free (dbusiface), NULL), _tmp8);
			break;
		}
	}
	if (dbusiface == NULL) {
		char** _tmp9;
		char** _tmp10;
		char** _tmp11;
		g_critical ("Manager.vala:117: Unknown adapter type");
		_tmp9 = NULL;
		_tmp10 = NULL;
		_tmp11 = NULL;
		return (_tmp11 = (_tmp10 = (_tmp9 = g_new0 (char*, 2 + 1), _tmp9[0] = g_strdup (""), _tmp9[1] = g_strdup (""), _tmp9), *result_length1 = 2, _tmp10), path = (g_free (path), NULL), (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), (reg_dev == NULL) ? NULL : (reg_dev = (g_object_unref (reg_dev), NULL)), dbusiface = (g_free (dbusiface), NULL), _tmp11);
	}
	if (!gee_map_contains ((GeeMap*) self->priv->scanners, path)) {
		DVBScanner* scanner;
		DBusGConnection* conn;
		scanner = NULL;
		switch (dvb_device_get_Type (device)) {
			case DVB_ADAPTER_TYPE_DVB_T:
			{
				DVBScanner* _tmp12;
				_tmp12 = NULL;
				scanner = (_tmp12 = (DVBScanner*) dvb_terrestrial_scanner_new (device), (scanner == NULL) ? NULL : (scanner = (g_object_unref (scanner), NULL)), _tmp12);
				break;
			}
			case DVB_ADAPTER_TYPE_DVB_S:
			{
				DVBScanner* _tmp13;
				_tmp13 = NULL;
				scanner = (_tmp13 = (DVBScanner*) dvb_satellite_scanner_new (device), (scanner == NULL) ? NULL : (scanner = (g_object_unref (scanner), NULL)), _tmp13);
				break;
			}
			case DVB_ADAPTER_TYPE_DVB_C:
			{
				DVBScanner* _tmp14;
				_tmp14 = NULL;
				scanner = (_tmp14 = (DVBScanner*) dvb_cable_scanner_new (device), (scanner == NULL) ? NULL : (scanner = (g_object_unref (scanner), NULL)), _tmp14);
				break;
			}
		}
		if (scanner == NULL) {
			char** _tmp15;
			char** _tmp16;
			char** _tmp17;
			g_critical ("Manager.vala:138: Unknown adapter type");
			_tmp15 = NULL;
			_tmp16 = NULL;
			_tmp17 = NULL;
			return (_tmp17 = (_tmp16 = (_tmp15 = g_new0 (char*, 2 + 1), _tmp15[0] = g_strdup (""), _tmp15[1] = g_strdup (""), _tmp15), *result_length1 = 2, _tmp16), (scanner == NULL) ? NULL : (scanner = (g_object_unref (scanner), NULL)), path = (g_free (path), NULL), (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), (reg_dev == NULL) ? NULL : (reg_dev = (g_object_unref (reg_dev), NULL)), dbusiface = (g_free (dbusiface), NULL), _tmp17);
		}
		g_signal_connect_object (scanner, "destroyed", (GCallback) _dvb_manager_on_scanner_destroyed_dvb_scanner_destroyed, self, 0);
		gee_map_set ((GeeMap*) self->priv->scanners, path, scanner);
		conn = dvb_manager_get_dbus_connection ();
		if (conn == NULL) {
			char** _tmp18;
			char** _tmp19;
			char** _tmp20;
			_tmp18 = NULL;
			_tmp19 = NULL;
			_tmp20 = NULL;
			return (_tmp20 = (_tmp19 = (_tmp18 = g_new0 (char*, 0 + 1), _tmp18), *result_length1 = 0, _tmp19), (scanner == NULL) ? NULL : (scanner = (g_object_unref (scanner), NULL)), (conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL)), path = (g_free (path), NULL), (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), (reg_dev == NULL) ? NULL : (reg_dev = (g_object_unref (reg_dev), NULL)), dbusiface = (g_free (dbusiface), NULL), _tmp20);
		}
		_vala_dbus_register_object (dbus_g_connection_get_connection (conn), path, (GObject*) scanner);
		g_debug ("Manager.vala:153: Created new Scanner D-Bus service for adapter %u, frontend %u (%s)", adapter, frontend, dbusiface);
		(scanner == NULL) ? NULL : (scanner = (g_object_unref (scanner), NULL));
		(conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL));
	}
	_tmp23 = NULL;
	_tmp22 = NULL;
	_tmp21 = NULL;
	_tmp24 = NULL;
	_tmp25 = NULL;
	return (_tmp25 = (_tmp24 = (_tmp23 = g_new0 (char*, 2 + 1), _tmp23[0] = (_tmp21 = path, (_tmp21 == NULL) ? NULL : g_strdup (_tmp21)), _tmp23[1] = (_tmp22 = dbusiface, (_tmp22 == NULL) ? NULL : g_strdup (_tmp22)), _tmp23), *result_length1 = 2, _tmp24), path = (g_free (path), NULL), (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), (reg_dev == NULL) ? NULL : (reg_dev = (g_object_unref (reg_dev), NULL)), dbusiface = (g_free (dbusiface), NULL), _tmp25);
}


/**
         * @returns: Device groups' ID
         */
static guint* dvb_manager_real_GetRegisteredDeviceGroups (DVBIDBusManager* base, int* result_length1) {
	DVBManager * self;
	guint* _tmp1;
	gint devs_size;
	gint devs_length1;
	gint _tmp0;
	guint* devs;
	gint i;
	guint* _tmp4;
	self = (DVBManager*) base;
	_tmp1 = NULL;
	devs = (_tmp1 = g_new0 (guint, _tmp0 = gee_map_get_size ((GeeMap*) self->priv->devices)), devs_length1 = _tmp0, devs_size = devs_length1, _tmp1);
	i = 0;
	{
		GeeSet* _tmp2;
		GeeIterator* _tmp3;
		GeeIterator* _key_it;
		_tmp2 = NULL;
		_tmp3 = NULL;
		_key_it = (_tmp3 = gee_iterable_iterator ((GeeIterable*) (_tmp2 = gee_map_get_keys ((GeeMap*) self->priv->devices))), (_tmp2 == NULL) ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)), _tmp3);
		while (gee_iterator_next (_key_it)) {
			guint key;
			key = GPOINTER_TO_UINT (gee_iterator_get (_key_it));
			devs[i] = key;
			i++;
		}
		(_key_it == NULL) ? NULL : (_key_it = (g_object_unref (_key_it), NULL));
	}
	_tmp4 = NULL;
	return (_tmp4 = devs, *result_length1 = devs_length1, _tmp4);
}


/**
         * @group_id: ID of device group
         * @returns: Name of adapter type the group holds
         * or an empty string when group with given id doesn't exist.
         */
static char* dvb_manager_real_GetTypeOfDeviceGroup (DVBIDBusManager* base, guint group_id) {
	DVBManager * self;
	self = (DVBManager*) base;
	if (gee_map_contains ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id))) {
		DVBDeviceGroup* devgroup;
		char* type_str;
		char* _tmp4;
		devgroup = (DVBDeviceGroup*) gee_map_get ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id));
		type_str = NULL;
		switch (dvb_device_group_get_Type (devgroup)) {
			case DVB_ADAPTER_TYPE_DVB_T:
			{
				char* _tmp0;
				_tmp0 = NULL;
				type_str = (_tmp0 = g_strdup ("DVB-T"), type_str = (g_free (type_str), NULL), _tmp0);
				break;
			}
			case DVB_ADAPTER_TYPE_DVB_S:
			{
				char* _tmp1;
				_tmp1 = NULL;
				type_str = (_tmp1 = g_strdup ("DVB-S"), type_str = (g_free (type_str), NULL), _tmp1);
				break;
			}
			case DVB_ADAPTER_TYPE_DVB_C:
			{
				char* _tmp2;
				_tmp2 = NULL;
				type_str = (_tmp2 = g_strdup ("DVB-C"), type_str = (g_free (type_str), NULL), _tmp2);
				break;
			}
			default:
			{
				char* _tmp3;
				_tmp3 = NULL;
				type_str = (_tmp3 = g_strdup (""), type_str = (g_free (type_str), NULL), _tmp3);
				break;
			}
		}
		_tmp4 = NULL;
		return (_tmp4 = type_str, (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), _tmp4);
	}
	return g_strdup ("");
}


/**
         * @group_id: ID of device group
         * @returns: Object path of the device's recorder
         * 
         * Returns the object path to the device's recorder.
         * The device group must be registered with AddDeviceToNewGroup () first.
         */
static char* dvb_manager_real_GetRecorder (DVBIDBusManager* base, guint group_id) {
	DVBManager * self;
	char* path;
	self = (DVBManager*) base;
	path = g_strdup_printf (DVB_CONSTANTS_DBUS_RECORDER_PATH, group_id);
	if (!gee_collection_contains ((GeeCollection*) self->priv->recorders, GUINT_TO_POINTER (group_id))) {
		DVBDeviceGroup* devgroup;
		DVBRecorder* _tmp1;
		DVBRecorder* recorder;
		DBusGConnection* conn;
		g_debug ("Manager.vala:205: Creating new Recorder D-Bus service for group %u", group_id);
		devgroup = dvb_manager_get_device_group_if_exists (self, group_id);
		if (devgroup == NULL) {
			char* _tmp0;
			_tmp0 = NULL;
			return (_tmp0 = g_strdup (""), (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), path = (g_free (path), NULL), _tmp0);
		}
		_tmp1 = NULL;
		recorder = (_tmp1 = dvb_device_group_get_recorder (devgroup), (_tmp1 == NULL) ? NULL : g_object_ref (_tmp1));
		conn = dvb_manager_get_dbus_connection ();
		if (conn == NULL) {
			char* _tmp2;
			_tmp2 = NULL;
			return (_tmp2 = g_strdup (""), (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), (recorder == NULL) ? NULL : (recorder = (g_object_unref (recorder), NULL)), (conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL)), path = (g_free (path), NULL), _tmp2);
		}
		_vala_dbus_register_object (dbus_g_connection_get_connection (conn), path, (GObject*) recorder);
		gee_collection_add ((GeeCollection*) self->priv->recorders, GUINT_TO_POINTER (group_id));
		(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
		(recorder == NULL) ? NULL : (recorder = (g_object_unref (recorder), NULL));
		(conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL));
	}
	return path;
}


/**
         * @adapter: Number of the device's adapter
         * @frontend: Number of the device's frontend
         * @channels_conf: Path to channels.conf for this device
         * @recordings_dir: Path where the recordings should be stored
         * @name: Name of group
         * @returns: TRUE when the device has been registered successfully
         *
         * Creates a new DeviceGroup and new DVB device whereas the
         * DVB device is the reference device of this group (i.e.
         * all other devices of this group will inherit the settings
         * of the reference device).
         */
static gboolean dvb_manager_real_AddDeviceToNewGroup (DVBIDBusManager* base, guint adapter, guint frontend, const char* channels_conf, const char* recordings_dir, const char* name) {
	DVBManager * self;
	DVBDevice* device;
	DVBDeviceGroup* devgroup;
	gboolean _tmp2;
	self = (DVBManager*) base;
	g_return_val_if_fail (channels_conf != NULL, FALSE);
	g_return_val_if_fail (recordings_dir != NULL, FALSE);
	g_return_val_if_fail (name != NULL, FALSE);
	device = dvb_manager_create_device (adapter, frontend, channels_conf, recordings_dir);
	if (device == NULL) {
		gboolean _tmp0;
		return (_tmp0 = FALSE, (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), _tmp0);
	}
	/* Check if device is already assigned to other group*/
	if (dvb_manager_device_is_in_any_group (self, device)) {
		gboolean _tmp1;
		return (_tmp1 = FALSE, (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), _tmp1);
	}
	self->priv->device_group_counter++;
	devgroup = dvb_device_group_new (self->priv->device_group_counter, device);
	dvb_device_group_set_Name (devgroup, name);
	dvb_manager_add_device_group (self, devgroup);
	g_signal_emit_by_name ((DVBIDBusManager*) self, "changed", self->priv->device_group_counter, (guint) DVB_CHANGE_TYPE_ADDED);
	return (_tmp2 = TRUE, (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), _tmp2);
}


/**
         * @adapter: Number of the device's adapter
         * @frontend: Number of the device's frontend
         * @group_id: ID of device group
         * @returns: TRUE when the device has been registered successfully
         *
         * Creates a new device and adds it to the specified DeviceGroup.
         * The new device will inherit all settings from the group's
         * reference device.
         */
static gboolean dvb_manager_real_AddDeviceToExistingGroup (DVBIDBusManager* base, guint adapter, guint frontend, guint group_id) {
	DVBManager * self;
	self = (DVBManager*) base;
	if (gee_map_contains ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id))) {
		DVBDevice* device;
		DVBDeviceGroup* devgroup;
		/* When the device is already registered we
		 might see some errors if the device is
		 currently in use*/
		device = dvb_device_new (adapter, frontend, TRUE);
		if (device == NULL) {
			gboolean _tmp0;
			return (_tmp0 = FALSE, (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), _tmp0);
		}
		if (dvb_manager_device_is_in_any_group (self, device)) {
			gboolean _tmp1;
			g_debug ("Device with adapter %u, frontend %u is" "already part of a group", adapter, frontend);
			return (_tmp1 = FALSE, (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), _tmp1);
		}
		g_debug ("Manager.vala:289: Adding device with adapter %u, frontend %u to group %u", adapter, frontend, group_id);
		devgroup = (DVBDeviceGroup*) gee_map_get ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id));
		if (dvb_device_group_add (devgroup, device)) {
			gboolean _tmp2;
			dvb_config_store_add_device_to_group (dvb_factory_get_config_store (), device, devgroup);
			g_signal_emit_by_name ((DVBIDBusManager*) self, "group-changed", group_id, adapter, frontend, (guint) DVB_CHANGE_TYPE_ADDED);
			return (_tmp2 = TRUE, (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), _tmp2);
		}
		(device == NULL) ? NULL : (device = (g_object_unref (device), NULL));
		(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
	}
	return FALSE;
}


/**
         * @adapter: Number of the device's adapter
         * @frontend: Number of the device's frontend
         * @group_id: ID of device group
         * @returns: TRUE when device has been removed successfully
         *
         * Removes the device from the specified group. If the group contains
         * no devices after the removal it's removed as well.
         */
static gboolean dvb_manager_real_RemoveDeviceFromGroup (DVBIDBusManager* base, guint adapter, guint frontend, guint group_id) {
	DVBManager * self;
	self = (DVBManager*) base;
	if (gee_map_contains ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id))) {
		DVBDeviceGroup* devgroup;
		DVBDevice* dev;
		devgroup = (DVBDeviceGroup*) gee_map_get ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id));
		dev = dvb_device_new (adapter, frontend, FALSE);
		if (dvb_device_group_contains (devgroup, dev)) {
			if (dvb_device_group_remove (devgroup, dev)) {
				DVBEPGScanner* _tmp0;
				DVBEPGScanner* epgscanner;
				gboolean _tmp1;
				/* Stop epgscanner, because it might use the
				 device we want to unregister*/
				_tmp0 = NULL;
				epgscanner = (_tmp0 = dvb_device_group_get_epgscanner (devgroup), (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0));
				if (epgscanner != NULL) {
					dvb_epg_scanner_stop (epgscanner);
				}
				dvb_config_store_remove_device_from_group (dvb_factory_get_config_store (), dev, devgroup);
				g_signal_emit_by_name ((DVBIDBusManager*) self, "group-changed", group_id, adapter, frontend, (guint) DVB_CHANGE_TYPE_DELETED);
				/* Group has no devices anymore, delete it*/
				if (dvb_device_group_get_size (devgroup) == 0) {
					dvb_manager_remove_group (self, devgroup);
				} else {
					/* We still have a device, start EPG scanner again*/
					if (epgscanner != NULL) {
						dvb_epg_scanner_start (epgscanner);
					}
				}
				return (_tmp1 = TRUE, (epgscanner == NULL) ? NULL : (epgscanner = (g_object_unref (epgscanner), NULL)), (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), (dev == NULL) ? NULL : (dev = (g_object_unref (dev), NULL)), _tmp1);
			}
		}
		(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
		(dev == NULL) ? NULL : (dev = (g_object_unref (dev), NULL));
	}
	return FALSE;
}


/**
         * @group_id: ID of device group
         * @returns: Name of specified device group or
         * empty string if group with given ID doesn't exist
         */
static char* dvb_manager_real_GetDeviceGroupName (DVBIDBusManager* base, guint group_id) {
	DVBManager * self;
	char* val;
	self = (DVBManager*) base;
	val = g_strdup ("");
	if (gee_map_contains ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id))) {
		DVBDeviceGroup* devgroup;
		char* _tmp1;
		const char* _tmp0;
		devgroup = (DVBDeviceGroup*) gee_map_get ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id));
		_tmp1 = NULL;
		_tmp0 = NULL;
		val = (_tmp1 = (_tmp0 = dvb_device_group_get_Name (devgroup), (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), val = (g_free (val), NULL), _tmp1);
		(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
	}
	return val;
}


/**
         * @group_id: ID of device group
         * @returns: Object path to the ChannelList service for this device
         *
         * The device group must be registered with AddDeviceToNewGroup () first.
         */
static char* dvb_manager_real_GetChannelList (DVBIDBusManager* base, guint group_id) {
	DVBManager * self;
	char* path;
	self = (DVBManager*) base;
	path = g_strdup_printf (DVB_CONSTANTS_DBUS_CHANNEL_LIST_PATH, group_id);
	if (!gee_collection_contains ((GeeCollection*) self->priv->channellists, GUINT_TO_POINTER (group_id))) {
		DVBDeviceGroup* devgroup;
		DVBChannelList* _tmp1;
		DVBChannelList* channels;
		DBusGConnection* conn;
		g_debug ("Manager.vala:375: Creating new ChannelList D-Bus service for group %u", group_id);
		devgroup = dvb_manager_get_device_group_if_exists (self, group_id);
		if (devgroup == NULL) {
			char* _tmp0;
			_tmp0 = NULL;
			return (_tmp0 = g_strdup (""), (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), path = (g_free (path), NULL), _tmp0);
		}
		_tmp1 = NULL;
		channels = (_tmp1 = dvb_device_group_get_Channels (devgroup), (_tmp1 == NULL) ? NULL : g_object_ref (_tmp1));
		conn = dvb_manager_get_dbus_connection ();
		if (conn == NULL) {
			char* _tmp2;
			_tmp2 = NULL;
			return (_tmp2 = g_strdup (""), (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), (channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL)), (conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL)), path = (g_free (path), NULL), _tmp2);
		}
		_vala_dbus_register_object (dbus_g_connection_get_connection (conn), path, (GObject*) channels);
		gee_collection_add ((GeeCollection*) self->priv->channellists, GUINT_TO_POINTER (group_id));
		(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
		(channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL));
		(conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL));
	}
	return path;
}


/**
         * @group_id: ID of device group
         * @returns: List of paths to the devices that are part of
         * the specified group (e.g. /dev/dvb/adapter0/frontend0)
         */
static char** dvb_manager_real_GetDeviceGroupMembers (DVBIDBusManager* base, guint group_id, int* result_length1) {
	DVBManager * self;
	gint groupdevs_size;
	gint groupdevs_length1;
	char** groupdevs;
	char** _tmp4;
	self = (DVBManager*) base;
	groupdevs = (groupdevs_length1 = 0, NULL);
	if (gee_map_contains ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id))) {
		DVBDeviceGroup* devgroup;
		char** _tmp1;
		gint _tmp0;
		gint i;
		devgroup = (DVBDeviceGroup*) gee_map_get ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id));
		_tmp1 = NULL;
		groupdevs = (_tmp1 = g_new0 (char*, (_tmp0 = dvb_device_group_get_size (devgroup)) + 1), groupdevs = (_vala_array_free (groupdevs, groupdevs_length1, (GDestroyNotify) g_free), NULL), groupdevs_length1 = _tmp0, groupdevs_size = groupdevs_length1, _tmp1);
		i = 0;
		{
			GeeIterator* _dev_it;
			_dev_it = gee_iterable_iterator ((GeeIterable*) devgroup);
			while (gee_iterator_next (_dev_it)) {
				DVBDevice* dev;
				char* _tmp2;
				dev = (DVBDevice*) gee_iterator_get (_dev_it);
				_tmp2 = NULL;
				groupdevs[i] = (_tmp2 = g_strdup_printf (DVB_CONSTANTS_DVB_DEVICE_PATH, dvb_device_get_Adapter (dev), dvb_device_get_Frontend (dev)), groupdevs[i] = (g_free (groupdevs[i]), NULL), _tmp2);
				i++;
				(dev == NULL) ? NULL : (dev = (g_object_unref (dev), NULL));
			}
			(_dev_it == NULL) ? NULL : (_dev_it = (g_object_unref (_dev_it), NULL));
		}
		(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
	} else {
		char** _tmp3;
		_tmp3 = NULL;
		groupdevs = (_tmp3 = g_new0 (char*, 0 + 1), groupdevs = (_vala_array_free (groupdevs, groupdevs_length1, (GDestroyNotify) g_free), NULL), groupdevs_length1 = 0, groupdevs_size = groupdevs_length1, _tmp3);
	}
	_tmp4 = NULL;
	return (_tmp4 = groupdevs, *result_length1 = groupdevs_length1, _tmp4);
}


/**
         * @adapter: Adapter of device
         * @frontend: Frontend of device
         * @returns: The name of the device or "Unknown"
         *
         * The device must be part of group, otherwise "Unknown"
         * is returned.
         */
static char* dvb_manager_real_GetNameOfRegisteredDevice (DVBIDBusManager* base, guint adapter, guint frontend) {
	DVBManager * self;
	DVBDevice* dev;
	self = (DVBManager*) base;
	dev = dvb_manager_get_registered_device (self, adapter, frontend);
	if (dev == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		return (_tmp0 = g_strdup ("Unknown"), (dev == NULL) ? NULL : (dev = (g_object_unref (dev), NULL)), _tmp0);
	} else {
		const char* _tmp1;
		char* _tmp2;
		_tmp1 = NULL;
		_tmp2 = NULL;
		return (_tmp2 = (_tmp1 = dvb_device_get_Name (dev), (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), (dev == NULL) ? NULL : (dev = (g_object_unref (dev), NULL)), _tmp2);
	}
	(dev == NULL) ? NULL : (dev = (g_object_unref (dev), NULL));
}


static char* dvb_manager_real_GetSchedule (DVBIDBusManager* base, guint group_id, guint channel_sid) {
	DVBManager * self;
	self = (DVBManager*) base;
	if (gee_map_contains ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id))) {
		DVBDeviceGroup* devgroup;
		devgroup = (DVBDeviceGroup*) gee_map_get ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id));
		if (dvb_channel_list_contains (dvb_device_group_get_Channels (devgroup), channel_sid)) {
			char* path;
			char* _tmp4;
			path = g_strdup_printf (DVB_CONSTANTS_DBUS_SCHEDULE_PATH, group_id, channel_sid);
			if (!gee_collection_contains ((GeeCollection*) self->priv->schedules, path)) {
				DBusGConnection* conn;
				DVBSchedule* _tmp2;
				DVBChannel* _tmp1;
				DVBSchedule* _tmp3;
				DVBSchedule* schedule;
				conn = dvb_manager_get_dbus_connection ();
				if (conn == NULL) {
					char* _tmp0;
					_tmp0 = NULL;
					return (_tmp0 = g_strdup (""), (conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL)), path = (g_free (path), NULL), (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), _tmp0);
				}
				_tmp2 = NULL;
				_tmp1 = NULL;
				_tmp3 = NULL;
				schedule = (_tmp3 = (_tmp2 = dvb_channel_get_Schedule (_tmp1 = dvb_channel_list_get_channel (dvb_device_group_get_Channels (devgroup), channel_sid)), (_tmp2 == NULL) ? NULL : g_object_ref (_tmp2)), (_tmp1 == NULL) ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)), _tmp3);
				_vala_dbus_register_object (dbus_g_connection_get_connection (conn), path, (GObject*) schedule);
				gee_collection_add ((GeeCollection*) self->priv->schedules, path);
				(conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL));
				(schedule == NULL) ? NULL : (schedule = (g_object_unref (schedule), NULL));
			}
			_tmp4 = NULL;
			return (_tmp4 = path, (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), _tmp4);
		}
		(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
	}
	return g_strdup ("");
}


/**
         * @returns: Whether the device has been added successfully
         *
         * Register device, create Recorder and ChannelList D-Bus service
         */
gboolean dvb_manager_add_device_group (DVBManager* self, DVBDeviceGroup* devgroup) {
	gboolean success;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (devgroup != NULL, FALSE);
	g_debug ("Manager.vala:472: Adding device group %u with %d devices", dvb_device_group_get_Id (devgroup), dvb_device_group_get_size (devgroup));
	success = FALSE;
	if (dvb_device_group_get_Type (devgroup) != DVB_ADAPTER_TYPE_UNKNOWN) {
		char* rec_path;
		char* channels_path;
		gboolean _tmp0;
		gee_map_set ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (dvb_device_group_get_Id (devgroup)), devgroup);
		rec_path = dvb_id_bus_manager_GetRecorder ((DVBIDBusManager*) self, dvb_device_group_get_Id (devgroup));
		channels_path = dvb_id_bus_manager_GetChannelList ((DVBIDBusManager*) self, dvb_device_group_get_Id (devgroup));
		dvb_config_store_add_device_group (dvb_factory_get_config_store (), devgroup);
		_tmp0 = FALSE;
		if (_vala_strcmp0 (rec_path, "") != 0) {
			_tmp0 = _vala_strcmp0 (channels_path, "") != 0;
		} else {
			_tmp0 = FALSE;
		}
		success = (_tmp0);
		rec_path = (g_free (rec_path), NULL);
		channels_path = (g_free (channels_path), NULL);
	} else {
		g_warning ("Manager.vala:486: Not adding device group %u of unknown type", dvb_device_group_get_Id (devgroup));
		success = FALSE;
	}
	if (dvb_device_group_get_Id (devgroup) > self->priv->device_group_counter) {
		self->priv->device_group_counter = dvb_device_group_get_Id (devgroup);
	}
	return success;
}


static DVBDevice* dvb_manager_create_device (guint adapter, guint frontend, const char* channels_conf, const char* recordings_dir) {
	GError * inner_error;
	GFile* channelsfile;
	GFile* recdir;
	DVBDevice* device;
	DVBChannelList* channels;
	DVBDevice* _tmp3;
	g_return_val_if_fail (channels_conf != NULL, NULL);
	g_return_val_if_fail (recordings_dir != NULL, NULL);
	inner_error = NULL;
	/* TODO Check if adapter and frontend exists*/
	channelsfile = g_file_new_for_path (channels_conf);
	recdir = g_file_new_for_path (recordings_dir);
	device = dvb_device_new (adapter, frontend, TRUE);
	dvb_device_set_RecordingsDirectory (device, recdir);
	channels = NULL;
	{
		DVBChannelList* _tmp0;
		DVBChannelList* _tmp1;
		_tmp0 = dvb_channel_list_restore_from_file (channelsfile, dvb_device_get_Type (device), &inner_error);
		if (inner_error != NULL) {
			goto __catch7_g_error;
			goto __finally7;
		}
		_tmp1 = NULL;
		channels = (_tmp1 = _tmp0, (channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL)), _tmp1);
	}
	goto __finally7;
	__catch7_g_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			DVBDevice* _tmp2;
			g_critical ("Manager.vala:510: Could not create channels list from %s: %s", channels_conf, e->message);
			_tmp2 = NULL;
			return (_tmp2 = NULL, (e == NULL) ? NULL : (e = (g_error_free (e), NULL)), (channelsfile == NULL) ? NULL : (channelsfile = (g_object_unref (channelsfile), NULL)), (recdir == NULL) ? NULL : (recdir = (g_object_unref (recdir), NULL)), (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), (channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL)), _tmp2);
		}
	}
	__finally7:
	if (inner_error != NULL) {
		(channelsfile == NULL) ? NULL : (channelsfile = (g_object_unref (channelsfile), NULL));
		(recdir == NULL) ? NULL : (recdir = (g_object_unref (recdir), NULL));
		(device == NULL) ? NULL : (device = (g_object_unref (device), NULL));
		(channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return NULL;
	}
	dvb_device_set_Channels (device, channels);
	_tmp3 = NULL;
	return (_tmp3 = device, (channelsfile == NULL) ? NULL : (channelsfile = (g_object_unref (channelsfile), NULL)), (recdir == NULL) ? NULL : (recdir = (g_object_unref (recdir), NULL)), (channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL)), _tmp3);
}


static DBusGConnection* dvb_manager_get_dbus_connection (void) {
	GError * inner_error;
	DBusGConnection* conn;
	inner_error = NULL;
	conn = NULL;
	{
		DBusGConnection* _tmp0;
		DBusGConnection* _tmp1;
		_tmp0 = dbus_g_bus_get (DBUS_BUS_SESSION, &inner_error);
		if (inner_error != NULL) {
			goto __catch8_g_error;
			goto __finally8;
		}
		_tmp1 = NULL;
		conn = (_tmp1 = _tmp0, (conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL)), _tmp1);
	}
	goto __finally8;
	__catch8_g_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			DBusGConnection* _tmp2;
			g_error ("Manager.vala:525: Could not get D-Bus session bus: %s", e->message);
			_tmp2 = NULL;
			return (_tmp2 = NULL, (e == NULL) ? NULL : (e = (g_error_free (e), NULL)), (conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL)), _tmp2);
		}
	}
	__finally8:
	if (inner_error != NULL) {
		(conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return NULL;
	}
	return conn;
}


DVBDeviceGroup* dvb_manager_get_device_group_if_exists (DVBManager* self, guint group_id) {
	g_return_val_if_fail (self != NULL, NULL);
	;
	if (gee_map_contains ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id))) {
		return (DVBDeviceGroup*) gee_map_get ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id));
	} else {
		return NULL;
	}
}


static gboolean dvb_manager_device_is_in_any_group (DVBManager* self, DVBDevice* device) {
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (device != NULL, FALSE);
	{
		GeeSet* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* _group_id_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		_group_id_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = gee_map_get_keys ((GeeMap*) self->priv->devices))), (_tmp0 == NULL) ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (_group_id_it)) {
			guint group_id;
			DVBDeviceGroup* devgroup;
			group_id = GPOINTER_TO_UINT (gee_iterator_get (_group_id_it));
			devgroup = (DVBDeviceGroup*) gee_map_get ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id));
			if (dvb_device_group_contains (devgroup, device)) {
				gboolean _tmp2;
				return (_tmp2 = TRUE, (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), (_group_id_it == NULL) ? NULL : (_group_id_it = (g_object_unref (_group_id_it), NULL)), _tmp2);
			}
			(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
		}
		(_group_id_it == NULL) ? NULL : (_group_id_it = (g_object_unref (_group_id_it), NULL));
	}
	return FALSE;
}


static void dvb_manager_on_scanner_destroyed (DVBManager* self, DVBScanner* scanner) {
	guint adapter;
	guint frontend;
	char* path;
	DVBDeviceGroup* devgroup;
	g_return_if_fail (self != NULL);
	g_return_if_fail (scanner != NULL);
	adapter = dvb_device_get_Adapter (dvb_scanner_get_Device (scanner));
	frontend = dvb_device_get_Frontend (dvb_scanner_get_Device (scanner));
	path = g_strdup_printf (DVB_CONSTANTS_DBUS_SCANNER_PATH, adapter, frontend);
	gee_map_remove ((GeeMap*) self->priv->scanners, path);
	g_debug ("Manager.vala:553: Destroying scanner for adapter %u, frontend %u (%s)", adapter, frontend, path);
	/* Start epgscanner for device again if there was one*/
	devgroup = dvb_manager_get_device_group_of_device (self, dvb_scanner_get_Device (scanner));
	if (devgroup != NULL) {
		DVBEPGScanner* _tmp0;
		DVBEPGScanner* epgscanner;
		_tmp0 = NULL;
		epgscanner = (_tmp0 = dvb_device_group_get_epgscanner (devgroup), (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0));
		if (epgscanner != NULL) {
			dvb_epg_scanner_start (epgscanner);
		}
		(epgscanner == NULL) ? NULL : (epgscanner = (g_object_unref (epgscanner), NULL));
	}
	path = (g_free (path), NULL);
	(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
}


static DVBDevice* dvb_manager_get_registered_device (DVBManager* self, guint adapter, guint frontend) {
	DVBDevice* fake_device;
	DVBDevice* _tmp3;
	g_return_val_if_fail (self != NULL, NULL);
	fake_device = dvb_device_new (adapter, frontend, FALSE);
	{
		GeeSet* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* _group_id_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		_group_id_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = gee_map_get_keys ((GeeMap*) self->priv->devices))), (_tmp0 == NULL) ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (_group_id_it)) {
			guint group_id;
			DVBDeviceGroup* devgroup;
			group_id = GPOINTER_TO_UINT (gee_iterator_get (_group_id_it));
			devgroup = (DVBDeviceGroup*) gee_map_get ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id));
			if (dvb_device_group_contains (devgroup, fake_device)) {
				{
					GeeIterator* _device_it;
					_device_it = gee_iterable_iterator ((GeeIterable*) devgroup);
					while (gee_iterator_next (_device_it)) {
						DVBDevice* device;
						device = (DVBDevice*) gee_iterator_get (_device_it);
						if (dvb_device_equal (fake_device, device)) {
							DVBDevice* _tmp2;
							_tmp2 = NULL;
							return (_tmp2 = device, (_device_it == NULL) ? NULL : (_device_it = (g_object_unref (_device_it), NULL)), (devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL)), (_group_id_it == NULL) ? NULL : (_group_id_it = (g_object_unref (_group_id_it), NULL)), (fake_device == NULL) ? NULL : (fake_device = (g_object_unref (fake_device), NULL)), _tmp2);
						}
						(device == NULL) ? NULL : (device = (g_object_unref (device), NULL));
					}
					(_device_it == NULL) ? NULL : (_device_it = (g_object_unref (_device_it), NULL));
				}
			}
			(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
		}
		(_group_id_it == NULL) ? NULL : (_group_id_it = (g_object_unref (_group_id_it), NULL));
	}
	_tmp3 = NULL;
	return (_tmp3 = NULL, (fake_device == NULL) ? NULL : (fake_device = (g_object_unref (fake_device), NULL)), _tmp3);
}


static DVBDeviceGroup* dvb_manager_get_device_group_of_device (DVBManager* self, DVBDevice* device) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (device != NULL, NULL);
	{
		GeeSet* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* _group_id_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		_group_id_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = gee_map_get_keys ((GeeMap*) self->priv->devices))), (_tmp0 == NULL) ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (_group_id_it)) {
			guint group_id;
			DVBDeviceGroup* devgroup;
			group_id = GPOINTER_TO_UINT (gee_iterator_get (_group_id_it));
			devgroup = (DVBDeviceGroup*) gee_map_get ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id));
			if (dvb_device_group_contains (devgroup, device)) {
				{
					GeeIterator* _grp_device_it;
					_grp_device_it = gee_iterable_iterator ((GeeIterable*) devgroup);
					while (gee_iterator_next (_grp_device_it)) {
						DVBDevice* grp_device;
						grp_device = (DVBDevice*) gee_iterator_get (_grp_device_it);
						if (dvb_device_equal (grp_device, device)) {
							DVBDeviceGroup* _tmp2;
							_tmp2 = NULL;
							return (_tmp2 = devgroup, (grp_device == NULL) ? NULL : (grp_device = (g_object_unref (grp_device), NULL)), (_grp_device_it == NULL) ? NULL : (_grp_device_it = (g_object_unref (_grp_device_it), NULL)), (_group_id_it == NULL) ? NULL : (_group_id_it = (g_object_unref (_group_id_it), NULL)), _tmp2);
						}
						(grp_device == NULL) ? NULL : (grp_device = (g_object_unref (grp_device), NULL));
					}
					(_grp_device_it == NULL) ? NULL : (_grp_device_it = (g_object_unref (_grp_device_it), NULL));
				}
			}
			(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
		}
		(_group_id_it == NULL) ? NULL : (_group_id_it = (g_object_unref (_group_id_it), NULL));
	}
	return NULL;
}


static void dvb_manager_remove_group (DVBManager* self, DVBDeviceGroup* devgroup) {
	guint group_id;
	g_return_if_fail (self != NULL);
	g_return_if_fail (devgroup != NULL);
	group_id = dvb_device_group_get_Id (devgroup);
	if (gee_map_remove ((GeeMap*) self->priv->devices, GUINT_TO_POINTER (group_id))) {
		DVBEPGScanner* _tmp0;
		DVBEPGScanner* epgscanner;
		gee_collection_remove ((GeeCollection*) self->priv->recorders, GUINT_TO_POINTER (group_id));
		gee_collection_remove ((GeeCollection*) self->priv->channellists, GUINT_TO_POINTER (group_id));
		gee_collection_remove ((GeeCollection*) self->priv->epgscanners, GUINT_TO_POINTER (group_id));
		_tmp0 = NULL;
		epgscanner = (_tmp0 = dvb_device_group_get_epgscanner (devgroup), (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0));
		/* Remove EPG scanner, too*/
		if (epgscanner != NULL) {
			gee_collection_remove ((GeeCollection*) self->priv->epgscanners, GUINT_TO_POINTER (group_id));
		}
		dvb_device_group_destroy (devgroup);
		dvb_config_store_remove_device_group (dvb_factory_get_config_store (), devgroup);
		g_signal_emit_by_name ((DVBIDBusManager*) self, "changed", group_id, (guint) DVB_CHANGE_TYPE_DELETED);
		(epgscanner == NULL) ? NULL : (epgscanner = (g_object_unref (epgscanner), NULL));
	}
}


DVBManager* dvb_manager_construct (GType object_type) {
	DVBManager * self;
	self = g_object_newv (object_type, 0, NULL);
	return self;
}


DVBManager* dvb_manager_new (void) {
	return dvb_manager_construct (DVB_TYPE_MANAGER);
}


static GObject * dvb_manager_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	DVBManagerClass * klass;
	GObjectClass * parent_class;
	DVBManager * self;
	klass = DVB_MANAGER_CLASS (g_type_class_peek (DVB_TYPE_MANAGER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = DVB_MANAGER (obj);
	{
		GeeHashMap* _tmp1;
		GeeHashMap* _tmp2;
		GeeHashSet* _tmp3;
		GeeHashSet* _tmp4;
		GeeHashSet* _tmp5;
		GeeHashSet* _tmp6;
		_tmp1 = NULL;
		self->priv->scanners = (_tmp1 = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, DVB_TYPE_SCANNER, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_str_hash, g_str_equal, g_direct_equal), (self->priv->scanners == NULL) ? NULL : (self->priv->scanners = (g_object_unref (self->priv->scanners), NULL)), _tmp1);
		_tmp2 = NULL;
		self->priv->devices = (_tmp2 = gee_hash_map_new (G_TYPE_UINT, NULL, NULL, DVB_TYPE_DEVICE_GROUP, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_hash, g_direct_equal, g_direct_equal), (self->priv->devices == NULL) ? NULL : (self->priv->devices = (g_object_unref (self->priv->devices), NULL)), _tmp2);
		_tmp3 = NULL;
		self->priv->schedules = (_tmp3 = gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal), (self->priv->schedules == NULL) ? NULL : (self->priv->schedules = (g_object_unref (self->priv->schedules), NULL)), _tmp3);
		_tmp4 = NULL;
		self->priv->recorders = (_tmp4 = gee_hash_set_new (G_TYPE_UINT, NULL, NULL, g_direct_hash, g_direct_equal), (self->priv->recorders == NULL) ? NULL : (self->priv->recorders = (g_object_unref (self->priv->recorders), NULL)), _tmp4);
		_tmp5 = NULL;
		self->priv->channellists = (_tmp5 = gee_hash_set_new (G_TYPE_UINT, NULL, NULL, g_direct_hash, g_direct_equal), (self->priv->channellists == NULL) ? NULL : (self->priv->channellists = (g_object_unref (self->priv->channellists), NULL)), _tmp5);
		_tmp6 = NULL;
		self->priv->epgscanners = (_tmp6 = gee_hash_set_new (G_TYPE_UINT, NULL, NULL, g_direct_hash, g_direct_equal), (self->priv->epgscanners == NULL) ? NULL : (self->priv->epgscanners = (g_object_unref (self->priv->epgscanners), NULL)), _tmp6);
		self->priv->device_group_counter = (guint) 0;
	}
	return obj;
}


void _dvb_manager_dbus_unregister (DBusConnection* connection, void* user_data) {
}


static DBusMessage* _dbus_dvb_manager_introspect (DVBManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessage* reply;
	DBusMessageIter iter;
	GString* xml_data;
	char** children;
	int i;
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	xml_data = g_string_new ("<!DOCTYPE node PUBLIC \"-//freedesktop//DTD D-BUS Object Introspection 1.0//EN\" \"http://www.freedesktop.org/standards/dbus/1.0/introspect.dtd\">\n");
	g_string_append (xml_data, "<node>\n<interface name=\"org.freedesktop.DBus.Introspectable\">\n  <method name=\"Introspect\">\n    <arg name=\"data\" direction=\"out\" type=\"s\"/>\n  </method>\n</interface>\n<interface name=\"org.freedesktop.DBus.Properties\">\n  <method name=\"Get\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"propname\" direction=\"in\" type=\"s\"/>\n    <arg name=\"value\" direction=\"out\" type=\"v\"/>\n  </method>\n  <method name=\"Set\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"propname\" direction=\"in\" type=\"s\"/>\n    <arg name=\"value\" direction=\"in\" type=\"v\"/>\n  </method>\n  <method name=\"GetAll\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"props\" direction=\"out\" type=\"a{sv}\"/>\n  </method>\n</interface>\n<interface name=\"org.gnome.DVB.Manager\">\n  <method name=\"GetScannerForDevice\">\n    <arg name=\"adapter\" type=\"u\" direction=\"in\"/>\n    <arg name=\"frontend\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"as\" direction=\"out\"/>\n  </method>\n  <method name=\"GetRegisteredDeviceGroups\">\n    <arg name=\"result\" type=\"au\" direction=\"out\"/>\n  </method>\n  <method name=\"GetTypeOfDeviceGroup\">\n    <arg name=\"group_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"GetRecorder\">\n    <arg name=\"group_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"AddDeviceToNewGroup\">\n    <arg name=\"adapter\" type=\"u\" direction=\"in\"/>\n    <arg name=\"frontend\" type=\"u\" direction=\"in\"/>\n    <arg name=\"channels_conf\" type=\"s\" direction=\"in\"/>\n    <arg name=\"recordings_dir\" type=\"s\" direction=\"in\"/>\n    <arg name=\"name\" type=\"s\" direction=\"in\"/>\n    <arg name=\"result\" type=\"b\" direction=\"out\"/>\n  </method>\n  <method name=\"AddDeviceToExistingGroup\">\n    <arg name=\"adapter\" type=\"u\" direction=\"in\"/>\n    <arg name=\"frontend\" type=\"u\" direction=\"in\"/>\n    <arg name=\"group_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"b\" direction=\"out\"/>\n  </method>\n  <method name=\"RemoveDeviceFromGroup\">\n    <arg name=\"adapter\" type=\"u\" direction=\"in\"/>\n    <arg name=\"frontend\" type=\"u\" direction=\"in\"/>\n    <arg name=\"group_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"b\" direction=\"out\"/>\n  </method>\n  <method name=\"GetDeviceGroupName\">\n    <arg name=\"group_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"GetChannelList\">\n    <arg name=\"group_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"GetDeviceGroupMembers\">\n    <arg name=\"group_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"as\" direction=\"out\"/>\n  </method>\n  <method name=\"GetNameOfRegisteredDevice\">\n    <arg name=\"adapter\" type=\"u\" direction=\"in\"/>\n    <arg name=\"frontend\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"GetSchedule\">\n    <arg name=\"group_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"channel_sid\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <signal name=\"Changed\">\n    <arg name=\"group_id\" type=\"u\"/>\n    <arg name=\"change_type\" type=\"u\"/>\n  </signal>\n  <signal name=\"GroupChanged\">\n    <arg name=\"group_id\" type=\"u\"/>\n    <arg name=\"adapter\" type=\"u\"/>\n    <arg name=\"frontend\" type=\"u\"/>\n    <arg name=\"change_type\" type=\"u\"/>\n  </signal>\n</interface>\n");
	dbus_connection_list_registered (connection, g_object_get_data ((GObject *) self, "dbus_object_path"), &children);
	for (i = 0; children[i]; i++) {
		g_string_append_printf (xml_data, "<node name=\"%s\"/>\n", children[i]);
	}
	g_string_append (xml_data, "</node>\n");
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &xml_data->str);
	g_string_free (xml_data, TRUE);
	return reply;
}


DBusHandlerResult dvb_manager_dbus_message (DBusConnection* connection, DBusMessage* message, void* object) {
	DBusMessage* reply;
	reply = NULL;
	if (dbus_message_is_method_call (message, "org.freedesktop.DBus.Introspectable", "Introspect")) {
		reply = _dbus_dvb_manager_introspect (object, connection, message);
	}
	if (reply) {
		dbus_connection_send (connection, reply, NULL);
		dbus_message_unref (reply);
		return DBUS_HANDLER_RESULT_HANDLED;
	} else if (dvb_id_bus_manager_dbus_message (connection, message, object) == DBUS_HANDLER_RESULT_HANDLED) {
		return DBUS_HANDLER_RESULT_HANDLED;
	} else {
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}
}


void dvb_manager_dbus_register_object (DBusConnection* connection, const char* path, void* object) {
	if (!g_object_get_data (object, "dbus_object_path")) {
		g_object_set_data (object, "dbus_object_path", g_strdup (path));
		dbus_connection_register_object_path (connection, path, &_dvb_manager_dbus_path_vtable, object);
	}
	dvb_id_bus_manager_dbus_register_object (connection, path, object);
}


static void dvb_manager_class_init (DVBManagerClass * klass) {
	GStaticRecMutex _tmp0 = {0};
	dvb_manager_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (DVBManagerPrivate));
	G_OBJECT_CLASS (klass)->constructor = dvb_manager_constructor;
	G_OBJECT_CLASS (klass)->finalize = dvb_manager_finalize;
	g_type_set_qdata (DVB_TYPE_MANAGER, g_quark_from_static_string ("DBusObjectVTable"), (void*) (&_dvb_manager_dbus_vtable));
	dvb_manager_instance_mutex = (g_static_rec_mutex_init (&_tmp0), _tmp0);
}


static void dvb_manager_dvb_id_bus_manager_interface_init (DVBIDBusManagerIface * iface) {
	dvb_manager_dvb_id_bus_manager_parent_iface = g_type_interface_peek_parent (iface);
	iface->GetScannerForDevice = dvb_manager_real_GetScannerForDevice;
	iface->GetRegisteredDeviceGroups = dvb_manager_real_GetRegisteredDeviceGroups;
	iface->GetTypeOfDeviceGroup = dvb_manager_real_GetTypeOfDeviceGroup;
	iface->GetRecorder = dvb_manager_real_GetRecorder;
	iface->AddDeviceToNewGroup = dvb_manager_real_AddDeviceToNewGroup;
	iface->AddDeviceToExistingGroup = dvb_manager_real_AddDeviceToExistingGroup;
	iface->RemoveDeviceFromGroup = dvb_manager_real_RemoveDeviceFromGroup;
	iface->GetDeviceGroupName = dvb_manager_real_GetDeviceGroupName;
	iface->GetChannelList = dvb_manager_real_GetChannelList;
	iface->GetDeviceGroupMembers = dvb_manager_real_GetDeviceGroupMembers;
	iface->GetNameOfRegisteredDevice = dvb_manager_real_GetNameOfRegisteredDevice;
	iface->GetSchedule = dvb_manager_real_GetSchedule;
}


static void dvb_manager_instance_init (DVBManager * self) {
	self->priv = DVB_MANAGER_GET_PRIVATE (self);
}


static void dvb_manager_finalize (GObject* obj) {
	DVBManager * self;
	self = DVB_MANAGER (obj);
	(self->priv->scanners == NULL) ? NULL : (self->priv->scanners = (g_object_unref (self->priv->scanners), NULL));
	(self->priv->devices == NULL) ? NULL : (self->priv->devices = (g_object_unref (self->priv->devices), NULL));
	(self->priv->schedules == NULL) ? NULL : (self->priv->schedules = (g_object_unref (self->priv->schedules), NULL));
	(self->priv->recorders == NULL) ? NULL : (self->priv->recorders = (g_object_unref (self->priv->recorders), NULL));
	(self->priv->channellists == NULL) ? NULL : (self->priv->channellists = (g_object_unref (self->priv->channellists), NULL));
	(self->priv->epgscanners == NULL) ? NULL : (self->priv->epgscanners = (g_object_unref (self->priv->epgscanners), NULL));
	G_OBJECT_CLASS (dvb_manager_parent_class)->finalize (obj);
}


GType dvb_manager_get_type (void) {
	static GType dvb_manager_type_id = 0;
	if (dvb_manager_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBManagerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_manager_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBManager), 0, (GInstanceInitFunc) dvb_manager_instance_init, NULL };
		static const GInterfaceInfo dvb_id_bus_manager_info = { (GInterfaceInitFunc) dvb_manager_dvb_id_bus_manager_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		dvb_manager_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBManager", &g_define_type_info, 0);
		g_type_add_interface_static (dvb_manager_type_id, DVB_TYPE_ID_BUS_MANAGER, &dvb_id_bus_manager_info);
	}
	return dvb_manager_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
	g_free (array);
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}


static void _vala_dbus_register_object (DBusConnection* connection, const char* path, void* object) {
	const _DBusObjectVTable * vtable;
	vtable = g_type_get_qdata (G_TYPE_FROM_INSTANCE (object), g_quark_from_static_string ("DBusObjectVTable"));
	if (vtable) {
		vtable->register_object (connection, path, object);
	} else {
		g_warning ("Object does not implement any D-Bus interface");
	}
}




