/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>


#define DVB_TYPE_ID_BUS_MANAGER (dvb_id_bus_manager_get_type ())
#define DVB_ID_BUS_MANAGER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_ID_BUS_MANAGER, DVBIDBusManager))
#define DVB_IS_ID_BUS_MANAGER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_ID_BUS_MANAGER))
#define DVB_ID_BUS_MANAGER_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), DVB_TYPE_ID_BUS_MANAGER, DVBIDBusManagerIface))

typedef struct _DVBIDBusManager DVBIDBusManager;
typedef struct _DVBIDBusManagerIface DVBIDBusManagerIface;
typedef struct _DVBIDBusManagerDBusProxy DVBIDBusManagerDBusProxy;
typedef DBusGProxyClass DVBIDBusManagerDBusProxyClass;
typedef struct _DBusObjectVTable _DBusObjectVTable;

struct _DVBIDBusManagerIface {
	GTypeInterface parent_iface;
	char** (*GetScannerForDevice) (DVBIDBusManager* self, guint adapter, guint frontend, int* result_length1);
	char** (*GetRegisteredDeviceGroups) (DVBIDBusManager* self, int* result_length1);
	char* (*GetDeviceGroup) (DVBIDBusManager* self, guint group_id);
	gboolean (*AddDeviceToNewGroup) (DVBIDBusManager* self, guint adapter, guint frontend, const char* channels_conf, const char* recordings_dir, const char* name);
	char* (*GetNameOfRegisteredDevice) (DVBIDBusManager* self, guint adapter, guint frontend);
	gint (*GetDeviceGroupSize) (DVBIDBusManager* self);
};

struct _DVBIDBusManagerDBusProxy {
	DBusGProxy parent_instance;
	gboolean disposed;
};

struct _DBusObjectVTable {
	void (*register_object) (DBusConnection*, const char*, void*);
};



GType dvb_id_bus_manager_get_type (void);
char** dvb_id_bus_manager_GetScannerForDevice (DVBIDBusManager* self, guint adapter, guint frontend, int* result_length1);
char** dvb_id_bus_manager_GetRegisteredDeviceGroups (DVBIDBusManager* self, int* result_length1);
char* dvb_id_bus_manager_GetDeviceGroup (DVBIDBusManager* self, guint group_id);
gboolean dvb_id_bus_manager_AddDeviceToNewGroup (DVBIDBusManager* self, guint adapter, guint frontend, const char* channels_conf, const char* recordings_dir, const char* name);
char* dvb_id_bus_manager_GetNameOfRegisteredDevice (DVBIDBusManager* self, guint adapter, guint frontend);
gint dvb_id_bus_manager_GetDeviceGroupSize (DVBIDBusManager* self);
void dvb_id_bus_manager_dbus_register_object (DBusConnection* connection, const char* path, void* object);
void _dvb_id_bus_manager_dbus_unregister (DBusConnection* connection, void* user_data);
DBusHandlerResult dvb_id_bus_manager_dbus_message (DBusConnection* connection, DBusMessage* message, void* object);
static DBusMessage* _dbus_dvb_id_bus_manager_introspect (DVBIDBusManager* self, DBusConnection* connection, DBusMessage* message);
static DBusMessage* _dbus_dvb_id_bus_manager_property_get_all (DVBIDBusManager* self, DBusConnection* connection, DBusMessage* message);
static DBusMessage* _dbus_dvb_id_bus_manager_GetScannerForDevice (DVBIDBusManager* self, DBusConnection* connection, DBusMessage* message);
static DBusMessage* _dbus_dvb_id_bus_manager_GetRegisteredDeviceGroups (DVBIDBusManager* self, DBusConnection* connection, DBusMessage* message);
static DBusMessage* _dbus_dvb_id_bus_manager_GetDeviceGroup (DVBIDBusManager* self, DBusConnection* connection, DBusMessage* message);
static DBusMessage* _dbus_dvb_id_bus_manager_AddDeviceToNewGroup (DVBIDBusManager* self, DBusConnection* connection, DBusMessage* message);
static DBusMessage* _dbus_dvb_id_bus_manager_GetNameOfRegisteredDevice (DVBIDBusManager* self, DBusConnection* connection, DBusMessage* message);
static DBusMessage* _dbus_dvb_id_bus_manager_GetDeviceGroupSize (DVBIDBusManager* self, DBusConnection* connection, DBusMessage* message);
static void _dbus_dvb_id_bus_manager_group_added (GObject* _sender, guint group_id, DBusConnection* _connection);
static void _dbus_dvb_id_bus_manager_group_removed (GObject* _sender, guint group_id, DBusConnection* _connection);
GType dvb_id_bus_manager_dbus_proxy_get_type (void);
DVBIDBusManager* dvb_id_bus_manager_dbus_proxy_new (DBusGConnection* connection, const char* name, const char* path);
static void _dbus_handle_dvb_id_bus_manager_group_added (DVBIDBusManager* self, DBusConnection* connection, DBusMessage* message);
static void _dbus_handle_dvb_id_bus_manager_group_removed (DVBIDBusManager* self, DBusConnection* connection, DBusMessage* message);
DBusHandlerResult dvb_id_bus_manager_dbus_proxy_filter (DBusConnection* connection, DBusMessage* message, void* user_data);
enum  {
	DVB_ID_BUS_MANAGER_DBUS_PROXY_DUMMY_PROPERTY
};
static char** dvb_id_bus_manager_dbus_proxy_GetScannerForDevice (DVBIDBusManager* self, guint adapter, guint frontend, int* result_length1);
static char** dvb_id_bus_manager_dbus_proxy_GetRegisteredDeviceGroups (DVBIDBusManager* self, int* result_length1);
static char* dvb_id_bus_manager_dbus_proxy_GetDeviceGroup (DVBIDBusManager* self, guint group_id);
static gboolean dvb_id_bus_manager_dbus_proxy_AddDeviceToNewGroup (DVBIDBusManager* self, guint adapter, guint frontend, const char* channels_conf, const char* recordings_dir, const char* name);
static char* dvb_id_bus_manager_dbus_proxy_GetNameOfRegisteredDevice (DVBIDBusManager* self, guint adapter, guint frontend);
static gint dvb_id_bus_manager_dbus_proxy_GetDeviceGroupSize (DVBIDBusManager* self);
static void dvb_id_bus_manager_dbus_proxy_interface_init (DVBIDBusManagerIface* iface);
static void dvb_id_bus_manager_dbus_proxy_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void dvb_id_bus_manager_dbus_proxy_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_dbus_register_object (DBusConnection* connection, const char* path, void* object);
static void _vala_dbus_unregister_object (gpointer connection, GObject* object);

static const DBusObjectPathVTable _dvb_id_bus_manager_dbus_path_vtable = {_dvb_id_bus_manager_dbus_unregister, dvb_id_bus_manager_dbus_message};
static const _DBusObjectVTable _dvb_id_bus_manager_dbus_vtable = {dvb_id_bus_manager_dbus_register_object};


/**
         * @adapter: Number of the device's adapter
         * @frontend: Number of the device's frontend
         * @returns: Object path and interface to the scanner service
         *
         * Get the object path of the channel scanner for this device.
         */
char** dvb_id_bus_manager_GetScannerForDevice (DVBIDBusManager* self, guint adapter, guint frontend, int* result_length1) {
	return DVB_ID_BUS_MANAGER_GET_INTERFACE (self)->GetScannerForDevice (self, adapter, frontend, result_length1);
}


/**
         * @returns: Device groups' DBus path
         */
char** dvb_id_bus_manager_GetRegisteredDeviceGroups (DVBIDBusManager* self, int* result_length1) {
	return DVB_ID_BUS_MANAGER_GET_INTERFACE (self)->GetRegisteredDeviceGroups (self, result_length1);
}


/**
         * @group_id: A group ID
         * @returns: Device group's DBus path
         */
char* dvb_id_bus_manager_GetDeviceGroup (DVBIDBusManager* self, guint group_id) {
	return DVB_ID_BUS_MANAGER_GET_INTERFACE (self)->GetDeviceGroup (self, group_id);
}


/**
         * @adapter: Number of the device's adapter
         * @frontend: Number of the device's frontend
         * @channels_conf: Path to channels.conf for this device
         * @recordings_dir: Path where the recordings should be stored
         * @name: Name of group
         * @returns: TRUE when the device has been registered successfully
         *
         * Creates a new DeviceGroup and new DVB device whereas the
         * DVB device is the reference device of this group (i.e.
         * all other devices of this group will inherit the settings
         * of the reference device).
         */
gboolean dvb_id_bus_manager_AddDeviceToNewGroup (DVBIDBusManager* self, guint adapter, guint frontend, const char* channels_conf, const char* recordings_dir, const char* name) {
	return DVB_ID_BUS_MANAGER_GET_INTERFACE (self)->AddDeviceToNewGroup (self, adapter, frontend, channels_conf, recordings_dir, name);
}


/**
         * @adapter: Adapter of device
         * @frontend: Frontend of device
         * @returns: The name of the device or "Unknown"
         *
         * The device must be part of group, otherwise "Unknown"
         * is returned.
         */
char* dvb_id_bus_manager_GetNameOfRegisteredDevice (DVBIDBusManager* self, guint adapter, guint frontend) {
	return DVB_ID_BUS_MANAGER_GET_INTERFACE (self)->GetNameOfRegisteredDevice (self, adapter, frontend);
}


/**
         * @returns: the numner of configured device groups
         */
gint dvb_id_bus_manager_GetDeviceGroupSize (DVBIDBusManager* self) {
	return DVB_ID_BUS_MANAGER_GET_INTERFACE (self)->GetDeviceGroupSize (self);
}


void _dvb_id_bus_manager_dbus_unregister (DBusConnection* connection, void* user_data) {
}


static DBusMessage* _dbus_dvb_id_bus_manager_introspect (DVBIDBusManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessage* reply;
	DBusMessageIter iter;
	GString* xml_data;
	char** children;
	int i;
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	xml_data = g_string_new ("<!DOCTYPE node PUBLIC \"-//freedesktop//DTD D-BUS Object Introspection 1.0//EN\" \"http://www.freedesktop.org/standards/dbus/1.0/introspect.dtd\">\n");
	g_string_append (xml_data, "<node>\n<interface name=\"org.freedesktop.DBus.Introspectable\">\n  <method name=\"Introspect\">\n    <arg name=\"data\" direction=\"out\" type=\"s\"/>\n  </method>\n</interface>\n<interface name=\"org.freedesktop.DBus.Properties\">\n  <method name=\"Get\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"propname\" direction=\"in\" type=\"s\"/>\n    <arg name=\"value\" direction=\"out\" type=\"v\"/>\n  </method>\n  <method name=\"Set\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"propname\" direction=\"in\" type=\"s\"/>\n    <arg name=\"value\" direction=\"in\" type=\"v\"/>\n  </method>\n  <method name=\"GetAll\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"props\" direction=\"out\" type=\"a{sv}\"/>\n  </method>\n</interface>\n<interface name=\"org.gnome.DVB.Manager\">\n  <method name=\"GetScannerForDevice\">\n    <arg name=\"adapter\" type=\"u\" direction=\"in\"/>\n    <arg name=\"frontend\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"as\" direction=\"out\"/>\n  </method>\n  <method name=\"GetRegisteredDeviceGroups\">\n    <arg name=\"result\" type=\"ao\" direction=\"out\"/>\n  </method>\n  <method name=\"GetDeviceGroup\">\n    <arg name=\"group_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"o\" direction=\"out\"/>\n  </method>\n  <method name=\"AddDeviceToNewGroup\">\n    <arg name=\"adapter\" type=\"u\" direction=\"in\"/>\n    <arg name=\"frontend\" type=\"u\" direction=\"in\"/>\n    <arg name=\"channels_conf\" type=\"s\" direction=\"in\"/>\n    <arg name=\"recordings_dir\" type=\"s\" direction=\"in\"/>\n    <arg name=\"name\" type=\"s\" direction=\"in\"/>\n    <arg name=\"result\" type=\"b\" direction=\"out\"/>\n  </method>\n  <method name=\"GetNameOfRegisteredDevice\">\n    <arg name=\"adapter\" type=\"u\" direction=\"in\"/>\n    <arg name=\"frontend\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"GetDeviceGroupSize\">\n    <arg name=\"result\" type=\"i\" direction=\"out\"/>\n  </method>\n  <signal name=\"GroupAdded\">\n    <arg name=\"group_id\" type=\"u\"/>\n  </signal>\n  <signal name=\"GroupRemoved\">\n    <arg name=\"group_id\" type=\"u\"/>\n  </signal>\n</interface>\n");
	dbus_connection_list_registered (connection, g_object_get_data ((GObject *) self, "dbus_object_path"), &children);
	for (i = 0; children[i]; i++) {
		g_string_append_printf (xml_data, "<node name=\"%s\"/>\n", children[i]);
	}
	dbus_free_string_array (children);
	g_string_append (xml_data, "</node>\n");
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &xml_data->str);
	g_string_free (xml_data, TRUE);
	return reply;
}


static DBusMessage* _dbus_dvb_id_bus_manager_property_get_all (DVBIDBusManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessage* reply;
	DBusMessageIter iter, reply_iter, subiter;
	char* interface_name;
	const char* _tmp0_;
	if (strcmp (dbus_message_get_signature (message), "s")) {
		return NULL;
	}
	dbus_message_iter_init (message, &iter);
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &reply_iter);
	dbus_message_iter_get_basic (&iter, &_tmp0_);
	dbus_message_iter_next (&iter);
	interface_name = g_strdup (_tmp0_);
	if (strcmp (interface_name, "org.gnome.DVB.Manager") == 0) {
		dbus_message_iter_open_container (&reply_iter, DBUS_TYPE_ARRAY, "{sv}", &subiter);
		dbus_message_iter_close_container (&reply_iter, &subiter);
	} else {
		dbus_message_unref (reply);
		reply = NULL;
	}
	g_free (interface_name);
	return reply;
}


static DBusMessage* _dbus_dvb_id_bus_manager_GetScannerForDevice (DVBIDBusManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessageIter iter;
	GError* error;
	guint adapter;
	dbus_uint32_t _tmp1_;
	guint frontend;
	dbus_uint32_t _tmp2_;
	int result_length1;
	char** result;
	DBusMessage* reply;
	char** _tmp3_;
	DBusMessageIter _tmp4_;
	int _tmp5_;
	error = NULL;
	if (strcmp (dbus_message_get_signature (message), "uu")) {
		return NULL;
	}
	dbus_message_iter_init (message, &iter);
	adapter = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp1_);
	dbus_message_iter_next (&iter);
	adapter = _tmp1_;
	frontend = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp2_);
	dbus_message_iter_next (&iter);
	frontend = _tmp2_;
	result_length1 = 0;
	result = dvb_id_bus_manager_GetScannerForDevice (self, adapter, frontend, &result_length1);
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	_tmp3_ = result;
	dbus_message_iter_open_container (&iter, DBUS_TYPE_ARRAY, "s", &_tmp4_);
	for (_tmp5_ = 0; _tmp5_ < result_length1; _tmp5_++) {
		const char* _tmp6_;
		_tmp6_ = *_tmp3_;
		dbus_message_iter_append_basic (&_tmp4_, DBUS_TYPE_STRING, &_tmp6_);
		_tmp3_++;
	}
	dbus_message_iter_close_container (&iter, &_tmp4_);
	result = (_vala_array_free (result,  result_length1, (GDestroyNotify) g_free), NULL);
	return reply;
}


static DBusMessage* _dbus_dvb_id_bus_manager_GetRegisteredDeviceGroups (DVBIDBusManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessageIter iter;
	GError* error;
	int result_length1;
	char** result;
	DBusMessage* reply;
	char** _tmp7_;
	DBusMessageIter _tmp8_;
	int _tmp9_;
	error = NULL;
	if (strcmp (dbus_message_get_signature (message), "")) {
		return NULL;
	}
	dbus_message_iter_init (message, &iter);
	result_length1 = 0;
	result = dvb_id_bus_manager_GetRegisteredDeviceGroups (self, &result_length1);
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	_tmp7_ = result;
	dbus_message_iter_open_container (&iter, DBUS_TYPE_ARRAY, "o", &_tmp8_);
	for (_tmp9_ = 0; _tmp9_ < result_length1; _tmp9_++) {
		const char* _tmp10_;
		_tmp10_ = *_tmp7_;
		dbus_message_iter_append_basic (&_tmp8_, DBUS_TYPE_OBJECT_PATH, &_tmp10_);
		_tmp7_++;
	}
	dbus_message_iter_close_container (&iter, &_tmp8_);
	result = (_vala_array_free (result,  result_length1, (GDestroyNotify) g_free), NULL);
	return reply;
}


static DBusMessage* _dbus_dvb_id_bus_manager_GetDeviceGroup (DVBIDBusManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessageIter iter;
	GError* error;
	guint group_id;
	dbus_uint32_t _tmp11_;
	char* result;
	DBusMessage* reply;
	const char* _tmp12_;
	error = NULL;
	if (strcmp (dbus_message_get_signature (message), "u")) {
		return NULL;
	}
	dbus_message_iter_init (message, &iter);
	group_id = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp11_);
	dbus_message_iter_next (&iter);
	group_id = _tmp11_;
	result = dvb_id_bus_manager_GetDeviceGroup (self, group_id);
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	_tmp12_ = result;
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_OBJECT_PATH, &_tmp12_);
	result = (g_free (result), NULL);
	return reply;
}


static DBusMessage* _dbus_dvb_id_bus_manager_AddDeviceToNewGroup (DVBIDBusManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessageIter iter;
	GError* error;
	guint adapter;
	dbus_uint32_t _tmp13_;
	guint frontend;
	dbus_uint32_t _tmp14_;
	char* channels_conf;
	const char* _tmp15_;
	char* recordings_dir;
	const char* _tmp16_;
	char* name;
	const char* _tmp17_;
	gboolean result;
	DBusMessage* reply;
	dbus_bool_t _tmp18_;
	error = NULL;
	if (strcmp (dbus_message_get_signature (message), "uusss")) {
		return NULL;
	}
	dbus_message_iter_init (message, &iter);
	adapter = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp13_);
	dbus_message_iter_next (&iter);
	adapter = _tmp13_;
	frontend = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp14_);
	dbus_message_iter_next (&iter);
	frontend = _tmp14_;
	channels_conf = NULL;
	dbus_message_iter_get_basic (&iter, &_tmp15_);
	dbus_message_iter_next (&iter);
	channels_conf = g_strdup (_tmp15_);
	recordings_dir = NULL;
	dbus_message_iter_get_basic (&iter, &_tmp16_);
	dbus_message_iter_next (&iter);
	recordings_dir = g_strdup (_tmp16_);
	name = NULL;
	dbus_message_iter_get_basic (&iter, &_tmp17_);
	dbus_message_iter_next (&iter);
	name = g_strdup (_tmp17_);
	result = dvb_id_bus_manager_AddDeviceToNewGroup (self, adapter, frontend, channels_conf, recordings_dir, name);
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	channels_conf = (g_free (channels_conf), NULL);
	recordings_dir = (g_free (recordings_dir), NULL);
	name = (g_free (name), NULL);
	_tmp18_ = result;
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_BOOLEAN, &_tmp18_);
	return reply;
}


static DBusMessage* _dbus_dvb_id_bus_manager_GetNameOfRegisteredDevice (DVBIDBusManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessageIter iter;
	GError* error;
	guint adapter;
	dbus_uint32_t _tmp19_;
	guint frontend;
	dbus_uint32_t _tmp20_;
	char* result;
	DBusMessage* reply;
	const char* _tmp21_;
	error = NULL;
	if (strcmp (dbus_message_get_signature (message), "uu")) {
		return NULL;
	}
	dbus_message_iter_init (message, &iter);
	adapter = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp19_);
	dbus_message_iter_next (&iter);
	adapter = _tmp19_;
	frontend = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp20_);
	dbus_message_iter_next (&iter);
	frontend = _tmp20_;
	result = dvb_id_bus_manager_GetNameOfRegisteredDevice (self, adapter, frontend);
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	_tmp21_ = result;
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &_tmp21_);
	result = (g_free (result), NULL);
	return reply;
}


static DBusMessage* _dbus_dvb_id_bus_manager_GetDeviceGroupSize (DVBIDBusManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessageIter iter;
	GError* error;
	gint result;
	DBusMessage* reply;
	dbus_int32_t _tmp22_;
	error = NULL;
	if (strcmp (dbus_message_get_signature (message), "")) {
		return NULL;
	}
	dbus_message_iter_init (message, &iter);
	result = dvb_id_bus_manager_GetDeviceGroupSize (self);
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	_tmp22_ = result;
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_INT32, &_tmp22_);
	return reply;
}


DBusHandlerResult dvb_id_bus_manager_dbus_message (DBusConnection* connection, DBusMessage* message, void* object) {
	DBusMessage* reply;
	reply = NULL;
	if (dbus_message_is_method_call (message, "org.freedesktop.DBus.Introspectable", "Introspect")) {
		reply = _dbus_dvb_id_bus_manager_introspect (object, connection, message);
	} else if (dbus_message_is_method_call (message, "org.freedesktop.DBus.Properties", "GetAll")) {
		reply = _dbus_dvb_id_bus_manager_property_get_all (object, connection, message);
	} else if (dbus_message_is_method_call (message, "org.gnome.DVB.Manager", "GetScannerForDevice")) {
		reply = _dbus_dvb_id_bus_manager_GetScannerForDevice (object, connection, message);
	} else if (dbus_message_is_method_call (message, "org.gnome.DVB.Manager", "GetRegisteredDeviceGroups")) {
		reply = _dbus_dvb_id_bus_manager_GetRegisteredDeviceGroups (object, connection, message);
	} else if (dbus_message_is_method_call (message, "org.gnome.DVB.Manager", "GetDeviceGroup")) {
		reply = _dbus_dvb_id_bus_manager_GetDeviceGroup (object, connection, message);
	} else if (dbus_message_is_method_call (message, "org.gnome.DVB.Manager", "AddDeviceToNewGroup")) {
		reply = _dbus_dvb_id_bus_manager_AddDeviceToNewGroup (object, connection, message);
	} else if (dbus_message_is_method_call (message, "org.gnome.DVB.Manager", "GetNameOfRegisteredDevice")) {
		reply = _dbus_dvb_id_bus_manager_GetNameOfRegisteredDevice (object, connection, message);
	} else if (dbus_message_is_method_call (message, "org.gnome.DVB.Manager", "GetDeviceGroupSize")) {
		reply = _dbus_dvb_id_bus_manager_GetDeviceGroupSize (object, connection, message);
	}
	if (reply) {
		dbus_connection_send (connection, reply, NULL);
		dbus_message_unref (reply);
		return DBUS_HANDLER_RESULT_HANDLED;
	} else {
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}
}


static void _dbus_dvb_id_bus_manager_group_added (GObject* _sender, guint group_id, DBusConnection* _connection) {
	const char * _path;
	DBusMessage *_message;
	DBusMessageIter _iter;
	dbus_uint32_t _tmp23_;
	_path = g_object_get_data (_sender, "dbus_object_path");
	_message = dbus_message_new_signal (_path, "org.gnome.DVB.Manager", "GroupAdded");
	dbus_message_iter_init_append (_message, &_iter);
	_tmp23_ = group_id;
	dbus_message_iter_append_basic (&_iter, DBUS_TYPE_UINT32, &_tmp23_);
	dbus_connection_send (_connection, _message, NULL);
	dbus_message_unref (_message);
}


static void _dbus_dvb_id_bus_manager_group_removed (GObject* _sender, guint group_id, DBusConnection* _connection) {
	const char * _path;
	DBusMessage *_message;
	DBusMessageIter _iter;
	dbus_uint32_t _tmp24_;
	_path = g_object_get_data (_sender, "dbus_object_path");
	_message = dbus_message_new_signal (_path, "org.gnome.DVB.Manager", "GroupRemoved");
	dbus_message_iter_init_append (_message, &_iter);
	_tmp24_ = group_id;
	dbus_message_iter_append_basic (&_iter, DBUS_TYPE_UINT32, &_tmp24_);
	dbus_connection_send (_connection, _message, NULL);
	dbus_message_unref (_message);
}


void dvb_id_bus_manager_dbus_register_object (DBusConnection* connection, const char* path, void* object) {
	if (!g_object_get_data (object, "dbus_object_path")) {
		g_object_set_data (object, "dbus_object_path", g_strdup (path));
		dbus_connection_register_object_path (connection, path, &_dvb_id_bus_manager_dbus_path_vtable, object);
		g_object_weak_ref (object, _vala_dbus_unregister_object, connection);
	}
	g_signal_connect (object, "group-added", (GCallback) _dbus_dvb_id_bus_manager_group_added, connection);
	g_signal_connect (object, "group-removed", (GCallback) _dbus_dvb_id_bus_manager_group_removed, connection);
}


static void dvb_id_bus_manager_base_init (DVBIDBusManagerIface * iface) {
	static gboolean initialized = FALSE;
	if (!initialized) {
		initialized = TRUE;
		g_signal_new ("group_added", DVB_TYPE_ID_BUS_MANAGER, G_SIGNAL_RUN_LAST, 0, NULL, NULL, g_cclosure_marshal_VOID__UINT, G_TYPE_NONE, 1, G_TYPE_UINT);
		g_signal_new ("group_removed", DVB_TYPE_ID_BUS_MANAGER, G_SIGNAL_RUN_LAST, 0, NULL, NULL, g_cclosure_marshal_VOID__UINT, G_TYPE_NONE, 1, G_TYPE_UINT);
		g_type_set_qdata (DVB_TYPE_ID_BUS_MANAGER, g_quark_from_static_string ("DBusObjectVTable"), (void*) (&_dvb_id_bus_manager_dbus_vtable));
	}
}


GType dvb_id_bus_manager_get_type (void) {
	static GType dvb_id_bus_manager_type_id = 0;
	if (dvb_id_bus_manager_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBIDBusManagerIface), (GBaseInitFunc) dvb_id_bus_manager_base_init, (GBaseFinalizeFunc) NULL, (GClassInitFunc) NULL, (GClassFinalizeFunc) NULL, NULL, 0, 0, (GInstanceInitFunc) NULL, NULL };
		dvb_id_bus_manager_type_id = g_type_register_static (G_TYPE_INTERFACE, "DVBIDBusManager", &g_define_type_info, 0);
		g_type_interface_add_prerequisite (dvb_id_bus_manager_type_id, G_TYPE_OBJECT);
		g_type_set_qdata (dvb_id_bus_manager_type_id, g_quark_from_string ("ValaDBusInterfaceProxyType"), &dvb_id_bus_manager_dbus_proxy_get_type);
	}
	return dvb_id_bus_manager_type_id;
}


G_DEFINE_TYPE_EXTENDED (DVBIDBusManagerDBusProxy, dvb_id_bus_manager_dbus_proxy, DBUS_TYPE_G_PROXY, 0, G_IMPLEMENT_INTERFACE (DVB_TYPE_ID_BUS_MANAGER, dvb_id_bus_manager_dbus_proxy_interface_init));
DVBIDBusManager* dvb_id_bus_manager_dbus_proxy_new (DBusGConnection* connection, const char* name, const char* path) {
	DVBIDBusManager* self;
	self = g_object_new (dvb_id_bus_manager_dbus_proxy_get_type (), "connection", connection, "name", name, "path", path, "interface", "org.gnome.DVB.Manager", NULL);
	return self;
}


static GObject* dvb_id_bus_manager_dbus_proxy_construct (GType gtype, guint n_properties, GObjectConstructParam* properties) {
	GObject* self;
	DBusGConnection *connection;
	char* path;
	char* filter;
	self = G_OBJECT_CLASS (dvb_id_bus_manager_dbus_proxy_parent_class)->constructor (gtype, n_properties, properties);
	g_object_get (self, "connection", &connection, NULL);
	g_object_get (self, "path", &path, NULL);
	dbus_connection_add_filter (dbus_g_connection_get_connection (connection), dvb_id_bus_manager_dbus_proxy_filter, self, NULL);
	filter = g_strdup_printf ("type='signal',path='%s'", path);
	dbus_bus_add_match (dbus_g_connection_get_connection (connection), filter, NULL);
	dbus_g_connection_unref (connection);
	g_free (path);
	g_free (filter);
	return self;
}


static void _dbus_handle_dvb_id_bus_manager_group_added (DVBIDBusManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessageIter iter;
	guint group_id;
	dbus_uint32_t _tmp25_;
	DBusMessage* reply;
	if (strcmp (dbus_message_get_signature (message), "u")) {
		return;
	}
	dbus_message_iter_init (message, &iter);
	group_id = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp25_);
	dbus_message_iter_next (&iter);
	group_id = _tmp25_;
	g_signal_emit_by_name (self, "group-added", group_id);
}


static void _dbus_handle_dvb_id_bus_manager_group_removed (DVBIDBusManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessageIter iter;
	guint group_id;
	dbus_uint32_t _tmp26_;
	DBusMessage* reply;
	if (strcmp (dbus_message_get_signature (message), "u")) {
		return;
	}
	dbus_message_iter_init (message, &iter);
	group_id = 0U;
	dbus_message_iter_get_basic (&iter, &_tmp26_);
	dbus_message_iter_next (&iter);
	group_id = _tmp26_;
	g_signal_emit_by_name (self, "group-removed", group_id);
}


DBusHandlerResult dvb_id_bus_manager_dbus_proxy_filter (DBusConnection* connection, DBusMessage* message, void* user_data) {
	if (dbus_message_has_path (message, dbus_g_proxy_get_path (user_data))) {
		if (dbus_message_is_signal (message, "org.gnome.DVB.Manager", "GroupAdded")) {
			_dbus_handle_dvb_id_bus_manager_group_added (user_data, connection, message);
		} else if (dbus_message_is_signal (message, "org.gnome.DVB.Manager", "GroupRemoved")) {
			_dbus_handle_dvb_id_bus_manager_group_removed (user_data, connection, message);
		}
	}
	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}


static void dvb_id_bus_manager_dbus_proxy_dispose (GObject* self) {
	DBusGConnection *connection;
	if (((DVBIDBusManagerDBusProxy*) self)->disposed) {
		return;
	}
	((DVBIDBusManagerDBusProxy*) self)->disposed = TRUE;
	g_object_get (self, "connection", &connection, NULL);
	dbus_connection_remove_filter (dbus_g_connection_get_connection (connection), dvb_id_bus_manager_dbus_proxy_filter, self);
	G_OBJECT_CLASS (dvb_id_bus_manager_dbus_proxy_parent_class)->dispose (self);
}


static void dvb_id_bus_manager_dbus_proxy_class_init (DVBIDBusManagerDBusProxyClass* klass) {
	G_OBJECT_CLASS (klass)->constructor = dvb_id_bus_manager_dbus_proxy_construct;
	G_OBJECT_CLASS (klass)->dispose = dvb_id_bus_manager_dbus_proxy_dispose;
	G_OBJECT_CLASS (klass)->get_property = dvb_id_bus_manager_dbus_proxy_get_property;
	G_OBJECT_CLASS (klass)->set_property = dvb_id_bus_manager_dbus_proxy_set_property;
}


static void dvb_id_bus_manager_dbus_proxy_init (DVBIDBusManagerDBusProxy* self) {
}


static char** dvb_id_bus_manager_dbus_proxy_GetScannerForDevice (DVBIDBusManager* self, guint adapter, guint frontend, int* result_length1) {
	DBusGConnection *_connection;
	DBusMessage *_message, *_reply;
	DBusMessageIter _iter;
	dbus_uint32_t _tmp27_;
	dbus_uint32_t _tmp28_;
	char** _result;
	int _result_length1;
	char** _tmp29_;
	int _tmp29__length;
	int _tmp29__size;
	int _tmp29__length1;
	DBusMessageIter _tmp30_;
	if (((DVBIDBusManagerDBusProxy*) self)->disposed) {
		return NULL;
	}
	_message = dbus_message_new_method_call (dbus_g_proxy_get_bus_name ((DBusGProxy*) self), dbus_g_proxy_get_path ((DBusGProxy*) self), "org.gnome.DVB.Manager", "GetScannerForDevice");
	dbus_message_iter_init_append (_message, &_iter);
	_tmp27_ = adapter;
	dbus_message_iter_append_basic (&_iter, DBUS_TYPE_UINT32, &_tmp27_);
	_tmp28_ = frontend;
	dbus_message_iter_append_basic (&_iter, DBUS_TYPE_UINT32, &_tmp28_);
	g_object_get (self, "connection", &_connection, NULL);
	_reply = dbus_connection_send_with_reply_and_block (dbus_g_connection_get_connection (_connection), _message, -1, NULL);
	dbus_g_connection_unref (_connection);
	dbus_message_unref (_message);
	dbus_message_iter_init (_reply, &_iter);
	_result_length1 = 0;
	_tmp29_ = g_new (char*, 5);
	_tmp29__length = 0;
	_tmp29__size = 4;
	_tmp29__length1 = 0;
	dbus_message_iter_recurse (&_iter, &_tmp30_);
	for (; dbus_message_iter_get_arg_type (&_tmp30_); _tmp29__length1++) {
		const char* _tmp31_;
		if (_tmp29__size == _tmp29__length) {
			_tmp29__size = 2 * _tmp29__size;
			_tmp29_ = g_renew (char*, _tmp29_, _tmp29__size + 1);
		}
		dbus_message_iter_get_basic (&_tmp30_, &_tmp31_);
		dbus_message_iter_next (&_tmp30_);
		_tmp29_[_tmp29__length++] = g_strdup (_tmp31_);
	}
	_result_length1 = _tmp29__length1;
	_tmp29_[_tmp29__length] = NULL;
	dbus_message_iter_next (&_iter);
	_result = _tmp29_;
	*result_length1 = _result_length1;
	dbus_message_unref (_reply);
	return _result;
}


static char** dvb_id_bus_manager_dbus_proxy_GetRegisteredDeviceGroups (DVBIDBusManager* self, int* result_length1) {
	DBusGConnection *_connection;
	DBusMessage *_message, *_reply;
	DBusMessageIter _iter;
	char** _result;
	int _result_length1;
	char** _tmp32_;
	int _tmp32__length;
	int _tmp32__size;
	int _tmp32__length1;
	DBusMessageIter _tmp33_;
	if (((DVBIDBusManagerDBusProxy*) self)->disposed) {
		return NULL;
	}
	_message = dbus_message_new_method_call (dbus_g_proxy_get_bus_name ((DBusGProxy*) self), dbus_g_proxy_get_path ((DBusGProxy*) self), "org.gnome.DVB.Manager", "GetRegisteredDeviceGroups");
	dbus_message_iter_init_append (_message, &_iter);
	g_object_get (self, "connection", &_connection, NULL);
	_reply = dbus_connection_send_with_reply_and_block (dbus_g_connection_get_connection (_connection), _message, -1, NULL);
	dbus_g_connection_unref (_connection);
	dbus_message_unref (_message);
	dbus_message_iter_init (_reply, &_iter);
	_result_length1 = 0;
	_tmp32_ = g_new (char*, 5);
	_tmp32__length = 0;
	_tmp32__size = 4;
	_tmp32__length1 = 0;
	dbus_message_iter_recurse (&_iter, &_tmp33_);
	for (; dbus_message_iter_get_arg_type (&_tmp33_); _tmp32__length1++) {
		const char* _tmp34_;
		if (_tmp32__size == _tmp32__length) {
			_tmp32__size = 2 * _tmp32__size;
			_tmp32_ = g_renew (char*, _tmp32_, _tmp32__size + 1);
		}
		dbus_message_iter_get_basic (&_tmp33_, &_tmp34_);
		dbus_message_iter_next (&_tmp33_);
		_tmp32_[_tmp32__length++] = g_strdup (_tmp34_);
	}
	_result_length1 = _tmp32__length1;
	_tmp32_[_tmp32__length] = NULL;
	dbus_message_iter_next (&_iter);
	_result = _tmp32_;
	*result_length1 = _result_length1;
	dbus_message_unref (_reply);
	return _result;
}


static char* dvb_id_bus_manager_dbus_proxy_GetDeviceGroup (DVBIDBusManager* self, guint group_id) {
	DBusGConnection *_connection;
	DBusMessage *_message, *_reply;
	DBusMessageIter _iter;
	dbus_uint32_t _tmp35_;
	char* _result;
	const char* _tmp36_;
	if (((DVBIDBusManagerDBusProxy*) self)->disposed) {
		return NULL;
	}
	_message = dbus_message_new_method_call (dbus_g_proxy_get_bus_name ((DBusGProxy*) self), dbus_g_proxy_get_path ((DBusGProxy*) self), "org.gnome.DVB.Manager", "GetDeviceGroup");
	dbus_message_iter_init_append (_message, &_iter);
	_tmp35_ = group_id;
	dbus_message_iter_append_basic (&_iter, DBUS_TYPE_UINT32, &_tmp35_);
	g_object_get (self, "connection", &_connection, NULL);
	_reply = dbus_connection_send_with_reply_and_block (dbus_g_connection_get_connection (_connection), _message, -1, NULL);
	dbus_g_connection_unref (_connection);
	dbus_message_unref (_message);
	dbus_message_iter_init (_reply, &_iter);
	dbus_message_iter_get_basic (&_iter, &_tmp36_);
	dbus_message_iter_next (&_iter);
	_result = g_strdup (_tmp36_);
	dbus_message_unref (_reply);
	return _result;
}


static gboolean dvb_id_bus_manager_dbus_proxy_AddDeviceToNewGroup (DVBIDBusManager* self, guint adapter, guint frontend, const char* channels_conf, const char* recordings_dir, const char* name) {
	DBusGConnection *_connection;
	DBusMessage *_message, *_reply;
	DBusMessageIter _iter;
	dbus_uint32_t _tmp37_;
	dbus_uint32_t _tmp38_;
	const char* _tmp39_;
	const char* _tmp40_;
	const char* _tmp41_;
	gboolean _result;
	dbus_bool_t _tmp42_;
	if (((DVBIDBusManagerDBusProxy*) self)->disposed) {
		return FALSE;
	}
	_message = dbus_message_new_method_call (dbus_g_proxy_get_bus_name ((DBusGProxy*) self), dbus_g_proxy_get_path ((DBusGProxy*) self), "org.gnome.DVB.Manager", "AddDeviceToNewGroup");
	dbus_message_iter_init_append (_message, &_iter);
	_tmp37_ = adapter;
	dbus_message_iter_append_basic (&_iter, DBUS_TYPE_UINT32, &_tmp37_);
	_tmp38_ = frontend;
	dbus_message_iter_append_basic (&_iter, DBUS_TYPE_UINT32, &_tmp38_);
	_tmp39_ = channels_conf;
	dbus_message_iter_append_basic (&_iter, DBUS_TYPE_STRING, &_tmp39_);
	_tmp40_ = recordings_dir;
	dbus_message_iter_append_basic (&_iter, DBUS_TYPE_STRING, &_tmp40_);
	_tmp41_ = name;
	dbus_message_iter_append_basic (&_iter, DBUS_TYPE_STRING, &_tmp41_);
	g_object_get (self, "connection", &_connection, NULL);
	_reply = dbus_connection_send_with_reply_and_block (dbus_g_connection_get_connection (_connection), _message, -1, NULL);
	dbus_g_connection_unref (_connection);
	dbus_message_unref (_message);
	dbus_message_iter_init (_reply, &_iter);
	dbus_message_iter_get_basic (&_iter, &_tmp42_);
	dbus_message_iter_next (&_iter);
	_result = _tmp42_;
	dbus_message_unref (_reply);
	return _result;
}


static char* dvb_id_bus_manager_dbus_proxy_GetNameOfRegisteredDevice (DVBIDBusManager* self, guint adapter, guint frontend) {
	DBusGConnection *_connection;
	DBusMessage *_message, *_reply;
	DBusMessageIter _iter;
	dbus_uint32_t _tmp43_;
	dbus_uint32_t _tmp44_;
	char* _result;
	const char* _tmp45_;
	if (((DVBIDBusManagerDBusProxy*) self)->disposed) {
		return NULL;
	}
	_message = dbus_message_new_method_call (dbus_g_proxy_get_bus_name ((DBusGProxy*) self), dbus_g_proxy_get_path ((DBusGProxy*) self), "org.gnome.DVB.Manager", "GetNameOfRegisteredDevice");
	dbus_message_iter_init_append (_message, &_iter);
	_tmp43_ = adapter;
	dbus_message_iter_append_basic (&_iter, DBUS_TYPE_UINT32, &_tmp43_);
	_tmp44_ = frontend;
	dbus_message_iter_append_basic (&_iter, DBUS_TYPE_UINT32, &_tmp44_);
	g_object_get (self, "connection", &_connection, NULL);
	_reply = dbus_connection_send_with_reply_and_block (dbus_g_connection_get_connection (_connection), _message, -1, NULL);
	dbus_g_connection_unref (_connection);
	dbus_message_unref (_message);
	dbus_message_iter_init (_reply, &_iter);
	dbus_message_iter_get_basic (&_iter, &_tmp45_);
	dbus_message_iter_next (&_iter);
	_result = g_strdup (_tmp45_);
	dbus_message_unref (_reply);
	return _result;
}


static gint dvb_id_bus_manager_dbus_proxy_GetDeviceGroupSize (DVBIDBusManager* self) {
	DBusGConnection *_connection;
	DBusMessage *_message, *_reply;
	DBusMessageIter _iter;
	gint _result;
	dbus_int32_t _tmp46_;
	if (((DVBIDBusManagerDBusProxy*) self)->disposed) {
		return 0;
	}
	_message = dbus_message_new_method_call (dbus_g_proxy_get_bus_name ((DBusGProxy*) self), dbus_g_proxy_get_path ((DBusGProxy*) self), "org.gnome.DVB.Manager", "GetDeviceGroupSize");
	dbus_message_iter_init_append (_message, &_iter);
	g_object_get (self, "connection", &_connection, NULL);
	_reply = dbus_connection_send_with_reply_and_block (dbus_g_connection_get_connection (_connection), _message, -1, NULL);
	dbus_g_connection_unref (_connection);
	dbus_message_unref (_message);
	dbus_message_iter_init (_reply, &_iter);
	dbus_message_iter_get_basic (&_iter, &_tmp46_);
	dbus_message_iter_next (&_iter);
	_result = _tmp46_;
	dbus_message_unref (_reply);
	return _result;
}


static void dvb_id_bus_manager_dbus_proxy_interface_init (DVBIDBusManagerIface* iface) {
	iface->GetScannerForDevice = dvb_id_bus_manager_dbus_proxy_GetScannerForDevice;
	iface->GetRegisteredDeviceGroups = dvb_id_bus_manager_dbus_proxy_GetRegisteredDeviceGroups;
	iface->GetDeviceGroup = dvb_id_bus_manager_dbus_proxy_GetDeviceGroup;
	iface->AddDeviceToNewGroup = dvb_id_bus_manager_dbus_proxy_AddDeviceToNewGroup;
	iface->GetNameOfRegisteredDevice = dvb_id_bus_manager_dbus_proxy_GetNameOfRegisteredDevice;
	iface->GetDeviceGroupSize = dvb_id_bus_manager_dbus_proxy_GetDeviceGroupSize;
}


static void dvb_id_bus_manager_dbus_proxy_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
}


static void dvb_id_bus_manager_dbus_proxy_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}


static void _vala_dbus_register_object (DBusConnection* connection, const char* path, void* object) {
	const _DBusObjectVTable * vtable;
	vtable = g_type_get_qdata (G_TYPE_FROM_INSTANCE (object), g_quark_from_static_string ("DBusObjectVTable"));
	if (vtable) {
		vtable->register_object (connection, path, object);
	} else {
		g_warning ("Object does not implement any D-Bus interface");
	}
}


static void _vala_dbus_unregister_object (gpointer connection, GObject* object) {
	char* path;
	path = g_object_steal_data ((GObject*) object, "dbus_object_path");
	dbus_connection_unregister_object_path (connection, path);
	g_free (path);
}




