/* gnome-netinfo - A GUI Interface for network utilities
 * Copyright (C) 2002, 2003 by German Poo-Caaman~o
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <glade/glade.h>
#include "callbacks.h"
#include "ping.h"
#include "traceroute.h"
#include "info.h"
#include "netstat.h"

Netinfo *load_ping_widgets_from_xml (GladeXML * xml);
Netinfo *load_traceroute_widgets_from_xml (GladeXML * xml);
Netinfo *load_netstat_widgets_from_xml (GladeXML * xml);
Netinfo *load_scan_widgets_from_xml (GladeXML * xml);
netinfo_info *load_info_widgets_from_xml (GladeXML * xml);

int
main (int argc, char *argv[])
{
	GtkWidget *window;
	GladeXML *xml;
	GtkWidget *output, *entry;
	GtkWidget *scroll;
	const gchar *dialog = DATADIR "gnome-netinfo.glade";
	Netinfo *pinger;
	Netinfo *tracer;
	Netinfo *netstat;
	netinfo_info *info;
	Netinfo *scan;

#ifdef ENABLE_NLS
	bindtextdomain (GETTEXT_PACKAGE, GPING_LOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
#endif

	gnome_program_init (PACKAGE, VERSION, LIBGNOMEUI_MODULE,
			    argc, argv,
			    GNOME_PARAM_APP_DATADIR, DATADIR,
			    GNOME_PARAM_NONE);

	if (!g_file_test (dialog, G_FILE_TEST_EXISTS)) {
		g_critical (_("The file %s doesn't exist, "
			      "please check if gnome-netinfo is correcly installed"),
			    dialog);
		return -1;
	}

	xml = glade_xml_new (dialog, "main_window", NULL);
	window = glade_xml_get_widget (xml, "main_window");

	pinger = load_ping_widgets_from_xml (xml);
	tracer = load_traceroute_widgets_from_xml (xml);
	netstat = load_netstat_widgets_from_xml (xml);
	info = load_info_widgets_from_xml (xml);
	scan = load_scan_widgets_from_xml (xml);
	
	glade_xml_signal_autoconnect (xml);
	g_object_unref (G_OBJECT (xml));

	gtk_widget_show_all (window);

	gtk_main ();

	g_free (pinger);
	g_free (tracer);
	g_free (netstat);
	g_free (info);
	g_free (scan);

	return 0;
}

/* The value returned must be released from memory */
Netinfo *
load_ping_widgets_from_xml (GladeXML * xml)
{
	Netinfo *pinger;
	GtkWidget *vbox_ping;

	g_return_val_if_fail (xml != NULL, NULL);

	pinger = g_new0 (Netinfo, 1);

	pinger->running = FALSE;
	pinger->child_pid = 0;
	pinger->host = glade_xml_get_widget (xml, "ping_host");
	pinger->count = glade_xml_get_widget (xml, "ping_count");
	pinger->output = glade_xml_get_widget (xml, "ping_output");
	pinger->limited = glade_xml_get_widget (xml, "ping_limited");
	pinger->button = glade_xml_get_widget (xml, "ping_button");
	pinger->sensitive = pinger->host;
	pinger->label_run = _("Ping");
	pinger->label_stop = NULL;
	pinger->routing = NULL;
	pinger->protocol = NULL;
	pinger->multicast = NULL;
	pinger->min = glade_xml_get_widget (xml, "ping_minimum");
	pinger->avg = glade_xml_get_widget (xml, "ping_average");
	pinger->max = glade_xml_get_widget (xml, "ping_maximum");
	pinger->packets_received = glade_xml_get_widget (xml, "ping_packets_received");
	pinger->packets_loss = glade_xml_get_widget (xml, "ping_packets_loss");

	vbox_ping = glade_xml_get_widget (xml, "vbox_ping");
	
#ifdef PING_PROGRAM
	pinger->button_callback = G_CALLBACK (on_ping_activate);
	pinger->process_line = NETINFO_FOREACH_FUNC (ping_foreach_with_tree);
	
	g_signal_connect (G_OBJECT (pinger->host), "activate",
				  G_CALLBACK (on_ping_activate),
				  pinger);
	g_signal_connect (G_OBJECT (pinger->button), "clicked",
				  pinger->button_callback,
				  pinger);
#else
	gtk_widget_set_sensitive (vbox_ping, FALSE);
#endif

	return pinger;
}

/* The value returned must be released from memory */
Netinfo *
load_traceroute_widgets_from_xml (GladeXML * xml)
{
	Netinfo *tracer;
	GtkWidget *vbox_traceroute;

	g_return_val_if_fail (xml != NULL, NULL);

	tracer = g_new0 (Netinfo, 1);

	tracer->running = FALSE;
	tracer->child_pid = 0;
	tracer->host = glade_xml_get_widget (xml, "traceroute_host");
	tracer->output = glade_xml_get_widget (xml, "traceroute_output");
	tracer->button = glade_xml_get_widget (xml, "traceroute_button");
	tracer->count = NULL;
	tracer->limited = NULL;
	tracer->sensitive = tracer->host;
	tracer->label_run = _("Traceroute");
	tracer->label_stop = NULL;
	tracer->routing = NULL;
	tracer->protocol = NULL;
	tracer->multicast = NULL;
	
	vbox_traceroute = glade_xml_get_widget (xml, "vbox_traceroute");

#ifdef TCPTRACEROUTE_PROGRAM
	tracer->button_callback = G_CALLBACK (on_traceroute_activate);
	tracer->process_line = NETINFO_FOREACH_FUNC (traceroute_foreach_with_tree);

	g_signal_connect (G_OBJECT (tracer->host), "activate",
				  G_CALLBACK (on_traceroute_activate),
				  tracer);
	g_signal_connect (G_OBJECT (tracer->button), "clicked",
				  tracer->button_callback,
				  tracer);
#else
	gtk_widget_set_sensitive (vbox_traceroute, FALSE);
#endif

	return tracer;
}

Netinfo *
load_netstat_widgets_from_xml (GladeXML * xml)
{
	Netinfo *netstat;
	GtkWidget *vbox_netstat;

	g_return_val_if_fail (xml != NULL, NULL);

	netstat = g_new0 (Netinfo, 1);

	netstat->running = FALSE;
	netstat->child_pid = 0;
	netstat->host = NULL;
	netstat->count = NULL;
	netstat->output = glade_xml_get_widget (xml, "netstat_output");
	netstat->limited = NULL;
	netstat->button = glade_xml_get_widget (xml, "netstat_button");
	netstat->routing = glade_xml_get_widget (xml, "netstat_routing");
	netstat->protocol = glade_xml_get_widget (xml, "netstat_protocol");
	netstat->multicast = glade_xml_get_widget (xml, "netstat_multicast");
	netstat->sensitive = NULL;
	netstat->label_run = _("Netstat");
	netstat->label_stop = NULL;
	
	vbox_netstat = glade_xml_get_widget (xml, "vbox_netstat");
	
#ifdef NETSTAT_PROGRAM
	netstat->button_callback = G_CALLBACK (on_netstat_activate);
	netstat->process_line = NETINFO_FOREACH_FUNC (netstat_foreach);
	
	g_signal_connect (G_OBJECT (netstat->button), "clicked",
				  netstat->button_callback,
				  netstat);
#else
	gtk_widget_set_sensitive (vbox_netstat, FALSE);
#endif /* NETSTAT_PROGRAM */

	return netstat;
}

/* The value returned must be released from memory */
netinfo_info *
load_info_widgets_from_xml (GladeXML * xml)
{
	netinfo_info *info;
	GtkWidget *entry, *combo;
	GtkWidget *vbox_info;

	g_return_val_if_fail (xml != NULL, NULL);

	info = g_malloc (sizeof (netinfo_info));

	info->hw_address = glade_xml_get_widget (xml, "info_hw_address");
	info->ip_address = glade_xml_get_widget (xml, "info_ip_address");
	info->netmask = glade_xml_get_widget (xml, "info_netmask");
	info->broadcast = glade_xml_get_widget (xml, "info_broadcast");
	info->multicast = glade_xml_get_widget (xml, "info_multicast");
	info->link_speed = glade_xml_get_widget (xml, "info_link_speed");
	info->state = glade_xml_get_widget (xml, "info_state");
	info->mtu = glade_xml_get_widget (xml, "info_mtu");
	info->tx = glade_xml_get_widget (xml, "info_tx");
	info->tx_errors = glade_xml_get_widget (xml, "info_tx_errors");
	info->rx = glade_xml_get_widget (xml, "info_rx");
	info->rx_errors = glade_xml_get_widget (xml, "info_rx_errors");
	info->collisions = glade_xml_get_widget (xml, "info_collisions");

	vbox_info = glade_xml_get_widget (xml, "vbox_info");
	
#ifdef IFCONFIG_PROGRAM
	entry = glade_xml_get_widget (xml, "info_nic");
	combo = glade_xml_get_widget (xml, "info_combo");
	
	g_signal_connect (G_OBJECT (entry), "changed",
				  G_CALLBACK (info_nic_changed),
				  info);

	info_load_iface (info, combo);
#else
	gtk_widget_set_sensitive (vbox_info, FALSE);
#endif

	return info;
}

Netinfo *
load_scan_widgets_from_xml (GladeXML * xml)
{
	Netinfo *scan;

	g_return_val_if_fail (xml != NULL, NULL);

	scan = g_new0 (Netinfo, 1);

	scan->running = FALSE;
	scan->child_pid = 0;
	scan->host = glade_xml_get_widget (xml, "scan_host");
	scan->count = NULL;
	scan->output = glade_xml_get_widget (xml, "scan_output");
	scan->limited = NULL;
	scan->button = glade_xml_get_widget (xml, "scan_button");
	scan->routing = NULL;
	scan->protocol = NULL;
	scan->multicast = NULL;
	scan->sensitive = NULL;
	scan->label_run = _("Scan");
	scan->label_stop = NULL;
	
	scan->button_callback = G_CALLBACK (on_scan_activate);
	scan->process_line = NULL; /* NETINFO_FOREACH_FUNC (scan_foreach);*/

	g_signal_connect (G_OBJECT (scan->host), "activate",
				  scan->button_callback,
				  scan);
	g_signal_connect (G_OBJECT (scan->button), "clicked",
				  scan->button_callback,
				  scan);

	return scan;
}
