/* GnomeScan - Gnome Scanning Infrastructure
 *
 * gnome-scanutils.c
 *
 * Copyright © 2006 Étienne Bersac
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <glib.h>
#include "gnomescanutils.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/**
 * gs_strcmp:
 * @v1: a string
 * @v2: another string
 * 
 * This function is a safe wrapper to strcmp that allow to compare
 * NULL string.
 * 
 * Return value: strcmp results.
 */
gint gs_strcmp (const gchar *v1,
	       const gchar *v2)
{
  if (v1 == NULL)
    v1 = "";

  if (v2 == NULL)
    v2 = "";

  return strcmp (v1, v2);
}

/* How many mm in one inch ? */
/* Note that, i calculated this value testing a hp device. That's not
   an "official" value. */
#define	INCH	25.399999357

/**
 * gnome_scan_mm_from_pixel:
 * @pixel: a length in pixel
 * @resolution: the resolution in dpi
 * 
 * Convert a length from pixel to millimeter using a dpi resolution.
 * 
 * Return value: The length in mm
 */
gdouble
gnome_scan_mm_from_pixel (gint pixel,
			  gint resolution)
{
  return (gdouble) pixel / ((gdouble) resolution  / INCH);
}


/**
 * gnome_scan_pixel_from_mm:
 * @mm: a length in mm
 * @resolution: the resolution in dpi
 * 
 * The opposite for gnome_scan_mm_from_pixel(). Convert a length from
 * millimeter to pixel using a dpi resolution.
 * 
 * Return value: The length in pixel
 **/
gint
gnome_scan_pixel_from_mm (gdouble mm,
			  gint resolution)
{
  return (gint) ((mm * ((gdouble) resolution / INCH)) + 0.5);
}



/********************************
 * 	   GVALUE		*
 ********************************/

/**
 * gs_value_new_boolean:
 * @boolean: a gboolean
 * 
 * Convenient function that create a #GValue from a boolean.
 * 
 * Return value: a new #GValue
 */
GValue*
gs_value_new_boolean (gboolean boolean)
{
  GValue *value = g_new0 (GValue, 1);
  g_value_init (value, G_TYPE_BOOLEAN);
  g_value_set_boolean (value, boolean);
  return value;
}

/**
 * gs_value_new_int:
 * @integer: an integer
 * 
 * Convenient function that create a #GValue from an integer.
 * 
 * Return value: a new #GValue
 */
GValue*
gs_value_new_int (gint integer)
{
  GValue *value = g_new0 (GValue, 1);
  g_value_init (value, G_TYPE_INT);
  g_value_set_int (value, integer);
  return value;
}

/**
 * gs_value_new_string:
 * @string: a string
 * 
 * Convenient function that create a #GValue from a string.
 * 
 * Return value: a new #GValue
 */
GValue*
gs_value_new_string (const gchar *string)
{
  GValue *value = g_new0 (GValue, 1);
  g_value_init (value, G_TYPE_STRING);
  g_value_set_string (value, string);
  return value;
}

/**
 * gs_value_new_double:
 * @real: a gdouble value
 * 
 * Convenient function that create a #GValue from a double.
 * 
 * Return value: a new #GValue
 */
GValue*
gs_value_new_double (gdouble real)
{
  GValue *value = g_new0 (GValue, 1);
  g_value_init (value, G_TYPE_DOUBLE);
  g_value_set_double (value, real);
  return value;
}

/**
 * gs_value_new_enum:
 * @type: the enum type
 * @v_enum: the enum value
 * 
 * Convenient function that create a #GValue from an enum value.
 * 
 * Return value: a new #GValue
 */
GValue*
gs_value_new_enum (GType type,
		  gint v_enum)
{
  GValue *value = g_new0 (GValue, 1);
  g_value_init (value, type);
  g_value_set_enum (value, v_enum);
  return value;
}

/**
 * gs_enum_get_nick:
 * @type: an enum type
 * @value: a value
 * 
 * Convenient function that return the nick of an enum value.
 * 
 * Return value: the value nick
 */
gchar*
gs_enum_get_nick (GType type,
		 gint value)
{
  GEnumClass *klass;
  GEnumValue *enu;

  klass = g_type_class_ref (type);
  enu = g_enum_get_value (klass,
			  value);
  g_type_class_unref (klass);

  return g_strdup (enu->value_nick);
}

/**
 * gs_enum_get_name:
 * @type: an enum type
 * @value: a value
 * 
 * Convenient function that return the name of an enum value.
 * 
 * Return value: the value name
 **/
gchar*
gs_enum_get_name (GType type,
		 gint value)
{
  GEnumClass *klass;
  GEnumValue *enu;

  klass = g_type_class_ref (type);
  enu = g_enum_get_value (klass,
			  value);
  g_type_class_unref (klass);

  return g_strdup (enu->value_name);
}

/**
 * gs_enum_foreach:
 * @type: An enum type
 * @func: a GFunc callback
 * @data: @func parameter
 * 
 * Loop the value of an enumeration and apply the user function @func
 * with param @data.
 */
void
gs_enum_foreach (GType type,
		 GSEFunc func,
		 gpointer data)
{
  GEnumClass *klass;
  GEnumValue *enu;
  gint value;

  klass = g_type_class_ref (type);

  for (value = klass->minimum; value <= klass->maximum; value++) {
    func (value, data);
  }

  g_type_class_unref (klass);
}
