/* GnomeScanUI - Widgets for scan dialogs
 *
 * gnomescanadvancedpreview.c
 *
 * Copyright © 2006 Étienne Bersac
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnomescan.h"
#include "gnomescanui.h"
#include "gnomescanui-intl.h"

#define GNOME_SCAN_ADVANCED_PREVIEW_ERROR		(g_type_qname (GNOME_TYPE_SCAN_ADVANCED_PREVIEW))
#define	GNOME_SCAN_ADVANCED_PREVIEW_PARENT_CLASS(klass)	(GTK_WIDGET_CLASS (g_type_class_peek_parent ((klass))))
#define GET_PRIVATE(obj)				(G_TYPE_INSTANCE_GET_PRIVATE ((obj), GNOME_TYPE_SCAN_ADVANCED_PREVIEW, GnomeScanAdvancedPreviewPrivate))

typedef struct _GnomeScanAdvancedPreviewPrivate		GnomeScanAdvancedPreviewPrivate;

struct _GnomeScanAdvancedPreviewPrivate {
  gboolean		dispose_has_run;
  GtkWidget		*hruler;
  GtkWidget		*vruler;
  GtkWidget		*gspa;
  /* PROPERTIES */
};


/********************************
 * 	    CALLBACKS		*
 ********************************/

void				gsap_vadj_changed					(GtkAdjustment *adj,
											 GnomeScanAdvancedPreview *gsap);

void				gsap_hadj_changed					(GtkAdjustment *adj,
											 GnomeScanAdvancedPreview *gsap);

void				gsap_refresh_clicked					(GtkWidget *button,
											 GnomeScanAdvancedPreview *gsap);

/********************************
 * 	    OTHERS		*
 ********************************/

void				gsap_adj_changed					(GtkAdjustment *adj,
											 GtkRuler *ruler,
											 gint resolution);

/********************************
 * 	      GOBJECT		*
 ********************************/

void				gnome_scan_advanced_preview_set_property 		(GObject *obj,
											 guint property_id,
											 const GValue *value,
											 GParamSpec *pspec);

void				gnome_scan_advanced_preview_get_property 		(GObject *obj,
											 guint property_id,
											 GValue *value,
											 GParamSpec *pspec);

void				gnome_scan_advanced_preview_finalize			(GObject *obj);

void				gnome_scan_advanced_preview_dispose			(GObject *obj);


enum {
  PROP_0,
  PROP_CONTEXT
};

enum {
  N_SIGNALS
};

static guint signals[N_SIGNALS];

G_DEFINE_TYPE (GnomeScanAdvancedPreview, gnome_scan_advanced_preview, GTK_TYPE_VBOX);

void
gnome_scan_advanced_preview_class_init (GnomeScanAdvancedPreviewClass *klass)
{
  GObjectClass* gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->set_property = gnome_scan_advanced_preview_set_property;
  gobject_class->get_property = gnome_scan_advanced_preview_get_property;
  gobject_class->dispose 	= gnome_scan_advanced_preview_dispose;
  /*   gobject_class->finalize 	= gnome_scan_advanced_preview_finalize; */

  g_type_class_add_private (gobject_class,
			    sizeof (GnomeScanAdvancedPreviewPrivate));

  /* Properties */
  g_object_class_install_property (gobject_class,
				   PROP_CONTEXT,
				   g_param_spec_object ("context",
							"Context",
							"The GnomeScanContext the widget is connected to.",
							GNOME_TYPE_SCAN_CONTEXT,
							G_PARAM_READWRITE));

  /* Signals */

}

void
gnome_scan_advanced_preview_init (GnomeScanAdvancedPreview *gsap)
{
  GnomeScanAdvancedPreviewPrivate *priv = GET_PRIVATE (gsap);

  gsap->context = NULL;
  priv->hruler	= NULL;
  priv->vruler	= NULL;
  priv->gspa	= NULL;
  priv->dispose_has_run = FALSE;
}

void
gnome_scan_advanced_preview_dispose (GObject *obj)
{
  GnomeScanAdvancedPreview *widget = GNOME_SCAN_ADVANCED_PREVIEW (obj);
  GnomeScanAdvancedPreviewPrivate *priv = GET_PRIVATE (widget);
  GnomeScanAdvancedPreviewClass *b_klass = GNOME_SCAN_ADVANCED_PREVIEW_GET_CLASS (obj);

  /* That would be nice if g_return_if_fail were noiseless. */
  if (priv->dispose_has_run == TRUE) {
    return;
  }

  /* unref devices */
  g_object_unref (widget->context);
  priv->dispose_has_run = TRUE;

  /* chain */
  /*   GNOME_SCAN_ADVANCED_PREVIEW_PARENT_CLASS (b_klass)->dispose (obj); */
}

void
gnome_scan_advanced_preview_finalize (GObject *obj)
{
  GnomeScanAdvancedPreview *gsap = GNOME_SCAN_ADVANCED_PREVIEW (obj);
  GnomeScanAdvancedPreviewClass *klass = GNOME_SCAN_ADVANCED_PREVIEW_GET_CLASS (gsap);

  /*   GNOME_SCAN_ADVANCED_PREVIEW_PARENT_CLASS (klass)->finalize (obj); */
}

void
gnome_scan_advanced_preview_set_property (GObject *obj,
					  guint property_id,
					  const GValue *value,
					  GParamSpec *pspec)
{
  GnomeScanAdvancedPreview *gsap = GNOME_SCAN_ADVANCED_PREVIEW (obj);
  GnomeScanAdvancedPreviewPrivate *priv = GET_PRIVATE (gsap);

  switch (property_id) {
  case PROP_CONTEXT:
    gsap->context = GNOME_SCAN_CONTEXT (g_value_dup_object (value));
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj,
				      property_id,
				      pspec);
    break;
  }
}

void
gnome_scan_advanced_preview_get_property (GObject *obj,
					  guint property_id,
					  GValue *value,
					  GParamSpec *pspec)
{
  GnomeScanAdvancedPreview *gsap = GNOME_SCAN_ADVANCED_PREVIEW (obj);
  GnomeScanAdvancedPreviewPrivate *priv = GET_PRIVATE (gsap);

  switch (property_id) {
  case PROP_CONTEXT:
    g_value_set_object (value, gsap->context);
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj,
				      property_id,
				      pspec);
    break;
  }
}




/********************************
 * 	      METHODS		*
 ********************************/

/**
 * gnome_scan_advanced_preview_new:
 * @context: a #GnomeScanContext
 * 
 * Create a new #GnomeScanAdvancedPreview connected to @context.
 * 
 * Return value: a new #GnomeScanAdvancedPreview
 **/
GtkWidget*
gnome_scan_advanced_preview_new (GnomeScanContext *context)
{
  GtkWidget *widget, *toolbar, *table, *gspa, *area;
  GtkToolItem *item;
  GnomeScanAdvancedPreview *gsap;
  GnomeScanAdvancedPreviewPrivate *priv;

  gsap = g_object_new (GNOME_TYPE_SCAN_ADVANCED_PREVIEW,
		       "context", context,
		       NULL);

  priv = GET_PRIVATE (gsap);

  gspa = priv->gspa = gnome_scan_preview_area_new (context);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (gspa), GTK_POLICY_NEVER, GTK_POLICY_NEVER);
  area = GNOME_SCAN_PREVIEW_AREA (gspa)->drawing_area;

  /* TOOLBAR */
  toolbar = gtk_toolbar_new ();
  gtk_box_pack_start (GTK_BOX (gsap), toolbar, FALSE, TRUE, 0);

  item = gtk_tool_button_new_from_stock (GTK_STOCK_REFRESH);
  gtk_toolbar_insert (GTK_TOOLBAR (toolbar), item,-1);
  g_signal_connect (GTK_TOOL_BUTTON (item),
		    "clicked",
		    G_CALLBACK (gsap_refresh_clicked),
		    gsap);


  item = gtk_tool_button_new_from_stock (GS_STOCK_ROTATE_COUNTER_CLOCKWISE);
  gtk_toolbar_insert (GTK_TOOLBAR (toolbar), item, -1);
  g_signal_connect_swapped (GTK_TOOL_BUTTON (item),
			    "clicked",
			    G_CALLBACK (GNOME_SCAN_PREVIEW_AREA_GET_CLASS (gspa)->rotate_counter_clockwise_clicked),
			    gspa);

  item = gtk_tool_button_new_from_stock (GS_STOCK_ROTATE_CLOCKWISE);
  gtk_toolbar_insert (GTK_TOOLBAR (toolbar), item,-1);
  g_signal_connect_swapped (GTK_TOOL_BUTTON (item),
			    "clicked",
			    G_CALLBACK (GNOME_SCAN_PREVIEW_AREA_GET_CLASS (gspa)->rotate_clockwise_clicked),
			    gspa);

  

  /* PREVIEW AREA */
#define CELL_SPACING	2

  table = gtk_table_new (3, 3, FALSE);
  gtk_box_pack_start (GTK_BOX (gsap), table, TRUE, TRUE, 0);

  gtk_table_attach (GTK_TABLE (table),
		    gspa,
		    1, 2,
		    1, 2,
		    GTK_EXPAND | GTK_FILL,
		    GTK_EXPAND | GTK_FILL,
		    CELL_SPACING, CELL_SPACING);

  /* HRULER */
  widget = priv->hruler = gtk_hruler_new ();
  /* TODO: allow to choose INCHES or CENTIMETER using gconf */
  gtk_ruler_set_metric (GTK_RULER (widget), GTK_CENTIMETERS);

  g_signal_connect_swapped (G_OBJECT (area),
			    "motion-notify-event",
			    G_CALLBACK (GTK_WIDGET_GET_CLASS (widget)->motion_notify_event),
			    G_OBJECT (widget));

  g_signal_connect (gtk_scrolled_window_get_hadjustment (GTK_SCROLLED_WINDOW (gspa)),
		    "changed",
		    G_CALLBACK (gsap_hadj_changed),
		    gsap);

  g_signal_connect (gtk_scrolled_window_get_hadjustment (GTK_SCROLLED_WINDOW (gspa)),
		    "value-changed",
		    G_CALLBACK (gsap_hadj_changed),
		    gsap);

  gtk_table_attach (GTK_TABLE (table),
		    widget,
		    1, 2,
		    0, 1,
		    GTK_EXPAND | GTK_FILL,
		    GTK_FILL,
		    CELL_SPACING, CELL_SPACING);

  /* VRULER */

  widget = priv->vruler = gtk_vruler_new ();
  gtk_ruler_set_metric (GTK_RULER (widget), GTK_CENTIMETERS);

  g_signal_connect_swapped (G_OBJECT (area),
			    "motion-notify-event",
			    G_CALLBACK (GTK_WIDGET_GET_CLASS (widget)->motion_notify_event),
			    G_OBJECT (widget));

  g_signal_connect (gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (gspa)),
		    "changed",
		    G_CALLBACK (gsap_vadj_changed),
		    gsap);

  g_signal_connect (gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (gspa)),
		    "value-changed",
		    G_CALLBACK (gsap_vadj_changed),
		    gsap);

  gtk_table_attach (GTK_TABLE (table),
		    widget,
		    0, 1,
		    1, 2,
		    GTK_FILL,
		    GTK_EXPAND | GTK_FILL,
		    CELL_SPACING, CELL_SPACING);

  /* SCROLLBARS */

  widget = gtk_hscrollbar_new (gtk_scrolled_window_get_hadjustment (GTK_SCROLLED_WINDOW (gspa)));
  gtk_table_attach (GTK_TABLE (table),
		    widget,
		    1, 2,
		    2, 3,
		    GTK_EXPAND | GTK_FILL,
		    GTK_FILL,
		    CELL_SPACING, CELL_SPACING);

  widget = gtk_vscrollbar_new (gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (gspa)));
  gtk_table_attach (GTK_TABLE (table),
		    widget,
		    2, 3,
		    1, 2,
		    GTK_FILL,
		    GTK_EXPAND | GTK_FILL,
		    CELL_SPACING, CELL_SPACING);

  return GTK_WIDGET (gsap);
}

/********************************
 * 	    CALLBACKS		*
 ********************************/


void
gsap_hadj_changed (GtkAdjustment *adj,
		   GnomeScanAdvancedPreview *gsap)
{
  GnomeScanAdvancedPreviewPrivate *priv = GET_PRIVATE (gsap);
  gsap_adj_changed (adj, GTK_RULER (priv->hruler), GNOME_SCAN_PREVIEW_AREA (priv->gspa)->resolution);
}

void
gsap_vadj_changed (GtkAdjustment *adj,
		   GnomeScanAdvancedPreview *gsap)
{
  GnomeScanAdvancedPreviewPrivate *priv = GET_PRIVATE (gsap);
  gsap_adj_changed (adj, GTK_RULER (priv->vruler), GNOME_SCAN_PREVIEW_AREA (priv->gspa)->resolution);
}

void
gsap_refresh_clicked (GtkWidget *button,
		      GnomeScanAdvancedPreview *gsap)
{
  gnome_scan_context_acquire_preview (gsap->context);
}


/********************************
 * 	    OTHERS		*
 ********************************/

/* translate from preview resolution to GtkRuler resolution (72dpi)  */
void
gsap_adj_changed (GtkAdjustment *adj,
		  GtkRuler *ruler,
		  gint resolution)
{
  gint from = 72;

#define transdpi(val)	((val)*from)/resolution
  gtk_ruler_set_range (ruler,
		       transdpi (adj->value),
		       transdpi (adj->value + adj->page_size),
		       G_MAXDOUBLE,
		       transdpi (adj->page_size));
#undef	transdpi
}
