/* GnomeScanUI - Widgets for scan dialogs
 *
 * gnomescanresolutionselector.c
 *
 * Copyright © 2006 Étienne Bersac
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnomescanui.h"
#include "gnomescanui-intl.h"

#define GNOME_SCAN_RESOLUTION_SELECTOR_ERROR			(g_type_qname (GNOME_TYPE_SCAN_RESOLUTION_SELECTOR))
#define	GNOME_SCAN_RESOLUTION_SELECTOR_PARENT_CLASS(klass)	(GTK_WIDGET_CLASS (g_type_class_peek_parent ((klass))))
#define GET_PRIVATE(obj)					(G_TYPE_INSTANCE_GET_PRIVATE ((obj), GNOME_TYPE_SCAN_RESOLUTION_SELECTOR, GnomeScanResolutionSelectorPrivate))

typedef struct _GnomeScanResolutionSelectorPrivate		GnomeScanResolutionSelectorPrivate;

struct _GnomeScanResolutionSelectorPrivate {
  gboolean		dispose_has_run;
  GnomeScanContext	*context;
  GtkAdjustment		*x;
  GtkWidget		*xscale;
  GtkWidget		*xspin;
  GtkAdjustment		*y;
  GtkWidget		*yscale;
  GtkWidget		*yspin;
  GtkWidget		*chain;
  gboolean		chained;
};

/********************************
 * 	      CALLBACKS		*
 ********************************/

void				gsrs_scanner_selected				(GnomeScanContext *context,
										 GnomeScanner *scanner,
										 GnomeScanResolutionSelector *gsrs);

gchar*				gsrs_format_value				(GtkScale *scale,
										 gdouble value);

void				gsrs_value_changed				(GtkRange *range,
										 GnomeScanResolutionSelector *gsrs);

void				gsrs_chain_clicked				(GtkButton *button,
										 GnomeScanResolutionSelector *gsrs);

/********************************
 * 	       OTHERS		*
 ********************************/


/********************************
 * 	      GOBJECT		*
 ********************************/

void				gnome_scan_resolution_selector_set_property	(GObject *obj,
										 guint property_id,
										 const GValue *value,
										 GParamSpec *pspec);

void				gnome_scan_resolution_selector_get_property	(GObject *obj,
										 guint property_id,
										 GValue *value,
										 GParamSpec *pspec);

void				gnome_scan_resolution_selector_finalize		(GObject *obj);

void				gnome_scan_resolution_selector_dispose		(GObject *obj);


enum {
  PROP_0,
  PROP_CONTEXT
};

enum {
  N_SIGNALS
};

static guint signals[N_SIGNALS];


G_DEFINE_TYPE (GnomeScanResolutionSelector, gnome_scan_resolution_selector, GTK_TYPE_TABLE);


void
gnome_scan_resolution_selector_class_init (GnomeScanResolutionSelectorClass *klass)
{
  GObjectClass* gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->set_property = gnome_scan_resolution_selector_set_property;
  gobject_class->get_property = gnome_scan_resolution_selector_get_property;
  gobject_class->dispose 	= gnome_scan_resolution_selector_dispose;
  /*   gobject_class->finalize 	= gnome_scan_resolution_selector_finalize; */

  g_type_class_add_private (gobject_class,
			    sizeof (GnomeScanResolutionSelectorPrivate));

  /* Properties */
  g_object_class_install_property (gobject_class,
				   PROP_CONTEXT,
				   g_param_spec_object ("context",
							"Context",
							"The GnomeScanContext the widget is connected to.",
							GNOME_TYPE_SCAN_CONTEXT,
							G_PARAM_READWRITE));

  /* Signals */
}

void
gnome_scan_resolution_selector_init (GnomeScanResolutionSelector *widget)
{
  GnomeScanResolutionSelectorPrivate *priv = GET_PRIVATE (widget);

  priv->dispose_has_run = FALSE;
  priv->context		= NULL;
  priv->x		= NULL;
  priv->xscale		= NULL;
  priv->xspin		= NULL;
  priv->y		= NULL;
  priv->yscale		= NULL;
  priv->yspin		= NULL;
  priv->chain		= NULL;
  priv->chained		= FALSE;
}

void
gnome_scan_resolution_selector_dispose (GObject *obj)
{
  GnomeScanResolutionSelector *widget = GNOME_SCAN_RESOLUTION_SELECTOR (obj);
  GnomeScanResolutionSelectorPrivate *priv = GET_PRIVATE (widget);
  GnomeScanResolutionSelectorClass *b_klass = GNOME_SCAN_RESOLUTION_SELECTOR_GET_CLASS (obj);

  /* That would be nice if g_return_if_fail were noiseless. */
  if (priv->dispose_has_run == TRUE) {
    return;
  }

  /* unref objects */
  g_object_unref (priv->context);

  priv->dispose_has_run = TRUE;

  /* chain */
  /*   GNOME_SCAN_RESOLUTION_SELECTOR_PARENT_CLASS (b_klass)->dispose (obj); */
}

void
gnome_scan_resolution_selector_finalize (GObject *obj)
{
  GnomeScanResolutionSelector *widget = GNOME_SCAN_RESOLUTION_SELECTOR (obj);
  GnomeScanResolutionSelectorClass *b_klass = GNOME_SCAN_RESOLUTION_SELECTOR_GET_CLASS (obj);

  /*   GNOME_SCAN_RESOLUTION_SELECTOR_PARENT_CLASS (b_klass)->finalize (obj); */
}

void
gnome_scan_resolution_selector_set_property (GObject *obj,
					     guint property_id,
					     const GValue *value,
					     GParamSpec *pspec)
{
  GnomeScanResolutionSelector *widget = GNOME_SCAN_RESOLUTION_SELECTOR (obj);
  GnomeScanResolutionSelectorPrivate *priv = GET_PRIVATE (widget);

  switch (property_id) {
  case PROP_CONTEXT:
    priv->context = GNOME_SCAN_CONTEXT (g_value_dup_object (value));
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj,
				      property_id,
				      pspec);
    break;
  }
}

void
gnome_scan_resolution_selector_get_property (GObject *obj,
					     guint property_id,
					     GValue *value,
					     GParamSpec *pspec)
{
  GnomeScanResolutionSelector *widget = GNOME_SCAN_RESOLUTION_SELECTOR (obj);
  GnomeScanResolutionSelectorPrivate *priv = GET_PRIVATE (widget);

  switch (property_id) {
  case PROP_CONTEXT:
    g_value_set_object (value, priv->context);
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj,
				      property_id,
				      pspec);
    break;
  }
}




/********************************
 * 	      METHODS		*
 ********************************/

/**
 * gnome_scan_resolution_selector_new:
 * @context: a #GnomeScanContext
 * 
 * Create a new #GnomeScanResolutionSelector connected to @context.
 * 
 * Return value: a new #GnomeScanResolutionSelector
 **/
GtkWidget*
gnome_scan_resolution_selector_new (GnomeScanContext *context)
{
  GtkWidget *widget, *button, *icon;
  GnomeScanResolutionSelector *gsrs;
  GnomeScanResolutionSelectorPrivate *priv;

  widget = GTK_WIDGET (g_object_new (GNOME_TYPE_SCAN_RESOLUTION_SELECTOR,
				     "context", context,
				     NULL));

  gsrs = GNOME_SCAN_RESOLUTION_SELECTOR (widget);
  priv = GET_PRIVATE (gsrs);
  gtk_table_set_col_spacings (GTK_TABLE (widget), 6);
  gtk_table_set_row_spacings (GTK_TABLE (widget), 6);

  /* X */
  priv->xscale = gtk_hscale_new (NULL);
  priv->x = gtk_range_get_adjustment (GTK_RANGE (priv->xscale));
  gtk_scale_set_draw_value (GTK_SCALE (priv->xscale), FALSE);
  gtk_range_set_update_policy (GTK_RANGE (priv->xscale), GTK_UPDATE_CONTINUOUS);
  gtk_range_set_increments (GTK_RANGE (priv->xscale),
			    5,	/* seems buggy, value is doubled */
			    100);

  priv->xspin = gtk_spin_button_new (priv->x, .5, 0);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (priv->xspin), GTK_UPDATE_CONTINUOUS);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (priv->xspin), TRUE);

  gtk_table_attach (GTK_TABLE (widget),
		    gtk_label_new (_("X resolution")),
		    0, 1,
		    0, 1,
		    GTK_FILL,
		    GTK_FILL,
		    0, 0);
  gtk_table_attach (GTK_TABLE (widget),
		    priv->xscale,
		    1, 2,
		    0, 1,
		    GTK_EXPAND | GTK_FILL,
		    GTK_FILL,
		    0, 0);
  gtk_table_attach (GTK_TABLE (widget),
		    priv->xspin,
		    2, 3,
		    0, 1,
		    GTK_FILL,
		    GTK_FILL,
		    0, 0);
  gtk_table_attach (GTK_TABLE (widget),
		    gtk_label_new (_("dpi")),
		    3, 4,
		    0, 1,
		    GTK_FILL,
		    GTK_FILL,
		    0, 0);

  g_signal_connect (GTK_RANGE (priv->xscale),
		    "value-changed",
		    G_CALLBACK (gsrs_value_changed),
		    gsrs);

  /* Y */
  priv->yscale = gtk_hscale_new (NULL);
  priv->y = gtk_range_get_adjustment (GTK_RANGE (priv->yscale));
  g_object_ref (priv->y);
  gtk_scale_set_draw_value (GTK_SCALE (priv->yscale), FALSE);
  gtk_range_set_update_policy (GTK_RANGE (priv->yscale), GTK_UPDATE_CONTINUOUS);
  gtk_range_set_increments (GTK_RANGE (priv->yscale),
			    5,	/* seems buggy, value is doubled */
			    100);

  priv->yspin = gtk_spin_button_new (priv->y, .5, 0);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (priv->yspin), GTK_UPDATE_CONTINUOUS);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (priv->yspin), TRUE);

  gtk_table_attach (GTK_TABLE (widget),
		    gtk_label_new (_("Y resolution")),
		    0, 1,
		    1, 2,
		    GTK_FILL,
		    GTK_FILL,
		    0, 0);
  gtk_table_attach (GTK_TABLE (widget),
		    priv->yscale,
		    1, 2,
		    1, 2,
		    GTK_EXPAND | GTK_FILL,
		    GTK_FILL,
		    0, 0);
  gtk_table_attach (GTK_TABLE (widget),
		    priv->yspin,
		    2, 3,
		    1, 2,
		    GTK_FILL,
		    GTK_FILL,
		    0, 0);
  gtk_table_attach (GTK_TABLE (widget),
		    gtk_label_new (_("dpi")),
		    3, 4,
		    1, 2,
		    GTK_FILL,
		    GTK_FILL,
		    0, 0);

  g_signal_connect (GTK_RANGE (priv->yscale),
		    "value-changed",
		    G_CALLBACK (gsrs_value_changed),
		    gsrs);

  /* chain */
  priv->chain = gtk_button_new ();
  gtk_button_set_relief (GTK_BUTTON (priv->chain),
			 GTK_RELIEF_NONE);
  g_signal_connect (priv->chain,
		    "clicked",
		    (GCallback) gsrs_chain_clicked,
		    gsrs);

  icon = gtk_image_new_from_stock (GS_STOCK_VCHAIN_BROKEN,
				   -1);

  gtk_container_add (GTK_CONTAINER (priv->chain),
		     icon);

  gtk_table_attach (GTK_TABLE (widget),
		    priv->chain,
		    4, 5,
		    0, 2,
		    0,
		    0,
		    0, 0);
 
  g_signal_connect (context,
		    "scanner-selected",
		    G_CALLBACK (gsrs_scanner_selected),
		    gsrs);

  return widget;
}

/********************************
 * 	      CALLBACKS		*
 ********************************/

void
gsrs_scanner_selected (GnomeScanContext *context,
		       GnomeScanner *scanner,
		       GnomeScanResolutionSelector *gsrs)
{
  GnomeScanResolutionSelectorPrivate *priv = GET_PRIVATE (gsrs);
  GnomeScanRange *range = gnome_scanner_get_resolution_range (scanner);
  gdouble xresolution, yresolution;

  if (range) {
    gtk_widget_set_sensitive (GTK_WIDGET (gsrs), TRUE);
    gnome_scan_context_get_resolutions (context, &xresolution, &yresolution);
    gtk_range_set_range (GTK_RANGE (priv->xscale), range->lower, range->upper);
    gtk_range_set_value (GTK_RANGE (priv->xscale), xresolution);

    if (gnome_scanner_has_xy_resolution (scanner)) {
      gtk_range_set_range (GTK_RANGE (priv->yscale), range->lower, range->upper);
      gtk_range_set_value (GTK_RANGE (priv->yscale), yresolution);
      gtk_widget_set_sensitive (priv->chain, TRUE);
    }
    else {
      gtk_widget_set_sensitive (priv->chain, FALSE);
      if (!priv->chained) {
	gtk_button_clicked (GTK_BUTTON (priv->chain));
      }
    }
  }
  else {
    gtk_widget_set_sensitive (GTK_WIDGET (gsrs), FALSE);
  }
}

void
gsrs_value_changed (GtkRange *range,
		    GnomeScanResolutionSelector *gsrs)
{
  gnome_scan_context_set_resolutions (GET_PRIVATE (gsrs)->context,
				      gtk_range_get_value (range),
				      gtk_range_get_value (range));
}

void
gsrs_chain_clicked (GtkButton *button,
		    GnomeScanResolutionSelector *gsrs)
{
  GnomeScanResolutionSelectorPrivate *priv = GET_PRIVATE (gsrs);
  static const gchar *stocks[] = { GS_STOCK_VCHAIN,
				   GS_STOCK_VCHAIN_BROKEN };

  GtkWidget *icon;
  priv->chained = !priv->chained;
  gint stock_n = priv->chained ? 0 : 1;

  icon = gtk_bin_get_child (GTK_BIN (button));
  gtk_image_set_from_stock (GTK_IMAGE (icon),
			    stocks[stock_n],
			    -1);

  if (priv->chained) {
    gtk_range_set_adjustment (GTK_RANGE (priv->yscale), priv->x);
    gtk_spin_button_set_adjustment (GTK_SPIN_BUTTON (priv->yspin), priv->x);
  }
  else {
    gtk_adjustment_set_value (priv->y,
			      gtk_adjustment_get_value (priv->x));
    gtk_range_set_adjustment (GTK_RANGE (priv->yscale), priv->y);
    gtk_spin_button_set_adjustment (GTK_SPIN_BUTTON (priv->yspin), priv->y);
  }
}


/********************************
 * 	       OTHERS		*
 ********************************/

