/* Gnome Scan - Scan as easy as you print
 * Copyright © 2007  Étienne Bersac <bersace03@laposte.net>
 *
 * Gnome Scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * gnome-scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with gnome-scan.  If not, write to:
 *
 *	the Free Software Foundation, Inc.
 *	51 Franklin Street, Fifth Floor
 *	Boston, MA 02110-1301, USA
 */

/**
 * SECTION: gnome-scan-module
 * @short_description: Module loading and initialization
 *
 * A module represent the library opened with #GModule where are
 * stored a list of #GObjects.
 **/
#include "gnome-scan-module.h"

enum
{
	PROP_0,
	PROP_FILENAME
};

static gboolean gnome_scan_module_load (GTypeModule *gmodule);
static void gnome_scan_module_unload (GTypeModule *gmodule);

static GTypeModuleClass* parent_class = NULL;

G_DEFINE_TYPE (GnomeScanModule, gnome_scan_module, G_TYPE_TYPE_MODULE);

static void
gnome_scan_module_init (GnomeScanModule *object)
{
	GnomeScanModule *module = GNOME_SCAN_MODULE (object);
	module->filename	= NULL;
	module->library		= NULL;
	module->init		= NULL;
	module->finalize	= NULL;
}

static void
gnome_scan_module_finalize (GObject *object)
{
	/* TODO: Add deinitalization code here */

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gnome_scan_module_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
	g_return_if_fail (GNOME_IS_SCAN_MODULE (object));
	GnomeScanModule *module = GNOME_SCAN_MODULE (object);
	
	switch (prop_id)
	{
	case PROP_FILENAME:
		g_free (module->filename);
		module->filename = g_value_dup_string (value);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

static void
gnome_scan_module_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
	g_return_if_fail (GNOME_IS_SCAN_MODULE (object));
	GnomeScanModule *module = GNOME_SCAN_MODULE (object);
	
	switch (prop_id)
	{
	case PROP_FILENAME:
			g_value_set_string (value, module->filename);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

static void
gnome_scan_module_class_init (GnomeScanModuleClass *klass)
{
	GObjectClass* object_class = G_OBJECT_CLASS (klass);
	parent_class = G_TYPE_MODULE_CLASS (klass);

	object_class->finalize = gnome_scan_module_finalize;
	object_class->set_property = gnome_scan_module_set_property;
	object_class->get_property = gnome_scan_module_get_property;
	
	parent_class->load = gnome_scan_module_load;
	parent_class->unload = gnome_scan_module_unload;
	
	/**
	 * GnomeScanModule:filename:
	 *
	 * The path to the shared object.
	 **/
	g_object_class_install_property (object_class,
	                                 PROP_FILENAME,
	                                 g_param_spec_string ("filename",
	                                                      "Filename",
	                                                      "Library filename for use with GModule",
	                                                      NULL,
	                                                      G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
}

static gboolean
gnome_scan_module_load (GTypeModule *gmodule)
{
	GnomeScanModule *module = GNOME_SCAN_MODULE (gmodule);
	
	g_return_val_if_fail (module->filename, FALSE);
	
	module->library = g_module_open (module->filename, G_MODULE_BIND_MASK);
	if (!module->library) {
		g_warning (g_module_error ());
		return FALSE;
	}
	
	if (!g_module_symbol (module->library, "gnome_scan_module_init",
						  (gpointer *) &module->init) ||
		!g_module_symbol (module->library, "gnome_scan_module_finalize",
						  (gpointer *) &module->finalize))
	{
		g_warning ("%s does not contains a valid %s", module->filename, G_OBJECT_TYPE_NAME (gmodule));
		return FALSE;
	}
	
	module->init (module);
	return TRUE;
}

static void
gnome_scan_module_unload (GTypeModule *gmodule)
{
	GnomeScanModule *module = GNOME_SCAN_MODULE (module);
	module->finalize (module);
}


/**
 * gnome_scan_module_new:
 * @filename: The path to shared object
 * 
 * Open and initialize a new module. This function should be used only
 * by #GnomeScanModuleManager.
 * 
 * Returns: a new #GnomeScanModule
 **/
GnomeScanModule*
gnome_scan_module_new (gchar *filename)
{
	return g_object_new (GNOME_TYPE_SCAN_MODULE,
						 "filename", filename,
						 NULL);
}

