/*
 * GNOME Speech - Speech services for the GNOME desktop
 *
 * Copyright 2002 Sun Microsystems Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * viavoicespeaker.c: Implements the ViavoiceSpeaker object--
 *                            a GNOME Speech driver for IBM's Viavoice TTS RTK
 *
 */

#include <unistd.h>
#include <libbonobo.h>
#include <glib/gmain.h>
#include <glib/gthread.h>
#include <gnome-speech/gnome-speech.h>
#include "viavoicespeaker.h"
 


typedef struct {
	GNOME_Speech_speech_callback_type type;
	gint text_id;
	gint offset;
} index_queue_entry;



static GObjectClass *parent_class;

/* Unique text id for each utterance spoken by this speaker */
static gint text_id = 0;



static void
viavoice_speaker_add_index (ViavoiceSpeaker *s,
			    GNOME_Speech_speech_callback_type type,
			    gint text_id,
			    gint offset)
{
	index_queue_entry *e;
	if (s->callback == CORBA_OBJECT_NIL)
		return;
	
	e = g_new (index_queue_entry, 1);
	e->type = type;
	e->text_id = text_id;
	e->offset = offset;
	s->index_queue = g_slist_append (s->index_queue, e);
	eciInsertIndex (s->handle, 0);
}



static gboolean
viavoice_speaker_timeout_callback (void *data)
{
	ViavoiceSpeaker *s = VIAVOICE_SPEAKER (data);

	eciSpeaking (s->handle);
	return TRUE;
}



static ECICallbackReturn
viavoice_speaker_index_callback (ECIHand handle,
			   ECIMessage msg,
			   long param,
			   void *data)
{
	ViavoiceSpeaker *s = VIAVOICE_SPEAKER(data);
	index_queue_entry *e = NULL;
	CORBA_Environment ev;
	
	if (s && s->index_queue) {
		e = (index_queue_entry *) s->index_queue->data;
		s->index_queue = g_slist_remove_link (s->index_queue, s->index_queue);
	}
	switch (msg) {
	case eciIndexReply:
		if (s->callback != CORBA_OBJECT_NIL && e) {
			CORBA_exception_init (&ev);
			GNOME_Speech_SpeechCallback_notify (s->callback,
							    e->type,
							    e->text_id,
							    e->offset,
							    &ev);
			g_free (e);
			e = NULL;
		}
		
		break;
	case eciWaveformBuffer:
	case eciPhonemeBuffer:
	case eciPhonemeIndexReply:
		break;
	}
	if (e)
		g_free (e);
	return eciDataProcessed;
}



static ViavoiceSpeaker *
viavoice_speaker_from_servant (PortableServer_Servant *servant)
{
	return VIAVOICE_SPEAKER(bonobo_object_from_servant (servant));
}



static void
viavoice_add_parameter (ViavoiceSpeaker *viavoice_speaker,
			ECIVoiceParam param,
			const gchar *parameter_name,
			gdouble min,
			gdouble max,
			parameter_set_func func)
{
	Speaker *speaker = SPEAKER(viavoice_speaker);
	gdouble current;
	current = (gdouble)
		eciGetVoiceParam (viavoice_speaker->handle, 0, param);
	speaker_add_parameter (speaker,
			       parameter_name,
			       min,
			       current,
			       max,
			       func);
}



static CORBA_boolean
viavoice_registerSpeechCallback (PortableServer_Servant servant,
				 const GNOME_Speech_SpeechCallback callback,
				 CORBA_Environment *ev)
{
	ViavoiceSpeaker *s = viavoice_speaker_from_servant (servant);

	/* Store reference to callback */

	s->callback = CORBA_Object_duplicate (callback, ev);

	/* Add a timeout callback to poke this instance of Viavoice */

	s->timeout_id = g_timeout_add_full (G_PRIORITY_HIGH_IDLE, 100,
			    viavoice_speaker_timeout_callback, s, NULL);

        /* Set up ECI callback */

	eciRegisterCallback (s->handle,
			     viavoice_speaker_index_callback,
			     s);

	return TRUE;
}



static CORBA_long
viavoice_say (PortableServer_Servant servant,
	      const CORBA_char *text,
	      CORBA_Environment *ev)
{
	ViavoiceSpeaker *speaker = viavoice_speaker_from_servant (servant);
  
	g_return_val_if_fail (speaker, FALSE);
	g_return_val_if_fail (speaker->handle != NULL_ECI_HAND, FALSE);

	text_id++;
	viavoice_speaker_add_index (speaker,
				    GNOME_Speech_speech_callback_speech_started,
				    text_id,
				    0);
	eciAddText (speaker->handle, text);
	viavoice_speaker_add_index (speaker,
				    GNOME_Speech_speech_callback_speech_ended,
				    text_id,
				    0);
	eciSynthesize (speaker->handle);
	return text_id;
}



static CORBA_boolean
viavoice_stop (PortableServer_Servant servant,
	       CORBA_Environment *ev)
{
	ViavoiceSpeaker *speaker = viavoice_speaker_from_servant (servant);
	g_return_val_if_fail (speaker->handle != NULL_ECI_HAND, FALSE);
	eciStop (speaker->handle);
	return TRUE;
}



static void
viavoice_speaker_init (ViavoiceSpeaker *speaker)
{
	speaker->handle = NULL_ECI_HAND;
}



static void
viavoice_speaker_finalize (GObject *obj)
{
	ViavoiceSpeaker *speaker = VIAVOICE_SPEAKER (obj);

        /* Destroy the ECI instance */

	if (speaker->handle != NULL_ECI_HAND) 
	{
		eciDelete (speaker->handle);
	}

	/* Remove timeout */

	if (speaker->timeout_id >= 0)
		g_source_remove (speaker->timeout_id);

	if (parent_class->finalize)
		parent_class->finalize (obj);
}



static void
viavoice_speaker_class_init (ViavoiceSpeaker *klass)
{
	SpeakerClass *class = SPEAKER_CLASS (klass);
	GObjectClass *object_class = G_OBJECT_CLASS(klass);
  
	parent_class = g_type_class_peek_parent (klass);
	object_class->finalize = viavoice_speaker_finalize;

	/* Initialize parent class epv table */

	class->epv.say = viavoice_say;
	class->epv.stop = viavoice_stop;
	class->epv.registerSpeechCallback = viavoice_registerSpeechCallback;
}




BONOBO_TYPE_FUNC (ViavoiceSpeaker,
		  speaker_get_type (),
		  viavoice_speaker);



static gboolean
viavoice_set_gender (Speaker *speaker,
		     gdouble new_gender)
{
	ViavoiceSpeaker *s = VIAVOICE_SPEAKER(speaker);
	eciSetVoiceParam (s->handle, 0, eciGender, (int) new_gender);
	return TRUE;
}



static gboolean
viavoice_set_head_size (Speaker *speaker,
			gdouble new_head_size)
{
	ViavoiceSpeaker *s = VIAVOICE_SPEAKER(speaker);
	eciSetVoiceParam (s->handle, 0, eciHeadSize, (int) new_head_size);
	return TRUE;
}



static gboolean
viavoice_set_pitch (Speaker *speaker,
		    gdouble new_pitch)
{
	ViavoiceSpeaker *s = VIAVOICE_SPEAKER(speaker);
	eciSetVoiceParam (s->handle, 0, eciPitchBaseline, (int) new_pitch);
	return TRUE;
}



static gboolean
viavoice_set_pitch_fluctuation (Speaker *speaker,
				gdouble new_pitch_fluctuation)
{
	ViavoiceSpeaker *s = VIAVOICE_SPEAKER(speaker);
	eciSetVoiceParam (s->handle, 0, eciPitchFluctuation, (int) new_pitch_fluctuation);
	return TRUE;
}



static gboolean
viavoice_set_roughness (Speaker *speaker,
			gdouble new_roughness)
{
	ViavoiceSpeaker *s = VIAVOICE_SPEAKER(speaker);
	eciSetVoiceParam (s->handle, 0, eciRoughness, (int) new_roughness);
	return TRUE;
}



static gboolean
viavoice_set_breathiness (Speaker *speaker,
			  gdouble new_breathiness)
{
	ViavoiceSpeaker *s = VIAVOICE_SPEAKER(speaker);
	eciSetVoiceParam (s->handle, 0, eciBreathiness, (int) new_breathiness);
	return TRUE;
}



static gboolean
viavoice_set_rate (Speaker *speaker,
		   gdouble new_rate)
{
	ViavoiceSpeaker *s = VIAVOICE_SPEAKER(speaker);
	eciSetVoiceParam (s->handle, 0, eciSpeed, (int) new_rate);
	return TRUE;
}



static gboolean
viavoice_set_volume (Speaker *speaker,
		     gdouble new_volume)
{
	ViavoiceSpeaker *s = VIAVOICE_SPEAKER(speaker);
	eciSetVoiceParam (s->handle, 0, eciVolume, (int) new_volume);
	return TRUE;
}



static gboolean
viavoice_set_voice (Speaker *speaker,
		    gdouble new_voice)
{
	ViavoiceSpeaker *s = VIAVOICE_SPEAKER(speaker);
	eciCopyVoice (s->handle,
		      (int) new_voice,
		      0);
	return TRUE;
}



ViavoiceSpeaker *
viavoice_speaker_new (const GNOME_Speech_VoiceInfo *voice_spec)
{
	ViavoiceSpeaker *speaker;
	ECIHand handle;
	char name[ECI_VOICE_NAME_LENGTH];
	gint i;
	speaker = g_object_new (VIAVOICE_SPEAKER_TYPE, NULL);
  
	/* Create a handle and initialize it */

	handle = eciNew ();

	/* Set sample rate */

	eciSetVoiceParam (handle, 0, eciSampleRate, 1);
        /* Initialize the index queue  */
   
	speaker->index_queue = NULL;
	speaker->timeout_id = -1;
	
        /* Find the specified voice */

	for (i = 1; i <= ECI_PRESET_VOICES; i++) {
		char name[30];

		eciGetVoiceName (handle, i, name);
		if (!g_strcasecmp (voice_spec->name, name))
			eciCopyVoice (handle, i, 0);
	}

	speaker->handle = handle;

	/* Add parameters */

	viavoice_add_parameter (speaker,
				eciGender,
				"gender",
				0,
				1,
				viavoice_set_gender);
	viavoice_add_parameter (speaker,
				eciHeadSize,
				"head size",
				0,
				100,
				viavoice_set_head_size);
	viavoice_add_parameter (speaker,
				eciPitchBaseline,
				"pitch",
				0,
				100,
				viavoice_set_pitch);  
	viavoice_add_parameter (speaker,
				eciPitchFluctuation,
				"pitch fluctuation",
				0,
				100,
				viavoice_set_pitch_fluctuation);
	viavoice_add_parameter (speaker,
				eciRoughness,
				"roughness",
				0,
				100,
				viavoice_set_roughness);
	viavoice_add_parameter (speaker,
				eciBreathiness,
				"breathiness",
				0,
				100,
				viavoice_set_breathiness);  
	viavoice_add_parameter (speaker,
				eciSpeed,
				"rate",
				0,
				100,
				viavoice_set_rate);  
	viavoice_add_parameter (speaker,
				eciVolume,
				"volume",
				0,
				100,
				viavoice_set_volume);  

	/* Add the voice parameter */

	speaker_add_parameter (SPEAKER(speaker),
			       "voice",
			       1,
			       1,
			       ECI_PRESET_VOICES,
			       viavoice_set_voice);

	/* Add the voice names to the parameter */

	for (i = 1; i <= ECI_PRESET_VOICES; i++) {
		eciGetVoiceName (handle, i, name);
		speaker_add_parameter_value_description (SPEAKER(speaker),
							 "voice",
							 (gdouble) i,
							 name);
	}
  
	return speaker;
}




