/* smb-method.h - VFS modules for SMB
 *
 *  Copyright (C) 2001 Red Hat Inc,
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Author: Alexander Larsson <alexl@redhat.com>
 *
 * Some inspiration comes from the smbfs implementation in midnight commander
 * by Wayne Roberts <wroberts1@home.com>. 
 */

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime.h>

#include <libgnomevfs/gnome-vfs-method.h>
#include <libgnomevfs/gnome-vfs-module.h>
#include <libgnomevfs/gnome-vfs-module-shared.h>

#include "smb-method.h"

#include <sys/types.h>

#include "samba/include/config.h"
/* don't load crap in "samba/include/includes.h" we don't use and which 
   conflicts with definitions in other includes */
#undef HAVE_LIBREADLINE
#define NO_CONFIG_H
#define BOOL_DEFINED

#include "samba/include/includes.h"
#undef strcpy

typedef struct _SmbConnection SmbConnection;
typedef struct _SmbHandle SmbHandle;
typedef struct _SmbDirHandle SmbDirHandle;
typedef struct _SmbDirContent SmbDirContent;
typedef struct _SmbBrowseContent SmbBrowseContent;

/* One connection per server,share and auth-info */
struct _SmbConnection {
	struct cli_state *cli;
	char *server;
	char *share;
	/* authinfo, maybe group in struct? */
	char *user;
	char *password;
};

struct _SmbHandle {
	SmbConnection *connection;
	gboolean is_data;
	char *file_data;
	int fnum;
	GnomeVFSFileOffset offset;
};

typedef enum {
	SMB_CONTENT_WORKGROUP,
	SMB_CONTENT_SERVER,
	SMB_CONTENT_SHARE,
} SmbBrowseType;

struct _SmbBrowseContent {
	SmbBrowseType type;
	char *name;
};

struct _SmbDirContent {
	char *name;
	time_t mtime;
	time_t atime;
	time_t ctime;
	uint16 mode;
	uid_t uid;
	gid_t gid;
	SMB_OFF_T size;
};

struct _SmbDirHandle {
	gboolean is_files;
	char *server;
	GList *contents;
	const GnomeVFSDirectoryFilter *filter;
};

/* Global vars: */
static GPrivate *dir_key = NULL;
static GPrivate *static_key = NULL;
static GPrivate *get_info_key = NULL;
static GMutex *samba_lock = NULL;

extern pstring global_myname;

#define LOCK_SAMBA() 	g_mutex_lock (samba_lock)
#define UNLOCK_SAMBA() 	g_mutex_unlock (samba_lock)

#ifdef DEBUG_SMB_ENABLE
#define DEBUG_SMB(x) g_print x
#else
#define DEBUG_SMB(x) 
#endif

static char *
unix_filename_to_dos (const char *filename)
{
	int len, i;
	char *dos_filename;
	
	dos_filename = g_strdup (filename);
	len = strlen (dos_filename);
	for (i = 0; i < len; i++)
		if (dos_filename[i] == '/')
			dos_filename[i] = '\\';
	
	return dos_filename;
}

static GnomeVFSResult
smb_server_connection_new_from_ip (struct in_addr *ip,
				   const char *fallback_server_name,
				   const char *share,
				   const char *user,
				   const char *password,
				   SmbConnection **connection)
{
	struct cli_state *cli;
	struct nmb_name called, calling;
	char servername[256];
	pstring workgroup;
	SmbConnection *smbcon;
	const char *server;
	GnomeVFSResult res;

	DEBUG_SMB (("smb_server_connection_new_from_ip (ip: %s, share: %s, user: %s, password: %s)\n",
		    inet_ntoa (*ip), (share)?share:"NULL", user, password));

	if (name_status_find (0x20, *ip, servername)) {
		server = servername;
	} else {
		server = fallback_server_name;
	}
	
	make_nmb_name (&calling, global_myname, 0x0);
	make_nmb_name (&called , server, 0x20);

	DEBUG_SMB (("smb_server_connection_new_from_ip: caller.name: %s, called.name: %s\n",
		    calling.name, called.name));
	
	/* Read the workgroup from the config file */
	pstrcpy (workgroup, lp_workgroup ());

	cli = cli_initialise (NULL);
	
	if (!cli) {
		g_warning ("Couldn't initialize cli");
		return GNOME_VFS_ERROR_INTERNAL;
	}	
	if (!cli_set_port (cli, SMB_PORT)) {
		g_warning ("Couldn't set port");
		res = gnome_vfs_result_from_errno_code (cli_error (cli, NULL, NULL, NULL));
		cli_shutdown (cli);
		free (cli);
		return res;
	}
	
	if (!cli_connect (cli, server, ip)) {
		g_warning ("Couldn't connect to server");
		cli_shutdown (cli);
		free (cli);
		return GNOME_VFS_ERROR_HOST_NOT_FOUND;
	}

	cli->protocol = PROTOCOL_NT1;

	if (!cli_session_request (cli, &calling, &called)) {
		g_warning ("Couldn't request session");
		res = gnome_vfs_result_from_errno_code (cli_error (cli, NULL, NULL, NULL));
		/* TODO: smbclient tries some alternative names here */ 
		cli_shutdown (cli);
		free (cli);
		return res;
	}
	
	if (!cli_negprot (cli)) {
		g_warning ("Couldn't negotiate protocol");
		cli_shutdown (cli);
		free (cli);
		return GNOME_VFS_ERROR_LOGIN_FAILED;
	}

	if (!user)
	  user = "GUEST";
	if (!password)
	  password = "";

	if (!cli_session_setup (cli,
				/* Username: */
				(char *)user,
				/* Password: */
				(char *)password, strlen (password),
				/* NT Password: */
				(char *)password, strlen (password),
				/* Workgroup: */
				workgroup)) {
		/* Try anon login */
		
		if (!cli_session_setup (cli,
					"",
					"", 0,
					"", 0,
					workgroup)) { 
			cli_shutdown (cli);
			free (cli);
			g_warning ("Couldn't log in\n");
			return GNOME_VFS_ERROR_LOGIN_FAILED;
		}
		/* Anon login ok */
		user = "";
		password = "";
	}

	/* Connect to the share */
	if (share && !cli_send_tconX (cli,
				      (char *)share,
				      "?????",
				      (char *)password, strlen (password)+1)) {
		g_warning ("couldn't send tconX: res: %s",
			   cli_errstr (cli));
		res = gnome_vfs_result_from_errno_code (cli_error (cli, NULL, NULL, NULL));
		cli_shutdown (cli);
		free(cli);
		return res;
	}

	smbcon = g_new (SmbConnection, 1);
	smbcon->cli = cli;
	smbcon->server = g_strdup (server);
	if (share)
		smbcon->share = g_strdup (share);
	else
		smbcon->share = NULL;
	smbcon->user = g_strdup (user);
	smbcon->password = g_strdup (password);

	*connection = smbcon;
	
	return GNOME_VFS_OK;
}


static GnomeVFSResult
smb_server_connection_new (const char *server,
			   const char *share,
			   const char *user,
			   const char *password,
			   SmbConnection **connection)
{
	struct in_addr ip;

	DEBUG_SMB (("smb_server_connection_new (server: %s, share: %s, user: %s, password: %s)\n",
		    server, (share)?share:"NULL", user, password));

	
	if (!resolve_name (server, &ip, 0x20)) {
		DEBUG_SMB (("smb_server_connection_new: Couldn't resolve name %s", server));
		return GNOME_VFS_ERROR_HOST_NOT_FOUND;
	}

	return smb_server_connection_new_from_ip (&ip,
						  server,
						  share,
						  user,
						  password,
						  connection);
}

static void
smb_connection_destroy (SmbConnection *connection)
{
	cli_shutdown (connection->cli);
	g_free (connection->cli);
	g_free (connection->server);
	g_free (connection->share);
	g_free (connection->user);
	g_free (connection->password);

	g_free (connection);
}


/* Only works on URIs that have both server and share specified */
static GnomeVFSResult
smb_server_connection_new_from_uri (GnomeVFSURI *uri,
				    char **path,
				    SmbConnection **connection)
{
	GnomeVFSResult res;
	char *server;
	char *share;
	char *share_and_filename;
	char *filename;
	const char *user;
	const char *password;
	char *tmp;

	*path = NULL;
	DEBUG_SMB (("smb_server_connection_new_from_uri() %s\n",
		    gnome_vfs_uri_to_string (uri, 0)));
	
	server = gnome_vfs_unescape_string (gnome_vfs_uri_get_host_name (uri), NULL);
	if (server == NULL) {
		g_warning ("NULL server");
		return GNOME_VFS_ERROR_INTERNAL;
	}
	
	share_and_filename =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_path (uri),
				     NULL);

	tmp = share_and_filename;
	/* Skip root '/' */
	if (*tmp == '/')
	  tmp++;

	/* The first directory entry is the share name */
	filename = strchr (tmp, '/');
	if (filename == NULL)
		filename = tmp + strlen (tmp);

	share = g_malloc (filename - tmp + 1);
	memcpy (share, tmp, filename - tmp);
	share[filename - tmp] = 0;

	if (*filename == 0)
		filename = "/";
	
	user = gnome_vfs_uri_get_user_name (uri);
	password = gnome_vfs_uri_get_password (uri);
	
	res = smb_server_connection_new (server,
					 share,
					 user,
					 password,
					 connection);
	g_free (share);
	g_free (server);

	if (res) {
		g_free (share_and_filename);
		return res;
	}

	*path = g_strdup (filename);
	g_free (share_and_filename);
	return GNOME_VFS_OK;
}

static void
smb_dir_content_destroy (SmbDirContent *content)
{
	g_free (content->name);
	g_free (content);
}

static void
smb_browse_content_destroy (SmbBrowseContent *content)
{
	g_free (content->name);
	g_free (content);
}

static void
smb_dir_handle_destroy (SmbDirHandle *handle)
{
	GList *list;
	SmbDirContent *dir_content;
	SmbBrowseContent *browse_content;
	if (!handle)
		return;
	
	list = handle->contents;
	while (list) {
		if (handle->is_files) {
			dir_content = (SmbDirContent *)list->data;
			smb_dir_content_destroy (dir_content);
		} else {
			browse_content = (SmbBrowseContent *)list->data;
			smb_browse_content_destroy (browse_content);
		}
		list = g_list_next (list);
	}
	g_free (handle->server);
	g_list_free (handle->contents);
}

static gboolean
get_a_master_browser (struct in_addr *ip)
{
	struct in_addr *bcast, *ip_list;
	int iface;
	int count;
	int fd;

	fd = open_socket_in (SOCK_DGRAM,
			     0, 3,
                             interpret_addr(lp_socket_address()),
			     TRUE);

	if (fd < 0)
		return FALSE;

	set_socket_options (fd, "SO_BROADCAST");

	for (iface = iface_count () -1; iface >= 0; iface--) {
		bcast = iface_n_bcast (iface);
		ip_list = name_query (fd,
				      "\01\02__MSBROWSE__\02", 1,
				      TRUE,
				      TRUE,
				      *bcast,
				      &count);
		if (ip_list) {
			close (fd);
			*ip = ip_list[0];
			free (ip_list);
			return TRUE;
		}
	}
	close (fd);
	return FALSE;
}

struct lookforserver {
	const char *server;
	gboolean found;
};

static void
look_for_server (const char *server,
		 uint32 m,
		 const char *comment)
{
	struct lookforserver *data;

	data = g_private_get (static_key);
	if (strcmp (server, data->server) == 0)
		data->found = TRUE;
}

static gboolean 
is_workgroup_server_link (const char *workgroup,
			  const char *server)
{
	GnomeVFSResult res;
	struct lookforserver data;
	struct in_addr ip;
	SmbConnection *connection;
	char *next_slash;
	const char *user, *password;
	gboolean result;
	/* Keep a cache of one to avoid network trafic */
	static char *cached_workgroup = NULL;
	static char *cached_server = NULL;
	static gboolean cached_result;
	

	DEBUG_SMB (("is_workgroup_server_link (//%s/%s)", workgroup, server));
	
	if (workgroup == NULL || server == NULL)
		return FALSE;

	if (server[0] == '/')
		server++;

	/* Make sure we only check for URIs on the form
	 * smb://foo/bar/ or smb://foo/bar
	 */
	next_slash = strchr (server, '/');
	if (next_slash && next_slash[1] != 0) {
		return FALSE;
	}

	if (cached_workgroup &&
	    strcmp (workgroup, cached_workgroup) == 0 &&
	    strcmp (server, cached_server) == 0) {
		return cached_result;
	}

	result = FALSE;
	if (find_master_ip ((char *)workgroup, &ip)) {
		res = smb_server_connection_new_from_ip (&ip,
							 "*SMBSERVER",
							 "IPC$",
							 NULL,
							 NULL,
							 &connection);
		if (!res) {
			data.server = server;
			data.found = FALSE;
			
			g_private_set (static_key, &data);
			cli_NetServerEnum (connection->cli,
					   connection->cli->server_domain,
					   SV_TYPE_ALL,
					   look_for_server);
			smb_connection_destroy (connection);
			g_private_set (static_key, NULL);
			result = data.found;
		}
	}

	/* Cache this value */
	g_free (cached_workgroup);
	g_free (cached_server);
	cached_workgroup = g_strdup (workgroup);
	cached_server = g_strdup (server);
	cached_result = result;
	
	return result;

}

char *
get_workgroup_data (char *name)
{
  return g_strdup_printf ("[Desktop Entry]\n"
			  "Encoding=Legacy-Mixed\n"
			  "Name=%s\n"
			  "Type=Link\n"
			  "URL=smb://%s/\n"
			  "Icon=gnome-computer.png\n",
			  name, name);
}

char *
get_computer_data (char *name)
{
  return g_strdup_printf ("[Desktop Entry]\n"
			  "Encoding=Legacy-Mixed\n"
			  "Name=%s\n"
			  "Type=Link\n"
			  "URL=smb://%s/\n"
			  "Icon=gnome-computer.png\n",
			  name, name);
}

static GnomeVFSResult
do_open (GnomeVFSMethod *method,
	 GnomeVFSMethodHandle **method_handle,
	 GnomeVFSURI *uri,
	 GnomeVFSOpenMode mode,
	 GnomeVFSContext *context)
{
	GnomeVFSResult res;
	SmbConnection *connection;
	SmbHandle *handle;
	int fnum;
	char *filename;
	char *dos_filename;
	int flags;
	char *server;
	char *path;

	LOCK_SAMBA();
	
	DEBUG_SMB (("do_open() %s mode %d\n",
		    gnome_vfs_uri_to_string (uri, 0), mode));
	
	if ((mode & GNOME_VFS_OPEN_READ) &&
	    (mode & GNOME_VFS_OPEN_WRITE))
		flags = O_RDWR;
	else if (mode & GNOME_VFS_OPEN_READ)
		flags = O_RDONLY;
	else if (mode & GNOME_VFS_OPEN_WRITE)
		flags = O_WRONLY;
	else
		return GNOME_VFS_ERROR_INVALID_OPEN_MODE;


	server =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_host_name (uri),
				     NULL);
	path  =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_path (uri),
				     NULL);
	if (server == NULL) {
		if (!path) {
			g_free (server);
			return GNOME_VFS_ERROR_INTERNAL;
		}
		handle = g_new (SmbHandle, 1);
		handle->is_data = TRUE;
		handle->offset = 0;
		handle->file_data = get_workgroup_data ((*path=='/')?path+1:path);
		handle->offset = 0;
	} else if (is_workgroup_server_link (server, path)) {
		handle = g_new (SmbHandle, 1);
		handle->is_data = TRUE;
		handle->offset = 0;
		handle->file_data = get_computer_data ((*path=='/')?path+1:path);
		handle->offset = 0;
	} else {
		res = smb_server_connection_new_from_uri (uri,
							  &filename,
							  &connection);
		if (res) {
			g_free (server);
			g_free (path);
			UNLOCK_SAMBA ();
			return res;
		}
		
		dos_filename = unix_filename_to_dos (filename);
		fnum = cli_open (connection->cli, dos_filename, flags, DENY_NONE);
		g_free (filename);
		g_free (dos_filename);
		if (fnum == -1) {
			g_free (server);
			g_free (path);
			res = gnome_vfs_result_from_errno_code (cli_error (connection->cli, NULL, NULL, NULL));
			smb_connection_destroy (connection);
			UNLOCK_SAMBA();
			return res;
		}
		
		handle = g_new (SmbHandle, 1);
		handle->is_data = FALSE;
		handle->connection = connection;
		handle->fnum = fnum;
		handle->offset = 0;
	}

	UNLOCK_SAMBA();

	g_free (server);
	g_free (path);
	
	*method_handle = (GnomeVFSMethodHandle *)handle;

	return GNOME_VFS_OK;
}


static GnomeVFSResult
do_close (GnomeVFSMethod *method,
	  GnomeVFSMethodHandle *method_handle,
	  GnomeVFSContext *context)

{
	SmbHandle *handle = (SmbHandle *)method_handle;
	size_t n;

	LOCK_SAMBA();
	DEBUG_SMB (("do_close() %p\n", method_handle));

	if (handle->is_data) {
		g_free (handle->file_data);
	} else {
		cli_close  (handle->connection->cli, handle->fnum);
		smb_connection_destroy (handle->connection);
	}
	
	g_free (handle);
	
	UNLOCK_SAMBA();

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_read (GnomeVFSMethod *method,
	 GnomeVFSMethodHandle *method_handle,
	 gpointer buffer,
	 GnomeVFSFileSize num_bytes,
	 GnomeVFSFileSize *bytes_read,
	 GnomeVFSContext *context)
{
	SmbHandle *handle = (SmbHandle *)method_handle;
	size_t n;
	int link_len;

	LOCK_SAMBA();
	DEBUG_SMB (("do_read() %p\n", method_handle));

	if (handle->is_data) {
		link_len = strlen (handle->file_data);
		if (handle->offset >= link_len)
			n = 0;
		else {
			n = MIN (num_bytes, link_len - handle->offset);
			memcpy (buffer, handle->file_data + handle->offset, n);
		}
	} else {
		n = cli_read (handle->connection->cli,
			      handle->fnum,
			      buffer,
			      handle->offset,
			      num_bytes);
	}

	UNLOCK_SAMBA();

	if (n < 0) {
		*bytes_read = 0;
		return gnome_vfs_result_from_errno_code (cli_error (handle->connection->cli,
								    NULL, NULL, NULL));
	}
	
	*bytes_read = n;
	
	if (n == 0)
		return GNOME_VFS_ERROR_EOF;
	
	handle->offset += n;
	
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_write (GnomeVFSMethod *method,
	  GnomeVFSMethodHandle *method_handle,
	  gconstpointer buffer,
	  GnomeVFSFileSize num_bytes,
	  GnomeVFSFileSize *bytes_written,
	  GnomeVFSContext *context)


{
	SmbHandle *handle = (SmbHandle *)method_handle;
	ssize_t n;
	int link_len;

	LOCK_SAMBA();
	DEBUG_SMB (("do_write() %p\n", method_handle));

	if (handle->is_data) {
	  return GNOME_VFS_ERROR_NOT_SUPPORTED;
	} else {
		n = cli_write (handle->connection->cli,
			       handle->fnum,
			       0,
			       (char *)buffer,
			       handle->offset,
			       num_bytes);
	}

	UNLOCK_SAMBA();

	if (n < 0) {
		*bytes_written = 0;
		return gnome_vfs_result_from_errno_code (cli_error (handle->connection->cli,
								    NULL, NULL, NULL));
	}
	
	*bytes_written = n;
	
	if (n == 0)
		return GNOME_VFS_ERROR_EOF;
	
	handle->offset += n;
	
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_create (GnomeVFSMethod *method,
	   GnomeVFSMethodHandle **method_handle,
	   GnomeVFSURI *uri,
	   GnomeVFSOpenMode mode,
	   gboolean exclusive,
	   guint perm,
	   GnomeVFSContext *context)

{
	GnomeVFSResult res;
	SmbConnection *connection;
	SmbHandle *handle;
	int fnum;
	char *filename;
	char *dos_filename;
	int flags;
	char *server;
	char *path;

	LOCK_SAMBA();
	
	DEBUG_SMB (("do_create() %s mode %d\n",
		    gnome_vfs_uri_to_string (uri, 0), mode));
	
	if ((mode & GNOME_VFS_OPEN_READ) &&
	    (mode & GNOME_VFS_OPEN_WRITE))
		flags = O_RDWR;
	else if (mode & GNOME_VFS_OPEN_READ)
		flags = O_RDONLY;
	else if (mode & GNOME_VFS_OPEN_WRITE)
		flags = O_WRONLY;
	else
		return GNOME_VFS_ERROR_INVALID_OPEN_MODE;


	flags |= O_CREAT;
	if (exclusive)
		flags |= O_EXCL;
	
	server =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_host_name (uri),
				     NULL);
	path =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_path (uri),
				     NULL);
	if ((server == NULL) ||
	    (is_workgroup_server_link (server, path))) {
		g_free (server);
		g_free (path);
		return GNOME_VFS_ERROR_ACCESS_DENIED;
	}
	    
	res = smb_server_connection_new_from_uri (uri,
						  &filename,
						  &connection);
	if (res) {
		g_free (server);
		g_free (path);
		UNLOCK_SAMBA ();
		return res;
	}
	
	dos_filename = unix_filename_to_dos (filename);
	DEBUG_SMB (("tryping to cli_open '%s', flags: %d\n", dos_filename, flags));
	fnum = cli_open (connection->cli, dos_filename, flags, DENY_NONE);
	g_free (filename);
	g_free (dos_filename);
	if (fnum == -1) {
		g_free (server);
		g_free (path);
		res = gnome_vfs_result_from_errno_code (cli_error (connection->cli, NULL, NULL, NULL));
		smb_connection_destroy (connection);
		UNLOCK_SAMBA();
		return res;
	}
	
	handle = g_new (SmbHandle, 1);
	handle->is_data = FALSE;
	handle->connection = connection;
	handle->fnum = fnum;
	handle->offset = 0;

	UNLOCK_SAMBA();

	*method_handle = (GnomeVFSMethodHandle *)handle;

	return GNOME_VFS_OK;
}

static void
get_info_helper (file_info *info, const char *dir)
{
	file_info *smb_info;

	smb_info = g_private_get (get_info_key);
	*smb_info = *info;
}


static GnomeVFSResult
share_file_info (GnomeVFSMethod *method,
		 GnomeVFSURI *uri,
		 GnomeVFSFileInfo *vfs_file_info,
		 GnomeVFSFileInfoOptions options,
		 GnomeVFSContext *context)
{
	GnomeVFSResult res;
	SmbConnection *connection;
	char *filename;
	char *dos_filename;
	file_info *smb_info;

	LOCK_SAMBA();
	DEBUG_SMB (("share_get_file_info() %s options: %d\n",
		    gnome_vfs_uri_to_string (uri, 0),
		    options));
	
	res = smb_server_connection_new_from_uri (uri,
						  &filename,
						  &connection);
	if (res) {
		UNLOCK_SAMBA();
		return res;
	}
	
	if (strcmp (filename, "/")==0) {
		vfs_file_info->name = g_strdup ("/");
		vfs_file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
		vfs_file_info->mime_type = g_strdup("x-directory/normal");
		vfs_file_info->flags = 0;
		vfs_file_info->permissions =
			GNOME_VFS_PERM_USER_ALL |
			GNOME_VFS_PERM_GROUP_ALL |
			GNOME_VFS_PERM_OTHER_ALL;
		vfs_file_info->valid_fields =
		  GNOME_VFS_FILE_INFO_FIELDS_TYPE |
		  GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS |
		  GNOME_VFS_FILE_INFO_FIELDS_FLAGS |
		  GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;

		smb_connection_destroy (connection);
		g_free (filename);
		UNLOCK_SAMBA();
		return GNOME_VFS_OK;
	}
	

	smb_info = g_private_get (get_info_key);
	if (!smb_info) {
		smb_info = g_new (file_info, 1);
		g_private_set (get_info_key, smb_info);
	}


	dos_filename = unix_filename_to_dos (filename);
	
	if (cli_list (connection->cli, dos_filename,
		      aDIR | aSYSTEM | aHIDDEN,
		      get_info_helper) == -1) {
		smb_connection_destroy (connection);
		g_free (dos_filename);
		g_free (filename);
		UNLOCK_SAMBA();
		return GNOME_VFS_ERROR_NOT_FOUND;
	}
	smb_connection_destroy (connection);
	UNLOCK_SAMBA();
	g_free (dos_filename);

	vfs_file_info->name = g_strdup (g_basename (filename));
	g_free (filename);

	if (smb_info->mode & aDIR) {
		vfs_file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
		vfs_file_info->mime_type = g_strdup("x-directory/normal");
	} else {
		vfs_file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
		vfs_file_info->mime_type =
			g_strdup (gnome_vfs_mime_type_from_name_or_default (vfs_file_info->name,
									    GNOME_VFS_MIME_TYPE_UNKNOWN));
	}

	if (smb_info->mode & aRONLY)
		vfs_file_info->permissions =
			GNOME_VFS_PERM_USER_READ | GNOME_VFS_PERM_USER_EXEC |
			GNOME_VFS_PERM_GROUP_READ | GNOME_VFS_PERM_GROUP_EXEC |
			GNOME_VFS_PERM_OTHER_READ | GNOME_VFS_PERM_OTHER_EXEC;
	else
		vfs_file_info->permissions =
			GNOME_VFS_PERM_USER_ALL |
			GNOME_VFS_PERM_GROUP_ALL |
			GNOME_VFS_PERM_OTHER_ALL;
	
	vfs_file_info->size = smb_info->size;
	vfs_file_info->uid = smb_info->uid;
	vfs_file_info->gid = smb_info->gid;

	vfs_file_info->mtime = smb_info->mtime;
	vfs_file_info->atime = smb_info->atime;
	vfs_file_info->ctime = smb_info->ctime;

	vfs_file_info->flags = 0;
	
	vfs_file_info->valid_fields =
		GNOME_VFS_FILE_INFO_FIELDS_TYPE |
		GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS |
		GNOME_VFS_FILE_INFO_FIELDS_FLAGS |
		GNOME_VFS_FILE_INFO_FIELDS_SIZE |
		GNOME_VFS_FILE_INFO_FIELDS_ATIME |
		GNOME_VFS_FILE_INFO_FIELDS_MTIME |
		GNOME_VFS_FILE_INFO_FIELDS_CTIME |
		GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;
	
	return GNOME_VFS_OK;
}


static GnomeVFSResult
root_info (GnomeVFSMethod *method,
	   GnomeVFSURI *uri,
	   GnomeVFSFileInfo *vfs_file_info,
	   GnomeVFSFileInfoOptions options,
	   GnomeVFSContext *context)
{
	char *path;
	char *data;

	path =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_path (uri),
				     NULL);
	vfs_file_info->name = (path)?path:g_strdup ("/");
		
	g_assert (vfs_file_info->name != NULL);
	
	vfs_file_info->valid_fields = 0;

	if (!path || (path[0] == '/' && path [1] == 0)) {
		vfs_file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
		vfs_file_info->flags = 0;
		vfs_file_info->mime_type = g_strdup("x-directory/normal");
	} else {
		vfs_file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
		vfs_file_info->mime_type = g_strdup("application/x-gnome-app-info");
		
		data = get_workgroup_data (path);
		vfs_file_info->size = strlen (data);
		g_free (data);
		vfs_file_info->valid_fields |=
		  GNOME_VFS_FILE_INFO_FIELDS_SIZE;
	}

	vfs_file_info->flags = GNOME_VFS_FILE_FLAGS_NONE;
	
	vfs_file_info->permissions =
		GNOME_VFS_PERM_USER_READ | GNOME_VFS_PERM_USER_WRITE |
		GNOME_VFS_PERM_GROUP_READ | GNOME_VFS_PERM_GROUP_WRITE |
		GNOME_VFS_PERM_OTHER_READ | GNOME_VFS_PERM_OTHER_WRITE ;


	vfs_file_info->valid_fields |=
		GNOME_VFS_FILE_INFO_FIELDS_TYPE |
		GNOME_VFS_FILE_INFO_FIELDS_FLAGS |
		GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;
	
	return GNOME_VFS_OK;
}

static GnomeVFSResult
server_workgroup_info (GnomeVFSMethod *method,
		       GnomeVFSURI *uri,
		       GnomeVFSFileInfo *vfs_file_info,
		       GnomeVFSFileInfoOptions options,
		       GnomeVFSContext *context)
{
	vfs_file_info->name = g_strdup ("/");
	
	vfs_file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
	vfs_file_info->mime_type = g_strdup("x-directory/normal");

	vfs_file_info->flags = 0;
	
	vfs_file_info->valid_fields |=
		GNOME_VFS_FILE_INFO_FIELDS_TYPE |
		GNOME_VFS_FILE_INFO_FIELDS_FLAGS |
		GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;
	
	return GNOME_VFS_OK;
}

static GnomeVFSResult
server_link_info (GnomeVFSMethod *method,
		  GnomeVFSURI *uri,
		  GnomeVFSFileInfo *vfs_file_info,
		  GnomeVFSFileInfoOptions options,
		  GnomeVFSContext *context)
{
	char *share;
	char *data;

	share =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_path (uri),
				     NULL);
	vfs_file_info->name = g_strdup (g_basename (share));
	g_free (share);
	
	vfs_file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
	vfs_file_info->mime_type = g_strdup("application/x-gnome-app-info");

	data = get_computer_data (share);
	vfs_file_info->size = strlen (data);
	g_free (data);

	vfs_file_info->flags = 0;
	
	vfs_file_info->valid_fields |=
		GNOME_VFS_FILE_INFO_FIELDS_TYPE |
		GNOME_VFS_FILE_INFO_FIELDS_FLAGS |
		GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE |
		GNOME_VFS_FILE_INFO_FIELDS_SIZE;
	
	return GNOME_VFS_OK;
}


static GnomeVFSResult
do_get_file_info (GnomeVFSMethod *method,
		  GnomeVFSURI *uri,
		  GnomeVFSFileInfo *vfs_file_info,
		  GnomeVFSFileInfoOptions options,
		  GnomeVFSContext *context)

{
	char *server;
	char *share;

	server =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_host_name (uri),
				     NULL);

	/* Is this of the form smb:/// or smb:///workgroup
	 */
	if (server == NULL)
		return root_info (method, uri,
				  vfs_file_info, options,
				  context);

	share =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_path (uri),
				     NULL);
	
	/* Is this of the form smb://foo/ , where foo can be
	 * a server and/or a workgroup */ 
	if (share == NULL || share[0] == 0 ||
	    (share[0] == '/' && share[1] == 0)) {
		g_free (server);
		g_free (share);
		return server_workgroup_info (method, uri,
					      vfs_file_info, options,
					      context);
	}
	
	/* Is this a //workgroup/server/ ?
	 */
	if (is_workgroup_server_link (server, share)) {
		g_free (server);
		g_free (share);
		return server_link_info (method, uri,
					 vfs_file_info, options,
					 context);
	}
	
	g_free (server);
	g_free (share);
	
	/* None of the above, then this is a normal shared dir/file */
	return share_file_info (method, uri,
				vfs_file_info, options,
				context);
}

static gboolean
do_is_local (GnomeVFSMethod *method,
	     const GnomeVFSURI *uri)


{
	return FALSE;
}

static void
open_dir_helper (file_info *info, const char *dir)
{
	SmbDirHandle *handle;
	SmbDirContent *content;
	GnomeVFSDirectoryFilterNeeds needs;
	GnomeVFSFileInfo *file_info;
	gboolean passed;

	handle = g_private_get (dir_key);

	if (handle->filter) {
		needs = gnome_vfs_directory_filter_get_needs (handle->filter);
		file_info = gnome_vfs_file_info_new ();
		
		if (needs & GNOME_VFS_DIRECTORY_FILTER_NEEDS_NAME) {
			file_info->name = g_strdup (info->name);
		}
			
		if (needs & GNOME_VFS_DIRECTORY_FILTER_NEEDS_TYPE) {
			if (content->mode & aDIR) 
				file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
			else
				file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
			file_info->valid_fields |=
				GNOME_VFS_FILE_INFO_FIELDS_TYPE;

		}
		
		if (needs & GNOME_VFS_DIRECTORY_FILTER_NEEDS_MIMETYPE) {
			if (content->mode & aDIR) 
				file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
			else
				file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
			file_info->valid_fields |=
				GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;
		}
		
		if (needs & GNOME_VFS_DIRECTORY_FILTER_NEEDS_STAT) {
			if (info->mode & aRONLY)
				file_info->permissions =
					GNOME_VFS_PERM_USER_READ | GNOME_VFS_PERM_USER_EXEC |
					GNOME_VFS_PERM_GROUP_READ | GNOME_VFS_PERM_GROUP_EXEC |
					GNOME_VFS_PERM_OTHER_READ | GNOME_VFS_PERM_OTHER_EXEC;
			else
				file_info->permissions =
					GNOME_VFS_PERM_USER_ALL |
					GNOME_VFS_PERM_GROUP_ALL |
					GNOME_VFS_PERM_OTHER_ALL;
	
			file_info->size = info->size;
			file_info->uid = info->uid;
			info->gid = info->gid;
			
			file_info->mtime = info->mtime;
			file_info->atime = info->atime;
			file_info->ctime = info->ctime;
			
			file_info->flags = 0;
			
			file_info->valid_fields |=
				GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS |
				GNOME_VFS_FILE_INFO_FIELDS_FLAGS |
				GNOME_VFS_FILE_INFO_FIELDS_SIZE |
				GNOME_VFS_FILE_INFO_FIELDS_ATIME |
				GNOME_VFS_FILE_INFO_FIELDS_MTIME |
				GNOME_VFS_FILE_INFO_FIELDS_CTIME;
		}

		passed =
			gnome_vfs_directory_filter_apply (handle->filter,
							  file_info);
		gnome_vfs_file_info_unref (file_info);

		if (!passed)
			return;
	}
	
	content = g_new (SmbDirContent, 1);
	content->name = g_strdup (info->name);
	content->mtime = info->mtime;
	content->atime = info->atime;
	content->ctime = info->ctime;
	content->mode = info->mode;
	content->size = info->size;
	content->uid = info->uid;
	content->gid = info->gid;
	handle->contents = g_list_prepend (handle->contents, content);
}

static char *
dirname_to_mask (const char *dirname)
{
	char *mask;
	int dirname_len;
	int i;

	dirname_len = strlen (dirname);
	mask = g_malloc (dirname_len + 3);
	strcpy (mask, dirname);
	for (i = 0; i < dirname_len; i++)
		if (mask[i]=='/')
			mask[i] = '\\';

	if (mask[dirname_len-1] != '\\') {
		dirname_len++;
		mask[dirname_len-1] = '\\';
	}
	mask[dirname_len] = '*';
	mask[dirname_len+1] = 0;
	
	return mask;
}

static GnomeVFSResult
open_share_dir (GnomeVFSMethod *method,
		GnomeVFSMethodHandle **method_handle,
		GnomeVFSURI *uri,
		GnomeVFSFileInfoOptions options,
		const GnomeVFSDirectoryFilter *filter,
		GnomeVFSContext *context)
{
	GnomeVFSResult res;
	SmbConnection *connection;
	SmbDirHandle *handle;
	char *filename;
	char *mask;
	uint16 attr;
	size_t size;

	DEBUG_SMB (("open_share_directory() %s options: %d filter: %p\n",
		    gnome_vfs_uri_to_string (uri, 0),
		    options, filter));
	
	res = smb_server_connection_new_from_uri (uri,
						  &filename,
						  &connection);
	if (res) {
		g_warning ("open_share_dir: connectiong res: %d, filename: '%s'\n", res, filename);
		UNLOCK_SAMBA();
		return res;
	}

	handle = g_new (SmbDirHandle, 1);
	handle->is_files = TRUE;
	handle->contents = NULL;
	handle->server = NULL;
	handle->filter = filter;
	
	g_private_set (dir_key, handle);

	mask = dirname_to_mask (filename);
	if (cli_list (connection->cli, mask,
		      aDIR | aSYSTEM | aHIDDEN,
		      open_dir_helper) == -1) {
		res = gnome_vfs_result_from_errno_code (cli_error (connection->cli, NULL, NULL, NULL));
		smb_connection_destroy (connection);
		g_free (mask);
		g_free (handle);
		g_free (filename);
		g_private_set (dir_key, NULL);
		g_warning ("open_share_dir: cli_list returned -1, returning %d %s", res, gnome_vfs_result_to_string (res));
		UNLOCK_SAMBA();
		return res;
	}
	g_free (mask);
	g_free (filename);
	g_private_set (dir_key, NULL);
	smb_connection_destroy (connection);

	*method_handle = (GnomeVFSMethodHandle *)handle;
	return GNOME_VFS_OK;
}


static void
root_browse_helper (const char *workgroup,
		    uint32 m,
		    const char *master)
{
	SmbDirHandle *handle;
	SmbBrowseContent *content;

	handle = g_private_get (dir_key);
	content = g_new (SmbBrowseContent, 1);
	content->type = SMB_CONTENT_WORKGROUP;
	content->name = g_strdup (workgroup);
	handle->contents =
		g_list_prepend (handle->contents, content);
}

static GnomeVFSResult
open_root_dir (GnomeVFSMethod *method,
		GnomeVFSMethodHandle **method_handle,
		GnomeVFSURI *uri,
		GnomeVFSFileInfoOptions options,
		const GnomeVFSDirectoryFilter *filter,
		GnomeVFSContext *context)
{
	GnomeVFSResult res;
	struct in_addr ip;
	SmbConnection *connection;
	SmbDirHandle *handle;
	
	if (!get_a_master_browser (&ip)) {
		g_warning ("Couldn't find a master browser");
		return GNOME_VFS_ERROR_HOST_NOT_FOUND;
	}
	
	res = smb_server_connection_new_from_ip (&ip,
						 "*SMBSERVER",
						 "IPC$",
						 NULL,
						 NULL,
						 &connection);
	if (res) {
		g_warning ("open_root_dir: Failed to connect");
		return res;
	}

	handle = g_new (SmbDirHandle, 1);
	handle->is_files = FALSE;
	handle->contents = NULL;
	handle->server = NULL;
	g_private_set (dir_key, handle);

	cli_NetServerEnum (connection->cli,
			   connection->cli->server_domain,
			   SV_TYPE_DOMAIN_ENUM,
			   root_browse_helper);
	
	g_private_set (dir_key, NULL);

	smb_connection_destroy (connection);
	*method_handle = (GnomeVFSMethodHandle *)handle;
	return GNOME_VFS_OK;
}

static void
wg_browse_helper (const char *server,
		  uint32 m,
		  const char *comment)
{
	SmbDirHandle *handle;
	SmbBrowseContent *content;

	handle = g_private_get (dir_key);
	content = g_new (SmbBrowseContent, 1);
	content->type = SMB_CONTENT_SERVER;
	content->name = g_strdup (server);
	handle->contents =
		g_list_prepend (handle->contents, content);
}

static void
server_browse_helper (const char *share,
		      uint32 m,
		      const char *comment)
{
	SmbDirHandle *handle;
	SmbBrowseContent *content;

	if (m != STYPE_DISKTREE)
		return;
	
	handle = g_private_get (dir_key);
	content = g_new (SmbBrowseContent, 1);
	content->type = SMB_CONTENT_SHARE;
	content->name = g_strdup (share);
	
	handle->contents =
		g_list_prepend (handle->contents, content);

}

static GnomeVFSResult
open_server_workgroup_dir (GnomeVFSMethod *method,
			   GnomeVFSMethodHandle **method_handle,
			   GnomeVFSURI *uri,
			   GnomeVFSFileInfoOptions options,
			   const GnomeVFSDirectoryFilter *filter,
			   GnomeVFSContext *context)
{
	GnomeVFSResult res;
	struct in_addr ip;
	SmbConnection *connection;
	SmbDirHandle *handle;
	char *server;
	const char *user;
	const char *password;
	
	DEBUG_SMB (("open_sw_dir ()\n"));
	
	server =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_host_name (uri),
				     NULL);

	handle = g_new (SmbDirHandle, 1);
	handle->is_files = FALSE;
	handle->contents = NULL;
	handle->server = server;
	
	g_private_set (dir_key, handle);
	
	user = gnome_vfs_uri_get_user_name (uri);
	password = gnome_vfs_uri_get_password (uri);
	
	if (find_master_ip ((char *)server, &ip)) {
		res = smb_server_connection_new_from_ip (&ip,
							 "*SMBSERVER",
							 "IPC$",
							 user,
							 password,
							 &connection);
		if (!res) {
			DEBUG_SMB (("o_sw_dir: Couldn't open for server listing"));
			cli_NetServerEnum (connection->cli,
					   connection->cli->server_domain,
					   SV_TYPE_ALL,
					   wg_browse_helper);
			smb_connection_destroy (connection);
		}
	}
	
		
	res = smb_server_connection_new (server,
					 "IPC$",
					 user,
					 password,
					 &connection);
	if (!res) {
		DEBUG_SMB (("o_sw_dir: Couldn't %s open for share listing", server));
		cli_RNetShareEnum (connection->cli,
				   server_browse_helper);
		smb_connection_destroy (connection);
	}


	g_private_set (dir_key, NULL);

	*method_handle = (GnomeVFSMethodHandle *)handle;
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_open_directory (GnomeVFSMethod *method,
		   GnomeVFSMethodHandle **method_handle,
		   GnomeVFSURI *uri,
		   GnomeVFSFileInfoOptions options,
		   const GnomeVFSDirectoryFilter *filter,
		   GnomeVFSContext *context)

{
	char *server;
	char *share;
	GnomeVFSResult res;

	LOCK_SAMBA();
	
	server =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_host_name (uri),
				     NULL);
	share =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_path (uri),
				     NULL);

	DEBUG_SMB (("do_open_directory () server: '%s'\n", server));
	
	if (server == NULL) {
		res = open_root_dir (method, method_handle,
				     uri, options, filter,
				     context);
	} else if (share == NULL || share[0] == 0 ||
		   (share[0] == '/' && share[1] == 0)) {
		res = open_server_workgroup_dir (method, method_handle,
						 uri, options, filter,
						 context);
	} else if (is_workgroup_server_link (server, share)) {
		res =  GNOME_VFS_ERROR_NOT_A_DIRECTORY;
	} else {
		res = open_share_dir (method, method_handle,
				      uri, options, filter,
				      context);
	}
	
	g_free (server);
	g_free (share);
			
	UNLOCK_SAMBA();
	return res;
}

static GnomeVFSResult
do_close_directory (GnomeVFSMethod *method,
		    GnomeVFSMethodHandle *method_handle,
		    GnomeVFSContext *context)
{
	SmbDirHandle *handle = (SmbDirHandle *)method_handle;

	DEBUG_SMB (("do_close_directory() %p\n", handle));
	
	smb_dir_handle_destroy (handle);

	return GNOME_VFS_OK;
}


static GnomeVFSResult
share_read_dir (GnomeVFSMethod *method,
		GnomeVFSMethodHandle *method_handle,
		GnomeVFSFileInfo *file_info,
		GnomeVFSContext *context)
{
	SmbDirHandle *handle = (SmbDirHandle *)method_handle;
	SmbDirContent *content;
	GList *node;

	if (handle->contents) {
		node = handle->contents;
		content = (SmbDirContent *)node->data;
		handle->contents =
			g_list_remove_link (handle->contents, node);

		file_info->name = content->name;
		g_assert (file_info->name != NULL);

		if (content->mode & aDIR) {
			file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
			file_info->mime_type = g_strdup("x-directory/normal");
		} else {
			file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
			file_info->mime_type =
				g_strdup (gnome_vfs_mime_type_from_name_or_default (file_info->name,
										    GNOME_VFS_MIME_TYPE_UNKNOWN));
		}

		if (content->mode & aRONLY)
			file_info->permissions =
				GNOME_VFS_PERM_USER_READ | GNOME_VFS_PERM_USER_EXEC |
				GNOME_VFS_PERM_GROUP_READ | GNOME_VFS_PERM_GROUP_EXEC |
				GNOME_VFS_PERM_OTHER_READ | GNOME_VFS_PERM_OTHER_EXEC;
		else
			file_info->permissions =
				GNOME_VFS_PERM_USER_ALL |
				GNOME_VFS_PERM_GROUP_ALL |
				GNOME_VFS_PERM_OTHER_ALL;

		file_info->uid = content->uid;
		file_info->gid = content->gid;
		file_info->mtime = content->mtime;
		file_info->ctime = content->ctime;
		file_info->atime = content->atime;
		file_info->size = content->size;
		
		file_info->flags = GNOME_VFS_FILE_FLAGS_NONE;


		file_info->valid_fields |=
			GNOME_VFS_FILE_INFO_FIELDS_TYPE |
			GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS |
			GNOME_VFS_FILE_INFO_FIELDS_FLAGS |
			GNOME_VFS_FILE_INFO_FIELDS_SIZE |
			GNOME_VFS_FILE_INFO_FIELDS_ATIME |
			GNOME_VFS_FILE_INFO_FIELDS_MTIME |
			GNOME_VFS_FILE_INFO_FIELDS_CTIME |
			GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;
		
		g_free (content);
		return GNOME_VFS_OK;
	} else {
		return GNOME_VFS_ERROR_EOF;
	}
}

static GnomeVFSResult
browse_read_dir (GnomeVFSMethod *method,
		GnomeVFSMethodHandle *method_handle,
		GnomeVFSFileInfo *file_info,
		GnomeVFSContext *context)
{
	SmbDirHandle *handle = (SmbDirHandle *)method_handle;
	SmbBrowseContent *content;
	GList *node;

	if (handle->contents) {
		node = handle->contents;
		content = (SmbBrowseContent *)node->data;
		handle->contents =
			g_list_remove_link (handle->contents, node);

		file_info->name = g_strdup (content->name);
		g_assert (file_info->name != NULL);

		file_info->flags = GNOME_VFS_FILE_FLAGS_NONE;
		
		if ((content->type == SMB_CONTENT_WORKGROUP) ||
		    (content->type == SMB_CONTENT_SERVER)) {
			/* Workgroups and servers are desktop files */
			file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
			file_info->mime_type = g_strdup("application/x-gnome-app-info");
		} else {
			/* It's a share */
			file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
			file_info->mime_type = g_strdup("x-directory/normal");
		} 

		file_info->permissions =
			GNOME_VFS_PERM_USER_READ | GNOME_VFS_PERM_USER_WRITE |
			GNOME_VFS_PERM_GROUP_READ | GNOME_VFS_PERM_GROUP_WRITE |
			GNOME_VFS_PERM_OTHER_READ | GNOME_VFS_PERM_OTHER_WRITE;

		file_info->valid_fields |=
			GNOME_VFS_FILE_INFO_FIELDS_TYPE |
			GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS |
			GNOME_VFS_FILE_INFO_FIELDS_FLAGS |
			GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;

		smb_browse_content_destroy (content);
		return GNOME_VFS_OK;
	} else {
		return GNOME_VFS_ERROR_EOF;
	}

}

static GnomeVFSResult
do_read_directory (GnomeVFSMethod *method,
		   GnomeVFSMethodHandle *method_handle,
		   GnomeVFSFileInfo *file_info,
		   GnomeVFSContext *context)
{
	SmbDirHandle *handle = (SmbDirHandle *)method_handle;
	DEBUG_SMB (("do_read_directory()\n"));
	if (handle->is_files) {
		return share_read_dir (method,
				       method_handle,
				       file_info,
				       context);
	} else {
		return browse_read_dir (method,
				       method_handle,
				       file_info,
				       context);
	}
}

static GnomeVFSResult
do_seek (GnomeVFSMethod *method,
	 GnomeVFSMethodHandle *method_handle,
	 GnomeVFSSeekPosition whence,
	 GnomeVFSFileOffset offset,
	 GnomeVFSContext *context)

{
	SmbHandle *handle = (SmbHandle *)method_handle;

	DEBUG_SMB (("do_seek() %d from %d\n", offset, whence));
	
	switch (whence) {
	case GNOME_VFS_SEEK_START:
		handle->offset = offset;
		break;
	case GNOME_VFS_SEEK_CURRENT:
		handle->offset += offset;
		break;
	case GNOME_VFS_SEEK_END:
		return GNOME_VFS_ERROR_NOT_SUPPORTED;
		break;
	}

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_tell (GnomeVFSMethod *method,
	 GnomeVFSMethodHandle *method_handle,
	 GnomeVFSFileOffset *offset_return)


{
	SmbHandle *handle = (SmbHandle *)method_handle;

	*offset_return = handle->offset;

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_unlink (GnomeVFSMethod *method,
	   GnomeVFSURI *uri,
	   GnomeVFSContext *context)
{
  	GnomeVFSResult res;
	SmbConnection *connection;
	char *filename;
	char *dos_filename;
	file_info *smb_info;
	BOOL ret;

	LOCK_SAMBA();
	DEBUG_SMB (("do_unlink() %s\n",
		    gnome_vfs_uri_to_string (uri, 0)));
	
	
	res = smb_server_connection_new_from_uri (uri,
						  &filename,
						  &connection);
	if (res) {
		UNLOCK_SAMBA();
		return res;
	}

	dos_filename = unix_filename_to_dos (filename);
	ret = cli_unlink (connection->cli, dos_filename);
	g_free (dos_filename);
	g_free (filename);
	if (!ret) {
		res = gnome_vfs_result_from_errno_code (cli_error (connection->cli, NULL, NULL, NULL));
		smb_connection_destroy (connection);
		UNLOCK_SAMBA();
		return res;
	}
		
	smb_connection_destroy (connection);
	UNLOCK_SAMBA();
	
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_check_same_fs (GnomeVFSMethod *method,
		  GnomeVFSURI *a,
		  GnomeVFSURI *b,
		  gboolean *same_fs_return,
		  GnomeVFSContext *context)
{
	char *server1;
	char *server2;
	char *path1;
	char *path2;
	char *p1, *p2;

	DEBUG_SMB (("do_check_same_fs()\n"));
	
	server1 =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_host_name (a),
				     NULL);
	server2 =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_host_name (b),
				     NULL);
	path1 =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_path (a), NULL);
	path2 =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_path (b), NULL);

	if (!server1 || !server2 || !path1 || !path2 ||
	    (strcmp (server1, server2) != 0)) {
		g_free (server1);
		g_free (server2);
		g_free (path1);
		g_free (path2);
		*same_fs_return = FALSE;
		return GNOME_VFS_OK;
	}

	p1 = path1;
	p2 = path2;
	if (*p1 == '/')
		p1++;
	if (*p2 == '/')
		p2++;

	/* Make sure both URIs are on the same share: */
	while (*p1 && *p2 && *p1 == *p2 && *p1 != '/') {
		p1++;
		p2++;
	}
	if (*p1 == 0 || *p2 == 0 || *p1 != *p2) {
		*same_fs_return = FALSE;
	} else {
		*same_fs_return = TRUE;
	}
	
	g_free (server1);
	g_free (server2);
	g_free (path1);
	g_free (path2);
	
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_move (GnomeVFSMethod *method,
	 GnomeVFSURI *old_uri,
	 GnomeVFSURI *new_uri,
	 gboolean force_replace,
	 GnomeVFSContext *context)
{
  	GnomeVFSResult res;
	SmbConnection *connection;
	char *filename1;
	char *filename2;
	char *dos_filename1;
	char *dos_filename2;
	file_info *smb_info;
	char *server1;
	char *server2;
	char *path1;
	char *path2;
	BOOL ret;

	LOCK_SAMBA();
	DEBUG_SMB (("do_move() %s %s\n",
		    gnome_vfs_uri_to_string (old_uri, 0),
		    gnome_vfs_uri_to_string (new_uri, 0)));


	server1 =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_host_name (old_uri),
				     NULL);
	server2 =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_host_name (new_uri),
				     NULL);
	path1 =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_path (old_uri),
				     NULL);
	path2 =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_path (new_uri),
				     NULL);

	if (!server1 || !server2 || !path1 || !path2 ||
	    (strcmp (server1, server2) != 0)) {
		g_free (server1);
		g_free (server2);
		g_free (path1);
		g_free (path2);
		return GNOME_VFS_ERROR_NOT_PERMITTED;
	}

	if (*path1 == '/')
		path1++;
	if (*path2 == '/')
		path2++;

	/* Make sure both URIs are on the same share: */
	while (*path1 && *path2 && *path1 == *path2 && *path1 != '/') {
		path1++;
		path2++;
	}
	if (*path1 == 0 || *path2 == 0 || *path1 != *path2) {
		g_free (server1);
		g_free (server2);
		g_free (path1);
		g_free (path2);
		return GNOME_VFS_ERROR_NOT_PERMITTED;
	}

	if (*path2 == '/')
		path2++;

	filename2 = path2;
		
	res = smb_server_connection_new_from_uri (old_uri,
						  &filename1,
						  &connection);
	if (res) {
		g_free (server1);
		g_free (server2);
		g_free (path1);
		g_free (path2);
		UNLOCK_SAMBA();
		return res;
	}

	dos_filename1 = unix_filename_to_dos (filename1);
	dos_filename2 = unix_filename_to_dos (filename2);
	ret = cli_rename (connection->cli, dos_filename1, dos_filename2);
	g_free (dos_filename1);
	g_free (dos_filename2);
	g_free (filename1);
	g_free (server1);
	g_free (server2);
	g_free (path1);
	g_free (path2);
	if (!ret) {
		res = gnome_vfs_result_from_errno_code (cli_error (connection->cli, NULL, NULL, NULL));
		smb_connection_destroy (connection);
		UNLOCK_SAMBA();
		return res;
	}
		
	smb_connection_destroy (connection);
	UNLOCK_SAMBA();
	
	return GNOME_VFS_OK;
}


static GnomeVFSResult
do_truncate_handle (GnomeVFSMethod *method,
		    GnomeVFSMethodHandle *method_handle,
		    GnomeVFSFileSize where,
		    GnomeVFSContext *context)

{
	g_warning ("do_truncate_handle\n");
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}


static GnomeVFSResult
do_get_file_info_from_handle (GnomeVFSMethod *method,
			      GnomeVFSMethodHandle *method_handle,
			      GnomeVFSFileInfo *file_info,
			      GnomeVFSFileInfoOptions options,
			      GnomeVFSContext *context)


{
	g_warning ("do_get_file_info_from_handle\n");
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}

static GnomeVFSResult
do_make_directory (GnomeVFSMethod *method,
		   GnomeVFSURI *uri,
		   guint perm,
		   GnomeVFSContext *context)
{
	GnomeVFSResult res;
	SmbConnection *connection;
	BOOL ret;
	char *filename;
	char *dos_filename;
	char *server;
	char *path;

	LOCK_SAMBA();
	
	DEBUG_SMB (("do_make_dir() %s\n",
		    gnome_vfs_uri_to_string (uri, 0)));
	
	server =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_host_name (uri),
				     NULL);
	path =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_path (uri),
				     NULL);
	if ((server == NULL) ||
	    (is_workgroup_server_link (server, path))) {
		g_free (server);
		g_free (path);
		return GNOME_VFS_ERROR_ACCESS_DENIED;
	}
	    
	res = smb_server_connection_new_from_uri (uri,
						  &filename,
						  &connection);
	if (res) {
		g_free (server);
		g_free (path);
		UNLOCK_SAMBA ();
		return res;
	}
	
	dos_filename = unix_filename_to_dos (filename);
	ret = cli_mkdir (connection->cli, dos_filename);
	g_free (filename);
	g_free (dos_filename);
	if (!ret) {
		g_free (server);
		g_free (path);
		res = gnome_vfs_result_from_errno_code (cli_error (connection->cli, NULL, NULL, NULL));
		smb_connection_destroy (connection);
		UNLOCK_SAMBA();
		return res;
	}
	
	UNLOCK_SAMBA();

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_remove_directory (GnomeVFSMethod *method,
		     GnomeVFSURI *uri,
		     GnomeVFSContext *context)
{
	GnomeVFSResult res;
	SmbConnection *connection;
	BOOL ret;
	char *filename;
	char *dos_filename;
	char *server;
	char *path;

	LOCK_SAMBA();
	
	DEBUG_SMB (("do_make_dir() %s\n",
		    gnome_vfs_uri_to_string (uri, 0)));
	
	server =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_host_name (uri),
				     NULL);
	path =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_path (uri),
				     NULL);
	if ((server == NULL) ||
	    (is_workgroup_server_link (server, path))) {
		g_free (server);
		g_free (path);
		return GNOME_VFS_ERROR_ACCESS_DENIED;
	}
	    
	res = smb_server_connection_new_from_uri (uri,
						  &filename,
						  &connection);
	if (res) {
		g_free (server);
		g_free (path);
		UNLOCK_SAMBA ();
		return res;
	}
	
	dos_filename = unix_filename_to_dos (filename);
	ret = cli_rmdir (connection->cli, dos_filename);
	g_free (filename);
	g_free (dos_filename);
	if (!ret) {
		g_free (server);
		g_free (path);
		res = gnome_vfs_result_from_errno_code (cli_error (connection->cli, NULL, NULL, NULL));
		smb_connection_destroy (connection);
		UNLOCK_SAMBA();
		return res;
	}
	
	UNLOCK_SAMBA();

	return GNOME_VFS_OK;
}


static GnomeVFSResult
do_set_file_info (GnomeVFSMethod *method,
		  GnomeVFSURI *uri,
		  const GnomeVFSFileInfo *info,
		  GnomeVFSSetFileInfoMask mask,
		  GnomeVFSContext *context)
{
  	GnomeVFSResult res;
	SmbConnection *connection;
	char *filename1;
	char *filename2;
	char *dos_filename1;
	char *dos_filename2;
	char *end;
	BOOL ret;

	DEBUG_SMB (("do_set_file_info mask: %d\n", mask));
	
	if (mask &
	    (GNOME_VFS_SET_FILE_INFO_PERMISSIONS |
	     GNOME_VFS_SET_FILE_INFO_OWNER |
	     GNOME_VFS_SET_FILE_INFO_TIME))
		return GNOME_VFS_ERROR_NOT_SUPPORTED;

	
	if (mask & GNOME_VFS_SET_FILE_INFO_NAME) {
		LOCK_SAMBA();
		DEBUG_SMB (("set_info: set new name: %s\n", info->name));
		
		res = smb_server_connection_new_from_uri (uri,
							  &filename1,
							  &connection);
		if (res) {
			UNLOCK_SAMBA();
			return res;
		}
		dos_filename1 = unix_filename_to_dos (filename1);
		filename2 = g_malloc (strlen (filename1) + strlen (info->name) + 1);

		strcpy (filename2, filename1);
		end = strrchr (filename2, '/');
		if (!end)
			strcpy (filename2, info->name);
		else
			strcpy (end+1, info->name);
		
		dos_filename2 = unix_filename_to_dos (filename2);
		DEBUG_SMB (("cli_rename:ing %s to %s\n", dos_filename1, dos_filename2));
		ret = cli_rename (connection->cli, dos_filename1, dos_filename2);
		g_free (filename1);
		g_free (filename2);
		g_free (dos_filename1);
		g_free (dos_filename2);
		if (!ret) {
			res = gnome_vfs_result_from_errno_code (cli_error (connection->cli, NULL, NULL, NULL));
			smb_connection_destroy (connection);
			UNLOCK_SAMBA();
			return res;
		}
		
		smb_connection_destroy (connection);
		UNLOCK_SAMBA();
	}

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_truncate (GnomeVFSMethod *method,
	     GnomeVFSURI *uri,
	     GnomeVFSFileSize where,
	     GnomeVFSContext *context)

{
	g_warning ("do_truncate\n");
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}

static GnomeVFSResult
do_find_directory (GnomeVFSMethod *method,
		   GnomeVFSURI *near_uri,
		   GnomeVFSFindDirectoryKind kind,
		   GnomeVFSURI **result_uri,
		   gboolean create_if_needed,
		   gboolean find_if_needed,
		   guint permissions,
		   GnomeVFSContext *context)
{
	g_warning ("do_find_directory\n");
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}

static GnomeVFSResult
do_create_symbolic_link (GnomeVFSMethod *method,
			 GnomeVFSURI *uri,
			 const char *target_reference,
			 GnomeVFSContext *context)
{
	g_warning ("do_create_symbolic_link\n");
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}



static GnomeVFSMethod method = {
	sizeof (GnomeVFSMethod),
	do_open,
	do_create,
	do_close,
	do_read,
	do_write,
	do_seek,
	do_tell,
	NULL, //do_truncate_handle,
	do_open_directory,
	do_close_directory,
	do_read_directory,
	do_get_file_info,
	NULL, //do_get_file_info_from_handle,
	do_is_local,
	do_make_directory,
	do_remove_directory,
	do_move,
	do_unlink,
	do_check_same_fs,
	do_set_file_info,
	NULL, //do_truncate,
	NULL, //do_find_directory,
	NULL //do_create_symbolic_link
};

GnomeVFSMethod *
vfs_module_init (const char *method_name, const char *args)
{
	TimeInit ();
	charset_initialise ();
	
	if (!lp_load ("/etc/samba/smb.conf", TRUE, FALSE, FALSE)) {
		DEBUG_SMB (("Couldn't load smb config file"));
	}
	
	codepage_initialise (lp_client_code_page());

	load_interfaces ();

	get_myname((*global_myname)?NULL:global_myname);

	dir_key = g_private_new ((GDestroyNotify)smb_dir_handle_destroy);
	static_key = g_private_new (NULL);
	get_info_key = g_private_new ((GDestroyNotify)g_free);
	
	samba_lock = g_mutex_new();

	return &method;
}

void
vfs_module_shutdown (GnomeVFSMethod *method)
{
	DEBUG_SMB (("<-- smb module shutdown called -->\n"));
}

