/*
 * src/manager.c - GNOME Volume Manager
 *
 * Robert Love <rml@ximian.com>
 *
 * gnome-volume-manager is a simple policy engine that implements a state
 * machine in response to events from HAL.  Responding to these events,
 * gnome-volume-manager implements automount, autorun, autoplay, automatic
 * photo management, and so on.
 *
 * Licensed under the GNU GPL v2.  See COPYING.
 *
 * (C) Copyright 2004 Novell, Inc.
 */

#include "config.h"

#include <gnome.h>
#include <gconf/gconf-client.h>
#include <gdk/gdkx.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <libhal.h>

#include "gvm.h"

#ifdef ENABLE_NLS
# include <libintl.h>
# define _(String) gettext (String)
# ifdef gettext_noop
#   define N_(String) gettext_noop (String)
# else
#   define N_(String) (String)
# endif
#else
# define _(String)
# define N_(String) (String)
#endif

//#define GVM_DEBUG
#ifdef GVM_DEBUG
# define dbg(fmt,arg...) fprintf(stderr, "%s/%d: " fmt,__FILE__,__LINE__,##arg)
#else
# define dbg(fmt,arg...) do { } while(0)
#endif

#define warn(fmt,arg...) g_warning("%s/%d: " fmt,__FILE__,__LINE__,##arg)

#define BIN_MOUNT	"/bin/mount"	/* what we mount with */
#define NAUTILUS_COMMAND "/usr/bin/nautilus -n --no-desktop %m"

static struct gvm_configuration config;
static LibHalContext *hal_ctx;

/*
 * gvm_load_config - synchronize gconf => config structure
 */
static void
gvm_load_config (void)
{
	config.automount_drives = gconf_client_get_bool (config.client,
			GCONF_ROOT "automount_drives", NULL);
	config.automount_media = gconf_client_get_bool (config.client,
			GCONF_ROOT "automount_media", NULL);
	config.autoplay_cda = gconf_client_get_bool (config.client,
			GCONF_ROOT "autoplay_cda", NULL);
	config.autobrowse = gconf_client_get_bool (config.client,
			GCONF_ROOT "autobrowse", NULL);
	config.autorun = gconf_client_get_bool(config.client,
			GCONF_ROOT "autorun", NULL);
	config.autophoto = gconf_client_get_bool(config.client,
			GCONF_ROOT "autophoto", NULL);
	config.autoplay_dvd = gconf_client_get_bool (config.client,
			GCONF_ROOT "autoplay_dvd", NULL);
	config.autoplay_cda_command = gconf_client_get_string (config.client,
			GCONF_ROOT "autoplay_cda_command", NULL);
	config.autorun_path = gconf_client_get_string (config.client,
			GCONF_ROOT "autorun_path", NULL);
	config.autoplay_dvd_command = gconf_client_get_string (config.client,
			GCONF_ROOT "autoplay_dvd_command", NULL);
	config.autoburn_cdr = gconf_client_get_bool (config.client,
			GCONF_ROOT "autoburn_cdr", NULL);
	config.autoburn_cdr_command = gconf_client_get_string (config.client,
			GCONF_ROOT "autoburn_cdr_command", NULL);
	config.autophoto_command = gconf_client_get_string (config.client,
			GCONF_ROOT "autophoto_command", NULL);
	config.eject_command = gconf_client_get_string (config.client,
			GCONF_ROOT "eject_command", NULL);

	/*
	 * If all of the options that control our policy are disabled, then we
	 * have no point in living.  Save the user some memory and exit.
	 */
	if (!(config.automount_drives || config.autobrowse || config.autorun 
			|| config.autoplay_cda 	|| config.autoplay_dvd 
			|| config.autophoto)) {
		dbg ("daemon exit: no point living\n");
		exit (EXIT_SUCCESS);
	}
}

/*
 * gvm_config_changed - gconf_client_notify_add () call back to reload config
 */
static void
gvm_config_changed (GConfClient *client __attribute__((__unused__)),
		    guint id __attribute__((__unused__)),
		    GConfEntry *entry __attribute__((__unused__)),
		    gpointer data __attribute__((__unused__)))
{
	g_free (config.autoplay_cda_command);
	g_free (config.autorun_path);
	g_free (config.autoplay_dvd_command);
	g_free (config.autoburn_cdr_command);
	g_free (config.autophoto_command);
	g_free (config.eject_command);
    
	gvm_load_config ();
}

/*
 * gvm_init_config - initialize gconf client and load config data
 */
static void
gvm_init_config (void)
{
	config.client = gconf_client_get_default ();

	gconf_client_add_dir (config.client, GCONF_ROOT_SANS_SLASH,
			      GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);

	gvm_load_config ();

	gconf_client_notify_add (config.client, GCONF_ROOT_SANS_SLASH,
				 gvm_config_changed, NULL, NULL, NULL);
}

/*
 * gvm_run_command - run the given command, replacing %d with the device node
 * and %m with the given path
 */
static void
gvm_run_command (const char *device, const char *command, const char *path)
{
	char *argv[4];
	gchar *new_command;
	GError *error = NULL;
	GString *exec = g_string_new (NULL);
	char *p, *q;

	/* perform s/%d/device/ and s/%m/path/ */
	new_command = g_strdup (command);
	q = new_command;
	p = new_command;
	while ((p = strchr (p, '%')) != NULL) {
		if (*(p + 1) == 'd') {
			*p = '\0';
			g_string_append (exec, q);
			g_string_append (exec, device);
			q = p + 2;
			p = p + 2;
		} else if (*(p + 1) == 'm') {
			*p = '\0';
			g_string_append (exec, q);
			g_string_append (exec, path);
			q = p + 2;
			p = p + 2;
		}
	}
	g_string_append (exec, q);

	argv[0] = "/bin/sh";
	argv[1] = "-c";
	argv[2] = exec->str;
	argv[3] = NULL;

	g_spawn_async (g_get_home_dir (), argv, NULL, 0, NULL, NULL,
		       NULL, &error);
	if (error)
		warn ("failed to exec %s: %s\n", exec->str, error->message);

	g_string_free (exec, TRUE);
	g_free (new_command);
}

/*
 * gvm_ask_autorun - ask the user if they want to autorun a specific file
 *
 * Returns TRUE if the user selected 'Yes' and FALSE otherwise
 */
static gboolean
gvm_ask_autorun (const char *path)
{
	GtkWidget *askme;
	gboolean retval;

	askme = gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_QUESTION,
					GTK_BUTTONS_YES_NO,
					_("Do you want to run %s?"), path);
	gtk_dialog_set_default_response (GTK_DIALOG (askme), GTK_RESPONSE_YES);

	switch (gtk_dialog_run (GTK_DIALOG (askme))) {
	case GTK_RESPONSE_YES:
		retval = TRUE;
		break;
	case GTK_RESPONSE_NO:
	default:
		retval = FALSE;
		break;
	}

	gtk_widget_destroy (askme);

	return retval;
}

/*
 * gvm_check_dvd - is this a Video DVD?  If so, do something about it.
 *
 * Returns TRUE if this was a Video DVD and FALSE otherwise.
 */
static gboolean
gvm_check_dvd (const char *device, const char *mount_point)
{
	char *path;
	gboolean retval;

	path = g_build_path (G_DIR_SEPARATOR_S, mount_point, "video_ts", NULL);
	retval = g_file_test (path, G_FILE_TEST_IS_DIR);
	g_free (path);

	/* try the other name, if needed */
	if (retval == FALSE) {
		path = g_build_path (G_DIR_SEPARATOR_S, mount_point,
				     "VIDEO_TS", NULL);
		retval = g_file_test (path, G_FILE_TEST_IS_DIR);
		g_free (path);
	}

	if (retval && config.autoplay_dvd)
		gvm_run_command (device, config.autoplay_dvd_command,
				 mount_point);

	return retval;
}

#define ASK_PHOTOS_MSG	"There are photographs on this device.\n\n" \
			"Would you like to import these photographs " \
			"into your photo album?"

/*
 * gvm_check_photos - check if this device is a digital camera or a storage
 * unit from a digital camera (e.g., a compact flash card).  If it is, then
 * ask the user if he wants to import the photos.
 *
 * Returns TRUE if there were photos on this device, FALSE otherwise
 *
 * FIXME: Should probably not prompt the user and just do it automatically.
 *        This now makes sense, as gphoto added an import mode.
 */
static gboolean
gvm_check_photos (const char *udi, const char *device, const char *mount_point)
{
	char *dcim_path;
	enum { IMPORT } action = -1;
	GtkWidget *askme;
	int retval = FALSE;

	dcim_path = g_build_path (G_DIR_SEPARATOR_S, mount_point, "dcim", NULL);

	if (!g_file_test (dcim_path, G_FILE_TEST_IS_DIR))
		goto out;

	retval = TRUE;
	dbg ("Photos detected: %s\n", dcim_path);

	/* add the "content.photos" capability to this device */
	if (!hal_device_add_capability (hal_ctx, udi, "content.photos"))
		warn ("failed to set content.photos on %s\n", device);

	if (config.autophoto) {
		askme = gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_QUESTION,
						GTK_BUTTONS_NONE,
						_(ASK_PHOTOS_MSG));
		gtk_dialog_add_buttons (GTK_DIALOG (askme),
					GTK_STOCK_CANCEL,
					GTK_RESPONSE_CANCEL, _("_Import"),
					IMPORT, NULL);
		action = gtk_dialog_run (GTK_DIALOG (askme));
		gtk_widget_destroy (askme);

		if (action == IMPORT)
			gvm_run_command (device, config.autophoto_command,
					 dcim_path);
	}

out:
	g_free (dcim_path);
	return retval;
}

/*
 * gvm_device_autorun - automatically execute stuff on the given UDI
 *
 * we currently autorun: autorun files, video DVD's, and digital photos
 */
static void
gvm_device_autorun (const char *udi)
{
	char *device = NULL, *mount_point = NULL;
	gboolean autorun_succeeded = FALSE;

	device = hal_device_get_property_string (hal_ctx, udi, "block.device");
	if (!device) {
		warn ("cannot get block.device\n");
		goto out;
	}

	mount_point = hal_device_get_property_string (hal_ctx, udi, 
						      "volume.mount_point");
	if (!mount_point) {
		warn ("cannot get volume.mount_point\n");
		goto out;
	}

	if (gvm_check_dvd (device, mount_point))
		goto out;

	if (gvm_check_photos (udi, device, mount_point))
		goto out;

	if (config.autorun == TRUE && config.autorun_path) {
		char **autorun_fns;
		int i;

		autorun_fns = g_strsplit (config.autorun_path, ":", -1);

		for (i = 0; autorun_fns[i]; i++) {
			char *path, *argv[2];

			path = g_strdup_printf ("%s/%s", mount_point,
						autorun_fns[i]);
			argv[0] = path;
			argv[1] = NULL;

			if (access (path, X_OK))
				continue;

			if (gvm_ask_autorun (path)) {
				GError *error = NULL;

				g_spawn_async (g_get_home_dir (), argv, NULL,
					       0, NULL, NULL, NULL, &error);
				if (error)
					warn ("failed to exec %s: %s\n", path,
					      error->message);
				else
					autorun_succeeded = TRUE;
				
				g_free (path);
				break;
			}

			g_free (path);
		}

		g_strfreev (autorun_fns);
	}
	
	if ((config.autobrowse == TRUE) && (autorun_succeeded == FALSE)) {
		gvm_run_command (device, NAUTILUS_COMMAND, mount_point);
	}

out:
	hal_free_string (device);
	hal_free_string (mount_point);
}

/*
 * gvm_device_mount - use BIN_MOUNT to mount the given device node.
 *
 * Note that this requires that the given device node is in /etc/fstab.  This
 * is intentional.
 */
static void
gvm_device_mount (char *device)
{
	char *argv[3];
	GError *error = NULL;

	argv[0] = BIN_MOUNT;
	argv[1] = device;
	argv[2] = NULL;

	g_spawn_async (g_get_home_dir (), argv, NULL, 0, NULL,
		       NULL, NULL, &error);
	if (error)
		warn ("failed to exec " BIN_MOUNT ": %s\n", error->message);
}

/*
 * gvm_run_cdplay - if so configured, execute the user-specified CD player on
 * the given device node
 */
static void
gvm_run_cdplayer (const char *device, const char *mount_point)
{
	if (config.autoplay_cda)
		gvm_run_command (device, config.autoplay_cda_command,
				 mount_point);
}

#define ASK_MIXED_MSG	"This CD has both audio tracks and data files.\n" \
			"Would you like to play the audio tracks or " \
			"browse the data files?"

/*
 * gvm_ask_mixed - if a mixed mode CD (CD Plus) is inserted, we can either
 * mount the data tracks or play the audio tracks.  How we handle that depends
 * on the user's configuration.  If the configuration allows either option,
 * we ask.
 */
static void
gvm_ask_mixed (const char *udi)
{
	enum { MOUNT, PLAY } action = -1;
	char *device = NULL, *mount_point = NULL;

	device = hal_device_get_property_string (hal_ctx, udi, "block.device");
	if (!device) {
		warn ("cannot get block.device\n");
		goto out;
	}

	if (config.automount_media && config.autoplay_cda) {
		GtkWidget *askme;

		askme = gtk_message_dialog_new (NULL, 0,
						GTK_MESSAGE_QUESTION,
						GTK_BUTTONS_NONE,
						_(ASK_MIXED_MSG));
		gtk_dialog_add_buttons (GTK_DIALOG (askme),
					GTK_STOCK_CANCEL,
					GTK_RESPONSE_CANCEL, _("_Play"),
					PLAY, _("_Browse"), MOUNT, NULL);
		action = gtk_dialog_run (GTK_DIALOG (askme));
		gtk_widget_destroy (askme);
	} else if (config.automount_media)
		action = MOUNT;
	else if (config.autoplay_cda)
		action = PLAY;

	switch (action) {
	case MOUNT:
		gvm_device_mount (device);
		break;
	case PLAY:
		gvm_run_cdplayer (device, device);
		break;
	default:
		break;
	}

out:
	hal_free_string (device);
	hal_free_string (mount_point);
}

/*
 * gvm_run_cdburner - execute the user-specified CD burner command on the
 * given device node, if so configured
 */
static void
gvm_run_cdburner (const char *device, const char *mount)
{
	if (config.autoburn_cdr)
		gvm_run_command (device, config.autoburn_cdr_command, mount);
}

/*
 * gvm_device_is_writer - is this device capable of writing CDs?
 */
static gboolean
gvm_device_is_writer (const char *udi)
{
	if ((hal_device_get_property_bool (hal_ctx, udi, "storage.cdrom.cdr")) ||
	    (hal_device_get_property_bool (hal_ctx, udi, "storage.cdrom.cdrw")) ||
	    (hal_device_get_property_bool (hal_ctx, udi, "storage.cdrom.dvdr")) ||
	    (hal_device_get_property_bool (hal_ctx, udi, "storage.cdrom.dvdram")) ||
	    (hal_device_get_property_bool (hal_ctx, udi, "storage.cdrom.dvdplusr")) ||
	    (hal_device_get_property_bool (hal_ctx, udi, "storage.cdrom.dvdplusrw")))
		return TRUE;

	return FALSE;
}

/*
 * gvm_cdrom_policy - There has been a media change event on the CD-ROM
 * associated with the given UDI.  Enforce policy.
 */
static void
gvm_cdrom_policy (const char *udi)
{
	char *device = NULL;
	char *drive_udi = NULL;
	dbus_bool_t has_audio;
	dbus_bool_t has_data;
	dbus_bool_t is_blank;

	has_audio = hal_device_get_property_bool (hal_ctx, udi,
						  "volume.disc.has_audio");
	has_data = hal_device_get_property_bool (hal_ctx, udi,
						  "volume.disc.has_data");
	is_blank = hal_device_get_property_bool (hal_ctx, udi,
						  "volume.disc.is_blank");
	drive_udi = hal_device_get_property_string(hal_ctx, udi,
			"info.parent");

	device = hal_device_get_property_string (hal_ctx, udi, "block.device");
	if (!device) {
		warn ("cannot get block.device\n");
		goto out;
	}

	if (has_audio && (!has_data)) {
		gvm_run_cdplayer (device, device);
	} else if (has_audio && has_data) {
		gvm_ask_mixed (udi);
	} else if (has_data) {
		if (config.automount_media)
			gvm_device_mount (device);
	} else if (is_blank) {
		if (gvm_device_is_writer (drive_udi))
			gvm_run_cdburner (device, device);
	}

	/** @todo enforce policy for all the new disc types now supported */

out:
	hal_free_string (device);
	hal_free_string (drive_udi);
}

/*
 * gvm_media_changed - generic media change handler.
 *
 * This is called on a UDI in response to a media change event.  We have to
 * decipher the storage media type to run the appropriate media-present check.
 * Then, if there is indeed media in the drive, we enforce the appropriate
 * policy.
 *
 * At the moment, we only handle CD-ROM and DVD drives.
 */
static void
gvm_media_changed (const char *udi)
{
	char *media_type;

	/* get HAL's interpretation of our media type */
	media_type = hal_device_get_property_string (hal_ctx, udi, 
						     "storage.drive_type");
	if (!media_type) {
		warn ("cannot get storage.drive_type\n");
		return;
	}

	if (!g_strcasecmp (media_type, "cdrom"))
		gvm_cdrom_policy (udi);

	/* other media_types go here */

	hal_free_string (media_type);
}

/** Invoked when a device is added to the Global Device List. 
 *
 *  @param  ctx                 LibHal context
 *  @param  udi                 Universal Device Id
 */
static void
hal_device_added (LibHalContext *ctx __attribute__((__unused__)), 
		  const char *udi)
{
	char *device = NULL, *storage_device = NULL;

	dbg ("New Device: %s\n", udi);

	if (!hal_device_query_capability(hal_ctx, udi, "block"))
		goto out;
	
	/* is this a mountable volume ? */
	if (!hal_device_get_property_bool (hal_ctx, udi, 
					   "block.is_volume"))
		goto out;
	
	/* if it is a volume, it must have a device node */
	device = hal_device_get_property_string (hal_ctx, udi, 
						 "block.device");
	if (!device) {
		dbg ("cannot get block.device\n");
		goto out;
	}
	
	/* get the backing storage device */
	storage_device = hal_device_get_property_string (
		hal_ctx, udi,
		"block.storage_device");
	if (!storage_device) {
		dbg ("cannot get block.storage_device\n");
		goto out;
	}
	
	/*
	 * Does this device support removable media?  Note that we
	 * check storage_device and not our own UDI
	 */
	if (hal_device_get_property_bool (hal_ctx, storage_device,
					  "storage.removable")) {
		/* we handle media change events separately */
		dbg ("Changed: %s\n", device);
		gvm_media_changed (udi);
		goto out;
	}
	
	/* folks, we have a new device! */
	dbg ("Added: %s\n", device);
	
	if (config.automount_drives)
		gvm_device_mount (device);
	
out:
	hal_free_string (device);
	hal_free_string (storage_device);
}

/** Invoked when a device is removed from the Global Device List. 
 *
 *  @param  ctx                 LibHal context
 *  @param  udi                 Universal Device Id
 */
static void
hal_device_removed (LibHalContext *ctx __attribute__((__unused__)), 
		    const char *udi)
{
	dbg ("Device removed: %s\n", udi);
}

/** Invoked when device in the Global Device List acquires a new capability.
 *
 *  @param  ctx                 LibHal context
 *  @param  udi                 Universal Device Id
 *  @param  capability          Name of capability
 */
static void
hal_device_new_capability (LibHalContext *ctx __attribute__((__unused__)),
			   const char *udi __attribute__((__unused__)), 
			   const char *capability __attribute__((__unused__)))
{
}

/** Invoked when device in the Global Device List loses a capability.
 *
 *  @param  ctx                 LibHal context
 *  @param  udi                 Universal Device Id
 *  @param  capability          Name of capability
 */
static void
hal_device_lost_capability (LibHalContext *ctx __attribute__((__unused__)),
			    const char *udi __attribute__((__unused__)), 
			    const char *capability __attribute__((__unused__)))
{
}

/** Invoked when a property of a device in the Global Device List is
 *  changed, and we have we have subscribed to changes for that device.
 *
 *  @param  ctx                 LibHal context
 *  @param  udi                 Univerisal Device Id
 *  @param  key                 Key of property
 */
static void
hal_property_modified (LibHalContext *ctx __attribute__((__unused__)),
		       const char *udi, 
		       const char *key,
		       dbus_bool_t is_removed __attribute__((__unused__)), 
		       dbus_bool_t is_added __attribute__((__unused__)))
{
	if (!g_strcasecmp (key, "volume.is_mounted")) {
		dbus_bool_t val;
		
		val = hal_device_get_property_bool (hal_ctx, udi, key);
		if (val == TRUE) {
			dbg ("Mounted: %s\n", udi);
			gvm_device_autorun (udi);
		} else
			dbg ("Unmounted: %s\n", udi);
	}	
}


/** Invoked when a device in the GDL emits a condition that cannot be
 *  expressed in a property (like when the processor is overheating)
 *
 *  @param  ctx                 LibHal context
 *  @param  udi                 Univerisal Device Id
 *  @param  condition_name      Name of condition
 *  @param  message             D-BUS message with parameters
 */
static void
hal_device_condition (LibHalContext *ctx __attribute__((__unused__)),
		      const char *udi __attribute__((__unused__)), 
		      const char *condition_name __attribute__((__unused__)),
		      DBusMessage * message __attribute__((__unused__)))
{
	if (!g_strcasecmp (condition_name, "EjectPressed")) {		
		char *argv[3];
		GError *error = NULL;
		char *device;

		device = hal_device_get_property_string (ctx, udi,
							 "block.device");
		if (!device)
			warn ("cannot get block.device\n");
		else if (config.eject_command) {
			argv[0] = config.eject_command;
			argv[1] = device;
			argv[2] = NULL;

			g_spawn_async (g_get_home_dir (), argv, NULL, 0, NULL,
				       NULL, NULL, &error);
			if (error)
				warn ("failed to exec %s: %s\n",
				      config.eject_command, error->message);
		}

		hal_free_string (device);
	}
}


/** Invoked by libhal for integration with our mainloop. 
 *
 *  @param  ctx                 LibHal context
 *  @param  dbus_connection     D-BUS connection to integrate
 */
static void
hal_mainloop_integration (LibHalContext *ctx __attribute__((__unused__)),
			  DBusConnection * dbus_connection)
{
	dbus_connection_setup_with_g_main (dbus_connection, NULL);
}


int
main (int argc, char *argv[])
{
	GnomeClient *client;
	LibHalFunctions hal_functions = { hal_mainloop_integration,
					  hal_device_added,
					  hal_device_removed,
					  hal_device_new_capability,
					  hal_device_lost_capability,
					  hal_property_modified,
					  hal_device_condition };

	gnome_program_init (PACKAGE, VERSION, LIBGNOMEUI_MODULE,
			    argc, argv, GNOME_PARAM_NONE);

	bindtextdomain(PACKAGE, GNOMELOCALEDIR);
	bind_textdomain_codeset(PACKAGE, "UTF-8");
	textdomain(PACKAGE);

	client = gnome_master_client ();
	if (gvm_get_clipboard ())
		gnome_client_set_restart_style (client, GNOME_RESTART_ANYWAY);
	else {
		gnome_client_set_restart_style (client, GNOME_RESTART_NEVER);
		warn ("already running?\n");
		return 1;
	}

	gtk_signal_connect (GTK_OBJECT (client), "die",
			    GTK_SIGNAL_FUNC (gtk_main_quit), NULL);

	hal_ctx = hal_initialize (&hal_functions, FALSE);
	if (!hal_ctx) {
		warn ("failed to initialize HAL!\n");
		return 1;
	}

	if (hal_device_property_watch_all (hal_ctx)) {
		warn ("failed to watch all HAL properties!\n");
		return 1;
	}

	gvm_init_config ();

	gtk_main ();

	return 0;
}
