/* GStreamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 * Copyright (C) <2003> David Schleef <ds@schleef.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <string.h>

#include <sys/time.h>

/*#define GST_DEBUG_FORCE_DISABLE*/

#include "xvideosink.h"

/* elementfactory information */
static GstElementDetails gst_xvideosink_details = {
  "Video sink",
  "Sink/Video",
  "LGPL",
  "A general X Window video sink",
  VERSION,
  "Wim Taymans <wim.taymans@chello.be>",
  "(C) 2000",
};

/* xvideosink signals and args */

enum {
  ARG_0,
  ARG_SCALE,
  ARG_DISABLE_XV,
  ARG_TOPLEVEL,
  ARG_AUTOSIZE,
  ARG_NEED_NEW_WINDOW,
};

static void		gst_xvideosink_class_init	(GstXVideoSinkClass *klass);
static void		gst_xvideosink_init		(GstXVideoSink *xvideosink);
static void 		gst_xvideosink_dispose 		(GObject *object);

static void		gst_xvideosink_chain		(GstPad *pad, GstData *_data);
static GstElementStateReturn
			gst_xvideosink_change_state 	(GstElement *element);

static gboolean		gst_xvideosink_release_locks	(GstElement *element);
static void		gst_xvideosink_set_property	(GObject *object, guint prop_id, 
							 const GValue *value, GParamSpec *pspec);
static void		gst_xvideosink_get_property	(GObject *object, guint prop_id, 
							 GValue *value, GParamSpec *pspec);

static void		gst_xvideosink_reset 		(GstXVideoSink *xvideosink);

/* these functions require the lock xvideosink->lock to be held */
static gboolean		gst_xvideosink_xwindow_new	(GstXVideoSink *xvideosink);
static void		gst_xvideosink_xwindow_free	(GstXVideoSink *xvideosink);
static gboolean		gst_xvideosink_image_new	(GstXVideoSink *xvideosink);
static void		gst_xvideosink_image_free	(GstXVideoSink *xvideosink);

/* default template - initiated with class struct to allow gst-register to work
   with X running */
GST_PAD_TEMPLATE_FACTORY (gst_xvideosink_sink_template_factory,
  "sink",
  GST_PAD_SINK,
  GST_PAD_ALWAYS,
  GST_CAPS_NEW ( "videosink_rgbsink", "video/x-raw-rgb",
    "framerate", GST_PROPS_FLOAT_RANGE(0, G_MAXFLOAT),
    "width", GST_PROPS_INT_RANGE(0, G_MAXINT),
    "height", GST_PROPS_INT_RANGE(0, G_MAXINT)),
  GST_CAPS_NEW ( "videosink_yuvsink", "video/x-raw-yuv",
    "framerate", GST_PROPS_FLOAT_RANGE(0, G_MAXFLOAT),
    "width", GST_PROPS_INT_RANGE(0, G_MAXINT),
    "height", GST_PROPS_INT_RANGE(0, G_MAXINT))
)

static GstElementClass *parent_class = NULL;

GType
gst_xvideosink_get_type (void)
{
  static GType xvideosink_type = 0;

  if (!xvideosink_type) {
    static const GTypeInfo xvideosink_info = {
      sizeof (GstXVideoSinkClass),
      NULL,
      NULL,
      (GClassInitFunc) gst_xvideosink_class_init,
      NULL,
      NULL,
      sizeof (GstXVideoSink),
      0,
      (GInstanceInitFunc) gst_xvideosink_init,
    };
    xvideosink_type = g_type_register_static(GST_TYPE_VIDEOSINK,
                                             "GstXVideoSink",
                                             &xvideosink_info, 0);
  }
  return xvideosink_type;
}

static void
gst_xvideosink_class_init (GstXVideoSinkClass *klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;
  GstVideoSinkClass *gstvs_class;

  gobject_class = (GObjectClass*) klass;
  gstelement_class = (GstElementClass*) klass;
  gstvs_class = (GstVideoSinkClass*) klass;

  parent_class = g_type_class_ref (GST_TYPE_ELEMENT);

  g_object_class_install_property (G_OBJECT_CLASS (klass), ARG_DISABLE_XV,
    g_param_spec_boolean ("disable_xv", "Disable XV", "Disable Xv images",
                          TRUE, G_PARAM_READWRITE));
  g_object_class_install_property (G_OBJECT_CLASS (klass), ARG_TOPLEVEL,
    g_param_spec_boolean ("toplevel", "Toplevel", "Create a toplevel window",
                          TRUE, G_PARAM_READWRITE));
  g_object_class_install_property (G_OBJECT_CLASS (klass), ARG_AUTOSIZE,
    g_param_spec_boolean ("auto_size", "Auto Size", "Resizes the window to negotiated size",
                          TRUE, G_PARAM_READWRITE));
  g_object_class_install_property (G_OBJECT_CLASS (klass), ARG_NEED_NEW_WINDOW,
    g_param_spec_boolean ("need_new_window", 
	                  "Needs a new X window", 
			  "Request that a new X window be created for embedding",
                          TRUE, G_PARAM_WRITABLE)); 

  gobject_class->set_property = gst_xvideosink_set_property;
  gobject_class->get_property = gst_xvideosink_get_property;

  gobject_class->dispose = gst_xvideosink_dispose;
  
  gstelement_class->change_state = gst_xvideosink_change_state;
  gstelement_class->release_locks = gst_xvideosink_release_locks;
  
  /*gstvs_class->set_video_out = gst_xvideosink_set_video_out;
  gstvs_class->push_ui_event = gst_xvideosink_push_ui_event;
  gstvs_class->set_geometry = gst_xvideosink_set_geometry;*/
}

static void
gst_xvideosink_imagepool_clear (GstXVideoSink *xvideosink)
{
  g_mutex_lock(xvideosink->pool_lock);
  while (xvideosink->image_pool) {
    GstImage *image = GST_IMAGE (xvideosink->image_pool->data);

    xvideosink->image_pool = g_slist_delete_link (xvideosink->image_pool, xvideosink->image_pool);

    _gst_image_destroy (image);
  }
  g_mutex_unlock(xvideosink->pool_lock);
}

G_GNUC_UNUSED static void
gst_xvideosink_buffer_pool_free (GstBufferPool *pool)
{
  GstXVideoSink *xvideosink = pool->user_data;
	
  gst_xvideosink_imagepool_clear (xvideosink);

  gst_buffer_pool_default_free (pool);
}

static GstBuffer*
gst_xvideosink_buffer_new (GstBufferPool *pool,  
		           gint64 location, guint size, gpointer user_data)
{
  GstXVideoSink *xvideosink;
  GstBuffer *buffer;
  GstImage *image;
  
  xvideosink = GST_XVIDEOSINK (user_data);

  g_mutex_lock (xvideosink->pool_lock);
  if (!xvideosink->image_pool) {
    g_mutex_unlock (xvideosink->pool_lock);

    /* we have to lock the X connection */
    g_mutex_lock (xvideosink->lock);
    if (xvideosink->image_format->type == GST_TYPE_XIMAGE) {
      GST_DEBUG ("creating ximage");
      image = GST_IMAGE (_gst_ximage_new ( 
		         xvideosink->window, 
		         GST_VIDEOSINK_WIDTH (xvideosink),
                         GST_VIDEOSINK_HEIGHT (xvideosink)));
    }
#ifdef HAVE_XVIDEO
    else if (xvideosink->image_format->type == GST_TYPE_XVIMAGE) {
      GST_DEBUG ("creating xvimage %08x", xvideosink->image_format->im_format);
      image = GST_IMAGE (_gst_xvimage_new (
		         xvideosink->image_format,
		         xvideosink->window,
		         GST_VIDEOSINK_WIDTH (xvideosink),
                         GST_VIDEOSINK_HEIGHT (xvideosink)));
    }
#endif
    else {
      g_assert_not_reached();
      return NULL;
    }
    g_mutex_unlock (xvideosink->lock);
  }
  else {
    image = xvideosink->image_pool->data;
    xvideosink->image_pool = g_slist_delete_link (xvideosink->image_pool, xvideosink->image_pool);
    g_mutex_unlock (xvideosink->pool_lock);
  }
  if (image == NULL) {
    gst_element_error(GST_ELEMENT (xvideosink), "image creation failed");
    /* FIXME: need better error handling here */
    return NULL;
  }

  buffer = gst_buffer_new ();
  GST_BUFFER_POOL_PRIVATE (buffer) = image;
  GST_BUFFER_DATA (buffer) = GST_IMAGE_DATA (image);
  GST_BUFFER_SIZE (buffer) = GST_IMAGE_SIZE (image);

  return buffer;
}

static void
gst_xvideosink_buffer_free (GstBufferPool *pool, GstBuffer *buffer, gpointer user_data)
{
  GstXVideoSink *xvideosink;
  gboolean keep_buffer = FALSE;
  GstImage *image;

  xvideosink = GST_XVIDEOSINK (user_data);

  /* get the image from the private data */
  image = GST_BUFFER_POOL_PRIVATE (buffer);

  /* the question is, to be or not to be, or rather: do the settings of the buffer still apply? */
  g_mutex_lock (xvideosink->lock);

  if (xvideosink->image) {
    gint image_type, image_size;
    GstXWindow *image_window;

    image_type = GST_IMAGE_TYPE (image);
    image_size = GST_IMAGE_SIZE (image);
    image_window = GST_IMAGE_WINDOW (image);
	    
    /* if size and type matches, keep the buffer */
    if (image_type == GST_IMAGE_TYPE (xvideosink->image) &&
        image_size == GST_IMAGE_SIZE (xvideosink->image) &&
        image_window == GST_IMAGE_WINDOW (xvideosink->image)) 
    { 
      keep_buffer = TRUE;
    } 
  }
  g_mutex_unlock(xvideosink->lock);

  if (keep_buffer) {
    g_mutex_lock (xvideosink->pool_lock);
    xvideosink->image_pool = g_slist_prepend (xvideosink->image_pool, image);
    g_mutex_unlock (xvideosink->pool_lock);
  } else {
    _gst_image_destroy (image);
  }

  GST_BUFFER_DATA (buffer) = NULL;

  gst_buffer_default_free (buffer);
}

static GstBufferPool*
gst_xvideosink_get_bufferpool (GstPad *pad)
{
  GstXVideoSink *xvideosink;
  
  xvideosink = GST_XVIDEOSINK (gst_pad_get_parent (pad));

  if (!xvideosink->bufferpool) {
    GST_DEBUG ("xvideosink: creating bufferpool");

    xvideosink->bufferpool = gst_buffer_pool_new (
		    NULL,		/* free */
		    NULL,		/* copy */
		    (GstBufferPoolBufferNewFunction)gst_xvideosink_buffer_new,
		    NULL,		/* buffer copy, the default is fine */
		    (GstBufferPoolBufferFreeFunction)gst_xvideosink_buffer_free,
		    xvideosink);

    xvideosink->image_pool = NULL;
  }

  gst_buffer_pool_ref (xvideosink->bufferpool);

  return xvideosink->bufferpool;
}

static void
gst_xvideosink_get_real_size (GstXVideoSink *xvideosink, 
		              gint *real_x, gint *real_y)
{
  gint pwidth, pheight;

  gst_video_sink_get_geometry (GST_VIDEOSINK (xvideosink), real_x, real_y);
  pwidth  = xvideosink->pixel_width;
  pheight = xvideosink->pixel_height;

  if (pwidth && pheight) {
    if (pwidth > pheight) {
      *real_x = (*real_x * pwidth) / pheight;
    }
    else if (pwidth < pheight) {
      *real_y = (*real_y * pheight) / pwidth;
    }
  }
}

static GstPadLinkReturn
gst_xvideosink_sinkconnect (GstPad *pad, GstCaps *caps)
{
  GstXVideoSink *xvideosink;
  GstCaps *icaps;
  int i;
  GstImageFormat *image_format;

  xvideosink = GST_XVIDEOSINK (gst_pad_get_parent (pad));

  /* we are not going to act on variable caps */
  if (!GST_CAPS_IS_FIXED (caps))
    return GST_PAD_LINK_DELAYED;

  if (GST_CAPS_IS_CHAINED (caps))
    return GST_PAD_LINK_DELAYED;

  GST_DEBUG ("sinkconnect %s", gst_caps_to_string(caps));

  image_format = NULL;
  icaps = NULL;
  for(i=0;i<xvideosink->window->image_formats->len;i++){
    image_format = g_ptr_array_index(xvideosink->window->image_formats, i);
    if(image_format->type == GST_TYPE_XVIMAGE && xvideosink->disable_xv){
      continue;
    }

    icaps = gst_caps_intersect (caps, image_format->caps);

    if(icaps != NULL){
      break;
    }
  }

  if(icaps == NULL){
    GST_DEBUG ("no format found");
    return GST_PAD_LINK_REFUSED;
  }

  gst_caps_unref(icaps);

  GST_DEBUG ("using format %d",i);
  xvideosink->image_format = image_format;

  gst_caps_get_int (caps, "width",
                    &(GST_VIDEOSINK_WIDTH (xvideosink)));
  gst_caps_get_int (caps, "height",
                    &(GST_VIDEOSINK_HEIGHT (xvideosink)));

  if (gst_caps_has_fixed_property (caps, "pixel_width")) {
    gst_caps_get_int (caps, "pixel_width", &xvideosink->pixel_width);
  } else {
    xvideosink->pixel_width = 1;
  }

  if (gst_caps_has_fixed_property (caps, "pixel_height")) {
    gst_caps_get_int (caps, "pixel_height", &xvideosink->pixel_height);
  } else {
    xvideosink->pixel_height = 1;
  }

  g_mutex_lock (xvideosink->lock);
  if (!gst_xvideosink_image_new (xvideosink)) {
    g_mutex_unlock (xvideosink->lock);
    xvideosink->image_format = NULL;
    return GST_PAD_LINK_REFUSED;
  }

  if (xvideosink->image == NULL) {
    /* FIXME: need better error handling? */
    g_mutex_unlock (xvideosink->lock);
    gst_element_error (GST_ELEMENT (xvideosink), "image creation failed");
    xvideosink->image_format = NULL;
    return GST_PAD_LINK_REFUSED;
  }

  g_mutex_unlock (xvideosink->lock);

  {
    gint real_x, real_y;

    gst_xvideosink_get_real_size (xvideosink, &real_x, &real_y);
  
    if (xvideosink->auto_size) {
      _gst_xwindow_resize (xvideosink->window, real_x, real_y);
    }
    gst_video_sink_got_video_size (GST_VIDEOSINK (xvideosink), real_x, real_y);
  }

  return GST_PAD_LINK_OK;
}

static GstCaps *
gst_xvideosink_getcaps (GstPad *pad, GstCaps *caps)
{
  GstXVideoSink *xvideosink;

  xvideosink = GST_XVIDEOSINK (gst_pad_get_parent (pad));

  if (xvideosink->window) {
    int i;
    GPtrArray *image_formats;
    GstImageFormat *image_format;

    caps = NULL;
    image_formats = xvideosink->window->image_formats;
    for(i=0;i<image_formats->len;i++){
      image_format = g_ptr_array_index(image_formats, i);
      caps = gst_caps_append(caps, gst_caps_copy(image_format->caps));
    }
  }else{
    caps = gst_caps_append(
	GST_CAPS_NEW ( "videosink_rgbsink", "video/x-raw-rgb",
	  "framerate", GST_PROPS_FLOAT_RANGE(0, G_MAXFLOAT),
	  "width", GST_PROPS_INT_RANGE(0, G_MAXINT),
	  "height", GST_PROPS_INT_RANGE(0, G_MAXINT)),
	GST_CAPS_NEW ( "videosink_yuvsink", "video/x-raw-yuv",
	  "framerate", GST_PROPS_FLOAT_RANGE(0, G_MAXFLOAT),
	  "width", GST_PROPS_INT_RANGE(0, G_MAXINT),
	  "height", GST_PROPS_INT_RANGE(0, G_MAXINT)));
  }

  GST_DEBUG ("getcaps returning %s",gst_caps_to_string(caps));

  return caps;
}

static void
gst_xvideosink_init (GstXVideoSink *xvideosink)
{
  GST_VIDEOSINK_PAD (xvideosink) = gst_pad_new_from_template (
                    GST_PAD_TEMPLATE_GET (gst_xvideosink_sink_template_factory), 
                    "sink");
  gst_element_add_pad (GST_ELEMENT (xvideosink),
                       GST_VIDEOSINK_PAD (xvideosink));
  gst_pad_set_chain_function (GST_VIDEOSINK_PAD (xvideosink),
                              gst_xvideosink_chain);
  gst_pad_set_link_function (GST_VIDEOSINK_PAD (xvideosink),
                             gst_xvideosink_sinkconnect);
  gst_pad_set_getcaps_function (GST_VIDEOSINK_PAD (xvideosink),
                                gst_xvideosink_getcaps);
  gst_pad_set_bufferpool_function (GST_VIDEOSINK_PAD (xvideosink),
                                   gst_xvideosink_get_bufferpool);

  /* one-time initialization */
  xvideosink->window = NULL;
  xvideosink->toplevel = TRUE;
  xvideosink->image_format = NULL;
  xvideosink->lock = g_mutex_new ();

  xvideosink->image_pool = NULL;
  xvideosink->pool_lock = g_mutex_new ();
  xvideosink->send_xid = FALSE;
  xvideosink->need_new_window = FALSE;

  xvideosink->image = NULL;

  gst_xvideosink_reset (xvideosink);
  
  GST_FLAG_SET(xvideosink, GST_ELEMENT_THREAD_SUGGESTED);
}

/* reset properties for first use or -> NULL transition */
static void
gst_xvideosink_reset (GstXVideoSink *xvideosink)
{
  GST_VIDEOSINK_WIDTH (xvideosink) = 100;
  GST_VIDEOSINK_HEIGHT (xvideosink) = 100;
  xvideosink->disable_xv = FALSE;
  xvideosink->auto_size = TRUE;
  xvideosink->correction = 0;
}

static void
gst_xvideosink_dispose (GObject *object)
{
  GstXVideoSink *xvideosink;

  xvideosink = GST_XVIDEOSINK (object);

  if (xvideosink->image) {
    _gst_image_destroy (GST_IMAGE (xvideosink->image));
  }
  
  g_mutex_free (xvideosink->lock);
  g_mutex_free (xvideosink->pool_lock);

  if (xvideosink->bufferpool) 
    gst_buffer_pool_free (xvideosink->bufferpool);

  G_OBJECT_CLASS (parent_class)->dispose (object);
}

static void
gst_xvideosink_chain (GstPad *pad, GstData *_data)
{
  GstBuffer *buf = GST_BUFFER (_data);
  GstXVideoSink *xvideosink;
  GstClockTime time = GST_BUFFER_TIMESTAMP (buf);
  gint64 jitter;

  g_return_if_fail (pad != NULL);
  g_return_if_fail (GST_IS_PAD (pad));
  g_return_if_fail (buf != NULL);

  xvideosink = GST_XVIDEOSINK (gst_pad_get_parent (pad));

  if (GST_IS_EVENT (buf)) {
    GstEvent *event = GST_EVENT (buf);
    gint64 offset;

    switch (GST_EVENT_TYPE (event)) {
      case GST_EVENT_DISCONTINUOUS:
	offset = GST_EVENT_DISCONT_OFFSET (event, 0).value;
	GST_DEBUG ("xvideo discont %" G_GINT64_FORMAT, offset);
	gst_clock_handle_discont (GST_VIDEOSINK_CLOCK (xvideosink),
                                  (guint64) offset);
	break;
      default:
	gst_pad_event_default (pad, event);
	break;
    }
    gst_event_unref (event);
    return;
  }

  if (GST_VIDEOSINK_CLOCK (xvideosink) && time != -1) {
    GstClockReturn ret;

    xvideosink->id = gst_clock_new_single_shot_id (
                       GST_VIDEOSINK_CLOCK (xvideosink), time);

    GST_DEBUG ("videosink: clock %s wait: %" G_GUINT64_FORMAT " %u", 
               GST_OBJECT_NAME (GST_VIDEOSINK_CLOCK (xvideosink)),
               time, GST_BUFFER_SIZE (buf));

    ret = gst_clock_id_wait (xvideosink->id, &jitter);
    gst_clock_id_free (xvideosink->id);
    xvideosink->id = NULL;
  }
  if (GST_VIDEOSINK_CLOCK (xvideosink))
    time = gst_clock_get_time (GST_VIDEOSINK_CLOCK (xvideosink));

  g_mutex_lock (xvideosink->lock);
  /* if we have a pool and the image is from this pool, simply _put it */
  if (xvideosink->bufferpool && GST_BUFFER_BUFFERPOOL (buf) == xvideosink->bufferpool) {
    _gst_image_put (xvideosink->window, GST_IMAGE (GST_BUFFER_POOL_PRIVATE (buf)));
  }
  else {
    /* else we have to copy the data into our private image, if we have one... */
    if (xvideosink->image) {
      memcpy (GST_IMAGE_DATA (xvideosink->image), 
	      GST_BUFFER_DATA (buf), 
	      MIN (GST_BUFFER_SIZE (buf), GST_IMAGE_SIZE (xvideosink->image)));

      _gst_image_put (xvideosink->window, xvideosink->image);
    }
    else {
      g_mutex_unlock (xvideosink->lock);
      gst_buffer_unref (buf);
      gst_element_error (GST_ELEMENT (xvideosink), "no image to draw (are you feeding me video?)");
      return;
    }
  }

  gst_video_sink_frame_displayed (GST_VIDEOSINK (xvideosink));
  g_mutex_unlock (xvideosink->lock);

  if (GST_VIDEOSINK_CLOCK (xvideosink)) {
    jitter = gst_clock_get_time (GST_VIDEOSINK_CLOCK (xvideosink)) - time;

    xvideosink->correction = (xvideosink->correction + jitter) >> 1;
    xvideosink->correction = 0;
  }

  if (xvideosink->send_xid){
    gst_video_sink_got_video_out (
      GST_VIDEOSINK (xvideosink),
      (gpointer) GST_XWINDOW_XID (xvideosink->window));
    xvideosink->send_xid = FALSE;
  }

  gst_buffer_unref (buf);
}


static void
gst_xvideosink_set_property (GObject *object, guint prop_id, 
		             const GValue *value, GParamSpec *pspec)
{
  GstXVideoSink *xvideosink;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail (GST_IS_XVIDEOSINK (object));

  xvideosink = GST_XVIDEOSINK (object);

  switch (prop_id) {
    case ARG_DISABLE_XV:
      xvideosink->disable_xv = g_value_get_boolean (value);
      break;
    case ARG_TOPLEVEL:
      xvideosink->toplevel = g_value_get_boolean (value);
      break;
    case ARG_AUTOSIZE:
      xvideosink->auto_size = g_value_get_boolean (value);
      break;
    case ARG_NEED_NEW_WINDOW:
      if (g_value_get_boolean (value)) {
        if (xvideosink->window) {
          g_mutex_lock (xvideosink->lock);
          gst_xvideosink_xwindow_free (xvideosink);
          gst_xvideosink_xwindow_new (xvideosink);
          g_mutex_unlock (xvideosink->lock);
	} else {
	  xvideosink->need_new_window = TRUE;
	}
      }
      break;
    default:
      break;
  }
}

static void
gst_xvideosink_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
  GstXVideoSink *xvideosink;

  /* it's not null if we got it, but it might not be ours */
  xvideosink = GST_XVIDEOSINK(object);

  switch (prop_id) {
    case ARG_DISABLE_XV:
      g_value_set_boolean (value, xvideosink->disable_xv);
      break;
    case ARG_TOPLEVEL:
      g_value_set_boolean (value, xvideosink->toplevel);
      break;
    case ARG_AUTOSIZE:
      g_value_set_boolean (value, xvideosink->auto_size);
      break;
    default: {
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
  }
}

static GstElementStateReturn
gst_xvideosink_change_state (GstElement *element)
{
  GstXVideoSink *xvideosink;

  xvideosink = GST_XVIDEOSINK (element);

  switch (GST_STATE_TRANSITION (element)) {
    case GST_STATE_NULL_TO_READY:
      if (!gst_xvideosink_xwindow_new (xvideosink))
	return GST_STATE_FAILURE;
      break;
    case GST_STATE_READY_TO_PAUSED:
      xvideosink->correction = 0;
      break;
    case GST_STATE_PAUSED_TO_PLAYING:
      break;
    case GST_STATE_PLAYING_TO_PAUSED:
      break;
    case GST_STATE_PAUSED_TO_READY:
      break;
    case GST_STATE_READY_TO_NULL:
      gst_xvideosink_xwindow_free (xvideosink);
      gst_xvideosink_reset (xvideosink);
      break;
  }

  if (parent_class->change_state)
    return parent_class->change_state (element);
  return GST_STATE_SUCCESS;
}

static gboolean
gst_xvideosink_release_locks (GstElement *element)
{
  GstXVideoSink *xvideosink;

  xvideosink = GST_XVIDEOSINK (element);

  if (xvideosink->id) {
    gst_clock_id_unlock (xvideosink->id);
  }

  return TRUE;
}

static gboolean
gst_xvideosink_xwindow_new (GstXVideoSink *xvideosink)
{
  xvideosink->window = _gst_xwindow_new (GST_VIDEOSINK_WIDTH (xvideosink), 
	                                 GST_VIDEOSINK_HEIGHT (xvideosink), 
					 xvideosink->toplevel);
  if (!xvideosink->window) {
    gst_element_error (GST_ELEMENT (xvideosink), "could not create X window");
    return FALSE;
  }
  
  gst_video_sink_got_video_size (GST_VIDEOSINK (xvideosink),
                                 GST_VIDEOSINK_WIDTH (xvideosink),
                                 GST_VIDEOSINK_HEIGHT (xvideosink));
  
  xvideosink->send_xid = TRUE;
  xvideosink->need_new_window = FALSE;

  if (!xvideosink->image_format) {
    return TRUE;
  } else {
    return gst_xvideosink_image_new (xvideosink);
  }
}

static void
gst_xvideosink_xwindow_free (GstXVideoSink *xvideosink)
{
  if (xvideosink->bufferpool) {
    gst_buffer_pool_unref (xvideosink->bufferpool);
    xvideosink->bufferpool = NULL;
  }

  gst_xvideosink_image_free (xvideosink);
  if (xvideosink->window) {  
    _gst_xwindow_destroy (xvideosink->window);
    xvideosink->window = NULL;
  }
}

static gboolean
gst_xvideosink_image_new (GstXVideoSink *xvideosink)
{
  if (!xvideosink->image_format)
    return FALSE;
  
  if (xvideosink->image)
    gst_xvideosink_image_free (xvideosink);

  /* create a new image to draw onto */
  if (xvideosink->image_format->type == GST_TYPE_XIMAGE) {
    xvideosink->image = GST_IMAGE (_gst_ximage_new ( 
		      xvideosink->window, 
		      GST_VIDEOSINK_WIDTH (xvideosink),
                      GST_VIDEOSINK_HEIGHT (xvideosink)));
    return TRUE;
  }
#ifdef HAVE_XVIDEO
  if (xvideosink->image_format->type == GST_TYPE_XVIMAGE && 
      !xvideosink->disable_xv) {
    xvideosink->image = GST_IMAGE (_gst_xvimage_new ( 
		      xvideosink->image_format,
		      xvideosink->window, 
		      GST_VIDEOSINK_WIDTH (xvideosink),
                      GST_VIDEOSINK_HEIGHT (xvideosink)));
    return TRUE;
  }
#endif
  return FALSE;
}
static void
gst_xvideosink_image_free (GstXVideoSink *xvideosink)
{
  if (xvideosink->image) {
    _gst_image_destroy (xvideosink->image);
    xvideosink->image = NULL;
    /* clear our imagepool too */
    gst_xvideosink_imagepool_clear (xvideosink);
  }
}

static gboolean
plugin_init (GModule *module, GstPlugin *plugin)
{
  GstElementFactory *factory;

  /* Loading the library containing GstVideoSink, our parent object */
  if (!gst_library_load ("gstvideo"))
    return FALSE;
  
  /* create an elementfactory for the xvideosink element */
  factory = gst_element_factory_new ("xvideosink", GST_TYPE_XVIDEOSINK,
                                     &gst_xvideosink_details);
  g_return_val_if_fail (factory != NULL, FALSE);

  gst_element_factory_add_pad_template (
    factory, 
    GST_PAD_TEMPLATE_GET (gst_xvideosink_sink_template_factory));

  gst_plugin_add_feature (plugin, GST_PLUGIN_FEATURE (factory));

  return TRUE;
}

GstPluginDesc plugin_desc = {
  GST_VERSION_MAJOR,
  GST_VERSION_MINOR,
  "xvideosink",
  plugin_init
};
