/* The Cairo CSS Drawing Library.
 * Copyright (C) 2008 Robert Staudinger
 *
 * This  library is free  software; you can  redistribute it and/or
 * modify it  under  the terms  of the  GNU Lesser  General  Public
 * License  as published  by the Free  Software  Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed  in the hope that it will be useful,
 * but  WITHOUT ANY WARRANTY; without even  the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License  along  with  this library;  if not,  write to  the Free
 * Software Foundation, Inc., 51  Franklin St, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

#include <string.h>
#include "ccss-function.h"

static ccss_function_t *_vtable = NULL;

void
ccss_function_set_vtable (ccss_function_t	const *vtable)
{
	ccss_function_t const	*iter;
	unsigned int		 i;

	if (_vtable) {
		g_free (_vtable);
		_vtable = NULL;
	}

	if (vtable) {

		iter = vtable;
		for (i = 0; iter->name; iter++, i++)
			;

		/* also copy NULL-terminator */
		i++;

		_vtable = g_new0 (ccss_function_t, i);
		memcpy (_vtable, vtable, sizeof (*vtable) * i);
	}
}

char *
ccss_function_invoke (char const		*name,
		     CRTerm const	*values)
{
	ccss_function_f	 function;
	GSList		*args;
	GSList		*iter;
	char		*val;
	char const	*unit;
	char		*ret;

	g_return_val_if_fail (_vtable && name, NULL);

	function = NULL;
	for (unsigned int i = 0; _vtable[i].name; i++) {
		if (0 == strcmp (name, _vtable[i].name)) {
			function = _vtable[i].function;
			break;
		}
	}

	if (!function) {
		g_warning ("Function `%s' could not be resolved", name);
		return NULL;
	}

	/* parse args */
	args = NULL;
	iter = NULL;
	while (values) {
		val = NULL;
		switch (values->type) {
		case TERM_NUMBER:
			switch (values->content.num->type) {
			case NUM_GENERIC:
				unit = "";
				break;
			case NUM_LENGTH_PX:
				unit = "px";
				break;
			case NUM_PERCENTAGE:
				unit = "%";
				break;
			default:
				g_assert_not_reached ();
				unit = "";
			}
			val = g_strdup_printf ("%f%s", values->content.num->val, unit);
			break;
		case TERM_STRING:
		case TERM_IDENT:
		case TERM_URI:
			val = g_strdup_printf ("%s", cr_string_peek_raw_str (values->content.str));			
			break;
		case TERM_HASH:
			val = g_strdup_printf ("#%s", cr_string_peek_raw_str (values->content.str));			
			break;
		case TERM_NO_TYPE:
		case TERM_FUNCTION:
		case TERM_RGB:
		case TERM_UNICODERANGE:
		default:
			g_assert_not_reached ();
			values = values->next;
			continue;
		}

		if (args == NULL) {
			args = g_slist_append (NULL, val);
			iter = args;
		} else {
			iter->next = g_slist_append (NULL, val);
		}

		values = values->next;
	}

	/* dispatch */
	ret = function (args);

	/* free args */
	iter = args;
	while (iter) {
		val = (char *) iter->data;
		iter = g_slist_remove (iter, val);
		g_free (val);
	}

	return ret;
}

