/* -*- mode: C; c-file-style: "gnu" -*- */
/*
 * Copyright (C) 2003 Johan Dahlin <jdahlin@async.com.br>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>
#include <popt.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <glib.h>
#include "song-db.h"

#define GETATTR(attr) \
    g_strdup (get_attribute_value (attr, attribute_names, attribute_values))
      
enum {
  NO_ERROR,
  PARSE_ERROR,
};

typedef enum {
  STATE_START,
  STATE_SONGS,
  STATE_SONG
} State;

typedef struct {
  GMarkupParseContext *context;
  GMarkupParser       *parser;
  Song                *song;
  GList               *songs;
  GList               *state_stack;
} ParserData;

static GQuark error_quark; 

static void start_element_handler (GMarkupParseContext  *context,
				   const char           *node_name,
				   const char          **attribute_names,
				   const char          **attribute_values,
				   gpointer              user_data,
				   GError              **error);
static void end_element_handler   (GMarkupParseContext  *context,
				   const char           *node_name,
				   gpointer              user_data,
				   GError              **error);
static void error_handler         (GMarkupParseContext  *context,
				   GError               *error,
				   gpointer              user_data);


static void
push_state (ParserData *data, State state)
{
  data->state_stack = g_list_prepend (data->state_stack, GINT_TO_POINTER (state));
}      

static State
pop_state (ParserData *data)
{
  State state;
  
  state = GPOINTER_TO_INT (data->state_stack->data);
  data->state_stack = g_list_delete_link (data->state_stack, data->state_stack);

  return state;
}      

static State
peek_state (ParserData *data)
{
  State state;
  
  state = GPOINTER_TO_INT (data->state_stack->data);

  return state;
}      

static void
set_error (GError              **err,
           GMarkupParseContext  *context,
           int                   error_code,
           const char           *format,
           ...)
{
  int line, ch;
  va_list args;
  char *str;
  
  g_markup_parse_context_get_position (context, &line, &ch);
  
  va_start (args, format);
  str = g_strdup_vprintf (format, args);
  va_end (args);

  g_set_error (err, error_quark, error_code, "Line %d character %d: %s", line, ch, str);
  
  g_free (str);
}

static const char *
get_attribute_value (const char  *name,
		     const char **names,
		     const char **values)
{
  int i = 0;

  while (names[i])
    {
      if (strcmp (name, names[i]) == 0)
	return values[i];
      i++;
    }

  return NULL;
}

static void
start_element_handler (GMarkupParseContext  *context,
                       const char           *element,
                       const char          **attribute_names,
                       const char          **attribute_values,
                       gpointer              user_data,
                       GError              **error)
{
  ParserData *data = user_data;
  State state;
  char buf[11];
  char *str;

  state = peek_state (data);
  
  switch (state)
    {
    case STATE_START:
      if (strcmp (element, "songs") != 0)
	{
	  set_error (error, context, PARSE_ERROR,
		     "Outermost element must be a <songs> not <%s>",
		     element);
	  break;
	}

      push_state (data, STATE_SONGS);
      break;

    case STATE_SONGS:
      if (strcmp (element, "song") != 0)
	{
	  set_error (error, context, PARSE_ERROR,
		     "<song> tag expected, not <%s>",
		     element);
	  break;
	}

      data->song = g_new0 (Song, 1);

      data->song->filename      = GETATTR ("filename");
      data->song->title         = GETATTR ("title");
      data->song->artist        = GETATTR ("artist");
      data->song->album         = GETATTR ("album");
      data->song->genre         = atoi (GETATTR ("genre"));
      data->song->year          = atoi (GETATTR ("year"));      
      data->song->length        = atoi (GETATTR ("length"));      
      data->song->filesize      = atol (GETATTR ("filesize"));
      data->song->track_number  = atoi (GETATTR ("track_number"));      
      data->song->play_count    = atoi (GETATTR ("play_count"));
      data->song->last_played   = atol (GETATTR ("last_played"));      
      data->song->date_added    = atol (GETATTR ("date_added"));
      data->song->date_modified = atol (GETATTR ("date_modified"));
      data->song->rating        = atoi (GETATTR ("rating"));

      str = GETATTR ("playlists");
      while (sscanf (str, "%10s\n", buf) > 0)
	{
	  data->song->playlists = g_list_append (data->song->playlists, GINT_TO_POINTER (atoi ((char*)&buf)));
	  str += strlen ((char*)&buf) + 1;
	}

      push_state (data, STATE_SONG);
      break;
      
    case STATE_SONG:
      
      push_state (data, STATE_SONGS);
      break;
    }
}

static void
end_element_handler (GMarkupParseContext  *context,
                     const char           *element_name,
                     gpointer              user_data,
                     GError              **error)
{
  ParserData *data = user_data;
  State state = pop_state (data);

  switch (state)
    {
    case STATE_START:
      break;

    case STATE_SONGS:
      data->songs = g_list_reverse (data->songs);
      break;
      
    case STATE_SONG:
      data->songs = g_list_prepend (data->songs, data->song);
      break;
    }
}

static void
error_handler (GMarkupParseContext *context,
	       GError              *error,
	       gpointer             user_data)
{
  g_warning ("%s", error->message);
}

static gboolean
song_db_import (SongDB *db,
		const char *filename)
{
  FILE *fp;
  char *buf;

  GList *l;
  ParserData data;
  static gboolean inited = FALSE;
  
  g_return_val_if_fail (IS_SONG_DB (db), FALSE);
  g_return_val_if_fail (filename != NULL, FALSE);

  if (!inited)
    {
      error_quark = g_quark_from_static_string ("query-xml-error-quark");
      inited = TRUE;
    }
  
  memset (&data, 0, sizeof (data));
  
  push_state (&data, STATE_START);
  
  data.parser = g_new0 (GMarkupParser, 1);

  data.parser->start_element = start_element_handler;
  data.parser->end_element = end_element_handler;
  data.parser->error = error_handler;

  data.context = g_markup_parse_context_new (data.parser, 0, &data, NULL);
  
  fp = fopen (filename, "r");
    
  g_file_get_contents (filename, &buf, NULL, NULL);
  g_markup_parse_context_parse (data.context, buf, -1, NULL);
  fclose (fp);
  g_free (data.parser);

  if (!data.songs)
    return FALSE;
      
  for (l = data.songs; l; l = l->next)
    song_db_update_song (db, l->data);

  song_db_set_version (db, SONG_DB_VERSION);
  
  return TRUE;
}

static gboolean
song_db_export (SongDB *db,
		const char *filename)
{
  FILE *fp;
  GList *l, *m;
  Song *song;
  char *tmp;
  
  g_return_val_if_fail (IS_SONG_DB (db), FALSE);
  g_return_val_if_fail (filename != NULL, FALSE);

  fp = fopen (filename, "w");
  fprintf (fp, "<songs>\n");
  for (l = db->songs; l; l = l->next)
    {
      GString *string = g_string_new ("");    
      song = l->data;

      for (m = song->playlists; m; m = m->next)
	{
	  g_string_append_printf (string, "%d", GPOINTER_TO_INT (m->data));
	  if (m->next)
	    g_string_append_c (string, '\n');
	}

      tmp = g_markup_escape_text (song->filename, -1);
      fprintf (fp, "  <song filename=\"%s\"", tmp);
      g_free (tmp);

      tmp = g_markup_escape_text (song->artist, -1);
      fprintf (fp, " artist=\"%s\"", tmp);
      g_free (tmp);
      
      tmp = g_markup_escape_text (song->title, -1);
      fprintf (fp, " title=\"%s\"", tmp);
      g_free (tmp);
      
      tmp = g_markup_escape_text (song->album, -1);
      fprintf (fp, " album=\"%s\"\n", tmp);
      g_free (tmp);

      fprintf (fp, "        genre=\"%d\" filesize=\"%lld\"\n"
	       "        year=\"%d\" length=\"%d\" track_number=\"%d\" play_count=\"%d\"\n"
	       "        last_played=\"%ld\" date_added=\"%ld\" date_modified=\"%ld\" rating=\"%d\"\n",
	       song->genre, song->filesize,
	       song->year, song->length, song->track_number, song->play_count,
	       song->last_played, song->date_added, song->date_modified, song->rating);
	       

      tmp = g_markup_escape_text (g_string_free (string, FALSE), -1);
      fprintf (fp, "        playlists=\"%s\"/>\n", tmp);
      g_free (tmp);
    }
  fprintf (fp, "</songs>\n");

  fclose (fp);
  
  return TRUE;
}

static gboolean
remove_file (SongDB     *db,
	     const char *filename)
{
  GList *l;
  Song *song;
  
  g_return_val_if_fail (IS_SONG_DB (db), FALSE);
  g_return_val_if_fail (filename != NULL, FALSE);

  for (l = db->songs; l; l = l->next)
    {
      song = l->data;

      if (strcmp (song->filename, filename) == 0)
	{
	  song_db_remove_song (db, song);
	  return TRUE;
	}
    }
  
  return FALSE;
}

int
main (int argc, char **argv)
{
  poptContext optCon;
  char *database_filename;
  SongDB *db;
  int append = 0;
  int remove = 0;
  int export = 0;
  int import = 0;

  struct poptOption optionsTable[] = {
    { "append", 'a', POPT_ARG_NONE, &append, 0, "append to database" },
    { "remove", 'r', POPT_ARG_NONE, &remove, 0, "remove from database" },
    { "export", 'e', POPT_ARG_NONE, &export, 0, "export database" },
    { "import", 'i', POPT_ARG_NONE, &import, 0, "import database" },
    POPT_AUTOHELP
    POPT_TABLEEND
  };
  
  optCon = poptGetContext (NULL, argc, (const char**)argv,
			   optionsTable, 0);

  if (argc < 2)
    {
      poptPrintUsage (optCon, stderr, 0);
      poptFreeContext (optCon);
      exit (0);
    }
  
  while (poptGetNextOpt (optCon) >= 0);

  g_type_init ();
  
  database_filename = g_build_filename (g_get_home_dir (),
					".gnome2", "jamboree", "songs.db",
					NULL);

  db = song_db_new (database_filename);
  if (!db)
    return 1;

  song_db_read_songs (db);
  
  if (export)
    {
      const char *filename;
      
      if (!poptPeekArg (optCon))
	{
	  fprintf (stderr, "-e requires a filename.\n");
	  return 2;
	}
      
      filename = poptGetArg (optCon);
      song_db_export (db, filename);
    }
  else if (import)
    {
      const char *filename;
      
      if (!poptPeekArg (optCon))
	{
	  fprintf (stderr, "-i requires a filename.\n");
	  return 2;
	}
      
      filename = poptGetArg (optCon);
      song_db_import (db, filename);
    }
  else if (append && remove)
    fprintf (stderr, "Can't append and remove at the same time.\n");
  else if (append || remove)
    {
      const char *filename;
      if (!poptPeekArg (optCon))
	{
	  fprintf (stderr, "-a and -r requires at least one argument.\n");
	  return 2;
	}
      
      while ((filename = poptGetArg (optCon)) != NULL)
	if (append)
	  song_db_add_file (db, filename);
	else
	  remove_file (db, filename);
    }
  else
    poptPrintUsage (optCon, stderr, 0);
  
  poptFreeContext (optCon);
  return 0;
}
