/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2004 Imendio HB
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <glib/gi18n.h>
#include <gst/gst.h>
#include <gst/gsttag.h>
#include "tag-reader.h"
#include "song-private.h"


typedef enum {
	PIPELINE_MP3,
	PIPELINE_OGG
} PipelineType;

typedef struct {
	PipelineType  type;

	GstElement   *pipeline;
	GstElement   *source;
	GstElement   *sink;

	GstElement   *tag_element;

	gboolean      got_eos;
	gboolean      got_handoff;
	gboolean      got_error;
	gboolean      got_tags;

	TagReader    *reader;
} PipelineData;

struct _TagReader {
	PipelineData *mp3_pipeline;
	PipelineData *ogg_pipeline;
	
	char         *title;
	char         *artist;
	char         *album;
	char         *genre;
	int           duration;
	int           track;
	int           year;
};

#define IS_TAG(tag,x) (strcmp(tag,x)==0)


static char *
get_title_from_path (const char *path)
{
	char     *utf8;
	char     *tmp;
	char     *p;
	gunichar  c;
  
	utf8 = g_filename_to_utf8 (path, -1, NULL, NULL, NULL);
	if (utf8) {
		tmp = g_path_get_basename (utf8);
		g_free (utf8);

		if (g_str_has_suffix (tmp, ".mp3") || g_str_has_suffix (tmp, ".ogg") ||
		    g_str_has_suffix (tmp, ".MP3") || g_str_has_suffix (tmp, ".OGG")) {
			p = strrchr (tmp, '.'); 
			if (p) {
				*p = 0;
			}
		}
      
		p = tmp;
		while (*p) {
			c = g_utf8_get_char (p);
	  
			if (c == '_') {
				*p = ' ';
			}
			
			p = g_utf8_next_char (p);
		}

		/* Extra check to catch bugs in this function. */
		if (g_utf8_validate (tmp, -1, NULL)) {
			return tmp;
		}

		g_free (tmp);
		return g_strdup (_("Invalid Unicode"));
	}

	return NULL;
}

static void
reader_load_tag (const GstTagList *list,
		 const char       *tag,
		 PipelineData     *data)
{
	int           count;
	const GValue *val;

	count = gst_tag_list_get_tag_size (list, tag);
	if (count < 1) {
		return;
	}

	val = gst_tag_list_get_value_index (list, tag, 0);

	/*g_print ("Tag: %s\n", tag);*/
	
	if (IS_TAG (tag, GST_TAG_TITLE)) {
		data->reader->title = g_value_dup_string (val);
		/*g_print ("%s\n", data->reader->title);*/
	}
	else if (IS_TAG (tag, GST_TAG_ALBUM)) {
		data->reader->album = g_value_dup_string (val);
	}
	else if (IS_TAG (tag, GST_TAG_ARTIST)) {
		data->reader->artist = g_value_dup_string (val);
	}
	else if (IS_TAG (tag, GST_TAG_GENRE)) {
		data->reader->genre = g_value_dup_string (val);
	}
	else if (IS_TAG (tag, GST_TAG_DURATION)) {
		data->reader->duration = g_value_get_uint64 (val) / GST_SECOND;
	}
	else if (IS_TAG (tag, GST_TAG_TRACK_NUMBER)) {
		data->reader->track = g_value_get_uint (val);
	}
	else if (IS_TAG (tag, GST_TAG_DATE)) {
		GDate     *date;
		GDateYear  year;

		date = g_date_new_julian (g_value_get_uint (val));
		year = g_date_get_year (date);
		
		data->reader->year = year;

		g_date_free (date);
	}
	/* FIXME: more... */
}

static void
reader_found_tag_cb (GObject      *pipeline,
		     GstElement   *element,
		     GstTagList   *tags,
		     PipelineData *data)
{
	if (element != data->tag_element) {
		return;
	}
	
	gst_tag_list_foreach (tags, (GstTagForeachFunc) reader_load_tag, data);
}

static void
reader_error_cb (GstElement   *element,
		 GstElement   *source,
		 GError       *error,
		 char         *debug,
		 PipelineData *data)
{
	g_print ("error: %s ", error->message);
	data->got_error = TRUE;
}

static void
reader_eos_cb (GstElement   *element,
	       PipelineData *data)
{
	gst_element_set_state (data->pipeline, GST_STATE_NULL);
	data->got_eos = TRUE;
}

static void
reader_handoff_cb (GstElement   *sink,
		   GstBuffer    *buf,
		   GstPad       *pad,
		   PipelineData *data)
{
	data->got_handoff = TRUE;
}

static void
mp3_pipeline_new_pad_cb (GstElement *id3demux, GstPad *pad, PipelineData *data) 
{
	GstElement *mad;

	gst_element_set_state (data->pipeline, GST_STATE_PAUSED);

	mad = gst_bin_get_by_name (GST_BIN (data->pipeline), "mad");
	gst_pad_link (pad, gst_element_get_pad (mad, "sink"));

	gst_element_set_locked_state (data->sink, FALSE);
	gst_element_set_locked_state (mad, FALSE);

	gst_element_set_state (data->pipeline, GST_STATE_PLAYING);
}

static gboolean
mp3_pipeline_setup (PipelineData *data)
{
	GstElement *mad;
	GstElement *id3demux;

	/* id3demux */
	id3demux = gst_bin_get_by_name (GST_BIN (data->pipeline), "id3demux");
	if (id3demux) {
		gst_bin_remove (GST_BIN (data->pipeline), id3demux);
	}
	
	id3demux = gst_element_factory_make ("id3demux", "id3demux");
	if (!id3demux) {
		return FALSE;
	}
	
	gst_bin_add (GST_BIN (data->pipeline), id3demux);
	data->tag_element = id3demux;

	g_signal_connect (id3demux,
			  "new_pad",
			  G_CALLBACK (mp3_pipeline_new_pad_cb),
			  data);

	if (!gst_element_link (data->source, id3demux)) {
		return FALSE;
	}

	/* mad */
	mad = gst_bin_get_by_name (GST_BIN (data->pipeline), "mad");
	if (mad) {
		gst_bin_remove (GST_BIN (data->pipeline), mad);
	}
	
	mad = gst_element_factory_make ("mad", "mad");
	if (!mad) {
		return FALSE;
	}
	gst_bin_add (GST_BIN (data->pipeline), mad);

	if (!gst_element_link (mad, data->sink)) {
		return FALSE;
	}

	gst_element_set_locked_state (data->sink, TRUE);
	gst_element_set_locked_state (mad, TRUE);

	return TRUE;
}

static void
ogg_pipeline_new_pad_cb (GstElement *oggdemux, GstPad *pad, PipelineData *data) 
{
	GstElement *vorbisdec;
	
	gst_element_set_state (data->pipeline, GST_STATE_PAUSED);

	vorbisdec = gst_bin_get_by_name (GST_BIN (data->pipeline), "vorbisdec");
	gst_pad_link (pad, gst_element_get_pad (vorbisdec, "sink"));

	gst_element_set_locked_state (data->sink, FALSE);
	gst_element_set_locked_state (vorbisdec, FALSE);

	gst_element_set_state (data->pipeline, GST_STATE_PLAYING);
}

static gboolean
ogg_pipeline_setup (PipelineData *data)
{
	GstElement *vorbisdec;
	GstElement *oggdemux;

	/* oggdemux */
	oggdemux = gst_bin_get_by_name (GST_BIN (data->pipeline), "oggdemux");
	if (oggdemux) {
		gst_bin_remove (GST_BIN (data->pipeline), oggdemux);
	}
	
	oggdemux = gst_element_factory_make ("oggdemux", "oggdemux");
	if (!oggdemux) {
		return FALSE;
	}

	g_signal_connect (oggdemux,
			  "new_pad",
			  G_CALLBACK (ogg_pipeline_new_pad_cb),
			  data);
	
	gst_bin_add (GST_BIN (data->pipeline), oggdemux);

	if (!gst_element_link (data->source, oggdemux)) {
		return FALSE;
	}

	/* vorbisdec */
	vorbisdec = gst_bin_get_by_name (GST_BIN (data->pipeline), "vorbisdec");
	if (vorbisdec) {
		gst_bin_remove (GST_BIN (data->pipeline), vorbisdec);
	}
	
	vorbisdec = gst_element_factory_make ("vorbisdec", "vorbisdec");
	if (!vorbisdec) {
		return FALSE;
	}
	gst_bin_add (GST_BIN (data->pipeline), vorbisdec);
	data->tag_element = vorbisdec;

	if (!gst_element_link (vorbisdec, data->sink)) {
		return FALSE;
	}

	gst_element_set_locked_state (data->sink, TRUE);
	gst_element_set_locked_state (vorbisdec, TRUE);

	return TRUE;
}

static void
reader_reset (TagReader *reader)
{
	reader->title = NULL;
	reader->artist = NULL;
	reader->album = NULL;
	reader->genre = NULL;
	reader->duration = 0;
	reader->track = 0;
}

static gboolean
reader_setup_pipeline (PipelineData *data, const char *path)
{
	g_object_set (data->source, "location", path, NULL);

	data->got_handoff = FALSE;
	data->got_eos = FALSE;
	data->got_error = FALSE;

	switch (data->type) {
	case PIPELINE_OGG:
		return ogg_pipeline_setup (data);
	case PIPELINE_MP3:
		return mp3_pipeline_setup (data);
	}

	return FALSE;
}

static PipelineData *
reader_create_pipeline (TagReader *reader, PipelineType type)
{
	PipelineData *data;

	data = g_new0 (PipelineData, 1);

	data->type = type;
	data->reader = reader;

	data->pipeline = gst_pipeline_new ("pipeline");
	if (!data->pipeline) {
		g_free (data);
		return NULL;
	}

	g_signal_connect (data->pipeline,
			  "error",
			  G_CALLBACK (reader_error_cb),
			  data);
	g_signal_connect (data->pipeline,
			  "found_tag",
			  G_CALLBACK (reader_found_tag_cb),
			  data);

	data->source = gst_element_factory_make ("filesrc", "source");
	if (!data->source) {
		goto fail;
	}
	gst_bin_add (GST_BIN (data->pipeline), data->source);

	data->sink = gst_element_factory_make ("fakesink", "fakesink");
	if (!data->sink) {
		goto fail;
	}
	gst_bin_add (GST_BIN (data->pipeline), data->sink);
	gst_element_set_locked_state (data->sink, TRUE);
	
	g_object_set (data->sink,
		      "signal_handoffs", TRUE,
		      NULL);
	g_signal_connect (data->sink,
			  "handoff",
			  G_CALLBACK (reader_handoff_cb),
			  data);
	g_signal_connect (data->sink,
			  "eos",
			  G_CALLBACK (reader_eos_cb),
			  data);

	return data;

 fail:
	g_object_unref (data->pipeline);
	g_free (data);

	return NULL;
}

static void
reader_fixup_tags (TagReader *reader, Song *song)
{
	if (!reader->title) {
		_song_set_title (song, get_title_from_path (song_get_path (song)));
	}

	if (!reader->artist) {
		_song_set_artist_const (song, _("Unknown artist"));
	}

	if (!reader->album) {
		_song_set_album_const (song, _("Unknown album"));
	}

	if (!reader->genre) {
		_song_set_genre (song, g_strdup (""));
	}
}

static Song *
reader_read_file (TagReader  *reader,
		  const char *path)
{
	PipelineData *pipeline;
	GstFormat     format = GST_FORMAT_TIME;
	gint64        duration;
	Song         *song;
	time_t        now;
	struct stat   buf;
	int           i;

	if (g_str_has_suffix (path, ".mp3") || g_str_has_suffix (path, ".MP3")) {
		pipeline = reader->mp3_pipeline;
	}
	else if (g_str_has_suffix (path, ".ogg") || g_str_has_suffix (path, ".OGG")) {
		pipeline = reader->ogg_pipeline;
	} else {
		pipeline = NULL;
	}

	if (!pipeline) {
		return NULL;
	}

	if (!reader_setup_pipeline (pipeline, path)) {
		return NULL;
	}
	
	gst_element_set_state (pipeline->pipeline, GST_STATE_PLAYING);

	i = 0;
	while (gst_bin_iterate (GST_BIN (pipeline->pipeline)) &&
	       !pipeline->got_handoff && !pipeline->got_eos && !pipeline->got_error && i++ < 1000) {
	}
	
	if (reader->duration == 0 && pipeline->got_handoff) {
		if (gst_element_query (pipeline->sink, GST_QUERY_TOTAL, &format, &duration)) {
			reader->duration = duration / GST_SECOND;
		}
	}

	gst_element_set_state (pipeline->pipeline, GST_STATE_NULL);

	song = song_new (path);

	/* These move the ownership of the strings to the song. */
	_song_set_title (song, reader->title);
	_song_set_album (song, reader->album);
	_song_set_artist (song, reader->artist);
	_song_set_duration (song, reader->duration);
	_song_set_genre (song, reader->genre);
	_song_set_track (song, reader->track);
	_song_set_year (song, reader->year);

	now = time (NULL);
	_song_set_time_added (song, now);

	if (stat (path, &buf) == 0) {
		_song_set_size (song, buf.st_size);
		_song_set_time_modified (song, buf.st_mtime);
	}
	
	reader_fixup_tags (reader, song);
	reader_reset (reader);

	return song;
}

static gboolean
reader_add_dir (TagReader              *reader,
		const char             *path,
		TagReaderSongFoundFunc  song_found_func,
		TagReaderSongReadFunc   song_read_func,
		TagReaderProgressFunc   progress_func,
		gpointer                user_data)
{
	GDir       *dir;
	const char *name;
	char       *full;
	Song       *song;
	gboolean    ret;

	dir = g_dir_open (path, 0, NULL);
	if (!dir) {
		return FALSE;
	}

	ret = FALSE;
	
	while ((name = g_dir_read_name (dir))) {
		full = g_build_filename (path, name, NULL);
      
		if (g_file_test (full, G_FILE_TEST_IS_DIR)){
			if (!progress_func (reader, full, user_data)) {
				g_free (full);
				g_dir_close (dir);
				
				return FALSE;
			}
			
			ret |= reader_add_dir (reader, full,
					       song_found_func,
					       song_read_func,
					       progress_func,
					       user_data);
		} else {
			if (!progress_func (reader, full, user_data)) {
				g_free (full);
				g_dir_close (dir);
				
				return FALSE;
			}

			if (!song_found_func || song_found_func (reader, full, user_data)) {
				song = reader_read_file (reader, full);
				if (song) {
					ret = TRUE;
					
					if (song_read_func) {
						song_read_func (reader, song, user_data);
					}
					song_unref (song);
				}
			}
		}
		
		g_free (full);
	}
	
	g_dir_close (dir);

	return ret;
}

gboolean
tag_reader_read_path (TagReader              *reader,
		      const char             *path,
		      TagReaderSongFoundFunc  song_found_func,
		      TagReaderSongReadFunc   song_read_func,
		      TagReaderProgressFunc   progress_func,
		      gpointer                user_data)
{
	Song *song;

	if (g_file_test (path, G_FILE_TEST_IS_DIR)){
		return reader_add_dir (reader,
				       path,
				       song_found_func,
				       song_read_func,
				       progress_func,
				       user_data);
	} else {
		if (!song_found_func || song_found_func (reader, path, user_data)) {
			song = reader_read_file (reader, path);

			if (song) {
				if (song_read_func) {
					song_read_func (reader, song, user_data);
				}
				song_unref (song);

				return TRUE;
			} else {
				/*g_print ("Couldn't read %s\n", path);*/
			}
		}
	}

	return FALSE;
}

TagReader *
tag_reader_new (void)
{
	TagReader *reader;
	
	reader = g_new0 (TagReader, 1);

#ifdef HAVE_MP3
	reader->mp3_pipeline = reader_create_pipeline (reader, PIPELINE_MP3);
#endif

#ifdef HAVE_OGG
	reader->ogg_pipeline = reader_create_pipeline (reader, PIPELINE_OGG);
#endif

	return reader;
}


