/* testhashmap.vala
 *
 * Copyright (C) 2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <tests/testarraylist.h>
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/collection.h>
#include <gee/iterable.h>
#include <gee/iterator.h>




static void _test_arraylist_get_gcallback (void);
static void _test_arraylist_set_gcallback (void);
static void _test_arraylist_insert_gcallback (void);
static void _test_arraylist_remove_at_gcallback (void);
static void _test_arraylist_index_of_gcallback (void);
static void _test_arraylist_add_gcallback (void);
static void _test_arraylist_clear_gcallback (void);
static void _test_arraylist_contains_gcallback (void);
static void _test_arraylist_remove_gcallback (void);
static void _test_arraylist_size_gcallback (void);
static void _test_arraylist_iterator_gcallback (void);
static int _vala_strcmp0 (const char * str1, const char * str2);



void test_arraylist_get (void) {
	GeeArrayList* arraylistOfString;
	char* _tmp1;
	char* _tmp4;
	char* _tmp5;
	char* _tmp6;
	arraylistOfString = gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal);
	/* Check get for empty list*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		char* _tmp0;
		_tmp0 = NULL;
		_tmp0 = (char*) gee_list_get ((GeeList*) arraylistOfString, 0);
		_tmp0 = (g_free (_tmp0), NULL);
		(arraylistOfString == NULL) ? NULL : (arraylistOfString = (g_object_unref (arraylistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	/* Check get for valid index in list with one element*/
	gee_collection_add ((GeeCollection*) arraylistOfString, "1");
	_tmp1 = NULL;
	g_assert (_vala_strcmp0 (_tmp1 = (char*) gee_list_get ((GeeList*) arraylistOfString, 0), "1") == 0);
	_tmp1 = (g_free (_tmp1), NULL);
	/* Check get for indexes out of range*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		char* _tmp2;
		_tmp2 = NULL;
		_tmp2 = (char*) gee_list_get ((GeeList*) arraylistOfString, 1);
		_tmp2 = (g_free (_tmp2), NULL);
		(arraylistOfString == NULL) ? NULL : (arraylistOfString = (g_object_unref (arraylistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	/* Check get for invalid index number*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		char* _tmp3;
		_tmp3 = NULL;
		_tmp3 = (char*) gee_list_get ((GeeList*) arraylistOfString, -1);
		_tmp3 = (g_free (_tmp3), NULL);
		(arraylistOfString == NULL) ? NULL : (arraylistOfString = (g_object_unref (arraylistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	/* Check get for valid indexes in list with multiple element*/
	gee_collection_add ((GeeCollection*) arraylistOfString, "2");
	gee_collection_add ((GeeCollection*) arraylistOfString, "3");
	_tmp4 = NULL;
	g_assert (_vala_strcmp0 (_tmp4 = (char*) gee_list_get ((GeeList*) arraylistOfString, 0), "1") == 0);
	_tmp4 = (g_free (_tmp4), NULL);
	_tmp5 = NULL;
	g_assert (_vala_strcmp0 (_tmp5 = (char*) gee_list_get ((GeeList*) arraylistOfString, 1), "2") == 0);
	_tmp5 = (g_free (_tmp5), NULL);
	_tmp6 = NULL;
	g_assert (_vala_strcmp0 (_tmp6 = (char*) gee_list_get ((GeeList*) arraylistOfString, 2), "3") == 0);
	_tmp6 = (g_free (_tmp6), NULL);
	/* Check get if list is cleared and empty again*/
	gee_collection_clear ((GeeCollection*) arraylistOfString);
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		char* _tmp7;
		_tmp7 = NULL;
		_tmp7 = (char*) gee_list_get ((GeeList*) arraylistOfString, 0);
		_tmp7 = (g_free (_tmp7), NULL);
		(arraylistOfString == NULL) ? NULL : (arraylistOfString = (g_object_unref (arraylistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	(arraylistOfString == NULL) ? NULL : (arraylistOfString = (g_object_unref (arraylistOfString), NULL));
}


void test_arraylist_set (void) {
	GeeArrayList* arraylistOfString;
	char* _tmp0;
	char* _tmp1;
	arraylistOfString = gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal);
	/* Check set when list is empty.*/
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 0);
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_list_set ((GeeList*) arraylistOfString, 0, "0");
		(arraylistOfString == NULL) ? NULL : (arraylistOfString = (g_object_unref (arraylistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 0);
	/* Check set when one item is in list*/
	gee_collection_add ((GeeCollection*) arraylistOfString, "1");
	/* Add item "1"*/
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 1);
	_tmp0 = NULL;
	g_assert (_vala_strcmp0 (_tmp0 = (char*) gee_list_get ((GeeList*) arraylistOfString, 0), "1") == 0);
	_tmp0 = (g_free (_tmp0), NULL);
	gee_list_set ((GeeList*) arraylistOfString, 0, "2");
	/* Set the item to value 2*/
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 1);
	_tmp1 = NULL;
	g_assert (_vala_strcmp0 (_tmp1 = (char*) gee_list_get ((GeeList*) arraylistOfString, 0), "2") == 0);
	_tmp1 = (g_free (_tmp1), NULL);
	/* Check set when index out of range*/
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 1);
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_list_set ((GeeList*) arraylistOfString, 1, "0");
		(arraylistOfString == NULL) ? NULL : (arraylistOfString = (g_object_unref (arraylistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 1);
	(arraylistOfString == NULL) ? NULL : (arraylistOfString = (g_object_unref (arraylistOfString), NULL));
}


void test_arraylist_insert (void) {
	GeeArrayList* arraylistOfString;
	char* _tmp0;
	char* _tmp1;
	char* _tmp2;
	char* _tmp3;
	char* _tmp4;
	char* _tmp5;
	char* _tmp6;
	char* _tmp7;
	char* _tmp8;
	char* _tmp9;
	arraylistOfString = gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal);
	/* Check inserting in empty list
	 Inserting at index 1*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_list_insert ((GeeList*) arraylistOfString, 1, "0");
		(arraylistOfString == NULL) ? NULL : (arraylistOfString = (g_object_unref (arraylistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	/* Inserting at index 0*/
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 0);
	gee_list_insert ((GeeList*) arraylistOfString, 0, "10");
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 1);
	_tmp0 = NULL;
	g_assert (_vala_strcmp0 (_tmp0 = (char*) gee_list_get ((GeeList*) arraylistOfString, 0), "10") == 0);
	_tmp0 = (g_free (_tmp0), NULL);
	/* Check insert to the beginning*/
	gee_list_insert ((GeeList*) arraylistOfString, 0, "5");
	_tmp1 = NULL;
	g_assert (_vala_strcmp0 (_tmp1 = (char*) gee_list_get ((GeeList*) arraylistOfString, 0), "5") == 0);
	_tmp1 = (g_free (_tmp1), NULL);
	_tmp2 = NULL;
	g_assert (_vala_strcmp0 (_tmp2 = (char*) gee_list_get ((GeeList*) arraylistOfString, 1), "10") == 0);
	_tmp2 = (g_free (_tmp2), NULL);
	/* Check insert in between*/
	gee_list_insert ((GeeList*) arraylistOfString, 1, "7");
	_tmp3 = NULL;
	g_assert (_vala_strcmp0 (_tmp3 = (char*) gee_list_get ((GeeList*) arraylistOfString, 0), "5") == 0);
	_tmp3 = (g_free (_tmp3), NULL);
	_tmp4 = NULL;
	g_assert (_vala_strcmp0 (_tmp4 = (char*) gee_list_get ((GeeList*) arraylistOfString, 1), "7") == 0);
	_tmp4 = (g_free (_tmp4), NULL);
	_tmp5 = NULL;
	g_assert (_vala_strcmp0 (_tmp5 = (char*) gee_list_get ((GeeList*) arraylistOfString, 2), "10") == 0);
	_tmp5 = (g_free (_tmp5), NULL);
	/* Check insert into index out of current range*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_list_insert ((GeeList*) arraylistOfString, 4, "20");
		(arraylistOfString == NULL) ? NULL : (arraylistOfString = (g_object_unref (arraylistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	/* Check insert to the end*/
	gee_list_insert ((GeeList*) arraylistOfString, 3, "20");
	_tmp6 = NULL;
	g_assert (_vala_strcmp0 (_tmp6 = (char*) gee_list_get ((GeeList*) arraylistOfString, 0), "5") == 0);
	_tmp6 = (g_free (_tmp6), NULL);
	_tmp7 = NULL;
	g_assert (_vala_strcmp0 (_tmp7 = (char*) gee_list_get ((GeeList*) arraylistOfString, 1), "7") == 0);
	_tmp7 = (g_free (_tmp7), NULL);
	_tmp8 = NULL;
	g_assert (_vala_strcmp0 (_tmp8 = (char*) gee_list_get ((GeeList*) arraylistOfString, 2), "10") == 0);
	_tmp8 = (g_free (_tmp8), NULL);
	_tmp9 = NULL;
	g_assert (_vala_strcmp0 (_tmp9 = (char*) gee_list_get ((GeeList*) arraylistOfString, 3), "20") == 0);
	_tmp9 = (g_free (_tmp9), NULL);
	/* Check insert into invalid index*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_list_insert ((GeeList*) arraylistOfString, -1, "0");
		(arraylistOfString == NULL) ? NULL : (arraylistOfString = (g_object_unref (arraylistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	(arraylistOfString == NULL) ? NULL : (arraylistOfString = (g_object_unref (arraylistOfString), NULL));
}


void test_arraylist_remove_at (void) {
	GeeArrayList* arraylistOfString;
	char* _tmp0;
	char* _tmp1;
	char* _tmp2;
	char* _tmp3;
	char* _tmp4;
	char* _tmp5;
	char* _tmp6;
	char* _tmp7;
	char* _tmp8;
	arraylistOfString = gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal);
	/* Check removing in empty list*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_list_remove_at ((GeeList*) arraylistOfString, 0);
		(arraylistOfString == NULL) ? NULL : (arraylistOfString = (g_object_unref (arraylistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_list_remove_at ((GeeList*) arraylistOfString, 1);
		(arraylistOfString == NULL) ? NULL : (arraylistOfString = (g_object_unref (arraylistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	/* add 5 items*/
	gee_collection_add ((GeeCollection*) arraylistOfString, "1");
	gee_collection_add ((GeeCollection*) arraylistOfString, "2");
	gee_collection_add ((GeeCollection*) arraylistOfString, "3");
	gee_collection_add ((GeeCollection*) arraylistOfString, "4");
	gee_collection_add ((GeeCollection*) arraylistOfString, "5");
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 5);
	/* Check remove_at first*/
	gee_list_remove_at ((GeeList*) arraylistOfString, 0);
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 4);
	_tmp0 = NULL;
	g_assert (_vala_strcmp0 (_tmp0 = (char*) gee_list_get ((GeeList*) arraylistOfString, 0), "2") == 0);
	_tmp0 = (g_free (_tmp0), NULL);
	_tmp1 = NULL;
	g_assert (_vala_strcmp0 (_tmp1 = (char*) gee_list_get ((GeeList*) arraylistOfString, 1), "3") == 0);
	_tmp1 = (g_free (_tmp1), NULL);
	_tmp2 = NULL;
	g_assert (_vala_strcmp0 (_tmp2 = (char*) gee_list_get ((GeeList*) arraylistOfString, 2), "4") == 0);
	_tmp2 = (g_free (_tmp2), NULL);
	_tmp3 = NULL;
	g_assert (_vala_strcmp0 (_tmp3 = (char*) gee_list_get ((GeeList*) arraylistOfString, 3), "5") == 0);
	_tmp3 = (g_free (_tmp3), NULL);
	/* Check remove_at last*/
	gee_list_remove_at ((GeeList*) arraylistOfString, 3);
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 3);
	_tmp4 = NULL;
	g_assert (_vala_strcmp0 (_tmp4 = (char*) gee_list_get ((GeeList*) arraylistOfString, 0), "2") == 0);
	_tmp4 = (g_free (_tmp4), NULL);
	_tmp5 = NULL;
	g_assert (_vala_strcmp0 (_tmp5 = (char*) gee_list_get ((GeeList*) arraylistOfString, 1), "3") == 0);
	_tmp5 = (g_free (_tmp5), NULL);
	_tmp6 = NULL;
	g_assert (_vala_strcmp0 (_tmp6 = (char*) gee_list_get ((GeeList*) arraylistOfString, 2), "4") == 0);
	_tmp6 = (g_free (_tmp6), NULL);
	/* Check remove_at in between*/
	gee_list_remove_at ((GeeList*) arraylistOfString, 1);
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 2);
	_tmp7 = NULL;
	g_assert (_vala_strcmp0 (_tmp7 = (char*) gee_list_get ((GeeList*) arraylistOfString, 0), "2") == 0);
	_tmp7 = (g_free (_tmp7), NULL);
	_tmp8 = NULL;
	g_assert (_vala_strcmp0 (_tmp8 = (char*) gee_list_get ((GeeList*) arraylistOfString, 1), "4") == 0);
	_tmp8 = (g_free (_tmp8), NULL);
	/* Check remove_at when index out of range*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_list_remove_at ((GeeList*) arraylistOfString, 2);
		(arraylistOfString == NULL) ? NULL : (arraylistOfString = (g_object_unref (arraylistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	/* Check remove_at when invalid index*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_list_remove_at ((GeeList*) arraylistOfString, -1);
		(arraylistOfString == NULL) ? NULL : (arraylistOfString = (g_object_unref (arraylistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	(arraylistOfString == NULL) ? NULL : (arraylistOfString = (g_object_unref (arraylistOfString), NULL));
}


void test_arraylist_index_of (void) {
	GeeArrayList* arraylistOfString;
	GeeArrayList* arraylistOfInt;
	GeeArrayList* arraylistOfObjects;
	GObject* object1;
	GObject* object2;
	GObject* object3;
	GObject* object4;
	arraylistOfString = gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_equal);
	/* Check empty list*/
	g_assert (gee_list_index_of ((GeeList*) arraylistOfString, "one") == (-1));
	/* Check one item*/
	gee_collection_add ((GeeCollection*) arraylistOfString, "one");
	g_assert (gee_list_index_of ((GeeList*) arraylistOfString, "one") == 0);
	g_assert (gee_list_index_of ((GeeList*) arraylistOfString, "two") == (-1));
	/* Check more items*/
	gee_collection_add ((GeeCollection*) arraylistOfString, "two");
	gee_collection_add ((GeeCollection*) arraylistOfString, "three");
	gee_collection_add ((GeeCollection*) arraylistOfString, "four");
	g_assert (gee_list_index_of ((GeeList*) arraylistOfString, "one") == 0);
	g_assert (gee_list_index_of ((GeeList*) arraylistOfString, "two") == 1);
	g_assert (gee_list_index_of ((GeeList*) arraylistOfString, "three") == 2);
	g_assert (gee_list_index_of ((GeeList*) arraylistOfString, "four") == 3);
	g_assert (gee_list_index_of ((GeeList*) arraylistOfString, "five") == (-1));
	/* Check list of ints*/
	arraylistOfInt = gee_array_list_new (G_TYPE_INT, NULL, NULL, g_direct_equal);
	/* Check more items*/
	gee_collection_add ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (1));
	gee_collection_add ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (2));
	gee_collection_add ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (3));
	g_assert (gee_list_index_of ((GeeList*) arraylistOfInt, GINT_TO_POINTER (1)) == 0);
	g_assert (gee_list_index_of ((GeeList*) arraylistOfInt, GINT_TO_POINTER (2)) == 1);
	g_assert (gee_list_index_of ((GeeList*) arraylistOfInt, GINT_TO_POINTER (3)) == 2);
	g_assert (gee_list_index_of ((GeeList*) arraylistOfInt, GINT_TO_POINTER (4)) == (-1));
	/* Check list of objects*/
	arraylistOfObjects = gee_array_list_new (G_TYPE_OBJECT, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_equal);
	object1 = g_object_new (G_TYPE_OBJECT, NULL);
	object2 = g_object_new (G_TYPE_OBJECT, NULL);
	object3 = g_object_new (G_TYPE_OBJECT, NULL);
	object4 = g_object_new (G_TYPE_OBJECT, NULL);
	gee_collection_add ((GeeCollection*) arraylistOfObjects, object1);
	gee_collection_add ((GeeCollection*) arraylistOfObjects, object2);
	gee_collection_add ((GeeCollection*) arraylistOfObjects, object3);
	g_assert (gee_list_index_of ((GeeList*) arraylistOfObjects, object1) == 0);
	g_assert (gee_list_index_of ((GeeList*) arraylistOfObjects, object2) == 1);
	g_assert (gee_list_index_of ((GeeList*) arraylistOfObjects, object3) == 2);
	(arraylistOfString == NULL) ? NULL : (arraylistOfString = (g_object_unref (arraylistOfString), NULL));
	(arraylistOfInt == NULL) ? NULL : (arraylistOfInt = (g_object_unref (arraylistOfInt), NULL));
	(arraylistOfObjects == NULL) ? NULL : (arraylistOfObjects = (g_object_unref (arraylistOfObjects), NULL));
	(object1 == NULL) ? NULL : (object1 = (g_object_unref (object1), NULL));
	(object2 == NULL) ? NULL : (object2 = (g_object_unref (object2), NULL));
	(object3 == NULL) ? NULL : (object3 = (g_object_unref (object3), NULL));
	(object4 == NULL) ? NULL : (object4 = (g_object_unref (object4), NULL));
}


void test_arraylist_add (void) {
	GeeArrayList* arraylistOfString;
	char* _tmp0;
	char* _tmp1;
	char* _tmp2;
	char* _tmp3;
	GeeArrayList* arrayListOfInt;
	GeeArrayList* arrayListOfGLibObject;
	GObject* fooObject;
	arraylistOfString = gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_equal);
	gee_collection_add ((GeeCollection*) arraylistOfString, "42");
	g_assert (gee_collection_contains ((GeeCollection*) arraylistOfString, "42"));
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 1);
	/* check for correct order of elements*/
	gee_collection_add ((GeeCollection*) arraylistOfString, "43");
	gee_collection_add ((GeeCollection*) arraylistOfString, "44");
	gee_collection_add ((GeeCollection*) arraylistOfString, "45");
	_tmp0 = NULL;
	g_assert (_vala_strcmp0 (_tmp0 = (char*) gee_list_get ((GeeList*) arraylistOfString, 0), "42") == 0);
	_tmp0 = (g_free (_tmp0), NULL);
	_tmp1 = NULL;
	g_assert (_vala_strcmp0 (_tmp1 = (char*) gee_list_get ((GeeList*) arraylistOfString, 1), "43") == 0);
	_tmp1 = (g_free (_tmp1), NULL);
	_tmp2 = NULL;
	g_assert (_vala_strcmp0 (_tmp2 = (char*) gee_list_get ((GeeList*) arraylistOfString, 2), "44") == 0);
	_tmp2 = (g_free (_tmp2), NULL);
	_tmp3 = NULL;
	g_assert (_vala_strcmp0 (_tmp3 = (char*) gee_list_get ((GeeList*) arraylistOfString, 3), "45") == 0);
	_tmp3 = (g_free (_tmp3), NULL);
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 4);
	/* check adding of ints*/
	arrayListOfInt = gee_array_list_new (G_TYPE_INT, NULL, NULL, g_direct_equal);
	gee_collection_add ((GeeCollection*) arrayListOfInt, GINT_TO_POINTER (42));
	g_assert (gee_collection_contains ((GeeCollection*) arrayListOfInt, GINT_TO_POINTER (42)));
	g_assert (gee_collection_get_size ((GeeCollection*) arrayListOfInt) == 1);
	/* check adding of objects*/
	arrayListOfGLibObject = gee_array_list_new (G_TYPE_OBJECT, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_equal);
	fooObject = g_object_new (G_TYPE_OBJECT, NULL);
	gee_collection_add ((GeeCollection*) arrayListOfGLibObject, fooObject);
	g_assert (gee_collection_contains ((GeeCollection*) arrayListOfGLibObject, fooObject));
	g_assert (gee_collection_get_size ((GeeCollection*) arrayListOfGLibObject) == 1);
	(arraylistOfString == NULL) ? NULL : (arraylistOfString = (g_object_unref (arraylistOfString), NULL));
	(arrayListOfInt == NULL) ? NULL : (arrayListOfInt = (g_object_unref (arrayListOfInt), NULL));
	(arrayListOfGLibObject == NULL) ? NULL : (arrayListOfGLibObject = (g_object_unref (arrayListOfGLibObject), NULL));
	(fooObject == NULL) ? NULL : (fooObject = (g_object_unref (fooObject), NULL));
}


void test_arraylist_clear (void) {
	GeeArrayList* arraylistOfString;
	arraylistOfString = gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_equal);
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 0);
	/* Check clear on empty list*/
	gee_collection_clear ((GeeCollection*) arraylistOfString);
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 0);
	/* Check clear one item*/
	gee_collection_add ((GeeCollection*) arraylistOfString, "1");
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 1);
	gee_collection_clear ((GeeCollection*) arraylistOfString);
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 0);
	/* Check clear multiple items*/
	gee_collection_add ((GeeCollection*) arraylistOfString, "1");
	gee_collection_add ((GeeCollection*) arraylistOfString, "2");
	gee_collection_add ((GeeCollection*) arraylistOfString, "3");
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 3);
	gee_collection_clear ((GeeCollection*) arraylistOfString);
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 0);
	(arraylistOfString == NULL) ? NULL : (arraylistOfString = (g_object_unref (arraylistOfString), NULL));
}


void test_arraylist_contains (void) {
	GeeArrayList* arraylistOfString;
	GeeArrayList* arraylistOfInt;
	arraylistOfString = gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_equal);
	/* Check on empty list*/
	g_assert (!gee_collection_contains ((GeeCollection*) arraylistOfString, "1"));
	/* Check items*/
	gee_collection_add ((GeeCollection*) arraylistOfString, "10");
	g_assert (gee_collection_contains ((GeeCollection*) arraylistOfString, "10"));
	g_assert (!gee_collection_contains ((GeeCollection*) arraylistOfString, "20"));
	g_assert (!gee_collection_contains ((GeeCollection*) arraylistOfString, "30"));
	gee_collection_add ((GeeCollection*) arraylistOfString, "20");
	g_assert (gee_collection_contains ((GeeCollection*) arraylistOfString, "10"));
	g_assert (gee_collection_contains ((GeeCollection*) arraylistOfString, "20"));
	g_assert (!gee_collection_contains ((GeeCollection*) arraylistOfString, "30"));
	gee_collection_add ((GeeCollection*) arraylistOfString, "30");
	g_assert (gee_collection_contains ((GeeCollection*) arraylistOfString, "10"));
	g_assert (gee_collection_contains ((GeeCollection*) arraylistOfString, "20"));
	g_assert (gee_collection_contains ((GeeCollection*) arraylistOfString, "30"));
	/* Clear and recheck*/
	gee_collection_clear ((GeeCollection*) arraylistOfString);
	g_assert (!gee_collection_contains ((GeeCollection*) arraylistOfString, "10"));
	g_assert (!gee_collection_contains ((GeeCollection*) arraylistOfString, "20"));
	g_assert (!gee_collection_contains ((GeeCollection*) arraylistOfString, "30"));
	arraylistOfInt = gee_array_list_new (G_TYPE_INT, NULL, NULL, g_direct_equal);
	/* Check items*/
	gee_collection_add ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (10));
	g_assert (gee_collection_contains ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (10)));
	g_assert (!gee_collection_contains ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (20)));
	g_assert (!gee_collection_contains ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (30)));
	gee_collection_add ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (20));
	g_assert (gee_collection_contains ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (10)));
	g_assert (gee_collection_contains ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (20)));
	g_assert (!gee_collection_contains ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (30)));
	gee_collection_add ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (30));
	g_assert (gee_collection_contains ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (10)));
	g_assert (gee_collection_contains ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (20)));
	g_assert (gee_collection_contains ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (30)));
	/* Clear and recheck*/
	gee_collection_clear ((GeeCollection*) arraylistOfInt);
	g_assert (!gee_collection_contains ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (10)));
	g_assert (!gee_collection_contains ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (20)));
	g_assert (!gee_collection_contains ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (30)));
	(arraylistOfString == NULL) ? NULL : (arraylistOfString = (g_object_unref (arraylistOfString), NULL));
	(arraylistOfInt == NULL) ? NULL : (arraylistOfInt = (g_object_unref (arraylistOfInt), NULL));
}


void test_arraylist_remove (void) {
	GeeArrayList* arraylistOfString;
	char* _tmp0;
	char* _tmp1;
	char* _tmp2;
	char* _tmp3;
	char* _tmp4;
	char* _tmp5;
	char* _tmp6;
	char* _tmp7;
	char* _tmp8;
	GeeArrayList* arraylistOfInt;
	arraylistOfString = gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_equal);
	/* Check remove if list is empty*/
	gee_collection_remove ((GeeCollection*) arraylistOfString, "42");
	/* Add 5 same elements*/
	gee_collection_add ((GeeCollection*) arraylistOfString, "42");
	gee_collection_add ((GeeCollection*) arraylistOfString, "42");
	gee_collection_add ((GeeCollection*) arraylistOfString, "42");
	gee_collection_add ((GeeCollection*) arraylistOfString, "42");
	gee_collection_add ((GeeCollection*) arraylistOfString, "42");
	/* Check remove one element*/
	gee_collection_remove ((GeeCollection*) arraylistOfString, "42");
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 4);
	g_assert (gee_collection_contains ((GeeCollection*) arraylistOfString, "42"));
	/* Check remove another one element*/
	gee_collection_remove ((GeeCollection*) arraylistOfString, "42");
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 3);
	g_assert (gee_collection_contains ((GeeCollection*) arraylistOfString, "42"));
	/* Clear the list to start from scratch*/
	gee_collection_clear ((GeeCollection*) arraylistOfString);
	/* Add 5 different elements*/
	gee_collection_add ((GeeCollection*) arraylistOfString, "42");
	gee_collection_add ((GeeCollection*) arraylistOfString, "43");
	gee_collection_add ((GeeCollection*) arraylistOfString, "44");
	gee_collection_add ((GeeCollection*) arraylistOfString, "45");
	gee_collection_add ((GeeCollection*) arraylistOfString, "46");
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 5);
	/* Check remove first*/
	gee_collection_remove ((GeeCollection*) arraylistOfString, "42");
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 4);
	_tmp0 = NULL;
	g_assert (_vala_strcmp0 (_tmp0 = (char*) gee_list_get ((GeeList*) arraylistOfString, 0), "43") == 0);
	_tmp0 = (g_free (_tmp0), NULL);
	_tmp1 = NULL;
	g_assert (_vala_strcmp0 (_tmp1 = (char*) gee_list_get ((GeeList*) arraylistOfString, 1), "44") == 0);
	_tmp1 = (g_free (_tmp1), NULL);
	_tmp2 = NULL;
	g_assert (_vala_strcmp0 (_tmp2 = (char*) gee_list_get ((GeeList*) arraylistOfString, 2), "45") == 0);
	_tmp2 = (g_free (_tmp2), NULL);
	_tmp3 = NULL;
	g_assert (_vala_strcmp0 (_tmp3 = (char*) gee_list_get ((GeeList*) arraylistOfString, 3), "46") == 0);
	_tmp3 = (g_free (_tmp3), NULL);
	/* Check remove last*/
	gee_collection_remove ((GeeCollection*) arraylistOfString, "46");
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 3);
	_tmp4 = NULL;
	g_assert (_vala_strcmp0 (_tmp4 = (char*) gee_list_get ((GeeList*) arraylistOfString, 0), "43") == 0);
	_tmp4 = (g_free (_tmp4), NULL);
	_tmp5 = NULL;
	g_assert (_vala_strcmp0 (_tmp5 = (char*) gee_list_get ((GeeList*) arraylistOfString, 1), "44") == 0);
	_tmp5 = (g_free (_tmp5), NULL);
	_tmp6 = NULL;
	g_assert (_vala_strcmp0 (_tmp6 = (char*) gee_list_get ((GeeList*) arraylistOfString, 2), "45") == 0);
	_tmp6 = (g_free (_tmp6), NULL);
	/* Check remove in between*/
	gee_collection_remove ((GeeCollection*) arraylistOfString, "44");
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfString) == 2);
	_tmp7 = NULL;
	g_assert (_vala_strcmp0 (_tmp7 = (char*) gee_list_get ((GeeList*) arraylistOfString, 0), "43") == 0);
	_tmp7 = (g_free (_tmp7), NULL);
	_tmp8 = NULL;
	g_assert (_vala_strcmp0 (_tmp8 = (char*) gee_list_get ((GeeList*) arraylistOfString, 1), "45") == 0);
	_tmp8 = (g_free (_tmp8), NULL);
	/* Check removing of int element*/
	arraylistOfInt = gee_array_list_new (G_TYPE_INT, NULL, NULL, g_direct_equal);
	/* Add 5 different elements*/
	gee_collection_add ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (42));
	gee_collection_add ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (43));
	gee_collection_add ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (44));
	gee_collection_add ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (45));
	gee_collection_add ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (46));
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfInt) == 5);
	/* Remove first*/
	gee_collection_remove ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (42));
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfInt) == 4);
	g_assert (GPOINTER_TO_INT (gee_list_get ((GeeList*) arraylistOfInt, 0)) == 43);
	g_assert (GPOINTER_TO_INT (gee_list_get ((GeeList*) arraylistOfInt, 1)) == 44);
	g_assert (GPOINTER_TO_INT (gee_list_get ((GeeList*) arraylistOfInt, 2)) == 45);
	g_assert (GPOINTER_TO_INT (gee_list_get ((GeeList*) arraylistOfInt, 3)) == 46);
	/* Remove last*/
	gee_collection_remove ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (46));
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfInt) == 3);
	g_assert (GPOINTER_TO_INT (gee_list_get ((GeeList*) arraylistOfInt, 0)) == 43);
	g_assert (GPOINTER_TO_INT (gee_list_get ((GeeList*) arraylistOfInt, 1)) == 44);
	g_assert (GPOINTER_TO_INT (gee_list_get ((GeeList*) arraylistOfInt, 2)) == 45);
	/* Remove in between*/
	gee_collection_remove ((GeeCollection*) arraylistOfInt, GINT_TO_POINTER (44));
	g_assert (gee_collection_get_size ((GeeCollection*) arraylistOfInt) == 2);
	g_assert (GPOINTER_TO_INT (gee_list_get ((GeeList*) arraylistOfInt, 0)) == 43);
	g_assert (GPOINTER_TO_INT (gee_list_get ((GeeList*) arraylistOfInt, 1)) == 45);
	(arraylistOfString == NULL) ? NULL : (arraylistOfString = (g_object_unref (arraylistOfString), NULL));
	(arraylistOfInt == NULL) ? NULL : (arraylistOfInt = (g_object_unref (arraylistOfInt), NULL));
}


void test_arraylist_size (void) {
	GeeArrayList* arraylist;
	arraylist = gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal);
	/* Check empty list*/
	g_assert (gee_collection_get_size ((GeeCollection*) arraylist) == 0);
	/* Check when one item*/
	gee_collection_add ((GeeCollection*) arraylist, "1");
	g_assert (gee_collection_get_size ((GeeCollection*) arraylist) == 1);
	/* Check when more items*/
	gee_collection_add ((GeeCollection*) arraylist, "2");
	g_assert (gee_collection_get_size ((GeeCollection*) arraylist) == 2);
	/* Check when items cleared*/
	gee_collection_clear ((GeeCollection*) arraylist);
	g_assert (gee_collection_get_size ((GeeCollection*) arraylist) == 0);
	(arraylist == NULL) ? NULL : (arraylist = (g_object_unref (arraylist), NULL));
}


void test_arraylist_iterator (void) {
	GeeArrayList* arraylistOfString;
	GeeIterator* iterator;
	GeeIterator* _tmp0;
	char* _tmp1;
	char* _tmp2;
	char* _tmp3;
	arraylistOfString = gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal);
	/* Check iterate empty list*/
	iterator = gee_iterable_iterator ((GeeIterable*) arraylistOfString);
	g_assert (!gee_iterator_next (iterator));
	/* Check iterate list*/
	gee_collection_add ((GeeCollection*) arraylistOfString, "42");
	gee_collection_add ((GeeCollection*) arraylistOfString, "43");
	gee_collection_add ((GeeCollection*) arraylistOfString, "44");
	_tmp0 = NULL;
	iterator = (_tmp0 = gee_iterable_iterator ((GeeIterable*) arraylistOfString), (iterator == NULL) ? NULL : (iterator = (g_object_unref (iterator), NULL)), _tmp0);
	g_assert (gee_iterator_next (iterator));
	_tmp1 = NULL;
	g_assert (_vala_strcmp0 (_tmp1 = (char*) gee_iterator_get (iterator), "42") == 0);
	_tmp1 = (g_free (_tmp1), NULL);
	g_assert (gee_iterator_next (iterator));
	_tmp2 = NULL;
	g_assert (_vala_strcmp0 (_tmp2 = (char*) gee_iterator_get (iterator), "43") == 0);
	_tmp2 = (g_free (_tmp2), NULL);
	g_assert (gee_iterator_next (iterator));
	_tmp3 = NULL;
	g_assert (_vala_strcmp0 (_tmp3 = (char*) gee_iterator_get (iterator), "44") == 0);
	_tmp3 = (g_free (_tmp3), NULL);
	g_assert (!gee_iterator_next (iterator));
	(arraylistOfString == NULL) ? NULL : (arraylistOfString = (g_object_unref (arraylistOfString), NULL));
	(iterator == NULL) ? NULL : (iterator = (g_object_unref (iterator), NULL));
}


static void _test_arraylist_get_gcallback (void) {
	test_arraylist_get ();
}


static void _test_arraylist_set_gcallback (void) {
	test_arraylist_set ();
}


static void _test_arraylist_insert_gcallback (void) {
	test_arraylist_insert ();
}


static void _test_arraylist_remove_at_gcallback (void) {
	test_arraylist_remove_at ();
}


static void _test_arraylist_index_of_gcallback (void) {
	test_arraylist_index_of ();
}


static void _test_arraylist_add_gcallback (void) {
	test_arraylist_add ();
}


static void _test_arraylist_clear_gcallback (void) {
	test_arraylist_clear ();
}


static void _test_arraylist_contains_gcallback (void) {
	test_arraylist_contains ();
}


static void _test_arraylist_remove_gcallback (void) {
	test_arraylist_remove ();
}


static void _test_arraylist_size_gcallback (void) {
	test_arraylist_size ();
}


static void _test_arraylist_iterator_gcallback (void) {
	test_arraylist_iterator ();
}


void _main (char** args, int args_length1) {
	g_test_init (&args_length1, &args, NULL);
	/* Methods of List interface*/
	g_test_add_func ("/Arraylist/List/get", _test_arraylist_get_gcallback);
	g_test_add_func ("/Arraylist/List/set", _test_arraylist_set_gcallback);
	g_test_add_func ("/Arraylist/List/insert", _test_arraylist_insert_gcallback);
	g_test_add_func ("/Arraylist/List/remove_at", _test_arraylist_remove_at_gcallback);
	g_test_add_func ("/Arraylist/List/index_of", _test_arraylist_index_of_gcallback);
	/* Methods of Collection interface*/
	g_test_add_func ("/Arraylist/Collection/add", _test_arraylist_add_gcallback);
	g_test_add_func ("/Arraylist/Collection/clear", _test_arraylist_clear_gcallback);
	g_test_add_func ("/Arraylist/Collection/contains", _test_arraylist_contains_gcallback);
	g_test_add_func ("/Arraylist/Collection/remove", _test_arraylist_remove_gcallback);
	g_test_add_func ("/Arraylist/Collection/size", _test_arraylist_size_gcallback);
	/* Methods of Iterable interface*/
	g_test_add_func ("/Arraylist/Iterable/iterator", _test_arraylist_iterator_gcallback);
	g_test_run ();
}


int main (int argc, char ** argv) {
	g_type_init ();
	_main (argv, argc);
	return 0;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




