#include <config.h>
#include "gnome-cups-request.h"
#include <glib.h>

#include <stdlib.h>
#include <cups/cups.h>
#include <cups/language.h>
#include <cups/http.h>
#include <cups/ipp.h>

#include "gnome-cups-util.h"
#include "gnome-cups-i18n.h"
#include "gnome-cups-printer.h"

/* Arbitrary. */
#define MAX_REQUEST_THREADS 10
#define STOP_UNUSED_THREADS_TIMEOUT 60
#define CLOSE_UNUSED_CONNECTIONS_TIMEOUT 30

typedef struct
{
	GMutex *mutex;
	gint refcount;
	char *server;
	GTimeVal use_time;
	http_t *http;
} GnomeCupsConnection;

typedef struct
{
	gboolean cancelled;
	gboolean direct_callback;
	guint id;
	GnomeCupsConnection *connection;
	
	ipp_t *response;
	GError **error; 
	GnomeCupsAsyncRequestCallback callback;
	gpointer cb_data;
	GDestroyNotify destroy_notify;

	ipp_t *request;
	char *path;
} GnomeCupsRequest;

static void request_thread_main (GnomeCupsRequest *request, gpointer unused);
static guint gnome_cups_request_execute_async_internal (ipp_t      *request, 
							const char *server, 
							const char *path, 
							gboolean direct_callback,
							GnomeCupsAsyncRequestCallback callback,
							gpointer cb_data,
							GDestroyNotify destroy_notify);
static void gnome_cups_request_connection_destroy (GnomeCupsConnection *conn);
static gboolean idle_stop_unused_threads (gpointer unused);
static gboolean idle_close_unused_connections (gpointer unused);

static const char *
get_error_string (ipp_status_t error)
{  
	static const char *status_oks[] =     /* "OK" status codes */
                {
			N_("Success"),
			N_("Success, some attributes were ignored"),
			N_("Success, some attributes conflicted"),
			N_("Success, ignored subscriptions"),
			N_("Success, ignored notifications"),
			N_("Success, too many events"),
			N_("Success, cancel subscription"),
                };
	static const char *status_400s[] =        /* Client errors */
		{
			N_("Bad request"),
			N_("Forbidden"),
			N_("The client has not been authenticated"),
			N_("You are not authorized to perform this operation"),
			N_("This operation cannot be completed"),
			N_("Timeout"),
			N_("The requested file was not found"),
			N_("The requested resource no longer exists"),
			N_("The request was too large"),
			N_("The request was too long"),
			N_("The document format is not supported"),
			N_("The value or attributes in this request are not supported"),
			N_("The URI scheme is not supported"),
			N_("The requested character set is not supported"),
			N_("There were conflicting attributes in the request"),
			N_("Compression is not supported"),
			N_("There was a compression error"),
			N_("There was an error in the format of the document"),
			N_("There was an error accessing the document"),
			N_("Some attributes could not be set"),
			N_("All subscriptions were ignored"),
			N_("Too many subscriptions"),
			N_("All notifications were ignored."),
			N_("A print support file was not found.")
		};
	
	static const char *status_500s[] =        /* Server errors */
                {
			N_("Internal server error"),
			N_("Operation not supported"),
			N_("Service unavailable"),
			N_("Version not supported"),
			N_("Device error"),
			N_("Temporary error"),
			N_("The printer is not accepting jobs"),
			N_("The printer is busy"),
			N_("The job has been cancelled"),
			N_("Multiple-document jobs are not supported"),
			N_("The printer is deactivated"),
		};                                                                             
	if (error >= IPP_OK && error <= IPP_OK_BUT_CANCEL_SUBSCRIPTION) {
		return _(status_oks[error]);
	} else if (error == IPP_REDIRECTION_OTHER_SITE) {
		return _("Redirected to another site");
	} else if (error >= IPP_BAD_REQUEST && error <= IPP_PRINT_SUPPORT_FILE_NOT_FOUND) {
		return _(status_400s[error - IPP_BAD_REQUEST]);
	} else if (error >= IPP_INTERNAL_ERROR && error <= IPP_PRINTER_IS_DEACTIVATED) {
		return _(status_500s[error - IPP_INTERNAL_ERROR]);
	}                                                                       
	return _("Unknown error");
}

/* Should be per thread with push/pop/user_data etc. (clearly) */
static GnomeCupsAuthFunction global_auth = NULL;

static const char *
cups_password_cb (const char *prompt)
{
	static char *hazard = NULL;

	g_free (hazard);
	hazard = NULL;

	if (global_auth) {
		char *password = NULL;
		char *username = g_strdup (g_get_user_name ());

		if (global_auth (prompt, &username, &password, NULL)) {

			if (username) {
				cupsSetUser (username);
			} else {
				cupsSetUser (g_get_user_name ());
			}
			hazard = password;
		}
		g_free (username);

	} else {
		g_warning ("Cannot prompt for password: '%s'", prompt);
	}

	return hazard;
}

GStaticMutex request_mutex = G_STATIC_MUTEX_INIT;
static guint request_serial_number = 0;
static guint request_system_refcount = 0;
static guint idle_stop_unused_threads_id = 0;
static guint idle_close_unused_connections_id = 0;
static GThreadPool *request_thread_pool;
static GHashTable *request_map = 0;
static GHashTable *connection_cache_map = 0;

void
_gnome_cups_request_init (GnomeCupsAuthFunction auth_fn)
{
	GError *error = NULL;
	global_auth = auth_fn;
	cupsSetPasswordCB (cups_password_cb);
	
	g_static_mutex_lock (&request_mutex);
	if (request_system_refcount == 0) {
		request_map = g_hash_table_new (NULL, NULL);
		connection_cache_map = g_hash_table_new_full (g_str_hash, g_str_equal,
							      (GDestroyNotify) g_free,
							      (GDestroyNotify) gnome_cups_request_connection_destroy);
		request_thread_pool = g_thread_pool_new ((GFunc) request_thread_main,
							 NULL,
							 MAX_REQUEST_THREADS,
							 FALSE,
							 &error);
		idle_stop_unused_threads_id = g_timeout_add (STOP_UNUSED_THREADS_TIMEOUT * 1000, (GSourceFunc) idle_stop_unused_threads, NULL);
		idle_close_unused_connections_id = g_timeout_add (CLOSE_UNUSED_CONNECTIONS_TIMEOUT * 1000, (GSourceFunc) idle_close_unused_connections, NULL);
	}
	request_system_refcount++;
	g_static_mutex_unlock (&request_mutex);

	if (error != NULL) {
		g_critical ("Error creating thread pool: %s", error->message);
		_gnome_cups_request_shutdown ();
	}
}

void
_gnome_cups_request_shutdown (void)
{
	g_static_mutex_lock (&request_mutex);
	request_system_refcount--;
	if (request_system_refcount == 0) {
		g_hash_table_destroy (request_map);
		g_hash_table_destroy (connection_cache_map);
		g_source_remove (idle_stop_unused_threads_id);
		g_source_remove (idle_close_unused_connections_id);
		g_thread_pool_free (request_thread_pool, TRUE, TRUE);
	}
	g_static_mutex_unlock (&request_mutex);
}

static gboolean
idle_stop_unused_threads (gpointer unused)
{
	g_static_mutex_lock (&request_mutex);

	if (request_system_refcount == 0) {
		g_static_mutex_unlock (&request_mutex);
		return FALSE;
	}

	g_thread_pool_stop_unused_threads ();

	g_static_mutex_unlock (&request_mutex);
	return TRUE;
}

static gboolean
close_unused_connection (const char *server,
			 GnomeCupsConnection *connection,
			 GTimeVal *current_time)
{
	gboolean ret;

	g_mutex_lock (connection->mutex);
	ret = (g_atomic_int_get (&connection->refcount) == 0
	       && (current_time->tv_sec - connection->use_time.tv_sec > 30));
	g_mutex_unlock (connection->mutex);
	return ret;
}

static gboolean
idle_close_unused_connections (gpointer unused)
{
	GTimeVal current_time;

	g_static_mutex_lock (&request_mutex);

	if (request_system_refcount == 0) {
		g_static_mutex_unlock (&request_mutex);
		return FALSE;
	}

	g_get_current_time (&current_time);
	g_hash_table_foreach_remove (connection_cache_map,
				     (GHRFunc) close_unused_connection,
				     &current_time);

	g_static_mutex_unlock (&request_mutex);
	return TRUE;
}

static void
gnome_cups_request_struct_free (GnomeCupsRequest *request)
{
	/* "request->connection" is destroyed indepenently -
	 *   it can be shared between multiple requests.
	 * "request->response" should be freed by the client.
	 * "request->request" has already been freed by the
	 *   CUPS libraries.
	 */
	g_free (request->path);
	g_free (request);
}	

static void
gnome_cups_request_connection_destroy (GnomeCupsConnection *connection)
{
	g_mutex_lock (connection->mutex);
	if (connection->http)
		httpClose (connection->http);
	g_free (connection->server);
	g_mutex_unlock (connection->mutex);
	g_mutex_free (connection->mutex);
	g_free (connection);
}

static gboolean
idle_signal_request_complete (GnomeCupsRequest *request)
{
	if (!request->cancelled && request->callback)
		request->callback (request->id,
				   request->path,
				   request->response,
				   request->error,
				   request->cb_data);
	else {
		ippDelete (request->response);
	}

	g_static_mutex_lock (&request_mutex);
	g_assert (g_hash_table_remove (request_map, GUINT_TO_POINTER (request->id)));
	g_static_mutex_unlock (&request_mutex);

	if (request->destroy_notify)
		request->destroy_notify (request->cb_data);

	gnome_cups_request_struct_free (request);

	return FALSE;
}

static void
do_signal_complete (GnomeCupsRequest *request)
{
	if (request->direct_callback)
		idle_signal_request_complete (request);
	else
		g_idle_add ((GSourceFunc) idle_signal_request_complete, request);
}

static void
request_thread_main (GnomeCupsRequest *request,
		     gpointer unused)
{
	ipp_t *response;
	ipp_status_t status;

	if (request->cancelled) {
		do_signal_complete (request);
		return;
	}

	g_mutex_lock (request->connection->mutex);
	
	g_get_current_time (&request->connection->use_time);

	/* This is a deferred open for the first connection */
	if (!request->connection->http)
		request->connection->http = httpConnectEncrypt (request->connection->server, ippPort(), cupsEncryption());


	response = cupsDoRequest (request->connection->http, request->request,
				  request->path);

	/* FIXME - not currently threadsafe, but cups returns NULL on
	 * any error.  Thus we just set the status to an internal error
	 * for now.
	 */
	status = cupsLastError ();
	if (response == NULL)
		status = IPP_INTERNAL_ERROR;

	g_atomic_int_dec_and_test (&request->connection->refcount);
	g_mutex_unlock (request->connection->mutex);

	if (status > IPP_OK_CONFLICT && request->error != NULL) {
		*(request->error) = g_error_new (GNOME_CUPS_ERROR, 
						 status,
						 get_error_string (status));
	}

	request->response = response;
	do_signal_complete (request);
	
	return;
}

guint
_gnome_cups_outstanding_request_count (void)
{
	guint ret;

	g_static_mutex_lock (&request_mutex);
	ret = g_hash_table_size (request_map);
	g_static_mutex_unlock (&request_mutex);

	return ret;
}

ipp_t *
gnome_cups_request_new (int operation_id)
{
	ipp_t *request;
	cups_lang_t *language;
	
	language = cupsLangDefault ();
	request = ippNew ();
	request->request.op.operation_id = operation_id;
	request->request.op.request_id = 1;
	
	ippAddString(request, IPP_TAG_OPERATION, IPP_TAG_CHARSET,
		     "attributes-charset", 
		     NULL, "utf-8");
	
	ippAddString(request, IPP_TAG_OPERATION, IPP_TAG_LANGUAGE,
		     "attributes-natural-language", 
		     NULL, language->language);
	
	return request;
}

ipp_t *
gnome_cups_request_new_for_printer (int operation_id, 
				    GnomeCupsPrinter *printer)
{
	ipp_t *request;
	char *printer_uri;

	g_return_val_if_fail (gnome_cups_printer_get_attributes_initialized (printer), NULL);

	printer_uri = g_strdup (gnome_cups_printer_get_uri (printer));
	if (!printer_uri)
		printer_uri = g_strdup_printf ("ipp://localhost/printers/%s",
					       gnome_cups_printer_get_name (printer));
	request = gnome_cups_request_new (operation_id);

	ippAddString (request, IPP_TAG_OPERATION, IPP_TAG_URI,
		      "printer-uri", NULL, printer_uri);
	g_free (printer_uri);

	return request;
}

ipp_t *
gnome_cups_request_new_for_job (int operation_id, int job)
{
	ipp_t *request;
	char *job_uri;
	
	request = gnome_cups_request_new (operation_id);

	job_uri = g_strdup_printf ("ipp://localhost/jobs/%d", job);

	ippAddString (request, IPP_TAG_OPERATION, IPP_TAG_URI,
		      "job-uri", NULL, gnome_cups_strdup (job_uri));

	/* FIXME: need a way to act as another user.  I guess. */
	ippAddString (request, 
		      IPP_TAG_OPERATION,
		      IPP_TAG_NAME, 
		      "requesting-user-name", NULL, 
		      gnome_cups_strdup (g_get_user_name ()));

	g_free (job_uri);

	return request;
}

void
gnome_cups_request_add_requested_attributes (ipp_t *request, 
					     ipp_tag_t group,
					     int n_attributes,
					     char **attributes)
{
	ipp_attribute_t *attr;
	int i;
	
	attr = ippAddStrings (request, 
			      group,
			      IPP_TAG_KEYWORD,
			      "requested-attributes",
			      n_attributes, NULL, NULL);

	for (i = 0; i < n_attributes; i++) {
		attr->values[i].string.text = gnome_cups_strdup (attributes[i]);
	}
}

typedef struct
{
	GMutex *mutex;
	GCond *cond;
	gboolean done;
	ipp_t *response;
	GError **error;
} GnomeCupsAsyncWrapperData;

static void
async_wrapper_cb (guint id, const char *path,
		  ipp_t *response, GError **error,
		  gpointer user_data)
{
	GnomeCupsAsyncWrapperData *data = user_data;
	g_mutex_lock (data->mutex);
	data->done = TRUE;
	data->response = response;
	if (data->error && error && *error)
		g_propagate_error (data->error, *error);
	g_cond_signal (data->cond);
	g_mutex_unlock (data->mutex);
}

ipp_t *
gnome_cups_request_execute (ipp_t *request, const char *server, const char *path, GError **err)
{
	guint id;
	GnomeCupsAsyncWrapperData data;

	data.mutex = g_mutex_new ();
	data.cond = g_cond_new ();
	data.done = FALSE;
	data.response = NULL;	
	data.error = err;

	id = gnome_cups_request_execute_async_internal (request, server, path,
							TRUE,
							async_wrapper_cb,
							&data,
							NULL);
	if (id > 0) {
		g_mutex_lock (data.mutex);
		while (!data.done)
			g_cond_wait (data.cond, data.mutex);
		g_mutex_unlock (data.mutex);
	}

	g_mutex_free (data.mutex);
	g_cond_free (data.cond);
	
	return data.response;
}

/**
 * gnome_cups_request_execute_async:
 * @request: An IPP request, allocated via gnome_cups_request_new
 * @server: The hostname of the IPP server to connect to
 * @path: The URI path to execute from
 * @callback: A #GnomeCupsAsyncRequestCallback.
 * @cb_data: Data for the callback
 * @destroy_notify: A function to free the callback data
 * @returns: an operation ID, suitable for passing to gnome_cups_request_cancel
 *
 * Creates a new asynchronous IPP operation, which will invoke @cb_data when
 * complete.
 **/
guint
gnome_cups_request_execute_async (ipp_t      *request, 
				  const char *server, 
				  const char *path, 
				  GnomeCupsAsyncRequestCallback callback,
				  gpointer cb_data,
				  GDestroyNotify destroy_notify)
{
	return gnome_cups_request_execute_async_internal (request, server,
							  path, FALSE,
							  callback, cb_data,
							  destroy_notify);
}

static guint
gnome_cups_request_execute_async_internal (ipp_t      *request, 
					   const char *server, 
					   const char *path, 
					   gboolean direct_callback,
					   GnomeCupsAsyncRequestCallback callback,
					   gpointer cb_data,
					   GDestroyNotify destroy_notify)
{
	GnomeCupsConnection *connection;
	GnomeCupsRequest *req;

	if (!server)
		server = cupsServer();
	if (!path)
		path = "/";

	g_static_mutex_lock (&request_mutex);

	/* Connections are shared between multiple threads; actual
	 * usage of the connection is protected by connection->mutex.
	 */
	if ((connection = g_hash_table_lookup (connection_cache_map, server)) == NULL) {
		connection = g_new0 (GnomeCupsConnection, 1);
		connection->mutex = g_mutex_new ();
		connection->server = g_strdup (server);
		/* Let the thread actually make the HTTP connection */
		connection->http = NULL;
		connection->refcount = 0;
		g_hash_table_insert (connection_cache_map, g_strdup (server),
				     connection);
	}
	g_atomic_int_add (&connection->refcount, 1);
	
	req = g_new0 (GnomeCupsRequest, 1);
	req->connection = connection;
	req->cancelled = FALSE;
	req->request = request;
	req->callback = callback;
	req->cb_data = cb_data;
	req->destroy_notify = destroy_notify;
	req->path = g_strdup (path);
	req->direct_callback = direct_callback;
	req->error = NULL;

	req->id = ++request_serial_number;

	g_thread_pool_push (request_thread_pool, req, NULL);

	g_hash_table_insert (request_map, GUINT_TO_POINTER (req->id), req);
	g_static_mutex_unlock (&request_mutex);

	return req->id;
}

void
gnome_cups_request_cancel (guint request_id)
{
	GnomeCupsRequest *request;
	
	g_static_mutex_lock (&request_mutex);
	if ((request = g_hash_table_lookup (request_map, GUINT_TO_POINTER (request_id))) != NULL) {
		request->cancelled = TRUE;
	}
	g_static_mutex_unlock (&request_mutex);
}
