/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <libgda/gda-config.h>
#include <libgnomedb/gnome-db-data-source-selector.h>
#include <libgnomedb/gnome-db-util.h>

#define PARENT_TYPE GTK_TYPE_OPTION_MENU

struct _GnomeDbDataSourceSelectorPrivate {
};

static void gnome_db_data_source_selector_class_init (GnomeDbDataSourceSelectorClass *klass);
static void gnome_db_data_source_selector_init       (GnomeDbDataSourceSelector *selector,
						      GnomeDbDataSourceSelectorClass *klass);
static void gnome_db_data_source_selector_finalize   (GObject *object);

static GObjectClass *parent_class = NULL;

/*
 * Private functions
 */

static void
show_data_sources (GnomeDbDataSourceSelector *selector, const gchar *name)
{
	GList *dsn_list;
	GList *l;

	g_return_if_fail (GNOME_DB_IS_DATA_SOURCE_SELECTOR (selector));

	gtk_option_menu_remove_menu (GTK_OPTION_MENU (selector));

	/* load data sources */
	dsn_list = gda_config_get_data_source_list ();
	for (l = dsn_list; l != NULL; l = l->next) {
		GdaDataSourceInfo *dsn_info = (GdaDataSourceInfo *) l->data;

		if (!dsn_info)
			continue;

		if (name != NULL && strcmp (name, dsn_info->name) != 0)
			continue;

		/* add the data source to the option menu */
		gnome_db_option_menu_add_item (GTK_OPTION_MENU (selector), dsn_info->name);
	}

	gda_config_free_data_source_list (dsn_list);
}

/*
 * GnomeDbDataSourceSelector class implementation
 */

static void
gnome_db_data_source_selector_class_init (GnomeDbDataSourceSelectorClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = gnome_db_data_source_selector_finalize;
}

static void
gnome_db_data_source_selector_init (GnomeDbDataSourceSelector *selector,
				    GnomeDbDataSourceSelectorClass *klass)
{
	g_return_if_fail (GNOME_DB_IS_DATA_SOURCE_SELECTOR (selector));

	/* allocate private structure */
	selector->priv = g_new0 (GnomeDbDataSourceSelectorPrivate, 1);
}

static void
gnome_db_data_source_selector_finalize (GObject *object)
{
	GnomeDbDataSourceSelector *selector = (GnomeDbDataSourceSelector *) object;

	g_return_if_fail (GNOME_DB_IS_DATA_SOURCE_SELECTOR (selector));

	/* free memory */
	g_free (selector->priv);
	selector->priv = NULL;

	parent_class->finalize;
}

GType
gnome_db_data_source_selector_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbDataSourceSelectorClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_data_source_selector_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbDataSourceSelector),
			0,
			(GInstanceInitFunc) gnome_db_data_source_selector_init
		};
		type = g_type_register_static (PARENT_TYPE,
					       "GnomeDbDataSourceSelector",
					       &info, 0);
	}
	return type;
}

/**
 * gnome_db_data_source_selector_new
 * @name: name of data source to display.
 *
 * Create a new #GnomeDbDataSourceSelector, which is just a #GtkOptionMenu
 * which displays, as its menu items, all the data sources currently
 * configured in the system. It is useful for connection and configuration
 * screens, where the user has to choose a data source to work with.
 *
 * Returns: the newly created widget.
 */
GtkWidget *
gnome_db_data_source_selector_new (const gchar *name)
{
	GnomeDbDataSourceSelector *selector;

	selector = g_object_new (GNOME_DB_TYPE_DATA_SOURCE_SELECTOR, NULL);
	show_data_sources (selector, name);

	return GTK_WIDGET (selector);
}
