/* GNOME DB library
 * Copyright (C) 1999 - 2005 The GNOME Foundation
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *      Vivien Malerba <malerba@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <gtk/gtklabel.h>
#include <gtk/gtktable.h>
#include <libgnomedb/gnome-db-dsn-config.h>
#include <libgnomedb/gnome-db-provider-selector.h>
#include <libgnomedb/gnome-db-util.h>
#include <glib/gi18n-lib.h>

#define PARENT_TYPE GTK_TYPE_VBOX

struct _GnomeDbDsnConfigPrivate {
	GtkWidget *wname;
	GtkWidget *wprovider;
	GtkWidget *wstring;
	GtkWidget *wdesc;
	GtkWidget *wuser;
	GtkWidget *wpassword;
	GtkWidget *is_global;
};

static void gnome_db_dsn_config_class_init (GnomeDbDsnConfigClass *klass);
static void gnome_db_dsn_config_init       (GnomeDbDsnConfig *config,
					    GnomeDbDsnConfigClass *klass);
static void gnome_db_dsn_config_finalize   (GObject *object);

enum {
	CHANGED,
	LAST_SIGNAL
};

static gint gnome_db_dsn_config_signals[LAST_SIGNAL] = { 0 };
static GObjectClass *parent_class = NULL;

/*
 * Private functions
 */

/*
 * GnomeDbDsnConfig class implementation
 */

static void
gnome_db_dsn_config_class_init (GnomeDbDsnConfigClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = gnome_db_dsn_config_finalize;
	klass->changed = NULL;

	/* add class signals */
	gnome_db_dsn_config_signals[CHANGED] =
		g_signal_new ("changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GnomeDbDsnConfigClass, changed),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);
}

static void
field_changed_cb (GtkWidget *widget, GnomeDbDsnConfig *config)
{
	g_signal_emit (config, gnome_db_dsn_config_signals[CHANGED], 0, NULL);
}

static void
field_toggled_cb (GtkWidget *widget, GnomeDbDsnConfig *config)
{
	g_signal_emit (config, gnome_db_dsn_config_signals[CHANGED], 0, NULL);
}

static void
gnome_db_dsn_config_init (GnomeDbDsnConfig *config, GnomeDbDsnConfigClass *klass)
{
	GtkWidget *table;
	GtkWidget *label;

	g_return_if_fail (GNOME_DB_IS_DSN_CONFIG (config));

	/* allocate private structure */
	config->priv = g_new0 (GnomeDbDsnConfigPrivate, 1);

	/* set up widgets */
	table = gnome_db_new_table_widget (7, 3, FALSE);
	gtk_box_pack_start (GTK_BOX (config), table, TRUE, TRUE, 0);

	label = gnome_db_new_label_widget (_("Data source _name:"));
	gtk_widget_show (label);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	config->priv->wname = gnome_db_new_entry_widget (0, TRUE);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), config->priv->wname);
	gtk_widget_show (config->priv->wname);
	gtk_widget_set_sensitive (config->priv->wname, FALSE); /* FIXME */
	g_signal_connect (G_OBJECT (config->priv->wname), "changed",
			  G_CALLBACK (field_changed_cb), config);
	gtk_table_attach (GTK_TABLE (table), config->priv->wname, 1, 2, 0, 1,
			  GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);

	label = gnome_db_new_label_widget (_("_System wide data source:"));
	gtk_widget_show (label);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
	config->priv->is_global = gtk_check_button_new ();
	gtk_widget_show (config->priv->is_global);
	g_signal_connect (G_OBJECT (config->priv->is_global), "toggled",
			  G_CALLBACK (field_toggled_cb), config);
	gtk_table_attach (GTK_TABLE (table), config->priv->is_global, 1, 2, 1, 2,
			  GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);

	label = gnome_db_new_label_widget (_("_Provider:"));
	gtk_widget_show (label);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 0, 0);
	config->priv->wprovider = gnome_db_provider_selector_new ();
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), config->priv->wprovider);
	gtk_widget_show (config->priv->wprovider);
	g_signal_connect (G_OBJECT (config->priv->wprovider), "changed",
			  G_CALLBACK (field_changed_cb), config);
	gtk_table_attach (GTK_TABLE (table), config->priv->wprovider, 1, 2, 2, 3,
			  GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);

	label = gnome_db_new_label_widget (_("Connection _string:"));
	gtk_widget_show (label);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 3, 4, GTK_FILL, GTK_FILL, 0, 0);
	config->priv->wstring = gnome_db_new_entry_widget (0, TRUE);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), config->priv->wstring);
	gtk_widget_show (config->priv->wstring);
	g_signal_connect (G_OBJECT (config->priv->wstring), "changed",
			  G_CALLBACK (field_changed_cb), config);
	gtk_table_attach (GTK_TABLE (table), config->priv->wstring, 1, 2, 3, 4,
			  GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);

	label = gnome_db_new_label_widget (_("_Description:"));
	gtk_widget_show (label);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 4, 5, GTK_FILL, GTK_FILL, 0, 0);
	config->priv->wdesc = gnome_db_new_entry_widget (0, TRUE);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), config->priv->wdesc);
	gtk_widget_show (config->priv->wdesc);
	g_signal_connect (G_OBJECT (config->priv->wdesc), "changed",
			  G_CALLBACK (field_changed_cb), config);
	gtk_table_attach (GTK_TABLE (table), config->priv->wdesc, 1, 2, 4, 5,
			  GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);

	label = gnome_db_new_label_widget (_("_Username:"));
	gtk_widget_show (label);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 5, 6, GTK_FILL, GTK_FILL, 0, 0);
	config->priv->wuser = gnome_db_new_entry_widget (0, TRUE);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), config->priv->wuser);
	gtk_widget_show (config->priv->wuser);
	g_signal_connect (G_OBJECT (config->priv->wuser), "changed",
			  G_CALLBACK (field_changed_cb), config);
	gtk_table_attach (GTK_TABLE (table), config->priv->wuser, 1, 2, 5, 6,
			  GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);

	label = gnome_db_new_label_widget (_("P_assword:"));
	gtk_widget_show (label);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 6, 7, GTK_FILL, GTK_FILL, 0, 0);
	config->priv->wpassword = gnome_db_new_entry_widget (0, TRUE);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), config->priv->wpassword);
	gtk_entry_set_visibility (GTK_ENTRY (config->priv->wpassword), FALSE);
	gtk_widget_show (config->priv->wpassword);
	g_signal_connect (G_OBJECT (config->priv->wpassword), "changed",
			  G_CALLBACK (field_changed_cb), config);
	gtk_table_attach (GTK_TABLE (table), config->priv->wpassword, 1, 2, 6, 7,
			  GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);
}

static void
gnome_db_dsn_config_finalize (GObject *object)
{
	GnomeDbDsnConfig *config = (GnomeDbDsnConfig *) object;

	g_return_if_fail (GNOME_DB_IS_DSN_CONFIG (config));

	/* free memory */
	g_free (config->priv);
	config->priv = NULL;

	/* chain to parent class */
	parent_class->finalize (object);
}

GType
gnome_db_dsn_config_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbDsnConfigClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_dsn_config_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbDsnConfig),
			0,
			(GInstanceInitFunc) gnome_db_dsn_config_init
		};
		type = g_type_register_static (PARENT_TYPE, "GnomeDbDsnConfig",
					       &info, 0);
	}
	return type;
}

/**
 * gnome_db_dsn_config_new
 *
 *
 *
 * Returns:
 */
GtkWidget *
gnome_db_dsn_config_new (void)
{
	GnomeDbDsnConfig *config;

	config = g_object_new (GNOME_DB_TYPE_DSN_CONFIG, NULL);
	return GTK_WIDGET (config);
}

/**
 * gnome_db_dsn_config_get_info
 * @config:
 *
 *
 *
 * Returns:
 */
GdaDataSourceInfo *
gnome_db_dsn_config_get_info (GnomeDbDsnConfig *config)
{
	gint i;
	GdaDataSourceInfo *dsn_info;

	g_return_val_if_fail (GNOME_DB_IS_DSN_CONFIG (config), NULL);

	dsn_info = g_new0 (GdaDataSourceInfo, 1);
	dsn_info->name = g_strdup (gtk_entry_get_text (GTK_ENTRY (config->priv->wname)));
	for (i = 0; i < strlen (dsn_info->name); i++) {
		if (!g_ascii_isalnum (dsn_info->name[i]))
			dsn_info->name[i] = '_';
	}

	dsn_info->provider = g_strdup (gnome_db_provider_selector_get_selected_provider (GNOME_DB_PROVIDER_SELECTOR (config->priv->wprovider)));
	dsn_info->cnc_string = g_strdup (gtk_entry_get_text (GTK_ENTRY (config->priv->wstring)));
	dsn_info->description = g_strdup (gtk_entry_get_text (GTK_ENTRY (config->priv->wdesc)));
	dsn_info->username = g_strdup (gtk_entry_get_text (GTK_ENTRY (config->priv->wuser)));
	dsn_info->password = g_strdup (gtk_entry_get_text (GTK_ENTRY (config->priv->wpassword)));
	dsn_info->is_global = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (config->priv->is_global));

	return dsn_info;
}

/**
 * gnome_db_dsn_config_set_info
 * @config:
 * @dsn_info:
 *
 */
void
gnome_db_dsn_config_set_info (GnomeDbDsnConfig *config, const GdaDataSourceInfo *dsn_info)
{
	g_return_if_fail (GNOME_DB_IS_DSN_CONFIG (config));
	g_return_if_fail (dsn_info != NULL);

	gtk_entry_set_text (GTK_ENTRY (config->priv->wname), dsn_info->name);
	gnome_db_provider_selector_set_selected_provider (GNOME_DB_PROVIDER_SELECTOR (config->priv->wprovider), dsn_info->provider);
	gtk_entry_set_text (GTK_ENTRY (config->priv->wstring), dsn_info->cnc_string ? dsn_info->cnc_string : "");
	gtk_entry_set_text (GTK_ENTRY (config->priv->wdesc), dsn_info->description ? dsn_info->description : "");
	gtk_entry_set_text (GTK_ENTRY (config->priv->wuser), dsn_info->username ? dsn_info->username : "");
	gtk_entry_set_text (GTK_ENTRY (config->priv->wpassword), dsn_info->password ? dsn_info->password : "");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (config->priv->is_global), dsn_info->is_global);

	if (dsn_info->is_global && !gda_config_can_modify_global_config ())
		gtk_widget_set_sensitive (GTK_WIDGET (config), FALSE);
	else {
		gtk_widget_set_sensitive (GTK_WIDGET (config), TRUE);
		gtk_widget_set_sensitive (config->priv->wname, FALSE); /* FIXME */
		gtk_widget_set_sensitive (config->priv->is_global, 
					  gda_config_can_modify_global_config ());
	}
}
