/* gnome-db-entry-time.c
 *
 * Copyright (C) 2003 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-entry-time.h"
#include <libgnomedb/gnome-db-data-handler.h>
#include <gdk/gdkkeysyms.h>
#include <string.h>

/* 
 * Main static functions 
 */
static void gnome_db_entry_time_class_init (GnomeDbEntryTimeClass * class);
static void gnome_db_entry_time_init (GnomeDbEntryTime * srv);
static void gnome_db_entry_time_dispose (GObject *object);
static void gnome_db_entry_time_finalize (GObject *object);

static void gnome_db_entry_time_set_property (GObject              *object,
					guint                 param_id,
					const GValue         *value,
					GParamSpec           *pspec);
static void gnome_db_entry_time_get_property (GObject              *object,
					guint                 param_id,
					GValue               *value,
					GParamSpec           *pspec);

/* properties */
enum
{
	PROP_0,
	PROP_EDITING_CANCELLED
};

/* GtkCellEditable interface */
static void gnome_db_entry_time_cell_editable_init (GtkCellEditableIface *iface);
static void gnome_db_entry_time_start_editing (GtkCellEditable *iface, GdkEvent *event);

/* virtual functions */
static GtkWidget *create_entry (GnomeDbEntryWrapper *mgwrap);
static void       real_set_value (GnomeDbEntryWrapper *mgwrap, const GdaValue *value);
static GdaValue  *real_get_value (GnomeDbEntryWrapper *mgwrap);
static void       connect_signals(GnomeDbEntryWrapper *mgwrap, GCallback callback);
static gboolean   expand_in_layout (GnomeDbEntryWrapper *mgwrap);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;


/* private structure */
struct _GnomeDbEntryTimePrivate
{
	/* for date */
	GtkWidget *entry_date;
	GtkWidget *date;
        GtkWidget *window;
        GtkWidget *date_button;

	/* for time */
	GtkWidget *entry_time;
	GtkWidget *legend;

	/* for timestamp */
	GtkWidget *hbox;

	/* Last value set */
	GdaValue  *last_value_set;
};

static void
gnome_db_entry_time_cell_editable_init (GtkCellEditableIface *iface)
{
	iface->start_editing = gnome_db_entry_time_start_editing;
}

guint
gnome_db_entry_time_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbEntryTimeClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_entry_time_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbEntryTime),
			0,
			(GInstanceInitFunc) gnome_db_entry_time_init
		};

		static const GInterfaceInfo cell_editable_info = {
			(GInterfaceInitFunc) gnome_db_entry_time_cell_editable_init,    /* interface_init */
			NULL,                                                 /* interface_finalize */
			NULL                                                  /* interface_data */
		};
	
		type = g_type_register_static (GNOME_DB_ENTRY_WRAPPER_TYPE, "GnomeDbEntryTime", &info, 0);
		g_type_add_interface_static (type, GTK_TYPE_CELL_EDITABLE, &cell_editable_info);
	}
	return type;
}

static void
gnome_db_entry_time_class_init (GnomeDbEntryTimeClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = gnome_db_entry_time_dispose;
	object_class->finalize = gnome_db_entry_time_finalize;

	GNOME_DB_ENTRY_WRAPPER_CLASS (class)->create_entry = create_entry;
	GNOME_DB_ENTRY_WRAPPER_CLASS (class)->real_set_value = real_set_value;
	GNOME_DB_ENTRY_WRAPPER_CLASS (class)->real_get_value = real_get_value;
	GNOME_DB_ENTRY_WRAPPER_CLASS (class)->connect_signals = connect_signals;
	GNOME_DB_ENTRY_WRAPPER_CLASS (class)->expand_in_layout = expand_in_layout;

	/* Properties */
	object_class->set_property = gnome_db_entry_time_set_property;
	object_class->get_property = gnome_db_entry_time_get_property;

	g_object_class_install_property (object_class, PROP_EDITING_CANCELLED,
					 g_param_spec_boolean ("editing_cancelled", NULL, NULL, FALSE, G_PARAM_READABLE));

}

static void
gnome_db_entry_time_init (GnomeDbEntryTime * gnome_db_entry_time)
{
	gnome_db_entry_time->priv = g_new0 (GnomeDbEntryTimePrivate, 1);
	gnome_db_entry_time->priv->entry_date = NULL;
	gnome_db_entry_time->priv->entry_time = NULL;
	gnome_db_entry_time->priv->date = NULL;
	gnome_db_entry_time->priv->window = NULL;
	gnome_db_entry_time->priv->date_button = NULL;
	gnome_db_entry_time->priv->legend = NULL;
	gnome_db_entry_time->priv->hbox = NULL;
	gnome_db_entry_time->priv->last_value_set = NULL;
}

/**
 * gnome_db_entry_time_new
 * @dh: the data handler to be used by the new widget
 * @type: the requested data type (compatible with @dh)
 *
 * Creates a new widget which is mainly a GtkEntry
 *
 * Returns: the new widget
 */
GtkWidget *
gnome_db_entry_time_new (GnomeDbDataHandler *dh, GdaValueType type)
{
	GObject *obj;
	GnomeDbEntryTime *mgtim;

	g_return_val_if_fail (dh && IS_GNOME_DB_DATA_HANDLER (dh), NULL);
	g_return_val_if_fail (type != GDA_VALUE_TYPE_UNKNOWN, NULL);
	g_return_val_if_fail (gnome_db_data_handler_accepts_gda_type (dh, type), NULL);

	obj = g_object_new (GNOME_DB_ENTRY_TIME_TYPE, "handler", dh, NULL);
	mgtim = GNOME_DB_ENTRY_TIME (obj);
	gnome_db_data_entry_set_value_type (GNOME_DB_DATA_ENTRY (mgtim), type);

	return GTK_WIDGET (obj);
}


static void
gnome_db_entry_time_dispose (GObject   * object)
{
	GnomeDbEntryTime *gnome_db_entry_time;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_ENTRY_TIME (object));

	gnome_db_entry_time = GNOME_DB_ENTRY_TIME (object);
	if (gnome_db_entry_time->priv) {

	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_entry_time_finalize (GObject   * object)
{
	GnomeDbEntryTime *gnome_db_entry_time;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_ENTRY_TIME (object));

	gnome_db_entry_time = GNOME_DB_ENTRY_TIME (object);
	if (gnome_db_entry_time->priv) {
		if (gnome_db_entry_time->priv->last_value_set) 
			gda_value_free (gnome_db_entry_time->priv->last_value_set);

		g_free (gnome_db_entry_time->priv);
		gnome_db_entry_time->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}

static void
gnome_db_entry_time_set_property (GObject              *object,
			    guint                 param_id,
			    const GValue         *value,
			    GParamSpec           *pspec)
{
	GnomeDbEntryTime *mgtim;

	mgtim = GNOME_DB_ENTRY_TIME (object);
	if (mgtim->priv) {
		switch (param_id) {
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
		}
	}
}

static void
gnome_db_entry_time_get_property (GObject              *object,
			    guint                 param_id,
			    GValue               *value,
			    GParamSpec           *pspec)
{
	GnomeDbEntryTime *mgtim;
	gboolean cancelled;

	mgtim = GNOME_DB_ENTRY_TIME (object);
	if (mgtim->priv) {
		switch (param_id) {
		case PROP_EDITING_CANCELLED:
			cancelled = FALSE;
			/* FIXME */
			if (mgtim->priv->entry_date)
				cancelled = GTK_ENTRY (mgtim->priv->entry_date)->editing_canceled;
			if (!cancelled && mgtim->priv->entry_time)
				cancelled = GTK_ENTRY (mgtim->priv->entry_time)->editing_canceled;
			g_value_set_boolean (value, cancelled);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
		}
	}
}




static GtkWidget *create_entry_date (GnomeDbEntryTime *mgtim);
static GtkWidget *create_entry_time (GnomeDbEntryTime *mgtim);
static GtkWidget *create_entry_ts (GnomeDbEntryTime *mgtim);
static GtkWidget *
create_entry (GnomeDbEntryWrapper *mgwrap)
{
	GnomeDbEntryTime *mgtim;
	GtkWidget *entry = NULL;

	g_return_val_if_fail (mgwrap && IS_GNOME_DB_ENTRY_TIME (mgwrap), NULL);
	mgtim = GNOME_DB_ENTRY_TIME (mgwrap);
	g_return_val_if_fail (mgtim->priv, NULL);

	switch (gnome_db_data_entry_get_value_type (GNOME_DB_DATA_ENTRY (mgtim))) {
	case GDA_VALUE_TYPE_DATE:
		entry = create_entry_date (mgtim);
		break;
	case GDA_VALUE_TYPE_TIME:
		entry = create_entry_time (mgtim);
		break;
	case GDA_VALUE_TYPE_TIMESTAMP:
		entry = create_entry_ts (mgtim);
		break;
	default:
		g_assert_not_reached ();
		break;
	}
		
	return entry;
}

static void
real_set_value (GnomeDbEntryWrapper *mgwrap, const GdaValue *value)
{
	GnomeDbEntryTime *mgtim;
	GdaValueType type;
	GnomeDbDataHandler *dh;

	g_return_if_fail (mgwrap && IS_GNOME_DB_ENTRY_TIME (mgwrap));
	mgtim = GNOME_DB_ENTRY_TIME (mgwrap);
	g_return_if_fail (mgtim->priv);

	dh = gnome_db_data_entry_get_handler (GNOME_DB_DATA_ENTRY (mgwrap));
	type = gnome_db_data_entry_get_value_type (GNOME_DB_DATA_ENTRY (mgtim));
	switch (type) {
	case GDA_VALUE_TYPE_DATE:
		if (value) {
			if (gda_value_is_null (value))
				gtk_entry_set_text (GTK_ENTRY (mgtim->priv->entry_date), "");
			else {
				gchar *str;
				
				str = gnome_db_data_handler_get_str_from_value (dh, value);
				gtk_entry_set_text (GTK_ENTRY (mgtim->priv->entry_date), str);
				g_free (str);
			}
		}
		else 
			gtk_entry_set_text (GTK_ENTRY (mgtim->priv->entry_date), "");
		break;
	case GDA_VALUE_TYPE_TIME:
		if (value) {
			if (gda_value_is_null (value))
				gtk_entry_set_text (GTK_ENTRY (mgtim->priv->entry_time), "");
			else {
				gchar *str;
				
				str = gnome_db_data_handler_get_str_from_value (dh, value);
				gtk_entry_set_text (GTK_ENTRY (mgtim->priv->entry_time), str);
				g_free (str);
			}
		}
		else 
			gtk_entry_set_text (GTK_ENTRY (mgtim->priv->entry_time), "");
		break;
	case GDA_VALUE_TYPE_TIMESTAMP:
		if (value) {
			if (gda_value_is_null (value))
				gtk_entry_set_text (GTK_ENTRY (mgtim->priv->entry_time), "");
			else {
				gchar *str, *ptr;
				
				str = gnome_db_data_handler_get_str_from_value (dh, value);
				ptr = strtok (str, " ");
				gtk_entry_set_text (GTK_ENTRY (mgtim->priv->entry_date), ptr);
				ptr = strtok (NULL, " ");
				gtk_entry_set_text (GTK_ENTRY (mgtim->priv->entry_time), ptr);
				g_free (str);
			}
		}
		else {
			gtk_entry_set_text (GTK_ENTRY (mgtim->priv->entry_date), "");
			gtk_entry_set_text (GTK_ENTRY (mgtim->priv->entry_time), "");
		}
		break;
	default:
		g_assert_not_reached ();
		break;	
	}

	/* keep track of the last value set */
	if (mgtim->priv->last_value_set) {
		gda_value_free (mgtim->priv->last_value_set);
		mgtim->priv->last_value_set = NULL;
	}
	if (value)
		mgtim->priv->last_value_set = gda_value_copy (value);
}

static GdaValue *
real_get_value (GnomeDbEntryWrapper *mgwrap)
{
	GdaValue *value = NULL;
	GnomeDbEntryTime *mgtim;
	GnomeDbDataHandler *dh;
	const gchar *str;
	gchar *str2;
	GdaValueType type;

	g_return_val_if_fail (mgwrap && IS_GNOME_DB_ENTRY_TIME (mgwrap), NULL);
	mgtim = GNOME_DB_ENTRY_TIME (mgwrap);
	g_return_val_if_fail (mgtim->priv, NULL);

	type = gnome_db_data_entry_get_value_type (GNOME_DB_DATA_ENTRY (mgtim));
	dh = gnome_db_data_entry_get_handler (GNOME_DB_DATA_ENTRY (mgwrap));
	switch (type) {
	case GDA_VALUE_TYPE_DATE:
		str = gtk_entry_get_text (GTK_ENTRY (mgtim->priv->entry_date));
		value = gnome_db_data_handler_get_value_from_str (dh, str, type); /* FIXME: not SQL but STR */
		break;
	case GDA_VALUE_TYPE_TIME:
		str = gtk_entry_get_text (GTK_ENTRY (mgtim->priv->entry_time));
		value = gnome_db_data_handler_get_value_from_str (dh, str, type);
		if (mgtim->priv->last_value_set) {
			/* copy the 'timezone' part, we we have not modified */
			const GdaTime *dgatime_set = gda_value_get_time (mgtim->priv->last_value_set);
			GdaTime *gdatime = g_new (GdaTime, 1);
			*gdatime = *(gda_value_get_time (value));
			gdatime->timezone = dgatime_set->timezone;
			gda_value_set_time (value, gdatime);
			g_free (gdatime);
		}
		break;
	case GDA_VALUE_TYPE_TIMESTAMP:
		str2 = g_strdup_printf ("%s %s",
					gtk_entry_get_text (GTK_ENTRY (mgtim->priv->entry_date)),
					gtk_entry_get_text (GTK_ENTRY (mgtim->priv->entry_time)));
		value = gnome_db_data_handler_get_value_from_str (dh, str2, type);
		g_free (str2);
		if (mgtim->priv->last_value_set) {
			/* copy the 'fraction' and 'timezone' parts, we we have not modified */
			const GdaTimestamp *dgatime_set = gda_value_get_timestamp (mgtim->priv->last_value_set);
			GdaTimestamp *gdatime = g_new (GdaTimestamp, 1);
			*gdatime = *(gda_value_get_timestamp (value));
			gdatime->fraction = dgatime_set->fraction;
			gdatime->timezone = dgatime_set->timezone;
			gda_value_set_timestamp (value, gdatime);
			g_free (gdatime);
		}

		break;
	default:
		g_assert_not_reached ();
		break;	
	}

	if (!value) {
		/* in case the gnome_db_data_handler_get_value_from_str() returned an error because
		   the contents of the GtkEntry cannot be interpreted as a GdaValue */
		value = gda_value_new_null ();
	}

	return value;
}

static void
connect_signals(GnomeDbEntryWrapper *mgwrap, GCallback callback)
{
	GnomeDbEntryTime *mgtim;
	GdaValueType type;

	g_return_if_fail (mgwrap && IS_GNOME_DB_ENTRY_TIME (mgwrap));
	mgtim = GNOME_DB_ENTRY_TIME (mgwrap);
	g_return_if_fail (mgtim->priv);

	type = gnome_db_data_entry_get_value_type (GNOME_DB_DATA_ENTRY (mgtim));
	switch (type) {
	case GDA_VALUE_TYPE_DATE:
		g_signal_connect (G_OBJECT (mgtim->priv->entry_date), "changed",
				  callback, mgwrap);
		break;
	case GDA_VALUE_TYPE_TIME:
		g_signal_connect (G_OBJECT (mgtim->priv->entry_time), "changed",
				  callback, mgwrap);
		break;
	case GDA_VALUE_TYPE_TIMESTAMP:
		g_signal_connect (G_OBJECT (mgtim->priv->entry_date), "changed",
				  callback, mgwrap);
		g_signal_connect (G_OBJECT (mgtim->priv->entry_time), "changed",
				  callback, mgwrap);
		break;
	default:
		g_assert_not_reached ();
		break;
	}
}

static gboolean
expand_in_layout (GnomeDbEntryWrapper *mgwrap)
{
	return FALSE;
}






/*
 * callbacks for the date 
 */

static gint date_delete_popup (GtkWidget *widget, GnomeDbEntryTime *mgtim);
static gint date_key_press_popup (GtkWidget *widget, GdkEventKey *event, GnomeDbEntryTime *mgtim);
static gint date_button_press_popup (GtkWidget *widget, GdkEventButton *event, GnomeDbEntryTime *mgtim);
static void date_day_selected (GtkCalendar *calendar, GnomeDbEntryTime *mgtim);
static void date_day_selected_double_click (GtkCalendar *calendar, GnomeDbEntryTime *mgtim);
static void date_calendar_choose_cb (GtkWidget *button, GnomeDbEntryTime *mgtim);

static GtkWidget *
create_entry_date (GnomeDbEntryTime *mgtim)
{
	GtkWidget *wid, *hb, *window;

	/* top widget */
	hb = gtk_hbox_new (FALSE, 3);

	/* text entry */
	wid = gtk_entry_new ();
	gtk_entry_set_max_length (GTK_ENTRY (wid), 10); /* for a numerical date format */
	gtk_entry_set_width_chars (GTK_ENTRY (wid), 10);
	gtk_box_pack_start (GTK_BOX (hb), wid, FALSE, FALSE, 0);
	gtk_widget_show (wid);
	mgtim->priv->entry_date = wid;
	
	/* window to hold the calendar popup */
	window = gtk_window_new (GTK_WINDOW_POPUP);
	gtk_widget_set_events (window, gtk_widget_get_events (window) | GDK_KEY_PRESS_MASK);
	gtk_window_set_resizable (GTK_WINDOW (window), FALSE);
	g_signal_connect (G_OBJECT (window), "delete_event",
			  G_CALLBACK (date_delete_popup), mgtim);
	g_signal_connect (G_OBJECT (window), "key_press_event",
			  G_CALLBACK (date_key_press_popup), mgtim);
	g_signal_connect (G_OBJECT (window), "button_press_event",
			  G_CALLBACK (date_button_press_popup), mgtim);
	mgtim->priv->window = window;
	
	/* calendar */
	wid = gtk_calendar_new ();
	mgtim->priv->date = wid;
	gtk_container_add (GTK_CONTAINER (window), wid);
	gtk_widget_show (wid);
	g_signal_connect (G_OBJECT (wid), "day_selected",
			  G_CALLBACK (date_day_selected), mgtim);
	g_signal_connect (G_OBJECT (wid), "day_selected_double_click",
			  G_CALLBACK (date_day_selected_double_click), mgtim);
	
	/* button to pop up the calendar */
	wid = gtk_button_new_with_label (_("Choose"));
	gtk_box_pack_start (GTK_BOX (hb), wid, FALSE, FALSE, 0);
	gtk_widget_show (wid);
	g_signal_connect (G_OBJECT (wid), "clicked",
			  G_CALLBACK (date_calendar_choose_cb), mgtim);
	mgtim->priv->date_button = wid;

	/* padding */
	wid = gtk_label_new ("");
	gtk_box_pack_start (GTK_BOX (hb), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);

	return hb;
}

static void hide_popup (GnomeDbEntryTime *mgtim)
{
        gtk_widget_hide (mgtim->priv->window);
        gtk_grab_remove (mgtim->priv->window);
}
static gint
date_delete_popup (GtkWidget *widget, GnomeDbEntryTime *mgtim)
{
	hide_popup (mgtim);
	return TRUE;
}

static gint
date_key_press_popup (GtkWidget *widget, GdkEventKey *event, GnomeDbEntryTime *mgtim)
{
	if (event->keyval != GDK_Escape)
                return FALSE;

        g_signal_stop_emission_by_name (widget, "key_press_event");
        hide_popup (mgtim);

        return TRUE;
}

static gint
date_button_press_popup (GtkWidget *widget, GdkEventButton *event, GnomeDbEntryTime *mgtim)
{
	GtkWidget *child;

        child = gtk_get_event_widget ((GdkEvent *) event);

        /* We don't ask for button press events on the grab widget, so
         *  if an event is reported directly to the grab widget, it must
         *  be on a window outside the application (and thus we remove
         *  the popup window). Otherwise, we check if the widget is a child
         *  of the grab widget, and only remove the popup window if it
         *  is not.
         */
        if (child != widget) {
                while (child) {
                        if (child == widget)
                                return FALSE;
                        child = child->parent;
                }
        }

        hide_popup (mgtim);

        return TRUE;
}

static void
date_day_selected (GtkCalendar *calendar, GnomeDbEntryTime *mgtim)
{
	char buffer [256];
        guint year, month, day;
        struct tm mtm = {0};
        char *str_utf8;

        gtk_calendar_get_date (calendar, &year, &month, &day);

        mtm.tm_mday = day;
        mtm.tm_mon = month;
        if (year > 1900)
                mtm.tm_year = year - 1900;
        else
                mtm.tm_year = year;

        if (strftime (buffer, sizeof (buffer), "%x", &mtm) == 0)
                strcpy (buffer, "???");
        buffer[sizeof(buffer)-1] = '\0';

        str_utf8 = g_locale_to_utf8 (buffer, -1, NULL, NULL, NULL);
        gtk_entry_set_text (GTK_ENTRY (mgtim->priv->entry_date),
                            str_utf8 ? str_utf8 : "");
        g_free (str_utf8);
}

static void
date_day_selected_double_click (GtkCalendar *calendar, GnomeDbEntryTime *mgtim)
{
	hide_popup (mgtim);
}


static gboolean popup_grab_on_window (GdkWindow *window, guint32 activate_time);
static void position_popup (GnomeDbEntryTime *mgtim);
static void
date_calendar_choose_cb (GtkWidget *button, GnomeDbEntryTime *mgtim)
{
	GdaValue *value;
        guint year=0, month=0, day=0;
	GnomeDbDataHandler *dh;

        /* setting the calendar to the latest displayed date */
	dh = gnome_db_data_entry_get_handler (GNOME_DB_DATA_ENTRY (mgtim));
	value = gnome_db_data_entry_get_value (GNOME_DB_DATA_ENTRY (mgtim));
	
        if (value && !gda_value_is_null (value)) {
		const GdaDate *date;
		const GdaTimestamp *ts;

		switch (gnome_db_data_entry_get_value_type (GNOME_DB_DATA_ENTRY (mgtim))) {
		case GDA_VALUE_TYPE_DATE:
			date = gda_value_get_date (value);
			year = date->year;
			month = date->month - 1;
			day = date->day;
			break;
		case GDA_VALUE_TYPE_TIMESTAMP:
			ts = gda_value_get_timestamp (value);
			year = ts->year;
			month = ts->month - 1;
			day = ts->day;
			break;
		default:
			g_assert_not_reached ();
			break;
		}
        }
        else {
                time_t now;
                struct tm *stm;

                now = time (NULL);
                stm = localtime (&now);
                year = stm->tm_year + 1900;
                month = stm->tm_mon;
                day = stm->tm_mday;
        }

        gtk_calendar_select_month (GTK_CALENDAR (mgtim->priv->date), month, year);
        gtk_calendar_select_day (GTK_CALENDAR (mgtim->priv->date), day);

        /* popup window */
        /* Temporarily grab pointer and keyboard, copied from GnomeDateEdit */
        if (!popup_grab_on_window (button->window, gtk_get_current_event_time ()))
                return;
        position_popup (mgtim);
        gtk_grab_add (mgtim->priv->window);
        gtk_widget_show (mgtim->priv->window);
        gtk_widget_grab_focus (mgtim->priv->date);
        popup_grab_on_window (mgtim->priv->window->window,
                              gtk_get_current_event_time ());
}

static gboolean
popup_grab_on_window (GdkWindow *window, guint32 activate_time)
{
        if ((gdk_pointer_grab (window, TRUE,
                               GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
                               GDK_POINTER_MOTION_MASK,
                               NULL, NULL, activate_time) == 0)) {
                if (gdk_keyboard_grab (window, TRUE,
                                       activate_time) == 0)
                        return TRUE;
                else {
                        gdk_pointer_ungrab (activate_time);
                        return FALSE;
                }
        }
        return FALSE;
}

static void
position_popup (GnomeDbEntryTime *mgtim)
{
        gint x, y;
        gint bwidth, bheight;
        GtkRequisition req;

        gtk_widget_size_request (mgtim->priv->window, &req);

        gdk_window_get_origin (mgtim->priv->date_button->window, &x, &y);

        x += mgtim->priv->date_button->allocation.x;
        y += mgtim->priv->date_button->allocation.y;
        bwidth = mgtim->priv->date_button->allocation.width;
        bheight = mgtim->priv->date_button->allocation.height;

        x += bwidth - req.width;
        y += bheight;

        if (x < 0)
                x = 0;

        if (y < 0)
                y = 0;

        gtk_window_move (GTK_WINDOW (mgtim->priv->window), x, y);
}




/*
 * callbacks for the time
 */
static GtkWidget *
create_entry_time (GnomeDbEntryTime *mgtim)
{
	GtkWidget *wid, *hb;

	hb = gtk_hbox_new (FALSE, 3);

	/* text entry */
        wid = gtk_entry_new ();
        gtk_box_pack_start (GTK_BOX (hb), wid, FALSE, FALSE, 0);
        gtk_entry_set_max_length (GTK_ENTRY (wid), 8); /* max for HH:MM:SS format */
	gtk_entry_set_width_chars (GTK_ENTRY (wid), 8);
        gtk_widget_show (wid);
        mgtim->priv->entry_time = wid;

        /* small label */
        wid = gtk_label_new (_("hh:mm:ss"));
        gtk_box_pack_start (GTK_BOX (hb), wid, FALSE, FALSE, 0);
        gtk_widget_show (wid);
	mgtim->priv->legend = wid;

        return hb;
}


/*
 * callbacks for the timestamp
 */
static GtkWidget *
create_entry_ts (GnomeDbEntryTime *mgtim)
{
	GtkWidget *hb, *wid;

	hb = gtk_hbox_new (FALSE, 0);
	
	/* date part */
	wid = create_entry_date (mgtim);
	gtk_box_pack_start (GTK_BOX (hb), wid, FALSE, FALSE, 0);
	gtk_widget_show (wid);

	/* time part */
	wid = create_entry_time (mgtim);
	gtk_box_pack_start (GTK_BOX (hb), wid, FALSE, FALSE, 0);
	gtk_widget_show (wid);

	mgtim->priv->hbox = hb;
	
	return hb;
}



/*
 * GtkCellEditable interface
 */
static void
gtk_cell_editable_entry_editing_done_cb (GtkEntry *entry, GnomeDbEntryTime *mgtim) 
{
	gtk_cell_editable_editing_done (GTK_CELL_EDITABLE (mgtim));
}

static void
gtk_cell_editable_entry_remove_widget_cb (GtkEntry *entry, GnomeDbEntryTime *mgtim) 
{
	gtk_cell_editable_remove_widget (GTK_CELL_EDITABLE (mgtim));
}

static void
gnome_db_entry_time_start_editing (GtkCellEditable *iface, GdkEvent *event)
{
	GnomeDbEntryTime *mgtim;

	g_return_if_fail (iface && IS_GNOME_DB_ENTRY_TIME (iface));
	mgtim = GNOME_DB_ENTRY_TIME (iface);
	g_return_if_fail (mgtim->priv);


	if (mgtim->priv->date_button) {
		gtk_widget_destroy (mgtim->priv->date_button);
		mgtim->priv->date_button = NULL;
	}
	if (mgtim->priv->legend) {
		gtk_widget_destroy (mgtim->priv->legend);
		mgtim->priv->legend = NULL;
	}
	if (mgtim->priv->hbox) {
		gtk_box_set_spacing (GTK_BOX (mgtim->priv->hbox), 0);
		gtk_container_set_border_width (GTK_CONTAINER (mgtim->priv->hbox), 0);
	}

	if (mgtim->priv->entry_date) {
		g_object_set (G_OBJECT (mgtim->priv->entry_date), "has_frame", FALSE, NULL);
		gtk_cell_editable_start_editing (GTK_CELL_EDITABLE (mgtim->priv->entry_date), event);
		g_signal_connect (G_OBJECT (mgtim->priv->entry_date), "editing_done",
				  G_CALLBACK (gtk_cell_editable_entry_editing_done_cb), mgtim);
		g_signal_connect (G_OBJECT (mgtim->priv->entry_date), "remove_widget",
				  G_CALLBACK (gtk_cell_editable_entry_remove_widget_cb), mgtim);
	}

	if (mgtim->priv->entry_time) {
		g_object_set (G_OBJECT (mgtim->priv->entry_time), "has_frame", FALSE, NULL);
		gtk_cell_editable_start_editing (GTK_CELL_EDITABLE (mgtim->priv->entry_time), event);
		g_signal_connect (G_OBJECT (mgtim->priv->entry_time), "editing_done",
				  G_CALLBACK (gtk_cell_editable_entry_editing_done_cb), mgtim);
		g_signal_connect (G_OBJECT (mgtim->priv->entry_time), "remove_widget",
				  G_CALLBACK (gtk_cell_editable_entry_remove_widget_cb), mgtim);
	}

	gnome_db_entry_shell_refresh (GNOME_DB_ENTRY_SHELL (mgtim));

	if (mgtim->priv->entry_date)
		gtk_widget_grab_focus (mgtim->priv->entry_date);
	else
		gtk_widget_grab_focus (mgtim->priv->entry_time);
	gtk_widget_queue_draw (GTK_WIDGET (mgtim));
}
