/* gnome-db-query.c
 *
 * Copyright (C) 2003 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-query.h"
#include "gnome-db-field.h"
#include "gnome-db-qfield.h"
#include "gnome-db-qf-field.h"
#include "gnome-db-qf-value.h"
#include "gnome-db-qf-all.h"
#include "gnome-db-ref-base.h"
#include "gnome-db-target.h"
#include "gnome-db-entity.h"
#include "gnome-db-table.h"
#include "gnome-db-database.h"
#include "gnome-db-constraint.h"
#include "gnome-db-xml-storage.h"
#include "gnome-db-referer.h"
#include "gnome-db-renderer.h"
#include "gnome-db-parameter.h"
#include "marshal.h"
#include "gnome-db-join.h"
#include <string.h>
#include "gnome-db-data-set.h"
#include "gnome-db-condition.h"
#include "gnome-db-server.h"
#include "gnome-db-server-function.h"
#include "gnome-db-server-aggregate.h"
#include "gnome-db-data-handler.h"
#include "gnome-db-qf-func.h"

#include "gnome-db-query-private.h"
#include "gnome-db-query-parsing.h"

struct _GnomeDbBasePrivate
{
	GnomeDbDict        *dict;     /* property: NOT NULL to use GnomeDbBase object */
	guint              id;	     /* 0 or UNIQUE, comes from DictManager->id_serial */

	gchar             *name;
	gchar             *descr;
	gchar             *owner;
	
	gboolean           nullified;/* TRUE if the object has been "nullified" */
	gboolean           changed_locked;
};

/* 
 * Main static functions 
 */
static void gnome_db_query_class_init (GnomeDbQueryClass * class);
static void gnome_db_query_init (GnomeDbQuery * srv);
static void gnome_db_query_dispose (GObject   * object);
static void gnome_db_query_finalize (GObject   * object);

static void gnome_db_query_set_property (GObject              *object,
					 guint                 param_id,
					 const GValue         *value,
					 GParamSpec           *pspec);
static void gnome_db_query_get_property (GObject              *object,
					 guint                 param_id,
					 GValue               *value,
					 GParamSpec           *pspec);
/* Entity interface */
static void        gnome_db_query_entity_init         (GnomeDbEntityIface *iface);
static gboolean    gnome_db_query_has_field           (GnomeDbEntity *iface, GnomeDbField *field);
static GSList     *gnome_db_query_get_fields          (GnomeDbEntity *iface);
static GnomeDbField    *gnome_db_query_get_field_by_name   (GnomeDbEntity *iface, const gchar *name);
static GnomeDbField    *gnome_db_query_get_field_by_xml_id (GnomeDbEntity *iface, const gchar *xml_id);
static GnomeDbField    *gnome_db_query_get_field_by_index  (GnomeDbEntity *iface, gint index);
static gint        gnome_db_query_get_field_index     (GnomeDbEntity *iface, GnomeDbField *field);
static void        gnome_db_query_add_field           (GnomeDbEntity *iface, GnomeDbField *field);
static void        gnome_db_query_add_field_before    (GnomeDbEntity *iface, GnomeDbField *field, GnomeDbField *field_before);
static void        gnome_db_query_swap_fields         (GnomeDbEntity *iface, GnomeDbField *field1, GnomeDbField *field2);
static void        gnome_db_query_remove_field        (GnomeDbEntity *iface, GnomeDbField *field);
static gboolean    gnome_db_query_is_writable         (GnomeDbEntity *iface);
static GSList     *gnome_db_query_get_parameters      (GnomeDbEntity *iface);

/* XML storage interface */
static void        gnome_db_query_xml_storage_init (GnomeDbXmlStorageIface *iface);
static gchar      *gnome_db_query_get_xml_id       (GnomeDbXmlStorage *iface);
static xmlNodePtr  gnome_db_query_save_to_xml      (GnomeDbXmlStorage *iface, GError **error);
static gboolean    gnome_db_query_load_from_xml    (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error);

/* Referer interface */
static void        gnome_db_query_referer_init        (GnomeDbRefererIface *iface);
static gboolean    gnome_db_query_activate            (GnomeDbReferer *iface);
static void        gnome_db_query_deactivate          (GnomeDbReferer *iface);
static gboolean    gnome_db_query_is_active           (GnomeDbReferer *iface);
static GSList     *gnome_db_query_get_ref_objects     (GnomeDbReferer *iface);
static void        gnome_db_query_replace_refs        (GnomeDbReferer *iface, GHashTable *replacements);

/* Renderer interface */
static void        gnome_db_query_renderer_init   (GnomeDbRendererIface *iface);
static GdaXqlItem *gnome_db_query_render_as_xql   (GnomeDbRenderer *iface, GnomeDbDataSet *context, GError **error);
static gchar      *gnome_db_query_render_as_sql   (GnomeDbRenderer *iface, GnomeDbDataSet *context, guint options, GError **error);
static gchar      *gnome_db_query_render_as_str   (GnomeDbRenderer *iface, GnomeDbDataSet *context);
static gboolean    gnome_db_query_is_valid        (GnomeDbRenderer *iface, GnomeDbDataSet *context, GError **error);

/* callbaks from sub objects */
static void nullified_target_cb (GnomeDbTarget *target, GnomeDbQuery *query);
static void nullified_field_cb (GnomeDbField *field, GnomeDbQuery *query);
static void nullified_join_cb (GnomeDbJoin *join, GnomeDbQuery *query);
static void nullified_cond_cb (GnomeDbCondition *cond, GnomeDbQuery *query);
static void nullified_sub_query_cb (GnomeDbQuery *sub_query, GnomeDbQuery *query);
static void nullified_parent_query (GnomeDbQuery *parent_query, GnomeDbQuery *query);

static void nullified_param_source_cb (GnomeDbQuery *param_source, GnomeDbQuery *query);

static void changed_target_cb (GnomeDbTarget *target, GnomeDbQuery *query);
static void changed_field_cb (GnomeDbField *field, GnomeDbQuery *query);
static void changed_join_cb (GnomeDbJoin *join, GnomeDbQuery *query);
static void changed_sub_query_cb (GnomeDbQuery *sub_query, GnomeDbQuery *query);
static void changed_cond_cb (GnomeDbCondition *cond, GnomeDbQuery *query);

static void change_parent_query (GnomeDbQuery *query, GnomeDbQuery *parent_query);

static void id_target_changed_cb (GnomeDbTarget *target, GnomeDbQuery *query);
static void id_field_changed_cb (GnomeDbQfield *field, GnomeDbQuery *query);
static void id_cond_changed_cb (GnomeDbCondition *cond, GnomeDbQuery *query);

static gboolean query_sql_forget (GnomeDbQuery *query, GError **error);
static void     query_clean_junk (GnomeDbQuery *query);

#ifdef debug
static void gnome_db_query_dump (GnomeDbQuery *table, guint offset);
#endif

typedef struct
{
	GSList *targets;
	GSList *joins;
} JoinsPack;
#define JOINS_PACK(x) ((JoinsPack *)x)


/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	TYPE_CHANGED,
	CONDITION_CHANGED,
	TARGET_ADDED,
	TARGET_REMOVED,
	TARGET_UPDATED,
	JOIN_ADDED,
	JOIN_REMOVED,
	JOIN_UPDATED,
	SUB_QUERY_ADDED,
	SUB_QUERY_REMOVED,
	SUB_QUERY_UPDATED,
	LAST_SIGNAL
};

static gint gnome_db_query_signals[LAST_SIGNAL] = { 0, 0 };

/* properties */
enum
{
	PROP_0,
	PROP_SERIAL_TARGET,
	PROP_SERIAL_FIELD,
	PROP_SERIAL_COND,
	PROP_REALLY_ALL_FIELDS,
	PROP_AUTO_CLEAN
};

/* module error */
GQuark gnome_db_query_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_query_error");
	return quark;
}


guint
gnome_db_query_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbQueryClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_query_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbQuery),
			0,
			(GInstanceInitFunc) gnome_db_query_init
		};

		static const GInterfaceInfo entity_info = {
			(GInterfaceInitFunc) gnome_db_query_entity_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo xml_storage_info = {
			(GInterfaceInitFunc) gnome_db_query_xml_storage_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo referer_info = {
			(GInterfaceInitFunc) gnome_db_query_referer_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo renderer_info = {
			(GInterfaceInitFunc) gnome_db_query_renderer_init,
			NULL,
			NULL
		};
		
		type = g_type_register_static (GNOME_DB_BASE_TYPE, "GnomeDbQuery", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_ENTITY_TYPE, &entity_info);
		g_type_add_interface_static (type, GNOME_DB_XML_STORAGE_TYPE, &xml_storage_info);
		g_type_add_interface_static (type, GNOME_DB_REFERER_TYPE, &referer_info);
		g_type_add_interface_static (type, GNOME_DB_RENDERER_TYPE, &renderer_info);
	}
	return type;
}

static void
gnome_db_query_entity_init (GnomeDbEntityIface *iface)
{
	iface->has_field = gnome_db_query_has_field;
	iface->get_fields = gnome_db_query_get_fields;
	iface->get_field_by_name = gnome_db_query_get_field_by_name;
	iface->get_field_by_xml_id = gnome_db_query_get_field_by_xml_id;
	iface->get_field_by_index = gnome_db_query_get_field_by_index;
	iface->get_field_index = gnome_db_query_get_field_index;
	iface->add_field = gnome_db_query_add_field;
	iface->add_field_before = gnome_db_query_add_field_before;
	iface->swap_fields = gnome_db_query_swap_fields;
	iface->remove_field = gnome_db_query_remove_field;
	iface->is_writable = gnome_db_query_is_writable;
	iface->get_parameters = gnome_db_query_get_parameters;
}

static void 
gnome_db_query_xml_storage_init (GnomeDbXmlStorageIface *iface)
{
	iface->get_xml_id = gnome_db_query_get_xml_id;
	iface->save_to_xml = gnome_db_query_save_to_xml;
	iface->load_from_xml = gnome_db_query_load_from_xml;
}

static void
gnome_db_query_referer_init (GnomeDbRefererIface *iface)
{
	iface->activate = gnome_db_query_activate;
	iface->deactivate = gnome_db_query_deactivate;
	iface->is_active = gnome_db_query_is_active;
	iface->get_ref_objects = gnome_db_query_get_ref_objects;
	iface->replace_refs = gnome_db_query_replace_refs;
}

static void
gnome_db_query_renderer_init (GnomeDbRendererIface *iface)
{
	iface->render_as_xql = gnome_db_query_render_as_xql;
	iface->render_as_sql = gnome_db_query_render_as_sql;
	iface->render_as_str = gnome_db_query_render_as_str;
	iface->is_valid = gnome_db_query_is_valid;
}

static void m_changed_cb (GnomeDbQuery *query);
static void
gnome_db_query_class_init (GnomeDbQueryClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	gnome_db_query_signals[TYPE_CHANGED] =
		g_signal_new ("type_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbQueryClass, type_changed),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	gnome_db_query_signals[CONDITION_CHANGED] =
		g_signal_new ("condition_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbQueryClass, condition_changed),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	gnome_db_query_signals[TARGET_ADDED] =
		g_signal_new ("target_added",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbQueryClass, target_added),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	gnome_db_query_signals[TARGET_REMOVED] =
		g_signal_new ("target_removed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbQueryClass, target_removed),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	gnome_db_query_signals[TARGET_UPDATED] =
		g_signal_new ("target_updated",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbQueryClass, target_updated),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	gnome_db_query_signals[JOIN_ADDED] =
		g_signal_new ("join_added",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbQueryClass, join_added),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	gnome_db_query_signals[JOIN_REMOVED] =
		g_signal_new ("join_removed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbQueryClass, join_removed),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	gnome_db_query_signals[JOIN_UPDATED] =
		g_signal_new ("join_updated",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbQueryClass, join_updated),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	gnome_db_query_signals[SUB_QUERY_ADDED] =
		g_signal_new ("sub_query_added",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbQueryClass, sub_query_added),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	gnome_db_query_signals[SUB_QUERY_REMOVED] =
		g_signal_new ("sub_query_removed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbQueryClass, sub_query_removed),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	gnome_db_query_signals[SUB_QUERY_UPDATED] =
		g_signal_new ("sub_query_updated",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbQueryClass, sub_query_updated),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);

	class->type_changed = m_changed_cb;
	class->condition_changed = m_changed_cb;
	class->target_added = m_changed_cb;
	class->target_removed = NULL;
	class->target_updated = m_changed_cb;
	class->join_added = m_changed_cb;
	class->join_removed = NULL;
	class->join_updated = m_changed_cb;
	class->sub_query_added = NULL;
	class->sub_query_removed = NULL;
	class->sub_query_updated = NULL;

	object_class->dispose = gnome_db_query_dispose;
	object_class->finalize = gnome_db_query_finalize;

	/* Properties */
	object_class->set_property = gnome_db_query_set_property;
	object_class->get_property = gnome_db_query_get_property;
	g_object_class_install_property (object_class, PROP_SERIAL_TARGET,
					 g_param_spec_uint ("target_serial", NULL, NULL, 
							    1, G_MAXUINT, 1, G_PARAM_READABLE));
	g_object_class_install_property (object_class, PROP_SERIAL_FIELD,
					 g_param_spec_uint ("field_serial", NULL, NULL, 
							    1, G_MAXUINT, 1, G_PARAM_READABLE));
	g_object_class_install_property (object_class, PROP_SERIAL_COND,
					 g_param_spec_uint ("cond_serial", NULL, NULL, 
							    1, G_MAXUINT, 1, G_PARAM_READABLE));
	g_object_class_install_property (object_class, PROP_REALLY_ALL_FIELDS,
					 g_param_spec_pointer ("really_all_fields", NULL, NULL, G_PARAM_READABLE));

	g_object_class_install_property (object_class, PROP_AUTO_CLEAN,
					 g_param_spec_boolean ("auto_clean", "Auto cleaning", 
							       "Determines if the query tries to clean unused objects", TRUE,
                                                               G_PARAM_READABLE | G_PARAM_WRITABLE));

	/* virtual functions */
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_query_dump;
#endif

}

static void
m_changed_cb (GnomeDbQuery *query)
{
	if (!query->priv->internal_transaction)
		gnome_db_base_changed (GNOME_DB_BASE (query));
}

static void
gnome_db_query_init (GnomeDbQuery * gnome_db_query)
{
	gnome_db_query->priv = g_new0 (GnomeDbQueryPrivate, 1);
	gnome_db_query->priv->query_type = GNOME_DB_QUERY_TYPE_SELECT;
	gnome_db_query->priv->targets = NULL;
	gnome_db_query->priv->joins_flat = NULL;
	gnome_db_query->priv->joins_pack = NULL;
	gnome_db_query->priv->fields = NULL;
	gnome_db_query->priv->sub_queries = NULL;
	gnome_db_query->priv->param_sources = NULL;
	gnome_db_query->priv->cond = NULL;
	gnome_db_query->priv->parent_query = NULL;
	gnome_db_query->priv->sql = NULL;
	gnome_db_query->priv->sql_exprs = NULL;
	gnome_db_query->priv->fields_order_by = NULL;

	gnome_db_query->priv->serial_target = 1;
	gnome_db_query->priv->serial_field = 1;
	gnome_db_query->priv->serial_cond = 1;
	gnome_db_query->priv->internal_transaction = 0;

	gnome_db_query->priv->all_conds = NULL;
	gnome_db_query->priv->auto_clean = TRUE;
}

/**
 * gnome_db_query_new
 * @dict: a #GnomeDbDict object
 *
 * Creates a new #GnomeDbQuery object
 *
 * Returns: the new object
 */
GObject*
gnome_db_query_new (GnomeDbDict *dict)
{
	GObject   *obj;
	GnomeDbQuery *gnome_db_query;
	guint id;

	g_return_val_if_fail (!dict || IS_GNOME_DB_DICT (dict), NULL);
	obj = g_object_new (GNOME_DB_QUERY_TYPE, "dict", ASSERT_DICT (dict), NULL);
	gnome_db_query = GNOME_DB_QUERY (obj);

	g_object_get (G_OBJECT (ASSERT_DICT (dict)), "query_serial", &id, NULL);
	gnome_db_base_set_id (GNOME_DB_BASE (obj), id);

	gnome_db_dict_declare_query (ASSERT_DICT (dict), gnome_db_query);	

	return obj;
}


/**
 * gnome_db_query_new_copy
 * @orig: a #GnomeDbQuery to make a copy of
 * @replacements: a hash table to store replacements, or %NULL
 * 
 * Copy constructor
 *
 * Returns: a the new copy of @orig
 */
GObject *
gnome_db_query_new_copy (GnomeDbQuery *orig, GHashTable *replacements)
{
	GObject *obj;
	GnomeDbQuery *gnome_db_query;
	GnomeDbDict *dict;
	GHashTable *repl;
	GSList *list;
	guint id;
	gint order_pos;

	g_return_val_if_fail (orig && IS_GNOME_DB_QUERY (orig), NULL);

	dict = gnome_db_base_get_dict (GNOME_DB_BASE (orig));
	obj = g_object_new (GNOME_DB_QUERY_TYPE, "dict", dict, NULL);
	gnome_db_query = GNOME_DB_QUERY (obj);

	gnome_db_query->priv->internal_transaction ++;

	g_object_get (G_OBJECT (dict), "query_serial", &id, NULL);
	gnome_db_base_set_id (GNOME_DB_BASE (obj), id);
	gnome_db_dict_declare_query (dict, gnome_db_query);	

	/* hash table for replacements */
	if (!replacements)
		repl = g_hash_table_new (NULL, NULL);
	else
		repl = replacements;

	g_hash_table_insert (repl, orig, gnome_db_query);

	/* REM: parent query link is handled by the caller of this method */
	
	/* copy starts */
	gnome_db_base_set_name (GNOME_DB_BASE (gnome_db_query), gnome_db_base_get_name (GNOME_DB_BASE (orig)));
	gnome_db_base_set_description (GNOME_DB_BASE (gnome_db_query), gnome_db_base_get_description (GNOME_DB_BASE (orig)));
	gnome_db_query->priv->query_type = orig->priv->query_type;
	if (orig->priv->sql)
		gnome_db_query->priv->sql = g_strdup (orig->priv->sql);
	if (orig->priv->sql_exprs)
		gnome_db_query->priv->sql_exprs = gnome_db_sql_parse_copy_statement (orig->priv->sql_exprs);
	gnome_db_query->priv->global_distinct = orig->priv->global_distinct;

	/* copy sub queries */
	list = orig->priv->sub_queries;
	while (list) {
		GnomeDbQuery *copy = GNOME_DB_QUERY (gnome_db_query_new_copy (GNOME_DB_QUERY (list->data), repl));
		gnome_db_referer_replace_refs (GNOME_DB_REFERER (copy), repl);
		gnome_db_query_add_sub_query (gnome_db_query, copy);
		g_object_unref (G_OBJECT (copy));
		
		list = g_slist_next (list);
	}

	/* copy param sources */
	list = orig->priv->param_sources;
	while (list) {
		GnomeDbQuery *copy = GNOME_DB_QUERY (gnome_db_query_new_copy (GNOME_DB_QUERY (list->data), repl));
		gnome_db_referer_replace_refs (GNOME_DB_REFERER (copy), repl);
		gnome_db_query_add_param_source (gnome_db_query, copy);
		g_object_unref (G_OBJECT (copy));
		
		list = g_slist_next (list);
	}

	/* copy targets */
	list = orig->priv->targets;
	while (list) {
		guint id;
		GnomeDbTarget *target = GNOME_DB_TARGET (gnome_db_target_new_copy (GNOME_DB_TARGET (list->data)));

		gnome_db_referer_replace_refs (GNOME_DB_REFERER (target), repl);
		gnome_db_query_add_target (gnome_db_query, target, NULL);
		g_object_get (G_OBJECT (gnome_db_query), "target_serial", &id, NULL);
		gnome_db_base_set_id (GNOME_DB_BASE (target), id);

		g_object_unref (G_OBJECT (target));
		g_hash_table_insert (repl, list->data, target);
		list = g_slist_next (list);
	}

	/* copy fields */
	list = orig->priv->fields;
	while (list) {
		guint id;
		GnomeDbQfield *qf = GNOME_DB_QFIELD (gnome_db_qfield_new_copy (GNOME_DB_QFIELD (list->data)));
		gnome_db_referer_replace_refs (GNOME_DB_REFERER (qf), repl);
		gnome_db_query_add_field (GNOME_DB_ENTITY (gnome_db_query), GNOME_DB_FIELD (qf));
		g_object_get (G_OBJECT (gnome_db_query), "field_serial", &id, NULL);
		gnome_db_base_set_id (GNOME_DB_BASE (qf), id);

		g_object_unref (G_OBJECT (qf));
		g_hash_table_insert (repl, list->data, qf);
		list = g_slist_next (list);
	}

	/* copy joins */
	list = orig->priv->joins_flat;
	while (list) {
		GnomeDbJoin *join = GNOME_DB_JOIN (gnome_db_join_new_copy (GNOME_DB_JOIN (list->data), repl));
		gnome_db_referer_replace_refs (GNOME_DB_REFERER (join), repl);
		gnome_db_query_add_join (gnome_db_query, join);
		g_object_unref (G_OBJECT (join));
		g_hash_table_insert (repl, list->data, join);
		list = g_slist_next (list);
	}

	/* copy condition */
	if (orig->priv->cond) {
		GnomeDbCondition *cond;
		guint id;

		cond = GNOME_DB_CONDITION (gnome_db_condition_new_copy (orig->priv->cond, repl));
		g_object_get (G_OBJECT (gnome_db_query), "cond_serial", &id, NULL);
		gnome_db_base_set_id (GNOME_DB_BASE (cond), id);
		gnome_db_query_set_condition (gnome_db_query, cond);

		g_object_unref (G_OBJECT (cond));
		g_hash_table_insert (repl, orig->priv->cond, cond);
	}

	/* copy ORDER BY fields list */
	order_pos = 0;
	list = orig->priv->fields_order_by;
	while (list) {
		GnomeDbQfield *field;
		gboolean asc;

		asc = g_object_get_data (G_OBJECT (list->data), "order_by_asc") ? TRUE : FALSE;
		field = g_hash_table_lookup (repl, list->data);
		gnome_db_query_set_order_by_field (gnome_db_query, field, order_pos, asc);

		order_pos++;
		list = g_slist_next (list);
	}
		
	/* Another loop to make sure all the references have been replaced */
	gnome_db_query_replace_refs (GNOME_DB_REFERER (gnome_db_query), repl);

	if (!replacements)
		g_hash_table_destroy (repl);

	gnome_db_query->priv->internal_transaction --;

#ifdef debug_NO
	g_print ("Query %p is a copy of %p\n", gnome_db_query, orig);
	gnome_db_base_dump (GNOME_DB_BASE (orig), 2);
	gnome_db_base_dump (GNOME_DB_BASE (gnome_db_query), 2);
#endif

	return obj;	
}

/**
 * gnome_db_query_new_from_sql
 * @dict: a #GnomeDbDict object
 * @sql: an SQL statement
 * @error: location to store error, or %NULL
 *
 * Creates a new #GnomeDbQuery object and fills its structure by parsing the @sql. If the parsing failed,
 * then the returned query is of type GNOME_DB_QUERY_TYPE_NON_PARSED_SQL.
 *
 * To be parsed successfully, the expected SQL must respect the SQL standard; some extensions have been
 * added to be able to define variables within the SQL statement. See the introduction to the #GnomeDbQuery
 * for more information. 
 *
 * The @error is set only if the SQL statement parsing produced an error; there is always a new #GnomeDbQuery
 * object which is returned.
 *
 * Returns: a new #GnomeDbQuery
 */
GObject *
gnome_db_query_new_from_sql (GnomeDbDict *dict, const gchar *sql, GError **error) 
{
	GnomeDbQuery *query;

	query = (GnomeDbQuery *) gnome_db_query_new (dict);
	gnome_db_query_set_sql_text (query, sql, error);

	return (GObject*) query;
}


static void gnome_db_query_clean (GnomeDbQuery *gnome_db_query);
static void
gnome_db_query_dispose (GObject *object)
{
	GnomeDbQuery *gnome_db_query;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_QUERY (object));

	gnome_db_query = GNOME_DB_QUERY (object);
	if (gnome_db_query->priv) {
		gnome_db_base_nullify_check (GNOME_DB_BASE (object));
		
		gnome_db_query_clean (gnome_db_query);
	}

	/* parent class */
	parent_class->dispose (object);
}

/* completely cleans the query's contents */
static void
gnome_db_query_clean (GnomeDbQuery *gnome_db_query)
{
	/* order by list */
	if (gnome_db_query->priv->fields_order_by) {
		g_slist_free (gnome_db_query->priv->fields_order_by);
		gnome_db_query->priv->fields_order_by = NULL;
	}
	
	/* parent query */
	if (gnome_db_query->priv->parent_query)
		change_parent_query (gnome_db_query, NULL);
	
	/* condition */
	if (gnome_db_query->priv->cond)
		gnome_db_base_nullify (GNOME_DB_BASE (gnome_db_query->priv->cond));
	
	/* parameter sources */
	while (gnome_db_query->priv->param_sources) 
		gnome_db_base_nullify (GNOME_DB_BASE (gnome_db_query->priv->param_sources->data));
	
	/* sub queries */
	while (gnome_db_query->priv->sub_queries) 
		gnome_db_base_nullify (GNOME_DB_BASE (gnome_db_query->priv->sub_queries->data));
	
	/* joins */
	while (gnome_db_query->priv->joins_flat) 
		gnome_db_base_nullify (GNOME_DB_BASE (gnome_db_query->priv->joins_flat->data));
	
	/* fields */
	while (gnome_db_query->priv->fields) 
		gnome_db_base_nullify (GNOME_DB_BASE (gnome_db_query->priv->fields->data));
	
	/* targets */
	while (gnome_db_query->priv->targets) 
		gnome_db_base_nullify (GNOME_DB_BASE (gnome_db_query->priv->targets->data));

	/* SQL statement */
	if (gnome_db_query->priv->sql) {
		g_free (gnome_db_query->priv->sql);
		gnome_db_query->priv->sql = NULL;
	}
	if (gnome_db_query->priv->sql_exprs) {
		gnome_db_sql_destroy (gnome_db_query->priv->sql_exprs);
		gnome_db_query->priv->sql_exprs = NULL;
	}

	/* restore safe default values */
	gnome_db_query->priv->query_type = GNOME_DB_QUERY_TYPE_SELECT;
	gnome_db_query->priv->serial_target = 1;
	gnome_db_query->priv->serial_field = 1;
	gnome_db_query->priv->serial_cond = 1;

	/* internal coherence test */
#ifdef debug
	if (gnome_db_query->priv->all_conds) {
		g_print ("Query incoherence error!!!\n");
		gnome_db_base_dump (gnome_db_query, 0);
	}
#endif
	g_assert (!gnome_db_query->priv->all_conds);

	if (!gnome_db_query->priv->internal_transaction)
		gnome_db_base_changed (GNOME_DB_BASE (gnome_db_query));
}

static void
gnome_db_query_finalize (GObject *object)
{
	GnomeDbQuery *gnome_db_query;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_QUERY (object));

	gnome_db_query = GNOME_DB_QUERY (object);
	if (gnome_db_query->priv) {
		g_free (gnome_db_query->priv);
		gnome_db_query->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_query_set_property (GObject              *object,
			     guint                 param_id,
			     const GValue         *value,
			     GParamSpec           *pspec)
{
	GnomeDbQuery *gnome_db_query;

	gnome_db_query = GNOME_DB_QUERY (object);
	if (gnome_db_query->priv) {
		switch (param_id) {
		case PROP_SERIAL_TARGET:
			break;
		case PROP_AUTO_CLEAN:
			gnome_db_query->priv->auto_clean = g_value_get_boolean (value);
			break;
		}
	}
}

static void
gnome_db_query_get_property (GObject              *object,
			     guint                 param_id,
			     GValue               *value,
			     GParamSpec           *pspec)
{
	GnomeDbQuery *gnome_db_query;
	gnome_db_query = GNOME_DB_QUERY (object);
	
	if (gnome_db_query->priv) {
		switch (param_id) {
		case PROP_SERIAL_TARGET:
			g_value_set_uint (value, gnome_db_query->priv->serial_target++);
			break;
		case PROP_SERIAL_FIELD:
			g_value_set_uint (value, gnome_db_query->priv->serial_field++);
			break;
		case PROP_SERIAL_COND:
			g_value_set_uint (value, gnome_db_query->priv->serial_cond++);
			break;
		case PROP_REALLY_ALL_FIELDS:
			g_value_set_pointer (value, gnome_db_query->priv->fields);
			break;
		case PROP_AUTO_CLEAN:
			g_value_set_boolean (value, gnome_db_query->priv->auto_clean);
			break;
		}	
	}
}

static void cond_weak_ref_notify (GnomeDbQuery *query, GnomeDbCondition *cond);


/**
 * gnome_db_query_declare_condition
 * @query: a #GnomeDbQuery object
 * @cond: a #GnomeDbCondition object
 *
 * Declares the existence of a new condition to @query. All the #GnomeDbCondition objects MUST
 * be declared to the corresponding #GnomeDbQuery object for the library to work correctly.
 * Once @cond has been declared, @query does not hold any reference to @cond.
 *
 * This functions is called automatically from each gnome_db_condition_new* function, and it should not be necessary 
 * to call it except for classes extending the #GnomeDbCondition class.
 */
void
gnome_db_query_declare_condition (GnomeDbQuery *query, GnomeDbCondition *cond)
{
	g_return_if_fail (query && IS_GNOME_DB_QUERY (query));
	g_return_if_fail (query->priv);
	g_return_if_fail (cond && IS_GNOME_DB_CONDITION (cond));
	
	/* we don't take any reference on condition */
	if (g_slist_find (query->priv->all_conds, cond))
		return;	
	else {
		query->priv->all_conds = g_slist_append (query->priv->all_conds, cond);
		g_object_weak_ref (G_OBJECT (cond), (GWeakNotify) cond_weak_ref_notify, query);

		/* make sure the query->priv->serial_cond value is always 1 above this cond's id */
		id_cond_changed_cb (cond, query);
		g_signal_connect (G_OBJECT (cond), "id_changed",
				  G_CALLBACK (id_cond_changed_cb), query);
	}
}

/**
 * gnome_db_query_undeclare_condition
 * @query: a #GnomeDbQuery object
 * @cond: a #GnomeDbCondition object
 *
 * Explicitely ask @query to forget about the existence of @cond. This function is used by the
 * #GnomeDbCondition implementation, and should not be called directly
 */
void
gnome_db_query_undeclare_condition (GnomeDbQuery *query, GnomeDbCondition *cond)
{
	g_return_if_fail (query && IS_GNOME_DB_QUERY (query));
	g_return_if_fail (query->priv);
	g_return_if_fail (cond && IS_GNOME_DB_CONDITION (cond));

	cond_weak_ref_notify (query, cond);
	g_object_weak_unref (G_OBJECT (cond), (GWeakNotify) cond_weak_ref_notify, query);
}

static void
cond_weak_ref_notify (GnomeDbQuery *query, GnomeDbCondition *cond)
{
	query->priv->all_conds = g_slist_remove (query->priv->all_conds, cond);
	g_signal_handlers_disconnect_by_func (G_OBJECT (cond),
					      G_CALLBACK (id_cond_changed_cb), query);
}

static void
id_cond_changed_cb (GnomeDbCondition *cond, GnomeDbQuery *query)
{
	if (query->priv->serial_cond <= gnome_db_base_get_id (GNOME_DB_BASE (cond)))
		query->priv->serial_cond = gnome_db_base_get_id (GNOME_DB_BASE (cond)) + 1;
}

/*
 * This function is called whenever the structure of the query is changed (adding/removing targets, etc), and
 * make sure that the query is either in the following modes:
 * 1- the query structure is described using targets, fields, etc and its type is anything but GNOME_DB_QUERY_TYPE_NON_PARSED_SQL
 *    and query->priv->sql is NULL and query->priv->sql_exprs is NULL
 * or
 * 2- the query is described by a SQL text (in query->priv->sql), query->priv->sql_exprs may be NULL or not, and
 *    there might be some GnomeDbQfValue query fields (to describe parameters, in which case query->priv->sql_exprs is not NULL).
 *    In this mode the query type is GNOME_DB_QUERY_TYPE_NON_PARSED_SQL.
 *
 * If the query is a GNOME_DB_QUERY_TYPE_NON_PARSED_SQL, then nothing can be done and
 * an error is returned
 */
static gboolean
query_sql_forget (GnomeDbQuery *query, GError **error)
{
	/* return FALSE if we can't switch to non SQL mode */
	if (query->priv->query_type == GNOME_DB_QUERY_TYPE_NON_PARSED_SQL) {
		g_set_error (error, GNOME_DB_QUERY_ERROR, GNOME_DB_QUERY_STRUCTURE_ERROR, 
			     _("Can't modify the structure of a non parsed SQL query"));
		return FALSE;
	}

	/* free SQL if any */
	if (query->priv->sql) {
		g_free (query->priv->sql);
		query->priv->sql = NULL;
	}
	if (query->priv->sql_exprs) {
		gnome_db_sql_destroy (query->priv->sql_exprs);
		query->priv->sql_exprs = NULL;
	}

	return TRUE;
}

/*
 * Removes query fields which are useles: hidden fields not used by anything
 *
 * Note that this function doesn't don anything when @query's type
 * is GNOME_DB_QUERY_TYPE_NON_PARSED_SQL.
 */
static void
query_clean_junk (GnomeDbQuery *query)
{
	GSList *list;
	
	/* while in an internal transaction, don't do anything */
	if (query->priv->internal_transaction || !query->priv->auto_clean)
		return;

	query->priv->internal_transaction++;

	if (query->priv->query_type == GNOME_DB_QUERY_TYPE_NON_PARSED_SQL)
		return;

	/* 
	 * finding which fields need to be removed
	 */
	list = query->priv->fields;
	while (list) {
		GnomeDbQfield *qfield = GNOME_DB_QFIELD (list->data);
		gboolean used = TRUE;

		if (! gnome_db_qfield_is_visible (qfield)) {
			GSList *list2;

			used = FALSE;

			/* looking if field is used in WHERE clause */
			if (query->priv->cond) {
				list2 = gnome_db_condition_get_ref_objects_all (query->priv->cond);
				if (g_slist_find (list2, qfield))
					used = TRUE;
				g_slist_free (list2);
			}

			/* looking if field is used in a join's condition */
			list2 = query->priv->joins_flat;
			while (list2 && !used) {
				GnomeDbCondition *cond = gnome_db_join_get_condition (GNOME_DB_JOIN (list2->data));
				if (cond) {
					GSList *list3 = gnome_db_condition_get_ref_objects_all (cond);
					if (g_slist_find (list3, qfield))
						used = TRUE;
					g_slist_free (list3);
				}
				list2 = g_slist_next (list2);
			}

			/* looking for other fields using that field */
			if (!used && query->priv->sub_queries) {
				GSList *queries = query->priv->sub_queries;
				while (queries && !used) {
					list2 = gnome_db_referer_get_ref_objects (GNOME_DB_REFERER (queries->data));
					if (g_slist_find (list2, qfield))
						used = TRUE;
					g_slist_free (list2);
					
					queries = g_slist_next (queries);
				}
			}

			list2 = query->priv->fields;
			while (list2 && !used) {
				if (list2->data != (gpointer) qfield) {
					GSList *list3 = gnome_db_referer_get_ref_objects (GNOME_DB_REFERER (list2->data));
					if (g_slist_find (list3, qfield))
						used = TRUE;
					g_slist_free (list3);
				}
				list2 = g_slist_next (list2);
			}

			if (!used) {
#ifdef debug_NO
				g_print ("Removed field %p:\n", qfield);
#endif
				gnome_db_base_nullify (GNOME_DB_BASE (qfield));
				list = query->priv->fields;
			}
		}
		
		if (used)
			list = g_slist_next (list);
	}

	/* 
	 * finding which sub queries need to be removed: sub queries are used by:
	 * - GnomeDbTarget objects
	 * - GnomeDbQfSelect kind of field
	 */
	list = query->priv->sub_queries;
	while (list) {
		GSList *list2;
		gboolean used = FALSE;
		GnomeDbQuery *subq = GNOME_DB_QUERY (list->data);
		
		/* looking if field is used in any target */
		list2 = query->priv->targets;
		while (list2 && !used) {
			GSList *list3 = gnome_db_referer_get_ref_objects (GNOME_DB_REFERER (list2->data));
			if (g_slist_find (list3, subq))
				used = TRUE;
			g_slist_free (list3);
			list2 = g_slist_next (list2);
		}
		
		if (!used) {
			gnome_db_base_nullify (GNOME_DB_BASE (subq));
			list = query->priv->sub_queries;
		}
		else
			list = g_slist_next (list);
	}

	query->priv->internal_transaction--;

	gnome_db_base_changed (GNOME_DB_BASE (query));
}

static void
id_target_changed_cb (GnomeDbTarget *target, GnomeDbQuery *query)
{
	if (query->priv->serial_target <= gnome_db_base_get_id (GNOME_DB_BASE (target)))
		query->priv->serial_target = gnome_db_base_get_id (GNOME_DB_BASE (target)) + 1;
}

static void
id_field_changed_cb (GnomeDbQfield *field, GnomeDbQuery *query)
{
	if (query->priv->serial_field <= gnome_db_base_get_id (GNOME_DB_BASE (field))) 
		query->priv->serial_field = gnome_db_base_get_id (GNOME_DB_BASE (field)) + 1;
}

/**
 * gnome_db_query_set_query_type
 * @query: a #GnomeDbQuery object
 * @type: the new type of query
 *
 * Sets the type of @query
 */
void
gnome_db_query_set_query_type (GnomeDbQuery *query, GnomeDbQueryType type)
{
	g_return_if_fail (query && IS_GNOME_DB_QUERY (query));
	g_return_if_fail (query->priv);

	if (query->priv->query_type != type) {
		query->priv->internal_transaction ++;
		gnome_db_query_clean (query);
		query->priv->internal_transaction --;
		query->priv->query_type = type;
		    
#ifdef debug_signal
		g_print (">> 'TYPE_CHANGED' from %s\n", __FUNCTION__);
#endif
		g_signal_emit_by_name (G_OBJECT (query), "type_changed");
#ifdef debug_signal
		g_print ("<< 'TYPE_CHANGED' from %s\n", __FUNCTION__);
#endif	
	}
}

/**
 * gnome_db_query_get_query_type
 * @query: a #GnomeDbQuery object
 *
 * Get the type of a query
 *
 * Returns: the type of @query
 */
GnomeDbQueryType
gnome_db_query_get_query_type (GnomeDbQuery *query)
{
	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), GNOME_DB_QUERY_TYPE_SELECT);
	g_return_val_if_fail (query->priv, GNOME_DB_QUERY_TYPE_SELECT);
	
	return query->priv->query_type;
}

/**
 * gnome_db_query_get_query_type_string
 * @query: a #GnomeDbQuery object
 *
 * Get the type of a query as a human readable string
 *
 * Returns: a string for the type of @query
 */
const gchar *
gnome_db_query_get_query_type_string  (GnomeDbQuery *query)
{
	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (query->priv, NULL);

	switch (query->priv->query_type) {
	case GNOME_DB_QUERY_TYPE_SELECT:
		return _("Select");
	case GNOME_DB_QUERY_TYPE_INSERT:
		return _("Insert");
	case GNOME_DB_QUERY_TYPE_UPDATE:
		return _("Update");
	case GNOME_DB_QUERY_TYPE_DELETE:
		return _("Delete");
	case GNOME_DB_QUERY_TYPE_UNION:
		return _("Select (union)");
	case GNOME_DB_QUERY_TYPE_INTERSECT:
		return _("Select (intersection)");
	case GNOME_DB_QUERY_TYPE_EXCEPT:
		return _("Select (exception)");
	case GNOME_DB_QUERY_TYPE_NON_PARSED_SQL:
		return _("SQL text");
	default:
		g_assert_not_reached ();
	}

	return NULL;
}

/**
 * gnome_db_query_is_select_query
 * @query: a # GnomeDbQuery object
 *
 * Tells if @query is a SELECTION query (a simple SELECT, UNION, INTERSECT or EXCEPT); pure SQL
 * queries are not handled and will always return FALSE.
 *
 * Returns: TRUE if @query is a selection query
 */
gboolean
gnome_db_query_is_select_query (GnomeDbQuery *query)
{
	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), FALSE);
	g_return_val_if_fail (query->priv, FALSE);

	if ((query->priv->query_type == GNOME_DB_QUERY_TYPE_SELECT) ||
	    (query->priv->query_type == GNOME_DB_QUERY_TYPE_UNION) ||
	    (query->priv->query_type == GNOME_DB_QUERY_TYPE_INTERSECT) ||
	    (query->priv->query_type == GNOME_DB_QUERY_TYPE_EXCEPT))
		return TRUE;
	else
		return FALSE;
}

/**
 * gnome_db_query_is_modif_query
 * @query: a # GnomeDbQuery object
 *
 * Tells if @query is a modification query (a simple UPDATE, DELETE, INSERT).; pure SQL
 * queries are not handled and will always return FALSE.
 *
 * Returns: TRUE if @query is a modification query
 */
gboolean
gnome_db_query_is_modif_query (GnomeDbQuery *query)
{
	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), FALSE);
	g_return_val_if_fail (query->priv, FALSE);

	if ((query->priv->query_type == GNOME_DB_QUERY_TYPE_INSERT) ||
	    (query->priv->query_type == GNOME_DB_QUERY_TYPE_DELETE) ||
	    (query->priv->query_type == GNOME_DB_QUERY_TYPE_UPDATE))
		return TRUE;
	else
		return FALSE;
}


/**
* gnome_db_query_set_sql_text
* @query: a # GnomeDbQuery object
* @sql: the SQL statement
* @error: location to store parsing error, or %NULL
*
* Defines @query's contents from an SQL statement. The SQL text is parsed and the internal query structured
* is built from that; the query type is also set. If the SQL text cannot be parsed, then the internal structure
* of the query is emptied and the query type is set to GNOME_DB_QUERY_TYPE_NON_PARSED_SQL.
*
* To be parsed successfully, the expected SQL must respect the SQL standard; some extensions have been
* added to be able to define variables within the SQL statement. See the introduction to the #GnomeDbQuery
* for more information.
*/
void
gnome_db_query_set_sql_text (GnomeDbQuery *query, const gchar *sql, GError **error)
{
	sql_statement *result;
	gboolean err = FALSE;

	g_return_if_fail (query && IS_GNOME_DB_QUERY (query));
	g_return_if_fail (query->priv);

	if (query->priv->sql) {
		g_free (query->priv->sql);
		query->priv->sql = NULL;
	}
	if (query->priv->sql_exprs) {
		gnome_db_sql_destroy (query->priv->sql_exprs);
		query->priv->sql_exprs = NULL;
	}
	if (!sql || !(*sql))
		return;

	/* start internal transaction */
	query->priv->internal_transaction ++;

	/* try to parse the SQL */
	result = sql_parse_with_error (sql, error);
	if (result) {
#ifdef debug_NO
		sql_display (result);
#endif
		switch (result->type) {
		case SQL_select: 
			err = !parsed_create_select_query (query, (sql_select_statement *) result->statement, error);
			break;
		case SQL_insert:
			err = !parsed_create_insert_query (query, (sql_insert_statement *) result->statement, error);
			break;
		case SQL_delete:
			err = !parsed_create_delete_query (query, (sql_delete_statement *) result->statement, error);
			break;
		case SQL_update:
			err = !parsed_create_update_query (query, (sql_update_statement *) result->statement, error);
			break;
		}
		sql_destroy (result);
	}

	if (!result || err) {
		GnomeDbSqlStatement *stm;

		/* the query can't be parsed with libgda's parser */
		if (error && !(*error))
			g_set_error (error, GNOME_DB_QUERY_ERROR, GNOME_DB_QUERY_PARSE_ERROR,
				     _("Error during query parsing (%s)"), sql);
		gnome_db_query_clean (query); /* force cleaning */
		gnome_db_query_set_query_type (query, GNOME_DB_QUERY_TYPE_NON_PARSED_SQL);

		/* try the internal parser just to delimit the expressions and find the required parameters,
		* and check that we have everything we need to define a valid parameter.
		* If everything is OK, then create #GnomeDbQfValue internal fields for each required
		* parameter.
		*/
		stm = gnome_db_sql_parse (sql);
		if (stm) {
			GList *params;
			gboolean allok = TRUE;
			GnomeDbServer *srv = gnome_db_dict_get_server (gnome_db_base_get_dict (GNOME_DB_BASE (query)));

			params = stm->params_specs;
			while (params && allok) {
				GnomeDbServerDataType *dtype = NULL;
				GList *pspecs = (GList *)(params->data);
				
				while (pspecs && !dtype) {
					if (GNOME_DB_PARAM_SPEC (pspecs->data)->type == GNOME_DB_PARAM_TYPE) 
						dtype = gnome_db_server_get_data_type_by_name (srv,
							      GNOME_DB_PARAM_SPEC (pspecs->data)->content);
					pspecs = g_list_next (pspecs);
				}
				if (dtype) {
					GnomeDbQfield *field;

					/* create the #GnomeDbQfValue fields */
					field = GNOME_DB_QFIELD (gnome_db_qf_value_new (query, dtype));
					gnome_db_qfield_set_internal (field, TRUE);
					gnome_db_qfield_set_visible (field, FALSE);
					gnome_db_entity_add_field (GNOME_DB_ENTITY (query), GNOME_DB_FIELD (field));
					g_object_set_data (G_OBJECT (field), "pspeclist", params->data);
					g_object_unref (field);
					
					gnome_db_qf_value_set_is_parameter (GNOME_DB_QF_VALUE (field), TRUE);
					gnome_db_qf_value_set_not_null (GNOME_DB_QF_VALUE (field), FALSE);

					pspecs = (GList *) (params->data);
					while (pspecs) {
						GnomeDbParamSpec *ps = GNOME_DB_PARAM_SPEC (pspecs->data);
						switch (ps->type) {
						case GNOME_DB_PARAM_NAME:
							gnome_db_base_set_name (GNOME_DB_BASE (field), ps->content);
							break;
						case GNOME_DB_PARAM_DESCR:
							gnome_db_base_set_description (GNOME_DB_BASE (field), ps->content);
							break;
						case GNOME_DB_PARAM_NULLOK:
							gnome_db_qf_value_set_not_null (GNOME_DB_QF_VALUE (field),
										  (*(ps->content) == 'f') ||
										  (*(ps->content) == 'F') ? 
										  TRUE : FALSE);
							break;
						case GNOME_DB_PARAM_ISPARAM:
							gnome_db_qf_value_set_is_parameter (GNOME_DB_QF_VALUE (field), 
										      (*(ps->content) == 'f') || 
										      (*(ps->content) == 'F') ? 
										      FALSE : TRUE);
							break;
						case GNOME_DB_PARAM_TYPE:
							break;
						}
						pspecs = g_list_next (pspecs);
					}
				}
				else
					allok = FALSE;
				params = g_list_next (params);
			}

			if (allok) 
				query->priv->sql_exprs = stm;
			else {
				gnome_db_sql_destroy (stm);
				gnome_db_query_clean (query);
			}
		}
	}

	/* copy SQL */
	query->priv->sql = g_strdup (sql);

	/* close internal transaction */	
	query->priv->internal_transaction --;
	
	gnome_db_base_changed (GNOME_DB_BASE (query));
}

/**
 * gnome_db_query_get_sql_text
 * @query: a #GnomeDbQuery object
 *
 * Obtain a new string representing the SQL version of the query.
 *
 * WARNING: the returned SQL statement may contain some extensions which allow for the definition of
 * variables (see the introduction to the #GnomeDbQuery for more information). As such the returned SQL cannot
 * be executed as it may provoque errors. To get an executable statement, use the #GnomeDbRenderer interface's
 * methods.
 *
 * Returns: the new string
 */
const gchar *
gnome_db_query_get_sql_text (GnomeDbQuery *query)
{
	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (query->priv, NULL);

	return gnome_db_query_render_as_sql (GNOME_DB_RENDERER (query), NULL, GNOME_DB_RENDERER_EXTRA_VAL_ATTRS, NULL);
}





/**
 * gnome_db_query_get_parent_query
 * @query: a #GnomeDbQuery object
 *
 * Get the parent query of @query
 *
 * Returns: the parent query, or NULL if @query does not have any parent
 */
GnomeDbQuery *
gnome_db_query_get_parent_query (GnomeDbQuery *query)
{
	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (query->priv, NULL);

	return query->priv->parent_query;
}

/**
 * gnome_db_query_is_equal_to
 * @query: a #GnomeDbQuery object
 * @compare_to: a #GnomeDbQuery object
 *
 * Compares @query and @compare_to from the point of view of the result of the execution
 * of the two queries.
 *
 * Returns: TRUE is @query and @compare_to represent the same query
 */
gboolean
gnome_db_query_is_equal_to (GnomeDbQuery *query, GnomeDbQuery *compare_to)
{
	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), FALSE);
	g_return_val_if_fail (query->priv, FALSE);
	g_return_val_if_fail (compare_to && IS_GNOME_DB_QUERY (compare_to), FALSE);
	g_return_val_if_fail (compare_to->priv, FALSE);

	TO_IMPLEMENT;
	return FALSE;
}


/**
 * gnome_db_query_get_field_by_ref_field
 * @query: a #GnomeDbQuery object
 * @target: a #GnomeDbTarget, or %NULL
 * @ref_field: a #GnomeDbField object
 * @field_state: tells about the status of the requested field
 *
 * Finds the first #GnomeDbQfield object in @query which represents @ref_field.
 * The returned object will be a #GnomeDbQfField object which represents @ref_field.
 *
 * If @target is specified, then the returned field will be linked to that #GnomeDbTarget object.
 *
 * Returns: a #GnomeDbQfField object or %NULL
 */
GnomeDbQfield *
gnome_db_query_get_field_by_ref_field (GnomeDbQuery *query, GnomeDbTarget *target, GnomeDbField *ref_field, GnomeDbFieldState field_state)
{
	GnomeDbQfield *field = NULL;
	GSList *list;
	
	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (query->priv, NULL);
	if (target)
		g_return_val_if_fail (IS_GNOME_DB_TARGET (target), NULL);

	list = query->priv->fields;
	while (list && !field) {
		if (IS_GNOME_DB_QF_FIELD (list->data) &&
		    (gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (list->data)) == ref_field) &&
		    (!target || (gnome_db_qf_field_get_target (GNOME_DB_QF_FIELD (list->data)) == target))) {
			if (((field_state & GNOME_DB_FIELD_ANY) == GNOME_DB_FIELD_ANY) ||
			    ((field_state & GNOME_DB_FIELD_VISIBLE) && gnome_db_qfield_is_visible (list->data)) ||
			    ((field_state & GNOME_DB_FIELD_INVISIBLE) && !gnome_db_qfield_is_visible (list->data)))
				field = GNOME_DB_QFIELD (list->data);
		}
		list = g_slist_next (list);
	}

	return field;
}

/**
 * gnome_db_query_get_first_field_for_target
 * @query: a #GnomeDbQuery object
 * @target:
 *
 * Finds the first occurence of a #GnomeDbQfField object whose target is @target in @query
 *
 * Returns: the requested field, or %NULL
 */
GnomeDbQfield *
gnome_db_query_get_first_field_for_target (GnomeDbQuery *query, GnomeDbTarget *target)
{
	GnomeDbQfield *retval = NULL;
	GSList *list;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (query->priv, NULL);
	g_return_val_if_fail (!target || g_slist_find (query->priv->targets, target), NULL);

	list = query->priv->fields;
	while (list && !retval) {
		GnomeDbQfield *field = GNOME_DB_QFIELD (list->data);
		if (gnome_db_qfield_is_visible (field) &&
		    IS_GNOME_DB_QF_FIELD (field) &&
		    (gnome_db_qf_field_get_target (field) == target))
			retval = field;
		list = g_slist_next (list);
	}

	return retval;
}

/**
 * gnome_db_query_get_sub_queries
 * @query: a #GnomeDbQuery object
 *
 * Get a list of all the sub-queries managed by @query
 *
 * Returns: a new list of the sub-queries
 */
GSList *
gnome_db_query_get_sub_queries (GnomeDbQuery *query)
{
	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (query->priv, NULL);

	if (query->priv->sub_queries)
		return g_slist_copy (query->priv->sub_queries);
	else
		return NULL;
}

/**
 * gnome_db_query_add_param_source
 * @query: a #GnomeDbQuery object
 * @param_source: a #GnomeDbQuery object
 *
 * Tells @query that @param_source is a query which potentially will constraint the possible values
 * of one or more of @query's parameters. This implies that @query keeps a reference on @param_source.
 */
void 
gnome_db_query_add_param_source (GnomeDbQuery *query, GnomeDbQuery *param_source)
{
	g_return_if_fail (query && IS_GNOME_DB_QUERY (query));
        g_return_if_fail (query->priv);
	g_return_if_fail (param_source && IS_GNOME_DB_QUERY (param_source));
	g_return_if_fail (param_source->priv);

	if (g_slist_find (query->priv->param_sources, param_source))
		return;

	query->priv->param_sources = g_slist_append (query->priv->param_sources, param_source);
	change_parent_query (param_source, NULL);
	g_object_ref (G_OBJECT (param_source));

	gnome_db_base_connect_nullify (param_source, 
				       G_CALLBACK (nullified_param_source_cb), query);
}

static void
nullified_param_source_cb (GnomeDbQuery *param_source, GnomeDbQuery *query)
{
	g_assert (g_slist_find (query->priv->param_sources, param_source));

        query->priv->param_sources = g_slist_remove (query->priv->param_sources, param_source);
        g_signal_handlers_disconnect_by_func (G_OBJECT (param_source),
                                              G_CALLBACK (nullified_param_source_cb), query);

        g_object_unref (param_source);
}

/**
 * gnome_db_query_del_param_source
 * @query: a #GnomeDbQuery object
 * @param_source: a #GnomeDbQuery object
 *
 * Tells @query that it should no longer take care of @param_source.
 * The parameters which depend on @param_source will still depend on it, though.
 */
void
gnome_db_query_del_param_source (GnomeDbQuery *query, GnomeDbQuery *param_source)
{
	g_return_if_fail (query && IS_GNOME_DB_QUERY (query));
        g_return_if_fail (query->priv);
	g_return_if_fail (param_source && IS_GNOME_DB_QUERY (param_source));
	g_return_if_fail (g_slist_find (query->priv->param_sources, param_source));

	nullified_param_source_cb (param_source, query);
}

/**
 * gnome_db_query_get_param_sources
 * @query: a #GnomeDbQuery object
 *
 * Get a list of the parameter source queries that are references as such by @query.
 *
 * Returns: the list of #GnomeDbQuery objects
 */
const GSList *
gnome_db_query_get_param_sources (GnomeDbQuery *query)
{
	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
        g_return_val_if_fail (query->priv, NULL);

	return query->priv->param_sources;
}


/**
 * gnome_db_query_add_sub_query
 * @query: a #GnomeDbQuery object
 * @sub_query: a #GnomeDbQuery object
 *
 * Add @sub_query to @query. Sub queries are managed by their parent query, and as such they
 * are destroyed when their parent query is destroyed.
 */
void
gnome_db_query_add_sub_query (GnomeDbQuery *query, GnomeDbQuery *sub_query)
{
	g_return_if_fail (query && IS_GNOME_DB_QUERY (query));
        g_return_if_fail (query->priv);
	g_return_if_fail (sub_query && IS_GNOME_DB_QUERY (sub_query));
	g_return_if_fail (sub_query->priv);
        g_return_if_fail (!g_slist_find (query->priv->sub_queries, sub_query));

	query->priv->sub_queries = g_slist_append (query->priv->sub_queries, sub_query);
	change_parent_query (sub_query, query);
	g_object_ref (G_OBJECT (sub_query));

	gnome_db_base_connect_nullify (sub_query, 
				 G_CALLBACK (nullified_sub_query_cb), query);
        g_signal_connect (G_OBJECT (sub_query), "changed",
                          G_CALLBACK (changed_sub_query_cb), query);

#ifdef debug_signal
        g_print (">> 'SUB_QUERY_ADDED' from %s\n", __FUNCTION__);
#endif
        g_signal_emit_by_name (G_OBJECT (query), "sub_query_added", sub_query);
#ifdef debug_signal
        g_print ("<< 'SUB_QUERY_ADDED' from %s\n", __FUNCTION__);
#endif	
}

static void
nullified_sub_query_cb (GnomeDbQuery *sub_query, GnomeDbQuery *query)
{
	g_assert (g_slist_find (query->priv->sub_queries, sub_query));

        query->priv->sub_queries = g_slist_remove (query->priv->sub_queries, sub_query);
        g_signal_handlers_disconnect_by_func (G_OBJECT (sub_query),
                                              G_CALLBACK (nullified_sub_query_cb), query);
        g_signal_handlers_disconnect_by_func (G_OBJECT (sub_query),
                                              G_CALLBACK (changed_sub_query_cb), query);

	query->priv->internal_transaction ++;
#ifdef debug_signal
        g_print (">> 'SUB_QUERY_REMOVED' from %s\n", __FUNCTION__);
#endif
        g_signal_emit_by_name (G_OBJECT (query), "sub_query_removed", sub_query);
#ifdef debug_signal
        g_print ("<< 'SUB_QUERY_REMOVED' from %s\n", __FUNCTION__);
#endif
	query->priv->internal_transaction --;

        g_object_unref (sub_query);

	/* cleaning... */
	query_clean_junk (query);
}

static void
changed_sub_query_cb (GnomeDbQuery *sub_query, GnomeDbQuery *query)
{
#ifdef debug_signal
        g_print (">> 'SUB_QUERY_UPDATED' from %s\n", __FUNCTION__);
#endif
        g_signal_emit_by_name (G_OBJECT (query), "sub_query_updated", sub_query);
#ifdef debug_signal
        g_print ("<< 'SUB_QUERY_UPDATED' from %s\n", __FUNCTION__);
#endif
}

/**
 * gnome_db_query_del_sub_query
 * @query: a #GnomeDbQuery object
 * @sub_query: a #GnomeDbQuery object
 *
 * Removes @sub_query from @query. @sub_query MUST be present within @query.
 */
void
gnome_db_query_del_sub_query (GnomeDbQuery *query, GnomeDbQuery *sub_query)
{
	g_return_if_fail (query && IS_GNOME_DB_QUERY (query));
        g_return_if_fail (query->priv);
	g_return_if_fail (sub_query && IS_GNOME_DB_QUERY (sub_query));
	g_return_if_fail (g_slist_find (query->priv->sub_queries, sub_query));

	nullified_sub_query_cb (sub_query, query);
}

static void
change_parent_query (GnomeDbQuery *query, GnomeDbQuery *parent_query)
{
	GnomeDbDict *dict;
	g_return_if_fail (query && IS_GNOME_DB_QUERY (query));
	g_return_if_fail (query->priv);

	dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));

	/* if there was a parent query, then we break that link and give the query to the GnomeDbDict */
	if (query->priv->parent_query) {
		/* REM: we don't remove 'query' from 'parent_query' since that already has been done */
		g_signal_handlers_disconnect_by_func (G_OBJECT (query->priv->parent_query),
						      G_CALLBACK (nullified_parent_query), query);
		query->priv->parent_query = NULL;
	}
	
	if (parent_query) {
		g_return_if_fail (IS_GNOME_DB_QUERY (parent_query));
		query->priv->parent_query = parent_query;
		gnome_db_base_connect_nullify (parent_query,
					 G_CALLBACK (nullified_parent_query), query);
	}
}

static void
nullified_parent_query (GnomeDbQuery *parent_query, GnomeDbQuery *query)
{
	gnome_db_base_nullify (GNOME_DB_BASE (query));
}







/**
 * gnome_db_query_get_targets
 * @query: a #GnomeDbQuery object
 *
 * Get a list of all the targets used in @query
 *
 * Returns: a new list of the targets
 */
GSList *
gnome_db_query_get_targets (GnomeDbQuery *query)
{
	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (query->priv, NULL);

	if (query->priv->targets)
		return g_slist_copy (query->priv->targets);
	else
		return NULL;
}

static void gnome_db_query_assign_targets_aliases (GnomeDbQuery *query);

/**
 * gnome_db_query_add_target
 * @query: a #GnomeDbQuery object
 * @target: a #GnomeDbTarget to add to @query
 * @error: location to store error, or %NULL
 *
 * Adds a target to @query. A target represents a entity (it can actually be a table,
 * a view, or another query) which @query will use. 
 *
 * For a SELECT query, the targets appear
 * after the FROM clause. The targets can be joined two by two using #GnomeDbJoin objects
 *
 * For UPDATE, DELETE or INSERT queries, there can be only ONE #GnomeDbTarget object which is
 * the one where the data modifications are performed.
 *
 * For UNION and INTERSECT queries, there is no possible #GnomeDbTarget object.
 *
 * Returns: TRUE if no error occurred
 */
gboolean
gnome_db_query_add_target (GnomeDbQuery *query, GnomeDbTarget *target, GError **error)
{
	GnomeDbEntity *ent;
	const gchar *str;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), FALSE);
        g_return_val_if_fail (query->priv, FALSE);
	g_return_val_if_fail (query_sql_forget (query, error), FALSE);
        g_return_val_if_fail (target && IS_GNOME_DB_TARGET (target), FALSE);
        g_return_val_if_fail (!g_slist_find (query->priv->targets, target), FALSE);
	g_return_val_if_fail (gnome_db_target_get_query (target) == query, FALSE);

	/* if target represents another GnomeDbQuery, then make sure that other query is a sub query of @query */
	ent = gnome_db_target_get_represented_entity (target);
	if (ent && IS_GNOME_DB_QUERY (ent)) {
		if ((gnome_db_query_get_parent_query (GNOME_DB_QUERY (ent)) != query) ||
		    !g_slist_find (query->priv->sub_queries, ent)) {
			g_set_error (error,
				     GNOME_DB_QUERY_ERROR,
				     GNOME_DB_QUERY_TARGETS_ERROR,
				     _("The query represented by a target must be a sub query of the current query"));
			return FALSE;
		}
	}

	/* specific tests */
	switch (query->priv->query_type) {
	case GNOME_DB_QUERY_TYPE_INSERT:
	case GNOME_DB_QUERY_TYPE_UPDATE:
	case GNOME_DB_QUERY_TYPE_DELETE:
		/* if there is already one target, then refuse to add another one */
		if (query->priv->targets) {
			g_set_error (error,
				     GNOME_DB_QUERY_ERROR,
				     GNOME_DB_QUERY_TARGETS_ERROR,
				     _("Queries which update data can only have one target"));
			return FALSE;
		}
		break;
	case GNOME_DB_QUERY_TYPE_UNION:
	case GNOME_DB_QUERY_TYPE_INTERSECT:
	case GNOME_DB_QUERY_TYPE_EXCEPT:
		g_set_error (error,
			     GNOME_DB_QUERY_ERROR,
			     GNOME_DB_QUERY_TARGETS_ERROR,
			     _("Aggregation queries can't have any target, only sub queries"));
		return FALSE;
		break;
	default:
	case GNOME_DB_QUERY_TYPE_SELECT:
	case GNOME_DB_QUERY_TYPE_NON_PARSED_SQL:
		/* no specific test to be done */
		break;
	}

	query->priv->targets = g_slist_append (query->priv->targets, target);
	g_object_ref (G_OBJECT (target));
	gnome_db_base_connect_nullify (target, 
				 G_CALLBACK (nullified_target_cb), query);
        g_signal_connect (G_OBJECT (target), "changed",
                          G_CALLBACK (changed_target_cb), query);
        g_signal_connect (G_OBJECT (target), "id_changed",
                          G_CALLBACK (id_target_changed_cb), query);

	gnome_db_query_assign_targets_aliases (query);

	/* give the target a correct name */
	str = gnome_db_base_get_name (GNOME_DB_BASE (target));
	if ((!str || !(*str)) && ent) {
		str = gnome_db_base_get_name (GNOME_DB_BASE (ent));
		if (str && *str)
			gnome_db_base_set_name (GNOME_DB_BASE (target), str);
	}

	if (query->priv->serial_target <= gnome_db_base_get_id (GNOME_DB_BASE (target)))
		query->priv->serial_target = gnome_db_base_get_id (GNOME_DB_BASE (target)) + 1;

#ifdef debug_signal
        g_print (">> 'TARGET_ADDED' from %s\n", __FUNCTION__);
#endif
        g_signal_emit_by_name (G_OBJECT (query), "target_added", target);
#ifdef debug_signal
        g_print ("<< 'TARGET_ADDED' from %s\n", __FUNCTION__);
#endif
	return TRUE;
}

static void
nullified_target_cb (GnomeDbTarget *target, GnomeDbQuery *query)
{
	/* REM: when a GnomeDbTarget object is nullified, the GnomeDbJoin objects using it are 
	   also nullified, so we don't need to take care of them here */
	g_assert (g_slist_find (query->priv->targets, target));

        query->priv->targets = g_slist_remove (query->priv->targets, target);
        g_signal_handlers_disconnect_by_func (G_OBJECT (target),
                                              G_CALLBACK (nullified_target_cb), query);
        g_signal_handlers_disconnect_by_func (G_OBJECT (target),
                                              G_CALLBACK (changed_target_cb), query);
        g_signal_handlers_disconnect_by_func (G_OBJECT (target),
                                              G_CALLBACK (id_target_changed_cb), query);

	query->priv->internal_transaction ++;
#ifdef debug_signal
        g_print (">> 'TARGET_REMOVED' from %s\n", __FUNCTION__);
#endif
        g_signal_emit_by_name (G_OBJECT (query), "target_removed", target);
#ifdef debug_signal
        g_print ("<< 'TARGET_REMOVED' from %s\n", __FUNCTION__);
#endif
	query->priv->internal_transaction --;

        g_object_unref (target);
	gnome_db_query_assign_targets_aliases (query);

	/* cleaning... */
	query_clean_junk (query);
}

static void
changed_target_cb (GnomeDbTarget *target, GnomeDbQuery *query)
{
#ifdef debug_signal
        g_print (">> 'TARGET_UPDATED' from %s\n", __FUNCTION__);
#endif
        g_signal_emit_by_name (G_OBJECT (query), "target_updated", target);
#ifdef debug_signal
        g_print ("<< 'TARGET_UPDATED' from %s\n", __FUNCTION__);
#endif
}

static void
gnome_db_query_assign_targets_aliases (GnomeDbQuery *query)
{
	/* FIXME: add a targets assigning policy here; otherwise targets assign themselves the aliases as T<id>.
	 * The problem is with targets referencing other queries where there is a potential alias clash if the policy
	 * does not take care of the other queries
	 */
}

/**
 * gnome_db_query_del_target
 * @query: a #GnomeDbQuery object
 * @target: a #GnomeDbTarget object
 *
 * Removes @target from @query. @target MUST be present within @query. Warning:
 * All the joins and fields which depended on @target are also removed.
 */
void
gnome_db_query_del_target (GnomeDbQuery *query, GnomeDbTarget *target)
{
	g_return_if_fail (query && IS_GNOME_DB_QUERY (query));
        g_return_if_fail (query->priv);
	g_return_if_fail (query_sql_forget (query, NULL));
	g_return_if_fail (target && IS_GNOME_DB_TARGET (target));
	g_return_if_fail (g_slist_find (query->priv->targets, target));

	nullified_target_cb (target, query);
}


/**
 * gnome_db_query_get_target_by_xml_id
 * @query: a #GnomeDbQuery object
 * @xml_id: the XML Id of the requested #GnomeDbTarget object
 *
 * Get a pointer to a #GnomeDbTarget (which must be within @query) using
 * its XML Id
 *
 * Returns: the #GnomeDbTarget object, or NULL if not found
 */
GnomeDbTarget *
gnome_db_query_get_target_by_xml_id (GnomeDbQuery *query, const gchar *xml_id)
{
	GnomeDbTarget *target = NULL;
	GSList *list;
	gchar *str;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (query->priv, NULL);

	list = query->priv->targets;
	while (list && !target) {
		str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (list->data));
		if (!strcmp (str, xml_id))
			target = GNOME_DB_TARGET (list->data);
		g_free (str);
		list = g_slist_next (list);
	}

	return target;
}

/**
 * gnome_db_query_get_target_by_alias
 * @query: a #GnomeDbQuery object
 * @alias_or_name: the alias or name
 *
 * Get a pointer to a #GnomeDbTarget (which must be within @query) using
 * its alias (if not found then @alias_or_name is interpreted as the target name)
 *
 * Returns: the #GnomeDbTarget object, or NULL if not found
 */
GnomeDbTarget *
gnome_db_query_get_target_by_alias (GnomeDbQuery *query, const gchar *alias_or_name)
{
	GnomeDbTarget *target = NULL;
	GSList *list;
	gchar *str;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (query->priv, NULL);
	g_return_val_if_fail (alias_or_name && *alias_or_name, NULL);

	list = query->priv->targets;
	while (list && !target) {
		str = gnome_db_target_get_alias (GNOME_DB_TARGET (list->data));
		if (str && !strcmp (str, alias_or_name))
			target = GNOME_DB_TARGET (list->data);
		list = g_slist_next (list);
	}

	if (!target) {
		list = query->priv->targets;
		while (list && !target) {
			str = gnome_db_base_get_name (GNOME_DB_BASE (list->data));
			if (str && !strcmp (str, alias_or_name))
				target = GNOME_DB_TARGET (list->data);
			list = g_slist_next (list);
		}
	}

	return target;
}

/**
 * gnome_db_query_get_target_pkfields
 * @query: a #GnomeDbQuery object
 * @target: a #GnomeDbTarget object
 *
 * Makes a list of the #GnomeDbQfield objects which represent primary key fields of
 * the entity represented by @target.
 *
 * If the entity represented by @target does not have any primary key, or if the 
 * primary key's fields are not present in @query, then the returned value is %NULL.
 *
 * Returns: a new GSList, or %NULL.
 */
GSList *
gnome_db_query_get_target_pkfields (GnomeDbQuery *query, GnomeDbTarget *target)
{
	GnomeDbEntity *entity;
	GSList *pk_fields = NULL;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (query->priv, NULL);
	g_return_val_if_fail (target && IS_GNOME_DB_TARGET (target), NULL);
	g_return_val_if_fail (g_slist_find (query->priv->targets, target), NULL);

	entity = gnome_db_target_get_represented_entity (target);
	if (IS_GNOME_DB_TABLE (entity)) {
		GnomeDbConstraint *pkcons;
		gboolean allthere = TRUE;
		GSList *cons_pk_fields, *flist;
		
		pkcons = gnome_db_table_get_pk_constraint (GNOME_DB_TABLE (entity));
		if (pkcons) {
			cons_pk_fields = gnome_db_constraint_pkey_get_fields (pkcons);
			flist = cons_pk_fields;
			while (flist && allthere) {
				GnomeDbQfield *field;
				
				field = gnome_db_query_get_field_by_ref_field (query, target, flist->data, GNOME_DB_FIELD_VISIBLE);
				if (field)
					pk_fields = g_slist_append (pk_fields, field);
				else
					allthere = FALSE;
				flist = g_slist_next (flist);
			}
			g_slist_free (cons_pk_fields);

			if (!allthere) {
				g_slist_free (pk_fields);
				pk_fields = NULL;
			}
		}
	}
	else {
		/* not yet possible at the moment... */
		TO_IMPLEMENT;
	}

	return pk_fields;
}


/**
 * gnome_db_query_get_joins
 * @query: a #GnomeDbQuery object
 *
 * Get a list of all the joins used in @query
 *
 * Returns: a new list of the joins
 */
GSList *
gnome_db_query_get_joins (GnomeDbQuery *query)
{
	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (query->priv, NULL);

	if (query->priv->joins_flat)
		return g_slist_copy (query->priv->joins_flat);
	else
		return NULL;
}

/**
 * gnome_db_query_get_join_by_targets
 * @query: a #GnomeDbQuery object
 * @target1: a #GnomeDbTarget object
 * @target2: a #GnomeDbTarget object
 *
 * Find a join in @query which joins the @target1 and @target2 targets
 *
 * Returns: the #GnomeDbJoin object, or %NULL
 */
GnomeDbJoin *
gnome_db_query_get_join_by_targets (GnomeDbQuery *query, GnomeDbTarget *target1, GnomeDbTarget *target2)
{
	GnomeDbJoin *join = NULL;
	GSList *joins;
	GnomeDbTarget *lt1, *lt2;
	
	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (query->priv, NULL);
	g_return_val_if_fail (target1 && IS_GNOME_DB_TARGET (target1), NULL);
	g_return_val_if_fail (gnome_db_target_get_query (target1) == query, NULL);
	g_return_val_if_fail (target2 && IS_GNOME_DB_TARGET (target2), NULL);
	g_return_val_if_fail (gnome_db_target_get_query (target2) == query, NULL);

	joins = query->priv->joins_flat;
	while (joins && !join) {
		lt1 = gnome_db_join_get_target_1 (GNOME_DB_JOIN (joins->data));
		lt2 = gnome_db_join_get_target_2 (GNOME_DB_JOIN (joins->data));
		if (((lt1 == target1) && (lt2 == target2)) ||
		    ((lt1 == target2) && (lt2 == target1)))
			join = GNOME_DB_JOIN (joins->data);

		joins = g_slist_next (joins);
	}

	return join;
}

static gboolean gnome_db_query_are_joins_active (GnomeDbQuery *query);
#ifdef debug
static void joins_pack_dump (GnomeDbQuery *query);
#endif
static gboolean joins_pack_add_join (GnomeDbQuery *query, GnomeDbJoin *join);
static void joins_pack_del_join (GnomeDbQuery *query, GnomeDbJoin *join);

/**
 * gnome_db_query_add_join
 * @query: a #GnomeDbQuery object
 * @join : a #GnomeDbJoin object
 *
 * Add a join to @query. A join is defined by the two #GnomeDbTarget objects it joins and by
 * a join condition which MUST ONLY make use of fields of the two entities represented by the
 * targets.
 *
 * For any given couple of #GnomeDbTarget objects, there can exist ONLY ONE #GnomeDbJoin which joins the
 * two.
 *
 * Returns: TRUE on success, and FALSE otherwise
 */
gboolean
gnome_db_query_add_join (GnomeDbQuery *query, GnomeDbJoin *join)
{
	GSList *joins;
	GnomeDbTarget *t1, *t2, *lt1, *lt2;
	gboolean already_exists = FALSE;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), FALSE);
        g_return_val_if_fail (query->priv, FALSE);
	g_return_val_if_fail (query_sql_forget (query, NULL), FALSE);
        g_return_val_if_fail (join && IS_GNOME_DB_JOIN (join), FALSE);
        g_return_val_if_fail (!g_slist_find (query->priv->joins_flat, join), FALSE);
	g_return_val_if_fail (gnome_db_join_get_query (join) == query, FALSE);
	g_return_val_if_fail (gnome_db_referer_is_active (GNOME_DB_REFERER (join)), FALSE);
	g_return_val_if_fail (gnome_db_query_are_joins_active (query), FALSE);

	/* make sure there is not yet another join for the couple of #GnomeDbTarget objects
	   used by 'join' */
	t1 = gnome_db_join_get_target_1 (join);
	t2 = gnome_db_join_get_target_2 (join);

	joins = query->priv->joins_flat;
	while (joins && !already_exists) {
		lt1 = gnome_db_join_get_target_1 (GNOME_DB_JOIN (joins->data));
		lt2 = gnome_db_join_get_target_2 (GNOME_DB_JOIN (joins->data));
		if (((lt1 == t1) && (lt2 == t2)) ||
		    ((lt1 == t2) && (lt2 == t1)))
			already_exists = TRUE;

		joins = g_slist_next (joins);
	}
	g_return_val_if_fail (!already_exists, FALSE);

	g_return_val_if_fail (joins_pack_add_join (query, join), FALSE);
	query->priv->joins_flat = g_slist_append (query->priv->joins_flat, join);
	g_object_ref (G_OBJECT (join));
	gnome_db_base_connect_nullify (join,
				 G_CALLBACK (nullified_join_cb), query);
        g_signal_connect (G_OBJECT (join), "changed",
                          G_CALLBACK (changed_join_cb), query);

#ifdef debug_signal
        g_print (">> 'JOIN_ADDED' from %s\n", __FUNCTION__);
#endif
        g_signal_emit_by_name (G_OBJECT (query), "join_added", join);
#ifdef debug_signal
        g_print ("<< 'JOIN_ADDED' from %s\n", __FUNCTION__);
#endif	
	return TRUE;
}

static void
nullified_join_cb (GnomeDbJoin *join, GnomeDbQuery *query)
{
	g_assert (g_slist_find (query->priv->joins_flat, join));

        query->priv->joins_flat = g_slist_remove (query->priv->joins_flat, join);
	joins_pack_del_join (query, join);

        g_signal_handlers_disconnect_by_func (G_OBJECT (join),
                                              G_CALLBACK (nullified_join_cb), query);
        g_signal_handlers_disconnect_by_func (G_OBJECT (join),
                                              G_CALLBACK (changed_join_cb), query);

	query->priv->internal_transaction ++;
#ifdef debug_signal
        g_print (">> 'JOIN_REMOVED' from %s\n", __FUNCTION__);
#endif
        g_signal_emit_by_name (G_OBJECT (query), "join_removed", join);
#ifdef debug_signal
        g_print ("<< 'JOIN_REMOVED' from %s\n", __FUNCTION__);
#endif
	
	query->priv->internal_transaction --;
        g_object_unref (join);

	/* cleaning... */
	query_clean_junk (query);
}

static void
changed_join_cb (GnomeDbJoin *join, GnomeDbQuery *query)
{
#ifdef debug_signal
        g_print (">> 'JOIN_UPDATED' from %s\n", __FUNCTION__);
#endif
        g_signal_emit_by_name (G_OBJECT (query), "join_updated", join);
#ifdef debug_signal
        g_print ("<< 'JOIN_UPDATED' from %s\n", __FUNCTION__);
#endif
}

/**
 * gnome_db_query_del_join
 * @query: a #GnomeDbQuery object
 * @join: a #GnomeDbJoin object
 *
 * Removes @join from @query. @join MUST be present within @query.
 */
void
gnome_db_query_del_join (GnomeDbQuery *query, GnomeDbJoin *join)
{
	g_return_if_fail (query && IS_GNOME_DB_QUERY (query));
        g_return_if_fail (query->priv);
	g_return_if_fail (query_sql_forget (query, NULL));
	g_return_if_fail (join && IS_GNOME_DB_JOIN (join));
	g_return_if_fail (g_slist_find (query->priv->joins_flat, join));

	nullified_join_cb (join, query);
}

#ifdef debug
static void
joins_pack_dump (GnomeDbQuery *query)
{
	GSList *packs, *list;
	gchar *xml;

	packs = query->priv->joins_pack;
	while (packs) {
		g_print ("=== PACK ===\n");
		list = JOINS_PACK (packs->data)->targets;
		g_print ("TARGETS: ");
		while (list) {
			xml = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (list->data));
			g_print ("%s ", xml);
			g_free (xml);
			list = g_slist_next (list);
		}
		g_print ("\nJOINS: ");
		list = JOINS_PACK (packs->data)->joins;
		while (list) {
			g_print ("%p ", list->data);
			list = g_slist_next (list);
		}
		g_print ("\n");
		packs = g_slist_next (packs);
	}
}
#endif

/*
 * Adds a join in the joins pack structure, which is another way of storing joins
 * within the query.
 * It is assumed that the join itself is not yet present in the list of joins
 *
 * Returns: TRUE if no error
 */
static gboolean
joins_pack_add_join (GnomeDbQuery *query, GnomeDbJoin *join)
{
	GSList *pack_list;
	JoinsPack *pack;
	JoinsPack *pack1 = NULL, *pack2 = NULL;
	GnomeDbTarget *t1, *t2;

	/* we want an active join only */
	g_return_val_if_fail (gnome_db_referer_activate (GNOME_DB_REFERER (join)), FALSE);
	t1 = gnome_db_join_get_target_1 (join);
	t2 = gnome_db_join_get_target_2 (join);

	/* try fo identify existing joins packings in which the new join can go */
	pack_list = query->priv->joins_pack;
	while (pack_list && !pack1 && !pack2) {
		pack = JOINS_PACK (pack_list->data);
		if (!pack1) {
			if (g_slist_find (pack->targets, t2)) {
				GnomeDbTarget *tmp;
				gnome_db_join_swap_targets (join);
				tmp = t1;
				t1 = t2;
				t2 = tmp;
			}

			if (g_slist_find (pack->targets, t1)) 
				pack1 = pack;
		}
		else 
			if (g_slist_find (pack->targets, t2)) 
				pack2 = pack;
		
		pack_list = g_slist_next (pack_list);
	}

	/* updating the packings */
	if (!pack1) {
		/* a new JoinsPack is necessary */
		pack = g_new0 (JoinsPack, 1);
		pack->targets = g_slist_append (NULL, t1);
		pack->targets = g_slist_append (pack->targets, t2);
		pack->joins = g_slist_append (NULL, join);

		query->priv->joins_pack = g_slist_append (query->priv->joins_pack, pack);
	}
	else {
		/* append join to the identified pack1 */
		pack1->joins = g_slist_append (pack1->joins, join);
		pack1->targets = g_slist_append (pack1->targets, t2);

		if (pack2 && (pack2 != pack1)) {
			/* merge pack2 into pack1 */
			GSList *joins;
			GSList *targets;
			GnomeDbJoin *cjoin;
			
			/* reodering the joins to start with t2 */
			targets = g_slist_append (NULL, t2);
			while (pack2->joins) {
				joins = pack2->joins;
				cjoin = NULL;

				while (joins && !cjoin) {
					t1 = gnome_db_join_get_target_1 (GNOME_DB_JOIN (joins->data));
					t2 = gnome_db_join_get_target_2 (GNOME_DB_JOIN (joins->data));

					if (g_slist_find (targets, t1)) {
						cjoin = GNOME_DB_JOIN (joins->data);
						targets = g_slist_append (targets, t2);
						pack1->targets = g_slist_append (pack1->targets, t2);
					}
					else {
						if (g_slist_find (targets, t2)) {
							cjoin = GNOME_DB_JOIN (joins->data);
							gnome_db_join_swap_targets (cjoin);
							targets = g_slist_append (targets, t1);
							pack1->targets = g_slist_append (pack1->targets, t1);
						}
					}
				}

				g_assert (cjoin);
				pack2->joins = g_slist_remove (pack2->joins, cjoin);
				pack1->joins = g_slist_append (pack1->joins, cjoin);
			}
			
			g_slist_free (targets);
			/* getting rid of pack2 */
			query->priv->joins_pack = g_slist_remove (query->priv->joins_pack, pack2);
			g_slist_free (pack2->targets);
			g_free (pack2);
		}
	}

	return TRUE;
}

/* 
 * Removes a join from the joins packing structures.
 * It is assumed that the join itself IS present in the list of joins
 */
static void
joins_pack_del_join (GnomeDbQuery *query, GnomeDbJoin *join)
{
	JoinsPack *joinpack = NULL, *pack;
	GSList *pack_list, *list;

	/* identifying the pack in which join is; if the join is not activatable, then it
	   may already have been removed */
	pack_list = query->priv->joins_pack;
	while (pack_list && !joinpack) {
		pack = JOINS_PACK (pack_list->data);
		if (g_slist_find (pack->joins, join)) 
			joinpack = pack;
		pack_list = g_slist_next (pack_list);
	}
	
	if (g_slist_find (query->priv->joins_flat, join))
		g_assert (joinpack);

	/* removing the pack and adding again all the joins within that pack */
	if (joinpack) {
		query->priv->joins_pack = g_slist_remove (query->priv->joins_pack, joinpack);

		list = joinpack->joins;
		while (list) {
			if ((GNOME_DB_JOIN (list->data) != join) && 
			    gnome_db_referer_activate (GNOME_DB_REFERER (list->data)))
				joins_pack_add_join (query, GNOME_DB_JOIN (list->data));
			
			list = g_slist_next (list);
		}
		g_slist_free (joinpack->targets);
		g_slist_free (joinpack->joins);
		g_free (joinpack);
	}
}






/**
 * gnome_db_query_get_condition
 * @query: a #GnomeDbQuery object
 *
 * Get the query's associated condition
 *
 * Returns: the #GnomeDbCondition object
 */
GnomeDbCondition *
gnome_db_query_get_condition (GnomeDbQuery *query)
{
	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (query->priv, NULL);
	
	return query->priv->cond;
}

/**
 * gnome_db_query_set_condition
 * @query: a #GnomeDbQuery object
 * @cond: a #GnomeDbCondition object, or %NULL to remove condition
 *
 * Sets the query's associated condition; if there was already a query condition,
 * then the old one is trashed first.
 *
 * Pass %NULL as the @cond argument to remove any query condition
 */
void
gnome_db_query_set_condition (GnomeDbQuery *query, GnomeDbCondition *cond)
{
	g_return_if_fail (query && IS_GNOME_DB_QUERY (query));
	g_return_if_fail (query->priv);
	g_return_if_fail (query_sql_forget (query, NULL));
	if (cond)
		g_return_if_fail (IS_GNOME_DB_CONDITION (cond));

	if (query->priv->cond == cond)
		return;
	
	query->priv->internal_transaction ++;

	if (query->priv->cond) 
		nullified_cond_cb (query->priv->cond, query);

	if (cond) {
		g_object_ref (G_OBJECT (cond));
		query->priv->cond = cond;
		g_signal_connect (G_OBJECT (cond), "changed",
				  G_CALLBACK (changed_cond_cb), query);

		gnome_db_base_connect_nullify (cond,
					       G_CALLBACK (nullified_cond_cb), query);
	}

	query->priv->internal_transaction --;

	/* cleaning... */
	query_clean_junk (query);
}

static void
changed_cond_cb (GnomeDbCondition *cond, GnomeDbQuery *query)
{
	if (query->priv->auto_clean) {
		/* if query->priv->cond is limited to a single empty node, then remove it */
		if (query->priv->cond &&
		    ! gnome_db_condition_is_leaf (query->priv->cond)) {
			GSList *list = gnome_db_condition_get_children (query->priv->cond);
			if (!list) {
				query->priv->internal_transaction ++;
				nullified_cond_cb (query->priv->cond, query);
				query->priv->internal_transaction --;
			}
			g_slist_free (list);
		}
	}
	    
	if (! query->priv->internal_transaction)
		gnome_db_base_changed (GNOME_DB_BASE (query));
}

static void
nullified_cond_cb (GnomeDbCondition *cond, GnomeDbQuery *query)
{
	/* real nullify code */
	g_assert (query->priv->cond == cond);
	g_signal_handlers_disconnect_by_func (G_OBJECT (cond),
					      G_CALLBACK (nullified_cond_cb), query);
	g_signal_handlers_disconnect_by_func (G_OBJECT (cond),
					      G_CALLBACK (changed_cond_cb), query);
	query->priv->cond = NULL;
	g_object_unref (G_OBJECT (cond));
	
	/* cleaning... */
	query_clean_junk (query);
}

/**
 * gnome_db_query_add_field_from_sql
 * @query: a #GnomeDbQuery object
 * @field: a SQL expression
 * @error: place to store the error, or %NULL
 *
 * Parses @field and if it represents a valid SQL expression for a
 * field, then add it to @query.
 *
 * Returns: a new #GnomeDbQfield object, or %NULL
 */
GnomeDbQfield *
gnome_db_query_add_field_from_sql (GnomeDbQuery *query, const gchar *field, GError **error)
{
	GnomeDbQfield *qfield = NULL;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (query->priv, NULL);

	qfield = gnome_db_qfield_new_from_sql (query, field, error);
	if (qfield) {
		gnome_db_entity_add_field (GNOME_DB_ENTITY (query), GNOME_DB_FIELD (qfield));
		g_object_unref (G_OBJECT (qfield));
	}

	return qfield;
}


#ifdef debug
static void
gnome_db_query_dump (GnomeDbQuery *query, guint offset)
{
	gchar *str;
        guint i;
	GSList *list;
	GError *error = NULL;
	
	g_return_if_fail (query && IS_GNOME_DB_QUERY (query));

        /* string for the offset */
        str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;

        /* dump */
        if (query->priv) {
		gchar *sql;

                g_print ("%s" D_COL_H1 "GnomeDbQuery" D_COL_NOR " %p (type=%d, QU%d) %s",
                         str, query, query->priv->query_type, gnome_db_base_get_id (GNOME_DB_BASE (query)),
			 gnome_db_base_get_name (GNOME_DB_BASE (query)));
		if (gnome_db_query_is_active (GNOME_DB_REFERER (query)))
			g_print (": Active\n");
		else
			g_print (D_COL_ERR ": Non active\n" D_COL_NOR);
		
		/* targets */
		if (query->priv->targets)
			g_print ("%sTargets:\n", str);
		else
			g_print ("%sNo target defined.\n", str);
		list = query->priv->targets;
		while (list) {
			gnome_db_base_dump (GNOME_DB_BASE (list->data), offset+5);
			list = g_slist_next (list);
		}

		/* fields */
		if (query->priv->fields)
			g_print ("%sFields:\n", str);
		else
			g_print ("%sNo field defined.\n", str);
		list = query->priv->fields;
		while (list) {
			gnome_db_base_dump (GNOME_DB_BASE (list->data), offset+5);
			list = g_slist_next (list);
		}

		/* joins */
		if (query->priv->joins_flat)
			g_print ("%sJoins:\n", str);
		else
			g_print ("%sNo join defined.\n", str);
		list = query->priv->joins_flat;
		while (list) {
			gnome_db_base_dump (GNOME_DB_BASE (list->data), offset+5);
			list = g_slist_next (list);
		}
		/*joins_pack_dump (query);*/ /* RAW joins pack output */

		/* condition */
		if (query->priv->cond) {
			g_print ("%sCondition:\n", str);
			gnome_db_base_dump (GNOME_DB_BASE (query->priv->cond), offset+5);
		}
		else
			g_print ("%sNo Condition defined.\n", str);

		if (0 && query->priv->all_conds) {
			g_print ("%sReferenced conditions:\n", str);
			list = query->priv->all_conds;
			while (list) {
				gnome_db_base_dump (list->data, offset + 5);
				list = g_slist_next (list);
			}
		}


		/* sub queries */
		if (query->priv->sub_queries)
			g_print ("%sSub-queries:\n", str);
		else
			g_print ("%sNo sub-query defined.\n", str);
		list = query->priv->sub_queries;
		while (list) {
			gnome_db_base_dump (GNOME_DB_BASE (list->data), offset+5);
			list = g_slist_next (list);
		}

		/* parameters sources */
		if (query->priv->param_sources)
			g_print ("%sParameters sources:\n", str);
		list = query->priv->param_sources;
		while (list) {
			gnome_db_base_dump (GNOME_DB_BASE (list->data), offset+5);
			list = g_slist_next (list);
		}

		/* Rendered version of the query */
		sql = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (query), NULL, GNOME_DB_RENDERER_EXTRA_VAL_ATTRS, &error);
		if (sql) {
			g_print ("%sSQL=%s\n", str, sql);
			g_free (sql);
		}
		else {
			g_print ("%sError occurred:\n%s%s\n", str, str, error->message);
			g_error_free (error);
		}
	}
        else
                g_print ("%s" D_COL_ERR "Using finalized object %p" D_COL_NOR, str, query);
	g_free (str);
}
#endif


/**
 * gnome_db_query_set_order_by_field
 * @query: a #GnomeDbQuery
 * @field: a #GnomeDbQfield which is in @query
 * @order: the order in the list of ORDER BY fields (starts at 0), or -1
 * @ascendant: TRUE to sort ascending
 *
 * Sets @field to be used in the ORDER BY clause (using the @order and @ascendant attributes) if
 * @order >= 0. If @order < 0, then @field will not be used in the ORDER BY clause.
 */
void
gnome_db_query_set_order_by_field (GnomeDbQuery *query, GnomeDbQfield *field, gint order, gboolean ascendant)
{
	g_return_if_fail (query && IS_GNOME_DB_QUERY (query));
	g_return_if_fail (query->priv);
	g_return_if_fail (field && IS_GNOME_DB_QFIELD (field));
	g_return_if_fail (g_slist_find (query->priv->fields, field));

	if (! (IS_GNOME_DB_QF_VALUE (field) && 
	       (query->priv->query_type == GNOME_DB_QUERY_TYPE_NON_PARSED_SQL)))
		g_return_if_fail (query_sql_forget (query, NULL));
	
	if ((query->priv->query_type == GNOME_DB_QUERY_TYPE_INSERT) ||
	    (query->priv->query_type == GNOME_DB_QUERY_TYPE_DELETE) ||
	    (query->priv->query_type == GNOME_DB_QUERY_TYPE_UPDATE))
		return;
	
	if (g_slist_find (query->priv->fields_order_by, field))
		query->priv->fields_order_by = g_slist_remove (query->priv->fields_order_by, field);
	
	if (order < 0)
		g_object_set_data (G_OBJECT (field), "order_by_asc", NULL);
	else { /* add to the ORDER BY */
		g_object_set_data (G_OBJECT (field), "order_by_asc", GINT_TO_POINTER (ascendant));
		query->priv->fields_order_by = g_slist_insert (query->priv->fields_order_by, field, order);
	}

	if (!query->priv->internal_transaction)
		gnome_db_base_changed (GNOME_DB_BASE (query));
}

/**
 * gnome_db_query_get_order_by_field
 * @query: a #GnomeDbQuery
 * @field: a #GnomeDbQfield which is in @query
 * @ascendant: if not %NULL, will be set TRUE if ascendant sorting and FALSE otherwise
 *
 * Tells if @field (which MUST be in @query) is part of the ORDER BY clause.
 *
 * Returns: -1 if no, and the order where it appears in the ORDER BY list otherwise
 */
gint
gnome_db_query_get_order_by_field (GnomeDbQuery *query, GnomeDbQfield *field, gboolean *ascendant)
{
	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), -1);
	g_return_val_if_fail (query->priv, -1);
	g_return_val_if_fail (field && IS_GNOME_DB_QFIELD (field), -1);
	g_return_val_if_fail (g_slist_find (query->priv->fields, field), -1);

	if (ascendant)
		*ascendant = g_object_get_data (G_OBJECT (field), "order_by_asc") ? TRUE : FALSE;
	return g_slist_index (query->priv->fields_order_by, field);
}


/* NOTE: A query's field has several status:
   -> internal: such a field is added by libgnomedb for its own needs, should NEVER
      get out of the query
   -> visible: such fields make the external representation of the equivalent entity
   -> non visible: fields that take part in making the query (condition, values, function params, etc
*/


/**
 * gnome_db_query_get_all_fields
 * @query: a #GnomeDbQuery object
 *
 * Fetch a list of all the fields of @query: the ones which are visible, and
 * the ones which are not visible and are not internal query fields.
 *
 * Returns: a new list of fields
 */
GSList *
gnome_db_query_get_all_fields (GnomeDbQuery *query)
{
	GSList *list, *fields = NULL;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (GNOME_DB_QUERY (query)->priv, NULL);
	
	list = query->priv->fields;
	while (list) {
		if (gnome_db_qfield_is_visible (GNOME_DB_QFIELD (list->data)) ||
		    !gnome_db_qfield_is_internal (GNOME_DB_QFIELD (list->data)))
			fields = g_slist_append (fields, list->data);
		list = g_slist_next (list);
	}

	return fields;
}

/**
 * gnome_db_query_get_field_by_sql_naming
 * @query: a #GnomeDbQuery object
 * @sql_name: the SQL naming for the requested field
 *
 * Returns:
 */
GnomeDbQfield *
gnome_db_query_get_field_by_sql_naming (GnomeDbQuery *query, const gchar *sql_name) {
	
	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (query->priv, NULL);
	
	return gnome_db_query_get_field_by_sql_naming_fields (query, sql_name, query->priv->fields);
}


/**
 * gnome_db_query_get_field_by_sql_naming_fields
 * @query: a #GnomeDbQuery object
 * @sql_name: the SQL naming for the requested field
 * @fields_list: an explicit list of fields to search into
 *
 * Returns:
 */
GnomeDbQfield *
gnome_db_query_get_field_by_sql_naming_fields (GnomeDbQuery *query, const gchar *sql_name, GSList *fields_list)
{
	GnomeDbQfield *field = NULL;
	gboolean err = FALSE;
	GSList *list;

	g_return_val_if_fail (sql_name && *sql_name, NULL);

	list = fields_list;
	while (list && !err) {
		if (IS_GNOME_DB_QF_FIELD (list->data)) {
			/* split the sql_name into a list of strings, by the '.' separator */
			gchar **split = g_strsplit (sql_name, ".", 0);
			gint nb_tokens;
			GnomeDbField *ref_field = gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (list->data));
			gchar *str;

			for (nb_tokens = 0; split[nb_tokens] != NULL; nb_tokens++);
			if (nb_tokens == 1) {
				/* first try to find the table corresponding to a lower-case version of @sql_name, 
				   and if not found, try with the unchanged @sql_name */
				str = g_utf8_strdown (sql_name, -1);
				if (!strcmp (gnome_db_field_get_name (ref_field), str) ||
				    !strcmp (gnome_db_field_get_name (ref_field), sql_name)) {
					if (field)
						err = TRUE;
					else
						field = GNOME_DB_QFIELD (list->data);
				}
				g_free (str);
			}
			else {
				/* compare with target_alias.ref_field */
				GnomeDbTarget *target = gnome_db_qf_field_get_target (GNOME_DB_QF_FIELD (list->data));
				str = g_utf8_strdown (split[1], -1);
				if (!strcmp (gnome_db_target_get_alias (target), split[0]) &&
				    (!strcmp (gnome_db_field_get_name (ref_field), str) ||
				     !strcmp (gnome_db_field_get_name (ref_field), split[1]))) {
					if (field)
						err = TRUE;
					else
						field = GNOME_DB_QFIELD (list->data);
				}
				
				/* compare with target_ref_entity.ref_field */
				if (!field) {
					gchar *str2 = g_utf8_strdown (split[0], -1);
					gchar *entstr = gnome_db_base_get_name (GNOME_DB_BASE (gnome_db_target_get_represented_entity (target)));
					if (!err && !field &&
					    (!strcmp (entstr, str2) ||
					     !strcmp (entstr, split[0])) &&
					    (!strcmp (gnome_db_field_get_name (ref_field), str) ||
					     !strcmp (gnome_db_field_get_name (ref_field), split[1]) )) {
						if (field)
							err = TRUE;
						else
							field = GNOME_DB_QFIELD (list->data);
					}
					g_free (str2);
				}
				g_free (str);
			}
			g_strfreev (split);
		}

		if (IS_GNOME_DB_QF_ALL (list->data)) {
			/* split the sql_name into a list of strings, by the '.' separator */
			gchar **split = g_strsplit (sql_name, ".", 0);
			gint nb_tokens;
			
			for (nb_tokens = 0; split[nb_tokens] != NULL; nb_tokens++);
			if (nb_tokens == 1) {
				if (!strcmp ("*", sql_name)) {
					if (field)
						err = TRUE;
					else
						field = GNOME_DB_QFIELD (list->data);
				}
			}
			else {
				/* compare with target_alias.* */
				GnomeDbTarget *target = gnome_db_qf_all_get_target (GNOME_DB_QF_ALL (list->data));
				if (!strcmp (gnome_db_target_get_alias (target), split[0]) &&
				    !strcmp ("*", split[1])) {
					if (field)
						err = TRUE;
					else
						field = GNOME_DB_QFIELD (list->data);
				}
				
				/* compare with target_ref_entity.ref_field */
				if (!err && !field &&
				    !strcmp (gnome_db_base_get_name (GNOME_DB_BASE (gnome_db_target_get_represented_entity (target))), split[0]) &&
				    !strcmp ("*", split[1])) {
					if (field)
						err = TRUE;
					else
						field = GNOME_DB_QFIELD (list->data);
				}
			}
			g_strfreev (split);
		}

		if (IS_GNOME_DB_QF_FUNC (list->data)) {
			TO_IMPLEMENT;
		}
		if (IS_GNOME_DB_QF_VALUE (list->data)) {
			/* do nothing */
		}
		
		list = g_slist_next (list);
	}
	if (err)
		return NULL;
	return field;
}

/*
 * GnomeDbEntity interface implementation
 */
static gboolean
gnome_db_query_has_field (GnomeDbEntity *iface, GnomeDbField *field)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_QUERY (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_QUERY (iface)->priv, FALSE);
	g_return_val_if_fail (field && IS_GNOME_DB_QFIELD (field), FALSE);

	if (gnome_db_qfield_is_visible (GNOME_DB_QFIELD (field)) &&
	    g_slist_find (GNOME_DB_QUERY (iface)->priv->fields, field))
		return TRUE;
	else
		return FALSE;
}

static GSList *
gnome_db_query_get_fields (GnomeDbEntity *iface)
{
	GnomeDbQuery *query;
	GSList *list, *fields = NULL;

	g_return_val_if_fail (iface && IS_GNOME_DB_QUERY (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QUERY (iface)->priv, NULL);
	query = GNOME_DB_QUERY (iface);

	list = query->priv->fields;
	while (list) {
		if (gnome_db_qfield_is_visible (GNOME_DB_QFIELD (list->data)))
			fields = g_slist_append (fields, list->data);
		list = g_slist_next (list);
	}

	return fields;
}


static GnomeDbField *
gnome_db_query_get_field_by_name (GnomeDbEntity *iface, const gchar *name)
{
	GnomeDbQuery *query;
	GSList *list;
	GnomeDbField *field = NULL;
	gboolean err = FALSE;
	
	g_return_val_if_fail (iface && IS_GNOME_DB_QUERY (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QUERY (iface)->priv, NULL);
	query = GNOME_DB_QUERY (iface);
	
	/* first pass: using gnome_db_field_get_name () */
	list = query->priv->fields;
	while (list && !err) {
		if (!strcmp (gnome_db_field_get_name (GNOME_DB_FIELD (list->data)), name)) {
			if (field) 
				err = TRUE;
			else
				field = GNOME_DB_FIELD (list->data);
		}
		list = g_slist_next (list);
	}
	if (err)
		return NULL;
	if (field)
		return field;
	
	/* second pass: using SQL naming */
	return (GnomeDbField *) gnome_db_query_get_field_by_sql_naming (query, name);
}

static GnomeDbField *
gnome_db_query_get_field_by_xml_id (GnomeDbEntity *iface, const gchar *xml_id)
{
	GnomeDbQuery *query;
	GSList *list;
	GnomeDbField *field = NULL;
	gchar *str;

	g_return_val_if_fail (iface && IS_GNOME_DB_QUERY (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QUERY (iface)->priv, NULL);
	query = GNOME_DB_QUERY (iface);
	
	list = query->priv->fields;
	while (list && !field) {
		str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (list->data));
		if (!strcmp (str, xml_id))
			field = GNOME_DB_FIELD (list->data);
		list = g_slist_next (list);
	}

	return field;
}

static GnomeDbField *
gnome_db_query_get_field_by_index (GnomeDbEntity *iface, gint index)
{
	GnomeDbQuery *query;
	GSList *list;
	GnomeDbField *field = NULL;
	gint i = -1;

	g_return_val_if_fail (iface && IS_GNOME_DB_QUERY (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QUERY (iface)->priv, NULL);
	query = GNOME_DB_QUERY (iface);
	
	list = query->priv->fields;
	while (list && !field) {
		if (gnome_db_qfield_is_visible (GNOME_DB_QFIELD (list->data))) {
			i++;
			if (i == index)
				field = GNOME_DB_FIELD (list->data);
		}

		list = g_slist_next (list);
	}

	return field;
}

static gint
gnome_db_query_get_field_index (GnomeDbEntity *iface, GnomeDbField *field)
{
	GnomeDbQuery *query;
	GSList *list;
	gint current, pos = -1;

	g_return_val_if_fail (iface && IS_GNOME_DB_QUERY (iface), -1);
	g_return_val_if_fail (GNOME_DB_QUERY (iface)->priv, -1);
	g_return_val_if_fail (field && IS_GNOME_DB_QFIELD (field), -1);
	query = GNOME_DB_QUERY (iface);

	if (!g_slist_find (query->priv->fields, field))
		return -1;

	if (!gnome_db_qfield_is_visible (GNOME_DB_QFIELD (field)))
		return -1;

	current = 0;
	list = query->priv->fields;
	while (list && (pos==-1)) {
		if (list->data == (gpointer) field)
			pos = current;
		if (gnome_db_qfield_is_visible (GNOME_DB_QFIELD (list->data)))
			current++;
		list = g_slist_next (list);
	}

	return pos;
}

/**
 * gnome_db_query_get_fields_by_target
 * @query: a #GnomeDbQuery object
 * @target: a #GnomeDbTarget object representing a target in @query
 *
 * Get a list of all the #GnomeDbQfield objects in @query which depent on the existance of 
 * @target.
 *
 * Returns: a new list of #GnomeDbQfield objects
 */
GSList *
gnome_db_query_get_fields_by_target (GnomeDbQuery *query, GnomeDbTarget *target, gboolean visible_fields_only)
{
	GSList *retval = NULL;
	GSList *tmplist, *ptr;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (query->priv, NULL);
	g_return_val_if_fail (target && IS_GNOME_DB_TARGET (target), NULL);
	g_return_val_if_fail (g_slist_find (query->priv->targets, target), NULL);

	if (visible_fields_only)
		tmplist = gnome_db_query_get_fields (GNOME_DB_ENTITY (query));
	else
		tmplist = gnome_db_query_get_all_fields (query);

	ptr = tmplist;
	while (ptr) {
		if ((IS_GNOME_DB_QF_FIELD (ptr->data) && (gnome_db_qf_field_get_target (GNOME_DB_QF_FIELD (ptr->data)) == target)) ||
		    (IS_GNOME_DB_QF_ALL (ptr->data) && (gnome_db_qf_all_get_target (GNOME_DB_QF_ALL (ptr->data)) == target)))
			retval = g_slist_prepend (retval, ptr->data);
			
		ptr = g_slist_next (ptr);
	}
	g_slist_free (tmplist);
	retval = g_slist_reverse (retval);

	return retval;
}

static void
gnome_db_query_add_field (GnomeDbEntity *iface, GnomeDbField *field)
{
	gnome_db_query_add_field_before (iface, field, NULL);
}

static void
gnome_db_query_add_field_before (GnomeDbEntity *iface, GnomeDbField *field, GnomeDbField *field_before)
{
	GnomeDbQuery *query;
	gint pos = -1;

	g_return_if_fail (iface && IS_GNOME_DB_QUERY (iface));
	g_return_if_fail (GNOME_DB_QUERY (iface)->priv);
	query = GNOME_DB_QUERY (iface);

	g_return_if_fail (field && IS_GNOME_DB_QFIELD (field));
        g_return_if_fail (!g_slist_find (query->priv->fields, field));
	g_return_if_fail (gnome_db_field_get_entity (field) == GNOME_DB_ENTITY (query));

	if (! (IS_GNOME_DB_QF_VALUE (field) && 
	       (query->priv->query_type == GNOME_DB_QUERY_TYPE_NON_PARSED_SQL)))
		g_return_if_fail (query_sql_forget (query, NULL));

	if (field_before) {
		g_return_if_fail (field_before && IS_GNOME_DB_QFIELD (field_before));
		g_return_if_fail (g_slist_find (query->priv->fields, field_before));
		g_return_if_fail (gnome_db_field_get_entity (field_before) == GNOME_DB_ENTITY (query));
		pos = g_slist_index (query->priv->fields, field_before);
	}

	query->priv->fields = g_slist_insert (query->priv->fields, field, pos);
	g_object_ref (G_OBJECT (field));

	if (query->priv->serial_field <= gnome_db_base_get_id (GNOME_DB_BASE (field)))
		query->priv->serial_field = gnome_db_base_get_id (GNOME_DB_BASE (field)) + 1;
	
	gnome_db_base_connect_nullify (field, G_CALLBACK (nullified_field_cb), query);
        g_signal_connect (G_OBJECT (field), "changed",
                          G_CALLBACK (changed_field_cb), query);
        g_signal_connect (G_OBJECT (field), "id_changed",
                          G_CALLBACK (id_field_changed_cb), query);

#ifdef debug_signal
        g_print (">> 'FIELD_ADDED' from %s\n", __FUNCTION__);
#endif
        g_signal_emit_by_name (G_OBJECT (query), "field_added", field);
#ifdef debug_signal
        g_print ("<< 'FIELD_ADDED' from %s\n", __FUNCTION__);
#endif	
	m_changed_cb (query);
}

static void
nullified_field_cb (GnomeDbField *field, GnomeDbQuery *query)
{
	g_assert (g_slist_find (query->priv->fields, field));

	if (! (IS_GNOME_DB_QF_VALUE (field) && 
	       (query->priv->query_type == GNOME_DB_QUERY_TYPE_NON_PARSED_SQL)))
		g_return_if_fail (query_sql_forget (query, NULL));

	gnome_db_query_set_order_by_field (query, GNOME_DB_QFIELD (field), -1, FALSE);
        query->priv->fields = g_slist_remove (query->priv->fields, field);
        g_signal_handlers_disconnect_by_func (G_OBJECT (field),
                                              G_CALLBACK (nullified_field_cb), query);
        g_signal_handlers_disconnect_by_func (G_OBJECT (field),
                                              G_CALLBACK (changed_field_cb), query);
        g_signal_handlers_disconnect_by_func (G_OBJECT (field),
                                              G_CALLBACK (id_field_changed_cb), query);

#ifdef debug_signal
        g_print (">> 'FIELD_REMOVED' from %s\n", __FUNCTION__);
#endif
        g_signal_emit_by_name (G_OBJECT (query), "field_removed", field);
#ifdef debug_signal
        g_print ("<< 'FIELD_REMOVED' from %s\n", __FUNCTION__);
#endif

        g_object_unref (field);

	/* cleaning... */
	query_clean_junk (query);
}

static void
changed_field_cb (GnomeDbField *field, GnomeDbQuery *query)
{
#ifdef debug_signal
        g_print (">> 'FIELD_UPDATED' from %s\n", __FUNCTION__);
#endif
        g_signal_emit_by_name (G_OBJECT (query), "field_updated", field);
#ifdef debug_signal
        g_print ("<< 'FIELD_UPDATED' from %s\n", __FUNCTION__);
#endif
	gnome_db_base_changed (GNOME_DB_BASE (query));
}


static void
gnome_db_query_swap_fields (GnomeDbEntity *iface, GnomeDbField *field1, GnomeDbField *field2)
{
	GnomeDbQuery *query;
	GSList *ptr1, *ptr2;

	g_return_if_fail (iface && IS_GNOME_DB_QUERY (iface));
	g_return_if_fail (GNOME_DB_QUERY (iface)->priv);
	query = GNOME_DB_QUERY (iface);
	g_return_if_fail (query_sql_forget (query, NULL));

	g_return_if_fail (field1 && IS_GNOME_DB_QFIELD (field1));
	ptr1 = g_slist_find (query->priv->fields, field1);
	g_return_if_fail (ptr1);

	g_return_if_fail (field2 && IS_GNOME_DB_QFIELD (field2));
	ptr2 = g_slist_find (query->priv->fields, field2);
	g_return_if_fail (ptr2);

	ptr1->data = field2;
	ptr2->data = field1;

#ifdef debug_signal
        g_print (">> 'FIELDS_ORDER_CHANGED' from %s\n", __FUNCTION__);
#endif
        g_signal_emit_by_name (G_OBJECT (query), "fields_order_changed");
#ifdef debug_signal
        g_print ("<< 'FIELDS_ORDER_CHANGED' from %s\n", __FUNCTION__);
#endif

	gnome_db_base_changed (GNOME_DB_BASE (query));
}

static void
gnome_db_query_remove_field (GnomeDbEntity *iface, GnomeDbField *field)
{
	GnomeDbQuery *query;

	g_return_if_fail (iface && IS_GNOME_DB_QUERY (iface));
	g_return_if_fail (GNOME_DB_QUERY (iface)->priv);
	query = GNOME_DB_QUERY (iface);
	g_return_if_fail (field && IS_GNOME_DB_QFIELD (field));
	g_return_if_fail (g_slist_find (query->priv->fields, field));

	gnome_db_base_nullify (GNOME_DB_BASE (field));
}

static gboolean
gnome_db_query_is_writable (GnomeDbEntity *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_QUERY (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_QUERY (iface)->priv, FALSE);
	
	return FALSE;
}

static GSList *
gnome_db_query_get_parameters (GnomeDbEntity *iface)
{
	GnomeDbQuery *query;
	GSList *list, *tmplist, *retval = NULL;

	g_return_val_if_fail (iface && IS_GNOME_DB_QUERY (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QUERY (iface)->priv, NULL);
	query = GNOME_DB_QUERY (iface);
	
	/* parameters for value fields */
	list = query->priv->fields;
	while (list) {
		tmplist = gnome_db_qfield_get_parameters (GNOME_DB_QFIELD (list->data));
		if (tmplist)
			retval = g_slist_concat (retval, tmplist);
		list = g_slist_next (list);
	}

	/* parameters for sub queries */
	list = query->priv->sub_queries;
	while (list) {
		tmplist = gnome_db_query_get_parameters (GNOME_DB_ENTITY (list->data));
		if (tmplist)
			retval = g_slist_concat (retval, tmplist);
		list = g_slist_next (list);
	}

	return retval;
}

/**
 * gnome_db_query_get_main_conditions
 * @query: a #GnomeDbQuery object
 *
 * Makes a list of all the conditions (part of the WHERE clause) which
 * are always verified by @query when it is executed.
 *
 * Examples: if the WHERE clause is:
 * --> "A and B" then the list will contains {A, B}
 * --> "A and (B or C)" it will contain {A, B or C}
 * --> "A and (B and not C)", it will contain {A, B, not C}
 *
 * Returns: a new list of #GnomeDbCondition objects
 */
GSList *
gnome_db_query_get_main_conditions (GnomeDbQuery *query)
{
	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (query->priv, NULL);

	if (query->priv->cond)
		return gnome_db_condition_get_main_conditions (query->priv->cond);
	else
		return NULL;
}

/**
 * gnome_db_query_append_condition
 * @query: a #GnomeDbQuery object
 * @cond: a #GnomeDbCondition object
 * @append_as_and: mode of append if there is already a query condition
 *
 * Appends the @cond object to @query's condition. If @query does not yet
 * have any condition, then the result is the same as gnome_db_query_set_condition();
 * otherwise, @cond is added to @query's condition, using the AND operator
 * if @append_as_and is TRUE, and an OR operator if @append_as_and is FALSE.
 */
void
gnome_db_query_append_condition (GnomeDbQuery *query, GnomeDbCondition *cond, gboolean append_as_and)
{
	g_return_if_fail (query && IS_GNOME_DB_QUERY (query));
	g_return_if_fail (query->priv);
	g_return_if_fail (cond && IS_GNOME_DB_CONDITION (cond));

	if (!query->priv->cond)
		gnome_db_query_set_condition (query, cond);
	else {
		if ((append_as_and && 
		     (gnome_db_condition_get_cond_type (query->priv->cond) == GNOME_DB_CONDITION_NODE_AND)) ||
		    (!append_as_and && 
		     (gnome_db_condition_get_cond_type (query->priv->cond) == GNOME_DB_CONDITION_NODE_OR))) {
			/* add cond to query->priv->cond */
			g_assert (gnome_db_condition_node_add_child (query->priv->cond, cond, NULL));
		}
		else {
			GnomeDbCondition *nodecond, *oldcond;

			oldcond = query->priv->cond;
			nodecond = GNOME_DB_CONDITION (gnome_db_condition_new (query,
									       append_as_and ? 
									       GNOME_DB_CONDITION_NODE_AND : 
									       GNOME_DB_CONDITION_NODE_OR));
			g_object_ref (G_OBJECT (oldcond));
			query->priv->internal_transaction ++;
			gnome_db_query_set_condition (query, nodecond);
			query->priv->internal_transaction --;
			g_assert (gnome_db_condition_node_add_child (nodecond, oldcond, NULL));
			g_object_unref (G_OBJECT (oldcond));
			g_object_unref (G_OBJECT (nodecond));

			g_assert (gnome_db_condition_node_add_child (query->priv->cond, cond, NULL));
		}
	}
}

/* 
 * GnomeDbReferer interface implementation
 */

static gboolean
gnome_db_query_activate (GnomeDbReferer *iface)
{
	gboolean retval = TRUE;
	GnomeDbQuery *query;
	GSList *list;

	g_return_val_if_fail (iface && IS_GNOME_DB_QUERY (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_QUERY (iface)->priv, FALSE);
	query = GNOME_DB_QUERY (iface);


	list = query->priv->param_sources;
	while (list && retval) {
		retval = gnome_db_referer_activate (GNOME_DB_REFERER (list->data));
		list = g_slist_next (list);
	}

	list = query->priv->sub_queries;
	while (list && retval) {
		retval = gnome_db_referer_activate (GNOME_DB_REFERER (list->data));
		list = g_slist_next (list);
	}

	list = query->priv->targets;
	while (list && retval) {
		retval = gnome_db_referer_activate (GNOME_DB_REFERER (list->data));
		list = g_slist_next (list);
	}

	list = query->priv->fields;
	while (list && retval) {
		retval = gnome_db_referer_activate (GNOME_DB_REFERER (list->data));
		list = g_slist_next (list);
	}

	list = query->priv->joins_flat;
	while (list && retval) {
		retval = gnome_db_referer_activate (GNOME_DB_REFERER (list->data));
		list = g_slist_next (list);
	}

	if (retval && query->priv->cond)
		retval = gnome_db_referer_activate (GNOME_DB_REFERER (query->priv->cond));

	return retval;
}

static void
gnome_db_query_deactivate (GnomeDbReferer *iface)
{
	GnomeDbQuery *query;
	GSList *list;

	g_return_if_fail (iface && IS_GNOME_DB_QUERY (iface));
	g_return_if_fail (GNOME_DB_QUERY (iface)->priv);
	query = GNOME_DB_QUERY (iface);

	list = query->priv->param_sources;
	while (list) {
		gnome_db_referer_deactivate (GNOME_DB_REFERER (list->data));
		list = g_slist_next (list);
	}

	list = query->priv->sub_queries;
	while (list) {
		gnome_db_referer_deactivate (GNOME_DB_REFERER (list->data));
		list = g_slist_next (list);
	}

	list = query->priv->targets;
	while (list) {
		gnome_db_referer_deactivate (GNOME_DB_REFERER (list->data));
		list = g_slist_next (list);
	}

	list = query->priv->fields;
	while (list) {
		gnome_db_referer_deactivate (GNOME_DB_REFERER (list->data));
		list = g_slist_next (list);
	}

	list = query->priv->joins_flat;
	while (list) {
		gnome_db_referer_deactivate (GNOME_DB_REFERER (list->data));
		list = g_slist_next (list);
	}

	if (query->priv->cond)
		gnome_db_referer_deactivate (GNOME_DB_REFERER (query->priv->cond));
}

static gboolean
gnome_db_query_are_joins_active (GnomeDbQuery *query)
{
	gboolean retval = TRUE;
	GSList *list;
	list = query->priv->joins_flat;
	while (list && retval) {
		retval = gnome_db_referer_is_active (GNOME_DB_REFERER (list->data));
		list = g_slist_next (list);
	}

	return retval;
}

static gboolean
gnome_db_query_is_active (GnomeDbReferer *iface)
{
	gboolean retval = TRUE;
	GnomeDbQuery *query;
	GSList *list;

	g_return_val_if_fail (iface && IS_GNOME_DB_QUERY (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_QUERY (iface)->priv, FALSE);
	query = GNOME_DB_QUERY (iface);

	list = query->priv->param_sources;
	while (list && retval) {
		retval = gnome_db_referer_is_active (GNOME_DB_REFERER (list->data));
		list = g_slist_next (list);
	}

	list = query->priv->sub_queries;
	while (list && retval) {
		retval = gnome_db_referer_is_active (GNOME_DB_REFERER (list->data));
		list = g_slist_next (list);
	}

	list = query->priv->targets;
	while (list && retval) {
		retval = gnome_db_referer_is_active (GNOME_DB_REFERER (list->data));
		list = g_slist_next (list);
	}

	if (retval)
		retval = gnome_db_query_are_joins_active (query);

	list = query->priv->fields;
	while (list && retval) {
		retval = gnome_db_referer_is_active (GNOME_DB_REFERER (list->data));
		list = g_slist_next (list);
	}

	if (retval && query->priv->cond)
		retval = gnome_db_referer_is_active (GNOME_DB_REFERER (query->priv->cond));

	return retval;
}

static GSList *
gnome_db_query_get_ref_objects (GnomeDbReferer *iface)
{
	GSList *list = NULL, *sub, *retval = NULL;
	GnomeDbQuery *query;

	/* FIXME: do not take care of the objects which belong to the query itself */

	g_return_val_if_fail (iface && IS_GNOME_DB_QUERY (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QUERY (iface)->priv, NULL);
	query = GNOME_DB_QUERY (iface);

	list = query->priv->param_sources;
	while (list && retval) {
		sub = gnome_db_referer_get_ref_objects (GNOME_DB_REFERER (list->data));
		retval = g_slist_concat (retval, sub);
		list = g_slist_next (list);
	}

	list = query->priv->sub_queries;
	while (list && retval) {
		sub = gnome_db_referer_get_ref_objects (GNOME_DB_REFERER (list->data));
		retval = g_slist_concat (retval, sub);
		list = g_slist_next (list);
	}

	list = query->priv->targets;
	while (list && retval) {
		sub = gnome_db_referer_get_ref_objects (GNOME_DB_REFERER (list->data));
		retval = g_slist_concat (retval, sub);
		list = g_slist_next (list);
	}

	list = query->priv->fields;
	while (list && retval) {
		sub = gnome_db_referer_get_ref_objects (GNOME_DB_REFERER (list->data));
		retval = g_slist_concat (retval, sub);
		list = g_slist_next (list);
	}

	list = query->priv->joins_flat;
	while (list && retval) {
		sub = gnome_db_referer_get_ref_objects (GNOME_DB_REFERER (list->data));
		retval = g_slist_concat (retval, sub);
		list = g_slist_next (list);
	}

	if (query->priv->cond) {
		sub = gnome_db_referer_get_ref_objects (GNOME_DB_REFERER (query->priv->cond));
		retval = g_slist_concat (retval, sub);
	}

	return retval;
}

static void
gnome_db_query_replace_refs (GnomeDbReferer *iface, GHashTable *replacements)
{
	GSList *list;
	GnomeDbQuery *query;

	g_return_if_fail (iface && IS_GNOME_DB_QUERY (iface));
	g_return_if_fail (GNOME_DB_QUERY (iface)->priv);
	query = GNOME_DB_QUERY (iface);

	list = query->priv->param_sources;
	while (list) {
		gnome_db_referer_replace_refs (GNOME_DB_REFERER (list->data), replacements);
		list = g_slist_next (list);
	}

	list = query->priv->sub_queries;
	while (list) {
		gnome_db_referer_replace_refs (GNOME_DB_REFERER (list->data), replacements);
		list = g_slist_next (list);
	}

	list = query->priv->targets;
	while (list) {
		gnome_db_referer_replace_refs (GNOME_DB_REFERER (list->data), replacements);
		list = g_slist_next (list);
	}

	list = query->priv->fields;
	while (list) {
		gnome_db_referer_replace_refs (GNOME_DB_REFERER (list->data), replacements);
		list = g_slist_next (list);
	}

	list = query->priv->joins_flat;
	while (list) {
		gnome_db_referer_replace_refs (GNOME_DB_REFERER (list->data), replacements);
		list = g_slist_next (list);
	}

	if (query->priv->cond)
		gnome_db_referer_replace_refs (GNOME_DB_REFERER (query->priv->cond), replacements);
}

/* 
 * GnomeDbXmlStorage interface implementation
 */

static const gchar *convert_query_type_to_str (GnomeDbQueryType type);
static GnomeDbQueryType  convert_str_to_query_type (const gchar *str);

static gchar *
gnome_db_query_get_xml_id (GnomeDbXmlStorage *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_QUERY (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QUERY (iface)->priv, NULL);

	return g_strdup_printf ("QU%d", gnome_db_base_get_id (GNOME_DB_BASE (iface)));
}

static xmlNodePtr
gnome_db_query_save_to_xml (GnomeDbXmlStorage *iface, GError **error)
{
	xmlNodePtr node = NULL, psources = NULL;
	GnomeDbQuery *query;
	gchar *str;
	const gchar *type;
	GSList *list;

	g_return_val_if_fail (iface && IS_GNOME_DB_QUERY (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QUERY (iface)->priv, NULL);
	query = GNOME_DB_QUERY (iface);


	/* query itself */
	node = xmlNewNode (NULL, "GNOME_DB_QUERY");
	str = gnome_db_query_get_xml_id (GNOME_DB_XML_STORAGE (query));
	xmlSetProp (node, "id", str);
	g_free (str);
	xmlSetProp (node, "name", gnome_db_base_get_name (GNOME_DB_BASE (query)));
        xmlSetProp (node, "descr", gnome_db_base_get_description (GNOME_DB_BASE (query)));
	type = convert_query_type_to_str (query->priv->query_type);
	xmlSetProp (node, "query_type", type);

	/* param sources */
	list = query->priv->param_sources;
	if (list) 
		psources = xmlNewChild (node, NULL, "GNOME_DB_PARAM_SOURCES", NULL);
	
	while (list) {
		xmlNodePtr sub = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (list->data), error);
		if (sub)
			xmlAddChild (psources, sub);
		else {
			xmlFreeNode (node);
			return NULL;
		}
		list = g_slist_next (list);
	}

	if (query->priv->query_type != GNOME_DB_QUERY_TYPE_NON_PARSED_SQL) {
		if (query->priv->global_distinct)
			xmlSetProp (node, "distinct", "t");
		
		/* targets */
		list = query->priv->targets;
		while (list) {
			xmlNodePtr sub = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (list->data), error);
			if (sub)
				xmlAddChild (node, sub);
			else {
				xmlFreeNode (node);
				return NULL;
			}
			list = g_slist_next (list);
		}
		
		/* fields */
		list = query->priv->fields;
		while (list) {
			xmlNodePtr sub = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (list->data), error);
			if (sub)
				xmlAddChild (node, sub);
			else {
				xmlFreeNode (node);
				return NULL;
			}
			list = g_slist_next (list);
		}
		
		/* joins */
		list = query->priv->joins_flat;
		while (list) {
			xmlNodePtr sub = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (list->data), error);
			if (sub)
				xmlAddChild (node, sub);
			else {
				xmlFreeNode (node);
				return NULL;
			}
			list = g_slist_next (list);
		}

		/* condition */
		if (query->priv->cond) {
			xmlNodePtr sub = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (query->priv->cond), error);
			if (sub)
				xmlAddChild (node, sub);
			else {
				xmlFreeNode (node);
				return NULL;
			}
		}
		
		/* fields ORDER BY */
		if (query->priv->fields_order_by) {
			xmlNodePtr sub = xmlNewChild (node, NULL, "GNOME_DB_FIELDS_ORDER", NULL);
			
			list = query->priv->fields_order_by;
			while (list) {
				xmlNodePtr order = xmlNewChild (sub, NULL, "GNOME_DB_QF_REF", NULL);
				
				str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (list->data));
				xmlSetProp (order, "object", str);
				g_free (str);
				
				xmlSetProp (order, "order", 
					    g_object_get_data (G_OBJECT (list->data), "order_by_asc") ? "ASC" : "DES");
				list = g_slist_next (list);
			}
		}

		/* sub queries */
		list = query->priv->sub_queries;
		while (list) {
			xmlNodePtr sub = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (list->data), error);
			if (sub)
				xmlAddChild (node, sub);
			else {
				xmlFreeNode (node);
				return NULL;
			}
			list = g_slist_next (list);
		}
	}
	else {
		/* Text if SQL query */
		xmlNewChild (node, NULL, "GNOME_DB_QUERY_TEXT", query->priv->sql);
	}

	return node;
}


static gboolean
gnome_db_query_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error)
{
	GnomeDbQuery *query;
	gchar *prop;
	gboolean id = FALSE;
	xmlNodePtr children;

	g_return_val_if_fail (iface && IS_GNOME_DB_QUERY (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_QUERY (iface)->priv, FALSE);
	g_return_val_if_fail (node, FALSE);
	query = GNOME_DB_QUERY (iface);
	gnome_db_query_clean (query);
	g_return_val_if_fail (query_sql_forget (query, error), FALSE);

	if (strcmp (node->name, "GNOME_DB_QUERY")) {
		g_set_error (error,
			     GNOME_DB_QUERY_ERROR,
			     GNOME_DB_QUERY_XML_LOAD_ERROR,
			     _("XML Tag is not <GNOME_DB_QUERY>"));
		return FALSE;
	}

	/* query itself */
	prop = xmlGetProp (node, "id");
	if (prop) {
		gnome_db_base_set_id (GNOME_DB_BASE (query), atoi (prop+2));
		g_free (prop);
		id = TRUE;
	}	

	prop = xmlGetProp (node, "name");
        if (prop) {
                gnome_db_base_set_name (GNOME_DB_BASE (query), prop);
                g_free (prop);
        }

        prop = xmlGetProp (node, "descr");
        if (prop) {
                gnome_db_base_set_description (GNOME_DB_BASE (query), prop);
                g_free (prop);
        }

	prop = xmlGetProp (node, "query_type");
	if (prop) {
		query->priv->query_type = convert_str_to_query_type (prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "distinct");
	if (prop) {
		query->priv->global_distinct = *prop == 't' ? TRUE : FALSE;
		g_free (prop);
	}

	/* children nodes */
	children = node->children;
	while (children) {
		gboolean done = FALSE;

		/* parameters sources */
		if (!done && !strcmp (children->name, "GNOME_DB_PARAM_SOURCES")) {
			GnomeDbDict *dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));
			xmlNodePtr sparams = children->children;
			while (sparams) {
				if (!strcmp (sparams->name, "GNOME_DB_QUERY")) {
					GnomeDbQuery *squery;
					
					squery = GNOME_DB_QUERY (gnome_db_query_new (dict));
					if (gnome_db_xml_storage_load_from_xml (GNOME_DB_XML_STORAGE (squery), sparams, error)) {
						gnome_db_query_add_param_source (query, squery);
						g_object_unref (G_OBJECT (squery));
					}
					else
						return FALSE;
				}
				sparams = sparams->next;
			}
			done = TRUE;
                }

		/* targets */
		if (!done && !strcmp (children->name, "GNOME_DB_TARGET")) {
                        GnomeDbTarget *target;
			gchar *ent_id;

			ent_id = xmlGetProp (children, "entity_ref");
			if (ent_id) {
				target = GNOME_DB_TARGET (gnome_db_target_new_with_xml_id (query, ent_id));
				g_free (ent_id);
				if (gnome_db_xml_storage_load_from_xml (GNOME_DB_XML_STORAGE (target), children, error)) {
					if (!gnome_db_query_add_target (query, target, error)) {
						g_object_unref (G_OBJECT (target));
						return FALSE;
					}
					g_object_unref (G_OBJECT (target));
				}
				else 
					return FALSE;
			}
			else
				return FALSE;
			done = TRUE;
                }

		/* fields */
		if (!done && !strcmp (children->name, "GNOME_DB_QF")) {
			GObject *obj;

			obj = gnome_db_qfield_new_from_xml (query, children, error);
			if (obj) {
				gnome_db_query_add_field (GNOME_DB_ENTITY (query), GNOME_DB_FIELD (obj));
				g_object_unref (G_OBJECT (obj));
			}
			else
				return FALSE;
			done = TRUE;
                }

		/* joins */
		if (!done && !strcmp (children->name, "GNOME_DB_JOIN")) {
                        GnomeDbJoin *join;
			gchar *t1, *t2;

			t1 = xmlGetProp (children, "target1");
			t2 = xmlGetProp (children, "target2");

			if (t1 && t2) {
				join = GNOME_DB_JOIN (gnome_db_join_new_with_xml_ids (query, t1, t2));
				g_free (t1);
				g_free (t2);
				if (gnome_db_xml_storage_load_from_xml (GNOME_DB_XML_STORAGE (join), children, error)) {
					gnome_db_query_add_join (query, join);
					g_object_unref (G_OBJECT (join));
				}
				else
					return FALSE;
			}
			else
				return FALSE;
			done = TRUE;
                }

		/* condition */
		if (!done && !strcmp (children->name, "GNOME_DB_COND")) {
			GnomeDbCondition *cond;

			cond = GNOME_DB_CONDITION (gnome_db_condition_new (query, GNOME_DB_CONDITION_NODE_AND));
			if (gnome_db_xml_storage_load_from_xml (GNOME_DB_XML_STORAGE (cond), children, error)) {
				gnome_db_query_set_condition (query, cond);
				g_object_unref (G_OBJECT (cond));
			}
			else
				return FALSE;
			done = TRUE;
                }

		/* fields ORDER BY */
		if (!done && !strcmp (children->name, "GNOME_DB_FIELDS_ORDER")) {
			xmlNodePtr order = children->children;
			gint pos = 0;

			while (order) {
				if (!strcmp (order->name, "GNOME_DB_QF_REF")) {
					GnomeDbField *field = NULL;
					gboolean asc = TRUE;

					prop = xmlGetProp (order, "object");
					if (prop) {
						field = gnome_db_entity_get_field_by_xml_id (GNOME_DB_ENTITY (query), prop);
						if (!field)
							g_set_error (error,
								     GNOME_DB_QUERY_ERROR,
								     GNOME_DB_QUERY_XML_LOAD_ERROR,
								     _("Can't find field '%s'"), prop);
						g_free (prop);
						pos ++;
					}
					
					prop = xmlGetProp (order, "order");
					if (prop) {
						asc = (*prop == 'A');
						g_free (prop);
					}
					if (field) 
						gnome_db_query_set_order_by_field (query, GNOME_DB_QFIELD (field), pos, asc);
					else 
						return FALSE;
				}
				order = order->next;
			}

			done = TRUE;
                }

		/* textual query */
		if (!done && !strcmp (children->name, "GNOME_DB_QUERY_TEXT")) {
			gchar *contents;
			GError *error = NULL;

			contents = xmlNodeGetContent (children);
			gnome_db_query_set_sql_text (query, contents, &error);
			if (error)
				g_error_free (error);
			g_free (contents);
			done = TRUE;
                }

		/* sub queries */
		if (!done && !strcmp (children->name, "GNOME_DB_QUERY")) {
			GnomeDbQuery *squery;
			GnomeDbDict *dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));

			squery = GNOME_DB_QUERY (gnome_db_query_new (dict));
			if (gnome_db_xml_storage_load_from_xml (GNOME_DB_XML_STORAGE (squery), children, error)) {
				gnome_db_query_add_sub_query (query, squery);
				g_object_unref (G_OBJECT (squery));
			}
			else
				return FALSE;
			done = TRUE;
                }
		
		children = children->next;
	}

	if (0) {
		GSList *list;
		list = query->priv->fields;
		while (list) {
			g_print ("Query %p, field %p: refcount=%d\n", query, list->data, G_OBJECT (list->data)->ref_count);
			list = g_slist_next (list);
		}
	}

	if (id)
		return TRUE;
	else {
		g_set_error (error,
			     GNOME_DB_QUERY_ERROR,
			     GNOME_DB_QUERY_XML_LOAD_ERROR,
			     _("Problem loading <GNOME_DB_QUERY>"));
		return FALSE;
	}
}

static const gchar *
convert_query_type_to_str (GnomeDbQueryType type)
{
	switch (type) {
	default:
	case GNOME_DB_QUERY_TYPE_SELECT:
		return "SEL";
	case GNOME_DB_QUERY_TYPE_INSERT:
		return "INS";
	case GNOME_DB_QUERY_TYPE_UPDATE:
		return "UPD";
	case GNOME_DB_QUERY_TYPE_DELETE:
		return "DEL";
	case GNOME_DB_QUERY_TYPE_UNION:
		return "NION";
	case GNOME_DB_QUERY_TYPE_INTERSECT:
		return "ECT";
	case GNOME_DB_QUERY_TYPE_EXCEPT:
		return "XPT";
	case GNOME_DB_QUERY_TYPE_NON_PARSED_SQL:
		return "TXT";
	}
}

static GnomeDbQueryType
convert_str_to_query_type (const gchar *str)
{
	switch (*str) {
	case 'S':
	default:
		return GNOME_DB_QUERY_TYPE_SELECT;
	case 'I':
		return GNOME_DB_QUERY_TYPE_INSERT;
	case 'U':
		return GNOME_DB_QUERY_TYPE_UPDATE;
	case 'D':
		return GNOME_DB_QUERY_TYPE_DELETE;
	case 'N':
		return GNOME_DB_QUERY_TYPE_UNION;
	case 'E':
		return GNOME_DB_QUERY_TYPE_INTERSECT;
	case 'T':
		return GNOME_DB_QUERY_TYPE_NON_PARSED_SQL;
	case 'X':
		return GNOME_DB_QUERY_TYPE_EXCEPT;
	}
}





/*
 * GnomeDbRenderer interface implementation
 */
static GdaXqlItem *
gnome_db_query_render_as_xql (GnomeDbRenderer *iface, GnomeDbDataSet *context, GError **error)
{
	GnomeDbQuery *query;
	g_return_val_if_fail (iface && IS_GNOME_DB_QUERY (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QUERY (iface)->priv, NULL);
	query = GNOME_DB_QUERY (iface);

	TO_IMPLEMENT;
	return NULL;
}

static gchar *render_sql_select (GnomeDbQuery *query, GnomeDbDataSet *context, guint options, GError **error);
static gchar *render_sql_insert (GnomeDbQuery *query, GnomeDbDataSet *context, guint options, GError **error);
static gchar *render_sql_update (GnomeDbQuery *query, GnomeDbDataSet *context, guint options, GError **error);
static gchar *render_sql_delete (GnomeDbQuery *query, GnomeDbDataSet *context, guint options, GError **error);
static gchar *render_sql_union (GnomeDbQuery *query, GnomeDbDataSet *context, guint options, GError **error);
static gchar *render_sql_intersect (GnomeDbQuery *query, GnomeDbDataSet *context, guint options, GError **error);
static gchar *render_sql_except (GnomeDbQuery *query, GnomeDbDataSet *context, guint options, GError **error);
static gchar *render_sql_non_parsed_with_params (GnomeDbQuery *query, GnomeDbDataSet *context, guint options, GError **error);

static gboolean assert_coherence_all_params_present (GnomeDbQuery *query, GnomeDbDataSet *context, GError **error);
static gboolean assert_coherence_entities_same_fields (GnomeDbEntity *ent1, GnomeDbEntity *ent2);
static gboolean assert_coherence_sub_query_select (GnomeDbQuery *query, GnomeDbDataSet *context, GError **error);
static gboolean assert_coherence_data_select_query (GnomeDbQuery *query, GnomeDbDataSet *context, GError **error);
static gboolean assert_coherence_data_modify_query (GnomeDbQuery *query, GnomeDbDataSet *context, GError **error);
static gboolean assert_coherence_aggregate_query (GnomeDbQuery *query, GnomeDbDataSet *context, GError **error);

static gchar *
gnome_db_query_render_as_sql (GnomeDbRenderer *iface, GnomeDbDataSet *context, guint options, GError **error)
{
	GnomeDbQuery *query;
	gchar *sql = NULL;
	
	g_return_val_if_fail (iface && IS_GNOME_DB_QUERY (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QUERY (iface)->priv, NULL);
	if (!gnome_db_referer_activate (GNOME_DB_REFERER (iface))) {
		g_set_error (error,
			     GNOME_DB_QUERY_ERROR,
			     GNOME_DB_QUERY_RENDER_ERROR,
			     _("Can't resolve some references in the query: can't activate the query"));
		return NULL;
	}
	query = GNOME_DB_QUERY (iface);

	/* make sure all the required parameters are in @context */
	if (!assert_coherence_all_params_present (query, context, error))
		return NULL;

	switch (query->priv->query_type) {
	case GNOME_DB_QUERY_TYPE_SELECT:
		if (assert_coherence_data_select_query (query, context, error))
			sql = render_sql_select (query, context, options, error);
		break;
	case GNOME_DB_QUERY_TYPE_INSERT:
		if (assert_coherence_data_modify_query (query, context, error))
			sql = render_sql_insert (query, context, options, error);
		break;
	case GNOME_DB_QUERY_TYPE_UPDATE:
		if (assert_coherence_data_modify_query (query, context, error))
			sql = render_sql_update (query, context, options, error);
		break;
	case GNOME_DB_QUERY_TYPE_DELETE:
		if (assert_coherence_data_modify_query (query, context, error))
			sql = render_sql_delete (query, context, options, error);
		break;
	case GNOME_DB_QUERY_TYPE_UNION:
		if (assert_coherence_aggregate_query (query, context, error))
			sql = render_sql_union (query, context, options, error);
		break;
	case GNOME_DB_QUERY_TYPE_INTERSECT:
		if (assert_coherence_aggregate_query (query, context, error))
			sql = render_sql_intersect (query, context, options, error);
		break;
	case GNOME_DB_QUERY_TYPE_EXCEPT:
		if (assert_coherence_aggregate_query (query, context, error)) {
			if (g_slist_length (query->priv->sub_queries) != 2)
				g_set_error (error,
					     GNOME_DB_QUERY_ERROR,
					     GNOME_DB_QUERY_RENDER_ERROR,
					     _("More than two sub queries for an EXCEPT query"));
			else
				sql = render_sql_except (query, context, options, error);
		}
		break;
	case GNOME_DB_QUERY_TYPE_NON_PARSED_SQL:
		if (query->priv->sql && *(query->priv->sql)) {
			if (query->priv->sql_exprs && query->priv->sql_exprs->params_specs &&
			    assert_coherence_all_params_present (query, context, NULL))
				sql = render_sql_non_parsed_with_params (query, context, options, error);
			else
				sql = g_strdup (query->priv->sql);
		}
		else
			g_set_error (error,
				     GNOME_DB_QUERY_ERROR,
				     GNOME_DB_QUERY_RENDER_ERROR,
				     _("Query without any SQL code"));
		break;
	default:
		g_assert_not_reached ();
	}

	return sql;
}

/* make sure the context provides all the required values for the parameters */
static gboolean
assert_coherence_all_params_present (GnomeDbQuery *query, GnomeDbDataSet *context, GError **error)
{
	gboolean allok = TRUE;
	GSList *params, *plist;

	params = gnome_db_entity_get_parameters (GNOME_DB_ENTITY (query));

	plist = params;
	while (plist && allok) {
		GSList *for_fields = gnome_db_parameter_get_dest_fields (GNOME_DB_PARAMETER (plist->data));

		while (allok && for_fields) {
			if (gnome_db_field_get_entity (GNOME_DB_FIELD (for_fields->data)) == GNOME_DB_ENTITY (query)) {
				gboolean found = FALSE;
				GSList *clist = NULL;
				GnomeDbQfield *for_field = GNOME_DB_QFIELD (for_fields->data);
				GnomeDbParameter *invalid_param = NULL;

				if (context)
					clist = context->parameters;
				
				/* if the parameter has a value, then OK */
				if (IS_GNOME_DB_QF_VALUE (for_field) && gnome_db_qf_value_get_value (GNOME_DB_QF_VALUE (for_field)))
					found = TRUE;
				
				/* try to find a value within the context */
				while (clist && !found && !invalid_param) {
					if (g_slist_find (gnome_db_parameter_get_dest_fields (GNOME_DB_PARAMETER (clist->data)), 
							  for_field)) {
						if (gnome_db_parameter_is_valid (GNOME_DB_PARAMETER (clist->data)))
							found = TRUE;
						else
							invalid_param = GNOME_DB_PARAMETER (clist->data);
					}
					clist = g_slist_next (clist);
				}
				
				if (!found) {
					if (context) {
						allok = FALSE;
						if (invalid_param)
							g_set_error (error,
								     GNOME_DB_QUERY_ERROR,
								     GNOME_DB_QUERY_RENDER_ERROR,
								     _("Invalid parameter '%s'"),
								     gnome_db_base_get_name (GNOME_DB_BASE (invalid_param)));
						else {
							g_set_error (error,
								     GNOME_DB_QUERY_ERROR,
								     GNOME_DB_QUERY_RENDER_ERROR,
								     _("Missing parameters"));
#ifdef debug
							g_print ("QUERY MISSING PARAM: QU%d %s\n", 
								 gnome_db_base_get_id (GNOME_DB_BASE (query)),
								 gnome_db_base_get_name (GNOME_DB_BASE (query)));
#endif
						}	
					}
				}
			}
			for_fields = g_slist_next (for_fields);
		}
		
		plist = g_slist_next (plist);
	}

	/* free all the parameters which won't be used again */
	plist = params;
	while (plist) {
		g_object_unref (G_OBJECT (plist->data));
		plist = g_slist_next (plist);
	}

	g_slist_free (params);

	return allok;
}

/* makes sure the number of fields of the two entities is the same.
 * IMPROVE: test for the field types compatibilities, but we know nothing about
 * possible (implicit or not) conversions between data types
 */
static gboolean
assert_coherence_entities_same_fields (GnomeDbEntity *ent1, GnomeDbEntity *ent2)
{
	gboolean retval;
	GSList *list1, *list2;

	list1 = gnome_db_entity_get_fields (ent1);
	list2 = gnome_db_entity_get_fields (ent2);

	retval = g_slist_length (list1) == g_slist_length (list2) ? TRUE : FALSE;

	g_slist_free (list1);
	g_slist_free (list2);

	return retval;
}

/* makes sure that all the sub queries of @query are SELECT queries, and are valid.
 * It is assumed that @query is active.
 *
 * Fills error if provided
 */
static gboolean 
assert_coherence_sub_query_select (GnomeDbQuery *query, GnomeDbDataSet *context, GError **error)
{
	GSList *list;
	gboolean retval = TRUE;
	GnomeDbQuery *sub;

	list = query->priv->sub_queries;
	while (list && retval) {
		sub = GNOME_DB_QUERY (list->data);
		if ((sub->priv->query_type != GNOME_DB_QUERY_TYPE_SELECT) &&
		    (sub->priv->query_type != GNOME_DB_QUERY_TYPE_UNION) && 
		    (sub->priv->query_type != GNOME_DB_QUERY_TYPE_INTERSECT) &&
		    (sub->priv->query_type != GNOME_DB_QUERY_TYPE_EXCEPT)){
			gchar *str = gnome_db_query_render_as_str (GNOME_DB_RENDERER (sub), context);
			
			retval = FALSE;
			g_set_error (error,
				     GNOME_DB_QUERY_ERROR,
				     GNOME_DB_QUERY_RENDER_ERROR,
				     _("Query %s is not a selection query"), str);
			g_free (str);
		}
		else 
			retval = assert_coherence_sub_query_select (sub, context, error);
		list = g_slist_next (list);
	}
	
	return retval;
}


static gboolean
assert_coherence_data_select_query (GnomeDbQuery *query, GnomeDbDataSet *context, GError **error)
{
	gboolean retval;

	/* make sure all the sub queries are of type SELECT (recursively) and are also valid */
	retval = assert_coherence_sub_query_select (query, context, error);
	return retval;
}

static gboolean
assert_coherence_data_modify_query (GnomeDbQuery *query, GnomeDbDataSet *context, GError **error)
{
	gboolean retval = TRUE;

	/* make sure there is only 1 target and the represented entity can be modified */
	if (retval && (g_slist_length (query->priv->targets) == 0)) {
		g_set_error (error,
			     GNOME_DB_QUERY_ERROR,
			     GNOME_DB_QUERY_RENDER_ERROR,
			     _("No target defined to apply modifications"));
		retval = FALSE;
	}

	if (retval && (g_slist_length (query->priv->targets) > 1)) {
		g_set_error (error,
			     GNOME_DB_QUERY_ERROR,
			     GNOME_DB_QUERY_RENDER_ERROR,
			     _("More than one target defined to apply modifications"));
		retval = FALSE;
	}
	
	/* make sure entity is writable */
	if (retval) {
		GnomeDbEntity *entity = gnome_db_target_get_represented_entity (GNOME_DB_TARGET (query->priv->targets->data));
		if (!gnome_db_entity_is_writable (entity)) {
			g_set_error (error,
				     GNOME_DB_QUERY_ERROR,
				     GNOME_DB_QUERY_RENDER_ERROR,
				     _("Entity %s is not writable"), gnome_db_base_get_name (GNOME_DB_BASE (entity)));
			retval = FALSE;	
		}
	}

	/* make sure all the sub queries are of type SELECT (recursively) and are also valid */
	if (retval)
		retval = assert_coherence_sub_query_select (query, context, error);

	/* make sure all visible fields are of type GNOME_DB_QF_FIELD */
	if (retval) {
		GSList *list;
		list = query->priv->fields;
		while (list && retval) {
			if (gnome_db_qfield_is_visible (GNOME_DB_QFIELD (list->data))) {
				if (G_OBJECT_TYPE (list->data) != GNOME_DB_QF_FIELD_TYPE) {
					g_set_error (error,
						     GNOME_DB_QUERY_ERROR,
						     GNOME_DB_QUERY_RENDER_ERROR,
						     _("Modification query field has incompatible type"));
					retval = FALSE;
				}
			}
			list = g_slist_next (list);
		}
	}

	/* INSERT specific tests */
	if (retval && (query->priv->query_type == GNOME_DB_QUERY_TYPE_INSERT)) {
		/* I there is a sub query, make sure that:
		   - there is only ONE sub query of type SELECT
		   - that sub query has the same number and the same types of visible fields
		*/
		if (query->priv->sub_queries) {
			if (g_slist_length (query->priv->sub_queries) > 1) {
				g_set_error (error,
					     GNOME_DB_QUERY_ERROR,
					     GNOME_DB_QUERY_RENDER_ERROR,
					     _("An insertion query can only have one sub-query"));
				retval = FALSE;
			}
			if (retval && !assert_coherence_entities_same_fields (GNOME_DB_ENTITY (query),
									      GNOME_DB_ENTITY (query->priv->sub_queries->data))) {
				g_set_error (error,
					     GNOME_DB_QUERY_ERROR,
					     GNOME_DB_QUERY_RENDER_ERROR,
					     _("Insertion query fields incompatible with sub query's fields"));
				retval = FALSE;
			}
		}
		else {
			/* If there is no sub query, then make sure all the fields are of values */
			GSList *list;
			list = query->priv->fields;
			while (list && retval) {
				if (gnome_db_qfield_is_visible (GNOME_DB_QFIELD (list->data))) {
					GnomeDbBase *value_prov;
					g_object_get (G_OBJECT (list->data), "value_provider", &value_prov, NULL);
					if (value_prov && 
					    ((G_OBJECT_TYPE (value_prov) == GNOME_DB_QF_FIELD_TYPE) ||
					     (G_OBJECT_TYPE (value_prov) == GNOME_DB_QF_ALL_TYPE))) {
						g_set_error (error,
							     GNOME_DB_QUERY_ERROR,
							     GNOME_DB_QUERY_RENDER_ERROR,
							     _("Insertion query field has incompatible value assignment"));
						retval = FALSE;
					}
				}
				list = g_slist_next (list);
			}
		}
		
		/* make sure there is no condition */
		if (retval && query->priv->cond) {
			g_set_error (error,
				     GNOME_DB_QUERY_ERROR,
				     GNOME_DB_QUERY_RENDER_ERROR,
				     _("Insertion query can't have any condition"));
			retval = FALSE;
		}
	}


	/* DELETE specific tests */
	if (retval && (query->priv->query_type == GNOME_DB_QUERY_TYPE_DELETE)) {
		GSList *list;
		list = query->priv->fields;
		while (list && retval) {
			if (gnome_db_qfield_is_visible (GNOME_DB_QFIELD (list->data))) {
				g_set_error (error,
					     GNOME_DB_QUERY_ERROR,
					     GNOME_DB_QUERY_RENDER_ERROR,
					     _("Deletion query can't have any visible field"));
				retval = FALSE;
			}
			list = g_slist_next (list);
		}
	}

	/* UPDATE specific tests */
	if (retval && (query->priv->query_type == GNOME_DB_QUERY_TYPE_UPDATE)) {
		GSList *list;
		list = query->priv->fields;
		while (list && retval) {
			if (gnome_db_qfield_is_visible (GNOME_DB_QFIELD (list->data))) {
				GnomeDbBase *value_prov;
				g_object_get (G_OBJECT (list->data), "value_provider", &value_prov, NULL);
				if (value_prov && 
				    (G_OBJECT_TYPE (value_prov) == GNOME_DB_QF_ALL_TYPE)) {
					g_set_error (error,
						     GNOME_DB_QUERY_ERROR,
						     GNOME_DB_QUERY_RENDER_ERROR,
						     _("Update query field has incompatible value assignment"));
					retval = FALSE;
				}
			}
			list = g_slist_next (list);
		}
	}

	return retval;
}

static gboolean
assert_coherence_aggregate_query (GnomeDbQuery *query, GnomeDbDataSet *context, GError **error)
{
	gboolean retval;

	/* FIXME: 
	   - make sure all the fields in each sub query are of the same type (and same number of them)
	*/

	/* make sure all the sub queries are of type SELECT (recursively) and are also valid */
	retval = assert_coherence_sub_query_select (query, context, error);

	if (retval && (g_slist_length (query->priv->targets) != 0)) {
		g_set_error (error,
			     GNOME_DB_QUERY_ERROR,
			     GNOME_DB_QUERY_RENDER_ERROR,
			     _("An aggregate type (UNION, etc) of query can't have any target"));
		retval = FALSE;
	}

	if (retval && query->priv->cond) {
		g_set_error (error,
			     GNOME_DB_QUERY_ERROR,
			     GNOME_DB_QUERY_RENDER_ERROR,
			     _("An aggregate type (UNION, etc) of query can't have any condition"));
		retval = FALSE;
	}

	return retval;
}

/* Structure used while rendering a JoinsPack: each time a join is analysed, such a structure is added to a list.
 * --> 'target' is always not NULL and represents the GnomeDbTarget the node brings.
 * --> 'first_join', if present, is the join which will be used to get the join type (INNER, etc)
 * --> 'other_joins', if present is the list of other joins participating in 'cond'
 * --> 'cond' is the string representing the join condition.
 */
typedef struct {
	GnomeDbTarget *target;
	GnomeDbJoin   *first_join;
	GSList   *other_joins;
	GString  *cond;
} JoinRenderNode;
#define JOIN_RENDER_NODE(x) ((JoinRenderNode *) x)

static gchar *render_join_condition (GnomeDbJoin *join, GnomeDbDataSet *context, guint options, GError **error, GSList *fk_constraints);
static gchar *
render_sql_select (GnomeDbQuery *query, GnomeDbDataSet *context, guint options, GError **error)
{
	GString *sql;
	gchar *retval, *str;
	GSList *list;
	gboolean first;
	gboolean err = FALSE;
	GnomeDbDict *dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));
	GSList *fk_constraints = NULL;
	gboolean pprint = options & GNOME_DB_RENDERER_EXTRA_PRETTY_SQL;

	fk_constraints = gnome_db_database_get_all_constraints (gnome_db_dict_get_database (dict));

	/* query is supposed to be active and of a good constitution here */
	sql = g_string_new ("SELECT ");

	/* DISTINCT */
	if (query->priv->global_distinct) 
		g_string_append (sql, "DISTINCT ");
	else {
		/* FIXME */
	}

	/* fields */
	first = TRUE;
	list = query->priv->fields;
	while (list && !err) {
		if (gnome_db_qfield_is_visible (GNOME_DB_QFIELD (list->data))) {
			if (first)
				first = FALSE;
			else {
				if (pprint)
					g_string_append (sql, ",\n\t");
				else
					g_string_append (sql, ", ");
			}

			str = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (list->data), context, options, error);
			if (str) {
				g_string_append (sql, str);
				g_free (str);

				str = gnome_db_qfield_get_alias (GNOME_DB_QFIELD (list->data));
				if (str && *str) {
					if (strstr (str, " "))
						g_string_append_printf (sql, " AS \"%s\"", str);
					else
						g_string_append_printf (sql, " AS %s", str);
				}
			}
			else {
				if (error && *error)
					err = TRUE;
				else
					g_string_append (sql, "NULL");
			}
		}
		list = g_slist_next (list);
	}

	
	/* targets and joins */
	if (query->priv->targets && !err) {
		GSList *all_joined = NULL; /* all the joined targets */
		GSList *packs, *list;
		JoinsPack *pack;
		GnomeDbJoin *join;

		first = TRUE;
		if (pprint)
			g_string_append (sql, "\nFROM ");
		else
			g_string_append (sql, " FROM ");
		packs = query->priv->joins_pack;
		while (packs && !err) {
			/* preparing the list of JoinRenderNodes */
			GSList *join_nodes = NULL;

			if (first) 
				first = FALSE;
			else {
				if (pprint)
					g_string_append (sql, ",\n\t");
				else
					g_string_append (sql, ", ");
			}

			pack = JOINS_PACK (packs->data);
			list = pack->joins;
			while (list && !err) {
				GSList *jnode;
				gint targets_found = 0;
				GnomeDbTarget *l_target, *r_target;
				JoinRenderNode *node = NULL;
				gchar *str;

				join = GNOME_DB_JOIN (list->data);
				l_target = gnome_db_join_get_target_1 (join);
				r_target = gnome_db_join_get_target_2 (join);

				/* Find a JoinRenderNode for the current join (in 'node')*/
				jnode = join_nodes;
				while (jnode && (targets_found < 2)) {
					if (JOIN_RENDER_NODE (jnode->data)->target == l_target)
						targets_found ++;
					if (JOIN_RENDER_NODE (jnode->data)->target == r_target)
						targets_found ++;
					if (targets_found == 2)
						node = JOIN_RENDER_NODE (jnode->data);
					jnode = g_slist_next (jnode);
				}
				g_assert (targets_found <= 2);
				switch (targets_found) {
				case 0:
					node = g_new0 (JoinRenderNode, 1);
					node->target = l_target;
					join_nodes = g_slist_append (join_nodes, node);
				case 1:
					node = g_new0 (JoinRenderNode, 1);
					node->target = r_target;
					node->first_join = join;
					join_nodes = g_slist_append (join_nodes, node);
					break;
				default:
					node->other_joins = g_slist_append (node->other_joins, join);
					break;
				}

				/* render the join condition in 'node->cond' */
				str = render_join_condition (join, context, options, error, fk_constraints);
				if (str) {
					if (!node->cond)
						node->cond = g_string_new ("");
					if (node->other_joins)
						g_string_append (node->cond, " AND ");
					g_string_append (node->cond, str);
				}
				else 
					err = TRUE;

				list = g_slist_next (list);
			}
			
			/* actual rendering of the joins in the JoinsPack */
			list = join_nodes;
			while (list) {
				if (list != join_nodes) {
					if (pprint)
						g_string_append (sql, "\n\t");
					else
						g_string_append (sql, " ");
				}
				
				/* join type if applicable */
				if (JOIN_RENDER_NODE (list->data)->first_join) {
					g_string_append (sql, gnome_db_join_render_type (JOIN_RENDER_NODE (list->data)->first_join));
					g_string_append (sql, " ");
				}

				/* target */
				str = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (JOIN_RENDER_NODE (list->data)->target), 
								 context, options, error);
				
				if (str) {
					g_string_append (sql, str);
					g_free (str);
				}
				else
					err = TRUE;

				/* condition */
				if (JOIN_RENDER_NODE (list->data)->cond) 
					g_string_append_printf (sql, " ON (%s)", 
								JOIN_RENDER_NODE (list->data)->cond->str);
				list = g_slist_next (list);
			}

			/* free the list of JoinRenderNodes */
			list = join_nodes;
			while (list) {
				if (JOIN_RENDER_NODE (list->data)->other_joins)
					g_slist_free (JOIN_RENDER_NODE (list->data)->other_joins);
				if (JOIN_RENDER_NODE (list->data)->cond)
					g_string_free (JOIN_RENDER_NODE (list->data)->cond, TRUE);
				g_free (list->data);
				list = g_slist_next (list);
			}
			if (join_nodes)
				g_slist_free (join_nodes);

			/* update the all_joined list */
			all_joined = g_slist_concat (all_joined, g_slist_copy (pack->targets));

			packs = g_slist_next (packs);
		}

		/* Adding targets in no join at all */
		list = query->priv->targets;
		while (list && !err) {
			if (!g_slist_find (all_joined, list->data)) {
				if (first) 
					first = FALSE;
				else
					g_string_append (sql, ", ");
				str = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (list->data), context, 
								 options, error);
				if (str) {
					g_string_append (sql, str);
					g_free (str);
				}
				else
					err = TRUE;
			}
			list = g_slist_next (list);
		}
		g_string_append (sql, " ");
		g_slist_free (all_joined);
	}

	/* GROUP BY */
	/* FIXME */	

	/* condition */
	if (query->priv->cond) {
		if (pprint)
			g_string_append (sql, "\nWHERE ");
		else
			g_string_append (sql, "WHERE ");
		str = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (query->priv->cond), context, options, error);
		if (str) {
			g_string_append (sql, str);
			g_string_append (sql, " ");
			g_free (str);
		}
		else
			err = TRUE;
	}
	
	/* ORDER BY */
	list = query->priv->fields_order_by;
	while (list) {
		if (0) { /* render using field numbers */
			gint pos = gnome_db_query_get_field_index (GNOME_DB_ENTITY (query), GNOME_DB_FIELD (list->data));
			if (pos > -1) {
				if (list == query->priv->fields_order_by) { 
					if (pprint)
						g_string_append (sql, "\nORDER BY ");
					else
						g_string_append (sql, "ORDER BY ");
				}
				else
					g_string_append (sql, ", ");
				
				g_string_append_printf (sql, "%d", pos + 1);
				if (! g_object_get_data (G_OBJECT (list->data), "order_by_asc"))
					g_string_append (sql, " DESC");
			}
		}
		else { /* render using field names */
			str = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (list->data), context, options, error);
			if (str) {
				if (list == query->priv->fields_order_by){ 
					if (pprint)
						g_string_append (sql, "\nORDER BY ");
					else
						g_string_append (sql, "ORDER BY ");
				} 
				else
					g_string_append (sql, ", ");	
				g_string_append (sql, str);
				if (! g_object_get_data (G_OBJECT (list->data), "order_by_asc"))
					g_string_append (sql, " DESC");
			}
		}
			
			
		list = g_slist_next (list);
	}

	if (!err) 
		retval = sql->str;
	else 
		retval = NULL;
	g_string_free (sql, err);

	if (fk_constraints)
		g_slist_free (fk_constraints);

	return retval;
}

static gchar *
render_join_condition (GnomeDbJoin *join, GnomeDbDataSet *context, guint options, GError **error, GSList *fk_constraints)
{
	GString *string;
	gchar *retval = NULL;
	gboolean joincond = FALSE;
	GnomeDbCondition *cond;
	gboolean err = FALSE;
	
	string = g_string_new ("");
	
	if ((cond = gnome_db_join_get_condition (join))) {
		gchar *str = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (cond), context, options, error);
		joincond = TRUE;

		if (!str)
			err = TRUE;
		else {
			g_string_append (string, str);
			g_free (str);
		}
	}
	else {
		GnomeDbEntity *ent1, *ent2;
		ent1 = gnome_db_target_get_represented_entity (gnome_db_join_get_target_1 (join));
		ent2 = gnome_db_target_get_represented_entity (gnome_db_join_get_target_2 (join));

		if (IS_GNOME_DB_TABLE (ent1) && IS_GNOME_DB_TABLE (ent2)) {
			/* find a FK in the database constraints for the two targets */
			GSList *fklist;
			GnomeDbConstraint *fkcons = NULL, *fkptr;
			gboolean same_order = TRUE;

			fklist = fk_constraints;
			while (fklist && !fkcons) {
				fkptr = GNOME_DB_CONSTRAINT (fklist->data);
				if ((gnome_db_constraint_get_constraint_type (fkptr) == CONSTRAINT_FOREIGN_KEY) &&
				    (((gnome_db_constraint_get_table (fkptr) == GNOME_DB_TABLE (ent1)) &&
				      (gnome_db_constraint_fkey_get_ref_table (fkptr) == GNOME_DB_TABLE (ent2))) ||
				     ((gnome_db_constraint_get_table (fkptr) == GNOME_DB_TABLE (ent2)) &&
				      (gnome_db_constraint_fkey_get_ref_table (fkptr) == GNOME_DB_TABLE (ent1))))) {
					fkcons = fkptr;

					if ((gnome_db_constraint_get_table (fkptr) == GNOME_DB_TABLE (ent2)) &&
					    (gnome_db_constraint_fkey_get_ref_table (fkptr) == GNOME_DB_TABLE (ent1)))
						same_order = FALSE;
				}
				fklist = g_slist_next (fklist);
			}

			if (fkcons) {
				GSList *fkpairs;
				GnomeDbConstraintFkeyPair *pair;
				gboolean fkfirst = TRUE;

				fkpairs = gnome_db_constraint_fkey_get_fields (fkcons);
				fklist = fkpairs;
				while (fklist) {
					pair = GNOME_DB_CONSTRAINT_FK_PAIR (fklist->data);
					if (fkfirst)
						fkfirst = FALSE;
					else
						g_string_append (string, " AND ");
					if (same_order)
						g_string_append (string, gnome_db_target_get_alias (gnome_db_join_get_target_1 (join)));
					else
						g_string_append (string, gnome_db_target_get_alias (gnome_db_join_get_target_2 (join)));
					g_string_append (string, ".");
					g_string_append (string, gnome_db_field_get_name (GNOME_DB_FIELD (pair->fkey)));
					g_string_append (string, "=");
					if (same_order)
						g_string_append (string, gnome_db_target_get_alias (gnome_db_join_get_target_2 (join)));
					else
						g_string_append (string, gnome_db_target_get_alias (gnome_db_join_get_target_1 (join)));
					g_string_append (string, ".");
					g_string_append (string, gnome_db_field_get_name (GNOME_DB_FIELD (pair->ref_pkey)));
					g_free (fklist->data);
					fklist = g_slist_next (fklist);
				}
				g_slist_free (fkpairs);
				joincond = TRUE;
			}
		}
	}
				
	if (!joincond) {
		err = TRUE;
		g_set_error (error,
			     GNOME_DB_QUERY_ERROR,
			     GNOME_DB_QUERY_RENDER_ERROR,
			     _("Join has no joining condition"));
	}

	if (err) {
		retval = NULL;
		g_string_free (string, TRUE);
	}
	else {
		retval = string->str;
		g_string_free (string, FALSE);
	}

	return retval;
}


static gchar *
render_sql_insert (GnomeDbQuery *query, GnomeDbDataSet *context, guint options, GError **error)
{
	GString *sql;
	gchar *retval;
	GnomeDbEntity *ent;
	GSList *list;
	gboolean first, err = FALSE;
	gboolean pprint = options & GNOME_DB_RENDERER_EXTRA_PRETTY_SQL;
	GSList *printed_fields = NULL;
	GSList *printed_values = NULL;

	/* compute the list of fields apprearing after the INSERT: only the fields with non DEFAULT values */
	if (query->priv->sub_queries)
		printed_fields = gnome_db_entity_get_fields (GNOME_DB_ENTITY (query));
	else {
		GSList *vfields;
		GError *my_error = NULL;

		vfields = gnome_db_entity_get_fields (GNOME_DB_ENTITY (query));
		list = vfields;
		while (list && !err) {
			GnomeDbField *value_prov;
			gchar *str;
			gboolean add_field = TRUE;

			g_object_get (G_OBJECT (list->data), "value_provider", &value_prov, NULL);
			g_assert (value_prov);
			str = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (value_prov), context, 
							       options | GNOME_DB_RENDERER_ERROR_IF_DEFAULT, &my_error);
			if (!str) {
				if (my_error) {
					if (my_error->code == GNOME_DB_QF_VALUE_DEFAULT_PARAM_ERROR) {
						/* this value is a DEFAULT value, don't print it */
						add_field = FALSE;
					}
					else {
						/* make sure @error is set */
						gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (value_prov), context, 
										 options, error);
						err = TRUE;
					}
					g_error_free (my_error);
				}
				else
					str = g_strdup ("NULL");
			}
			
			if (add_field) {
				printed_fields = g_slist_append (printed_fields, list->data);
				printed_values = g_slist_append (printed_values, str);
			}
			else
				g_free (str);

			list = g_slist_next (list);
		}
		g_slist_free (vfields);
	}

	if (err) {
		if (printed_fields)
			g_slist_free (printed_fields);
		if (printed_values) {
			g_slist_foreach (printed_values, (GFunc) g_free, NULL);
			g_slist_free (printed_values);
		}

		return NULL;
	}

	/* query is supposed to be active and of a good constitution here */
	sql = g_string_new ("INSERT INTO ");
	ent = gnome_db_target_get_represented_entity (GNOME_DB_TARGET (query->priv->targets->data));
	g_string_append (sql, gnome_db_base_get_name (GNOME_DB_BASE (ent)));
	if (pprint) g_string_append (sql, "\n\t");

	/* list of fields */
	g_string_append (sql, " (");
	list = printed_fields;
	first = TRUE;
	while (list) {
		GnomeDbField *field;
		if (first) 
			first = FALSE;
		else {
			if (pprint)
				g_string_append (sql, ",\n\t");
			else
				g_string_append (sql, ", ");
		}
		field = gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (list->data));
		g_string_append (sql, gnome_db_field_get_name (field));
		
		list = g_slist_next (list);
	}
	g_string_append (sql, ") ");
	g_slist_free (printed_fields);

	if (pprint) g_string_append (sql, "\n");

	/* list of values */
	if (query->priv->sub_queries) {
		gchar *str = gnome_db_query_render_as_sql (GNOME_DB_RENDERER (query->priv->sub_queries->data), context, 
							   options, error);
		if (str) {
			g_string_append (sql, str);
			g_free (str);
		}
		else
			err = TRUE;
	}
	else {
		g_string_append (sql, "VALUES (");
		if (pprint) g_string_append (sql, "\n\t");
		
		first = TRUE;
		list = printed_values;
		while (list) {
			if (first) 
				first = FALSE;
			else {
				if (pprint)
					g_string_append (sql, ",\n\t");
				else
					g_string_append (sql, ", ");
			}
			
			g_string_append_printf (sql, (gchar *) list->data);

			list = g_slist_next (list);
		}

		g_slist_foreach (printed_values, (GFunc) g_free, NULL);
		g_slist_free (printed_values);
		g_string_append (sql, ")");
	}

	if (!err)
		retval = sql->str;
	else
		retval = NULL;

	g_string_free (sql, err);
	return retval;
}

static gchar *
render_sql_update (GnomeDbQuery *query, GnomeDbDataSet *context, guint options, GError **error)
{
	GString *sql;
	gchar *retval;
	GnomeDbEntity *ent;
	GSList *list;
	gboolean first, err = FALSE;
	gboolean pprint = options & GNOME_DB_RENDERER_EXTRA_PRETTY_SQL;

	/* query is supposed to be active and of a good constitution here */
	sql = g_string_new ("UPDATE ");
	ent = gnome_db_target_get_represented_entity (GNOME_DB_TARGET (query->priv->targets->data));
	g_string_append (sql, gnome_db_base_get_name (GNOME_DB_BASE (ent)));
	if (pprint)
		g_string_append (sql, "\nSET ");
	else
		g_string_append (sql, " SET ");
	list = query->priv->fields;
	first = TRUE;
	while (list && !err) {
		if (gnome_db_qfield_is_visible (GNOME_DB_QFIELD (list->data))) {
			GnomeDbField *field, *value_prov;
			if (first) 
				first = FALSE;
			else {
				if (pprint)
					g_string_append (sql, ",\n\t");
				else
					g_string_append (sql, ", ");
			}
			field = gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (list->data));
			g_string_append (sql, gnome_db_field_get_name (field));
			g_string_append (sql, "=");

			g_object_get (G_OBJECT (list->data), "value_provider", &value_prov, NULL);
			if (value_prov) {
				gchar *str;
				str = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (value_prov), context, 
								 options, error);
				if (str) {
					g_string_append (sql, str);
					g_free (str);
				}
				else {
					if (error && *error)
						err = TRUE;
					else
						g_string_append (sql, "NULL");
				}
			}
			else {
				/* signal an error */
				g_set_error (error,
					     GNOME_DB_QUERY_ERROR,
					     GNOME_DB_QUERY_RENDER_ERROR,
					     _("Missing values"));
				err = TRUE;
			}
		}
		list = g_slist_next (list);
	}
	g_string_append (sql, " ");


	if (!err && query->priv->cond) {
		gchar *str;
		if (pprint) 
			g_string_append (sql, "\nWHERE ");
		else
			g_string_append (sql, "WHERE ");
		str = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (query->priv->cond), context, 
						 options, error);
		if (str) {
			g_string_append (sql, str);
			g_free (str);
		}
		else
			err = TRUE;
	}

	if (!err)
		retval = sql->str;
	else
		retval = NULL;

	g_string_free (sql, err);
	return retval;
}

static gchar *
render_sql_delete (GnomeDbQuery *query, GnomeDbDataSet *context, guint options, GError **error)
{
	GString *sql;
	gchar *retval;
	GnomeDbEntity *ent;
	gboolean err = FALSE;
	gboolean pprint = options & GNOME_DB_RENDERER_EXTRA_PRETTY_SQL;

	/* query is supposed to be active and of a good constitution here */
	sql = g_string_new ("DELETE FROM ");
	ent = gnome_db_target_get_represented_entity (GNOME_DB_TARGET (query->priv->targets->data));
	g_string_append (sql, gnome_db_base_get_name (GNOME_DB_BASE (ent)));

	if (pprint) g_string_append (sql, "\n");
	if (query->priv->cond) {
		gchar *str;
		g_string_append (sql, " WHERE ");
		str = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (query->priv->cond), context, 
						 options, error);
		if (str) {
			g_string_append (sql, str);
			g_free (str);
		}
		else
			err = TRUE;
	}

	if (!err)
		retval = sql->str;
	else
		retval = NULL;

	g_string_free (sql, FALSE);
	return retval;
}

static gchar *
render_sql_union (GnomeDbQuery *query, GnomeDbDataSet *context, guint options, GError **error)
{
	GString *sql;
	gchar *retval, *str;
	GSList *list;
	gboolean first = TRUE;
	gboolean err = FALSE;

	/* query is supposed to be active here, and of good constitution */
	sql = g_string_new ("");
	list = query->priv->sub_queries;
	while (list && !err) {
		if (first)
			first = FALSE;
		else
			g_string_append (sql, " UNION ");
		str = gnome_db_query_render_as_sql (GNOME_DB_RENDERER (list->data), context, options, error);
		if (str) {
			g_string_append_printf (sql, "(%s)", str);
			g_free (str);
		}
		else err = TRUE;

		list = g_slist_next (list);
	}

	if (!err)
		retval = sql->str;
	else
		retval = NULL;

	g_string_free (sql, err);
	return retval;
}

static gchar *
render_sql_intersect (GnomeDbQuery *query, GnomeDbDataSet *context, guint options, GError **error)
{
	GString *sql;
	gchar *retval, *str;
	GSList *list;
	gboolean first = TRUE;
	gboolean err = FALSE;

	/* query is supposed to be active here, and of good constitution */
	sql = g_string_new ("");
	list = query->priv->sub_queries;
	while (list && !err) {
		if (first)
			first = FALSE;
		else
			g_string_append (sql, " INTERSECT ");
		str = gnome_db_query_render_as_sql (GNOME_DB_RENDERER (list->data), context, options, error);
		if (str) {
			g_string_append_printf (sql, "(%s)", str);
			g_free (str);
		}
		else err = TRUE;

		list = g_slist_next (list);
	}

	if (!err)
		retval = sql->str;
	else
		retval = NULL;

	g_string_free (sql, err);
	return retval;
}

static gchar *
render_sql_except (GnomeDbQuery *query, GnomeDbDataSet *context, guint options, GError **error)
{
	GString *sql;
	gchar *retval, *str;
	GSList *list;
	gboolean first = TRUE;
	gboolean err = FALSE;

	/* query is supposed to be active here, and of good constitution */
	sql = g_string_new ("");
	list = query->priv->sub_queries;
	while (list && !err) {
		if (first)
			first = FALSE;
		else
			g_string_append (sql, " EXCEPT ");
		str = gnome_db_query_render_as_sql (GNOME_DB_RENDERER (list->data), context, options, error);
		if (str) {
			g_string_append_printf (sql, "(%s)", str);
			g_free (str);
		}
		else err = TRUE;

		list = g_slist_next (list);
	}

	if (!err)
		retval = sql->str;
	else
		retval = NULL;

	g_string_free (sql, err);
	return retval;
}

static gchar *
render_sql_non_parsed_with_params (GnomeDbQuery *query, GnomeDbDataSet *context, guint options, GError **error)
{
	GString *string;
	gchar *retval;
	GList *list;
	
	g_return_val_if_fail (query->priv->sql_exprs, NULL);
	g_return_val_if_fail (query->priv->sql_exprs->params_specs, NULL);
	
	/* if no context, then return as we don't have any default value for the parameters */
	string = g_string_new("");
	list = query->priv->sql_exprs->expr_list;
	while (list) {
		GnomeDbSqlExpr *expr = GNOME_DB_SQL_EXPR (list->data);

		if (list != query->priv->sql_exprs->expr_list)
			g_string_append (string, " ");

		if (expr->sql_text)
			g_string_append (string, expr->sql_text);
		else {
			GnomeDbQfValue *found = NULL;
			GSList *fields;
			gchar *str = NULL;
			
			fields = query->priv->fields;
			while (fields && !found) {
				if (g_object_get_data (G_OBJECT (fields->data), "pspeclist") == expr->pspec_list)
					found = GNOME_DB_QF_VALUE (fields->data);
				fields = g_slist_next (fields);
			}
			
			if (found) 
				str = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (found), context, options, error);

			if (!str) {
				g_string_free (string, TRUE);
				return NULL;
			}
			g_string_append (string, str);
			g_free (str);
		}
		list = g_list_next (list);
	}

	retval = string->str;
	g_string_free (string, FALSE);
	return retval;
}

static gchar *
gnome_db_query_render_as_str (GnomeDbRenderer *iface, GnomeDbDataSet *context)
{
	GnomeDbQuery *query;
	gchar *str;
	const gchar *cstr;

	g_return_val_if_fail (iface && IS_GNOME_DB_QUERY (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QUERY (iface)->priv, NULL);
	query = GNOME_DB_QUERY (iface);

	cstr = gnome_db_base_get_name (GNOME_DB_BASE (query));
	if (cstr && *cstr)
		str = g_strdup_printf (_("Query '%s'"), cstr);
	else
		str = g_strdup (_("Unnamed Query"));
	       
	return str;
}

static gboolean
gnome_db_query_is_valid (GnomeDbRenderer *iface, GnomeDbDataSet *context, GError **error)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_QUERY (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_QUERY (iface)->priv, FALSE);

	return assert_coherence_all_params_present (GNOME_DB_QUERY (iface), context, error);
}

/**
 * gnome_db_query_expand_all_field
 * @query: a #GnomeDbQuery object
 * @target: a #GnomeDbTarget, or %NULL
 *
 * Converts each visible "target.*" (#GnomeDbQfAll) field into its list of fields. For example "t1.*" becomes "t1.a, t1.b"
 * if table t1 is composed of fields "a" and "b". The original GnomeDbQfAll field is not removed, but
 * simply rendered non visible.
 *
 * The returned list must be free'd by the caller using g_slist_free().
 *
 * Returns: a new list of the #GnomeDbQfield objects which have been created
 */
GSList *
gnome_db_query_expand_all_field (GnomeDbQuery *query, GnomeDbTarget *target)
{
	GSList *list, *retlist = NULL;
	
	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (query->priv, NULL);
	g_return_val_if_fail (!target || (IS_GNOME_DB_TARGET (target) && (gnome_db_target_get_query (target) == query)), NULL);

	list = query->priv->fields;
	while (list) {
		if (IS_GNOME_DB_QF_ALL (list->data) && gnome_db_qfield_is_visible (GNOME_DB_QFIELD (list->data))) {
			GnomeDbTarget *t;
			
			t = gnome_db_qf_all_get_target (GNOME_DB_QF_ALL (list->data));
			if (!target || (t == target)) {
				GSList *entfields, *list2;
				GnomeDbField *newfield;

				entfields = gnome_db_entity_get_fields (gnome_db_target_get_represented_entity (t));
				list2 = entfields;
				while (list2) {
					newfield = GNOME_DB_FIELD (gnome_db_qf_field_new_with_objects (query,
								   t, GNOME_DB_FIELD (list2->data)));
					g_object_set_data (G_OBJECT (newfield), "star_field", list->data);
					retlist = g_slist_append (retlist, newfield);
					gnome_db_entity_add_field_before (GNOME_DB_ENTITY (query), newfield,
								    GNOME_DB_FIELD (list->data));
					gnome_db_base_set_name (GNOME_DB_BASE (newfield), 
								gnome_db_base_get_name (GNOME_DB_BASE (list2->data)));
					gnome_db_base_set_description (GNOME_DB_BASE (newfield), 
								 gnome_db_base_get_description (GNOME_DB_BASE (list2->data)));
					g_object_unref (G_OBJECT (newfield));
					list2 = g_slist_next (list2);
				}
				g_slist_free (entfields);
				gnome_db_qfield_set_visible (GNOME_DB_QFIELD (list->data), FALSE);
			}
		}
		list = g_slist_next (list);
	}

	return retlist;
}


/**
 * gnome_db_query_order_fields_using_join_conds
 * @query: a #GnomeDbQuery object
 *
 * Re-orders the fields in @query using the joins' conditions: for each join condition,
 * the used query fields are grouped together near the 1st visible field.
 */
void
gnome_db_query_order_fields_using_join_conds (GnomeDbQuery *query)
{
	GSList *list;
	gboolean reordered = FALSE;

	g_return_if_fail (query && IS_GNOME_DB_QUERY (query));
	g_return_if_fail (query->priv);

	list = query->priv->joins_flat;
	while (list) {
		GnomeDbCondition *cond = gnome_db_join_get_condition (GNOME_DB_JOIN (list->data));
		if (cond) {
			GSList *refs, *ptr;
			gint minpos = G_MAXINT;

			refs = gnome_db_condition_get_ref_objects_all (cond);
			ptr = refs;
			while (ptr) {
				if (IS_GNOME_DB_QF_FIELD (ptr->data) && 
				    gnome_db_qfield_is_visible (GNOME_DB_QFIELD (ptr->data)) &&
				    g_slist_find (query->priv->fields, ptr->data)) {
					gint pos = g_slist_index (query->priv->fields, ptr->data);
					if (pos < minpos)
						minpos = pos;
				}
				ptr = g_slist_next (ptr);
			}

			if (minpos != G_MAXINT) {
				ptr = refs;
				while (ptr) {
					if (IS_GNOME_DB_QF_FIELD (ptr->data) && 
					    g_slist_find (query->priv->fields, ptr->data)) {
						if (g_slist_index (query->priv->fields, ptr->data) > minpos) {
							query->priv->fields = g_slist_remove (query->priv->fields, 
											      ptr->data);
							query->priv->fields = g_slist_insert (query->priv->fields, 
											      ptr->data, ++minpos);
							reordered = TRUE;
						}
					}
					ptr = g_slist_next (ptr);
				}
			}
			
			g_slist_free (refs);
		}
		list = g_slist_next (list);
	}

	if (reordered) {
#ifdef debug_signal
		g_print (">> 'FIELDS_ORDER_CHANGED' from %s\n", __FUNCTION__);
#endif
		g_signal_emit_by_name (G_OBJECT (query), "fields_order_changed");
#ifdef debug_signal
		g_print ("<< 'FIELDS_ORDER_CHANGED' from %s\n", __FUNCTION__);
#endif
	}
}
