/* gnome-db-canvas-field.c
 *
 * Copyright (C) 2002 - 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-canvas.h"
#include "gnome-db-canvas-field.h"
#include "gnome-db-canvas-entity.h"
#include <libgnomedb/gnome-db-decl.h>
#include <libgnomedb/gnome-db-field.h>
#include <libgnomedb/gnome-db-table.h>
#include <libgnomedb/gnome-db-table-field.h>
#include <libgnomedb/gnome-db-query.h>

static void gnome_db_canvas_field_class_init (GnomeDbCanvasFieldClass * class);
static void gnome_db_canvas_field_init       (GnomeDbCanvasField * drag);
static void gnome_db_canvas_field_finalize   (GObject   * object);

static void gnome_db_canvas_field_set_property (GObject              *object,
					  guint                 param_id,
					  const GValue         *value,
					  GParamSpec           *pspec);
static void gnome_db_canvas_field_get_property (GObject              *object,
					  guint                 param_id,
					  GValue               *value,
					  GParamSpec           *pspec);

static void gnome_db_canvas_field_extra_event  (GnomeDbCanvasItem * citem, GdkEventType event_type);
enum
{
	PROP_0,
	PROP_FIELD,
};

struct _GnomeDbCanvasFieldPrivate
{
	GnomeDbField *field;
};

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *field_parent_class = NULL;

GType
gnome_db_canvas_field_get_type (void)
{
	static GType type = 0;

        if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbCanvasFieldClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_canvas_field_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbCanvasField),
			0,
			(GInstanceInitFunc) gnome_db_canvas_field_init
		};		

		type = g_type_register_static (GNOME_DB_TYPE_CANVAS_TEXT, "GnomeDbCanvasField", &info, 0);
	}

	return type;
}

	

static void
gnome_db_canvas_field_class_init (GnomeDbCanvasFieldClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	field_parent_class = g_type_class_peek_parent (class);

	object_class->finalize = gnome_db_canvas_field_finalize;
	GNOME_DB_CANVAS_ITEM_CLASS (class)->extra_event = gnome_db_canvas_field_extra_event;

	/* Properties */
	object_class->set_property = gnome_db_canvas_field_set_property;
	object_class->get_property = gnome_db_canvas_field_get_property;

	g_object_class_install_property
                (object_class, PROP_FIELD,
                 g_param_spec_pointer ("field", NULL, NULL, (G_PARAM_READABLE | G_PARAM_WRITABLE)));
}

static void
gnome_db_canvas_field_init (GnomeDbCanvasField * field)
{
	field->priv = g_new0 (GnomeDbCanvasFieldPrivate, 1);
	field->priv->field = NULL;
}

static void
field_nullified_cb (GnomeDbField *field, GnomeDbCanvasField *cf)
{
	g_signal_handlers_disconnect_by_func (G_OBJECT (field),
					      G_CALLBACK (field_nullified_cb), cf);
	gtk_object_destroy (GTK_OBJECT (cf));
}

static void
gnome_db_canvas_field_finalize (GObject   * object)
{
	GnomeDbCanvasField *cf;
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_CANVAS_FIELD (object));

	cf = GNOME_DB_CANVAS_FIELD (object);
	if (cf->priv) {
		if (cf->priv->field) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (cf->priv->field),
							      G_CALLBACK (field_nullified_cb), cf);
			cf->priv->field = NULL;
		}
		g_free (cf->priv);
		cf->priv = NULL;
	}

	/* for the parent class */
	field_parent_class->finalize (object);
}

static void 
gnome_db_canvas_field_set_property    (GObject              *object,
				 guint                 param_id,
				 const GValue         *value,
				 GParamSpec           *pspec)
{
	GnomeDbCanvasField *cf;
	gpointer ptr;
	GString *string;
	GnomeDbServerDataType *dtype;
	GnomeDbEntity *ent;

	cf = GNOME_DB_CANVAS_FIELD (object);

	switch (param_id) {
	case PROP_FIELD:
		ptr = g_value_get_pointer (value);
		g_return_if_fail (IS_GNOME_DB_FIELD (ptr));

		/* signals management */
		if (cf->priv->field) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (ptr),
							      G_CALLBACK (field_nullified_cb), cf);
			cf->priv->field = NULL;
		}

		cf->priv->field = GNOME_DB_FIELD (ptr);
		gnome_db_base_connect_nullify (ptr, G_CALLBACK (field_nullified_cb), cf);
		
		/* attributes setting */
		string = g_string_new ("");
		dtype = gnome_db_field_get_data_type (GNOME_DB_FIELD (ptr));
		g_string_append_printf (string, _("Type: %s"), gnome_db_base_get_name (GNOME_DB_BASE (dtype)));

		g_object_set (object, "gnome_db_base", ptr, NULL);
		ent = gnome_db_field_get_entity (GNOME_DB_FIELD (ptr));
		if (IS_GNOME_DB_TABLE (ent)) {
			gboolean nullok = gnome_db_table_field_is_null_allowed (GNOME_DB_TABLE_FIELD (ptr));
			gboolean pkpart = gnome_db_table_field_is_pkey_part (GNOME_DB_TABLE_FIELD (ptr));
			g_object_set (object, 
				      "highlight_color", GNOME_DB_CANVAS_DB_TABLE_COLOR, 
				      "text_underline", !nullok,
				      "text_bold", pkpart,
				      NULL);
			if (gnome_db_table_field_is_pkey_alone (GNOME_DB_TABLE_FIELD (ptr)))
				g_string_append (string, _("\nPrimary key"));
			else
				if (pkpart)
					g_string_append (string, _("\nPart of primary key"));
				
		}
		if (IS_GNOME_DB_QUERY (ent))
			g_object_set (object, "highlight_color", GNOME_DB_CANVAS_QUERY_COLOR, NULL);

		g_object_set (object, "tip_text", string->str, NULL);
		g_string_free (string, TRUE);
		break;
	}
}

static void 
gnome_db_canvas_field_get_property    (GObject              *object,
				 guint                 param_id,
				 GValue               *value,
				 GParamSpec           *pspec)
{
	GnomeDbCanvasField *cf;
	gpointer ptr;

	cf = GNOME_DB_CANVAS_FIELD (object);

	switch (param_id) {
	case PROP_FIELD:
		g_object_get (object, "gnome_db_base", &ptr, NULL);
		g_value_set_pointer (value, ptr);
		break;
	}
}

static void
gnome_db_canvas_field_extra_event  (GnomeDbCanvasItem *citem, GdkEventType event_type)
{
	if (event_type == GDK_LEAVE_NOTIFY)
		gnome_db_canvas_text_set_highlight (GNOME_DB_CANVAS_TEXT (citem), FALSE);
}


/**
 * gnome_db_canvas_field_get_field
 * @cfield: a #GnomeDbCanvasField object
 *
 * Get the #GnomeDbField which @cfield represents
 *
 * Returns: the object implementing the #GnomeDbField interface
 */
GnomeDbField *
gnome_db_canvas_field_get_field (GnomeDbCanvasField *cfield)
{
	g_return_val_if_fail (cfield && IS_GNOME_DB_CANVAS_FIELD (cfield), NULL);
	g_return_val_if_fail (cfield->priv, NULL);

	return cfield->priv->field;
}


/**
 * gnome_db_canvas_field_get_parent_item
 * @cfield: a #GnomeDbCanvasField object
 *
 * Get the #GnomeDbCanvasEntity in which @cfield is
 *
 * Returns: the #GnomeDbCanvasEntity in which @cfield is, or %NULL
 */
GnomeDbCanvasEntity *
gnome_db_canvas_field_get_parent_item (GnomeDbCanvasField *cfield)
{
	GnomeCanvasItem *ci;

	g_return_val_if_fail (cfield && IS_GNOME_DB_CANVAS_FIELD (cfield), NULL);
	ci = GNOME_CANVAS_ITEM (cfield)->parent;

	while (ci && !IS_GNOME_DB_CANVAS_ENTITY (ci))
		ci = ci->parent;
	return (GnomeDbCanvasEntity *) ci;
}
