'''
Defines standard interfaces for inspecting, manipulating, and navigating 
accessibles and handling their events.

@author: Peter Parente
@author: Pete Brunet
@author: Larry Weiss
@author: Brett Clippingdale
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

from AccAdapt import Interface

def implements(thing, interface):
  '''
  Checks if the given thing implements the given interface. The thing may be
  an instance or class. The thing is checked using issubclass first, followed by
  isinstance next, and then by brute force method comparison last.
  
  @param thing: Class or object to be tested for an interface
  @type thing: class or object
  @param interface: Class representing an interface to be implemented
  @type interface: class
  @return: Does the thing implement the given interface?
  @rtype: boolean
  '''
  try:
    if issubclass(thing, interface): return True
  except:
    pass
  try:
    if isinstance(thing, interface): return True
  except:
    pass
  return False
  c = set((name for name in dir(thing) if not name.startswith('_')))
  i = set((name for name in dir(interface) if not name.startswith('_')))
  return c.issuperset(i)

class IAccessibleNav(Interface):
  '''
  Provides methods for navigating across accessible objects.
  '''
  def getNextAcc(): 
    '''
    Gets the next accessible relative to the one providing this interface.
    
    @return: Point of regard to the next accessible
    @rtype: L{POR}
    @raise IndexError: When there is no next accessible
    @raise LookupError: When lookup for the next accessible fails even though
      it may exist
    '''
    pass
    
  def getPrevAcc():
    '''
    Gets the previous accessible relative to the one providing this interface.
    
    @return: Point of regard to the previous accessible
    @rtype: L{POR}
    @raise IndexError: When there is no previous accessible
    @raise LookupError: When lookup for the previous accessible fails even 
      though it may exist
    '''
    pass
    
  def getParentAcc():
    '''
    Gets the parent accessible relative to the one providing this interface.
    
    @return: Point of regard to the parent accessible
    @rtype: L{POR}
    @raise LookupError: When lookup for the parent accessible fails because it
      does not exist
    '''
    pass
  
  def getFirstAccChild():
    '''
    Gets the first accessible child relative to the subject accessible.
    
    @return: Point of regard to the first child accessible
    @rtype: L{POR}
    @raise LookupError: When lookup for child fails because it does not exist
    '''
    pass

  def getLastAccChild():
    '''
    Gets the last accessible child relative to the subject accessible.
    
    @return: Point of regard to the last child accessible
    @rtype: L{POR}
    @raise LookupError: When lookup for child fails because it does not exist
    '''
    pass
  
  def getChildAcc(index):
    '''
    Gets the child accessible at the given index relative to the one providing
    this interface.
    
    @param index: Index of the child to retrieve
    @type index: integer
    @return: Point of regard to the child accessible
    @rtype: L{POR}
    @raise IndexError: When there is no child at the given index
    @raise LookupError: When the lookup for the child fails even though it may
      exist
    '''
    pass
    
class IItemNav(Interface):
  '''
  Provides methods for navigating items within accessible objects. The 
  definition of an item is left up to the object implementing this interface.
  ''' 
  def getNextItem(only_visible=True): 
    '''    
    Gets the next Item relative to the one indicated by the L{POR}
    providing this interface.
    
    @param only_visible: True when Item in the returned L{POR} must be visible
    @type only_visible: boolean
    @return: Point of regard to the next visible item in the same accessible
    @rtype: L{POR}
    @raise IndexError: When there is no next visible item
    @raise LookupError: When lookup for the next item fails even though it may 
      exist
    '''
    pass
    
  def getPrevItem(only_visible=True):
    '''
    Gets the previous Item relative to the one indicated by the L{POR} providing
    this interface.
    
    @param only_visible: True when Item in the returned L{POR} must be visible
    @type only_visible: boolean
    @return: Point of regard to the previous visible item in the same accessible
    @rtype: L{POR}
    @raise IndexError: When there is no previous visible item
    @raise LookupError: When lookup for the previous item fails even though it
      may exist
    '''
    pass
  
  def getFirstItem(only_visible=True):
    '''
    Gets the first Item relative to the one indicated by the L{POR} 
    providing this interface.
    
    @param only_visible: True when Item in the returned L{POR} must be visible
    @type only_visible: boolean
    @return: Point of regard to the first visible item in the same accessible
    @rtype: L{POR}
    @raise IndexError: When there is no last visible item
    @raise LookupError: When lookup for the last item fails even though it may 
      exist
    '''
    pass
  
  def getLastItem(only_visible=True):
    '''
    Gets the last Item relative to the one indicated by the L{POR} 
    providing this interface.
    
    @param only_visible: True when Item in the returned L{POR} must be visible
    @type only_visible: boolean
    @return: Point of regard to the last visible item in the same accessible
    @rtype: L{POR}
    @raise LookupError: When lookup for the last item fails because it does not
      exist
    '''
    pass

class IAccessibleInfo(Interface):
  '''
  Provides methods for accessing basic information about accessible objects.
  '''
  def getAccSelection():
    '''  
    Gets a list L{POR}s referring to the selected objects in the subject 
    accessible.
    
    @return: Points of regard to selected accessibles
    @rtype: list of L{POR}s
    @raise LookupError: When the subject accessible is dead
    '''
    pass
  
  def getAccChildCount():
    '''
    Gets the number of child accessibles for the object implementing this 
    interface.
    
    @return: Number of accessible children
    @rtype: integer
    @raise LookupError: When the accessible object is dead
    '''
    pass
  
  def getAccAppName():
    '''
    Gets the name of the application application to which the subject accessible 
    belongs.
    
    @return: Name of the subject's application
    @rtype: string
    @raise LookupError: When the accessible or any parent accessible up to the
      application is dead
    '''
    pass
  
  def getAccAppID():
    '''
    Gets a unique ID identifying the application to which the subject accessible 
    belongs.
    
    @return: Unique identifier for the subject's application
    @rtype: hashable
    @raise LookupError: When the accessible or any parent accessible up to the
      application is dead
    '''
    pass
   
  def getAccRoleName():
    '''
    Gets the unicode accessible role name for the object implementing this 
    interface.
    
    @return: Unicode role name
    @rtype: string
    @raise LookupError: When the accessible object is dead
    '''
    pass
  
  def getAccName():
    '''    
    Gets the accessible name for the object implementing this interface.
    
    @return: Name of the accessible
    @rtype: string
    @raise LookupError: When the accessible object is dead
    @raise IndexError: When the item offset is outside the bounds of the 
      number of children of the subject accessible
    '''
    pass

  def getAccDescription():
    '''
    Gets the accessible description for the object implementing this interface.
    
    @return: Accessible description of requested object
    @rtype: string
    @raise LookupError: When the subject accessible or the child is dead
    @raise IndexError: When the item offset is outside the bounds of the 
      number of children of the subject accessible
    '''
    pass
  
  def getAccStates():
    '''
    Gets a list of states for the object implementing this interface.
    
    @return: Names of all states
    @rtype: list of string
    @raise LookupError: When the subject accessible or the child is dead
    @raise IndexError: When the item offset is outside the bounds of the 
      number of children of the subject accessible
    '''
    pass
  
  def getAccItemText():
    '''
    Gets the accessible text for the object implementing this interface.
    The item text will be the accessible text if that is available, otherwise it
    will be the accessible name.
    
    @return: The accessible text or name
    @rtype: string
    @raise LookupError: When the accessible object or the child is dead
    @raise IndexError: When the item offset is invalid
    '''
    pass
  
  def getAccRelations(kind):
    '''
    Gets a list of accessibles related to the subject accessible in the manner
    given by kind.
    
    @param kind: Name specifying the kind of relations to retrieve
    @type kind: string
    @return: List of L{POR}s related to subject accessible in the manner 
      given by kind
    @rtype: list of L{POR}
    @raise LookupError: When the accessible object is dead
    '''
    pass
  
  def getAccTextAttr(name):
    '''
    Gets the value of a given attribute name.
    
    @param name: Name of the attribute
    @type name: string
    @return: Value of the accessible attribute
    @rtype: string
    @raise LookupError: When the accessible object is dead
    '''
    pass
  
  def getAccAllTextAttrs():
    '''
    Gets all of the name:value attribute pairs of an accessible.
    
    @return: List of name:value pairs in the accessible attributes
    @rtype: string
    @raise LookupError: When the accessible object is dead
    '''
    pass
  
  def isAccTrivial():
    '''
    Gets whether or not the accessible considers itself to be trivial.
    
    @return: Does the accessible consider itself trivial?
    @rtype: boolean
    @raise LookupError: When the accessible object is dead
    '''
    pass

  def isAccVisible():
    '''
    Gets whether or not the accessible considers itself visible.
    
    @return: Does the accessible consider itself visible?
    @rtype: boolean
    @raise LookupError: When the accessible object is dead
    '''
    pass

  def hasAccRole(role):
    '''
    Gets if the subject has the given role.
    
    @param role: Name of the role
    @type role: string
    @return: Does the accessible have the given role?
    @rtype: boolean
    '''
    pass
  
  def hasAccState(state):
    '''
    Gets if the subject has the given state.
    
    @param state: Name of the state
    @type state: string
    @return: Does the accessible have the given state?
    @rtype: boolean
    '''
    pass
  
  def hasAccOneState(*states):
    '''
    Gets if the subject has at least one of the given states.
    
    @param states: List of states
    @type states: list of string
    @return: Does the accessible have at least one of the given states?
    @rtype: boolean
    '''
    pass
  
  def hasAccAllStates(*states):
    '''
    Gets if the subject has all of the given states.
    
    @param states: List of states
    @type states: list of string
    @return: Does the accessible have all of the given states?
    @rtype: boolean
    '''
    pass
  
  def isAccTopLevelWindow():
    '''
    Gets whether or not the subject accessible is a top-level container in the
    accessible hierarchy.
    
    @return: Is the subject a top-level container or not?
    @rtype: boolean
    @raise LookupError: When the accessible object is dead
    '''
    pass

class IAccessibleAction(Interface):
  '''
  Provides methods for manipulating accessible objects.
  '''
  def setFocus():
    '''
    Gives the subject accessible the input focus. 
    
    @return: Did accessible accept (True) or refuse (False) the focus change?
    @rtype: boolean
    @raise LookupError: When the accessible object is dead or does not support
      setting the focus
    '''
    pass
  
  def selectChild():
    '''
    Selects the accessible object implementing this interface.
    
    @return: Did accessible accept (True) or refuse (False) the selection?
    @rtype: boolean
    @raise LookupError: When the accessible object is dead or does not support
      setting the selection
    '''
    pass
  
class IEventHandler(Interface):
  '''
  Provides methods for handling accessible events.
  '''
  def getAEEvents(event):
    '''
    Returns the L{AEEvent}s that need to be posted for this event.
    
    @param event: Raw event
    @type event: object
    @return: One or more L{AEEvent}s to be posted to the L{EventManager}
    @rtype: list, L{AEEvent}, or None
    '''
    pass
  
class IPORFactory(Interface):
  '''
  Provides methods for building L{POR}s.
  '''
  def create():
    '''
    Returns a complete L{POR} built based on the properties of the subject.
    
    @return: Point of regard
    @rtype: L{POR}
    '''
    pass
 