'''
Defines classes that store basic user preferences and state information for 
L{Task}s.

@author: Peter Parente
@author: Larry Weiss
@author: Pete Brunet
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

import AEState
from POR import POR
from LSRConstants import *
from i18n import _

class LSRSettings(AEState.AEState):
  '''
  Stores the state of built-in, LSR user preferences and settings.
  
  @note: This class is a structure and based on the LSR coding conventions its
    members can be accessed directly. Setters and getters are not needed.  
  @ivar Blank: When set to True, "blank" will be presented for blank Words and
      Items. Defaults to True.
  @type Blank: boolean
  @ivar CapExpand:  When set to True, a space will be inserted before
      imbedded capital letters in the presentation of a word. Defaults to True.
  @type CapExpand: boolean
  @ivar Caps: When set to True, capitalization is preserved when presenting
      text. Defaults to True.
  @type Caps: boolean
  @ivar Dictionary:  When set to True, defined replacements for specific words
      will be presented. Defaults to True.
  @type Dictionary: boolean
  @ivar ForceSpell:  When set to True, 1 character words and those that have no
      vowels will be spelled for presentation. Defaults to True.
  @type ForceSpell: boolean
  @ivar NumExpand:  When set to True, a space will be inserted before imbedded
      numbers in the presentation of a word. Defaults to False.
  @type NumExpand: boolean
  @ivar SpellCaps: When set to True, to fully capitalized words will be spelled
      for presentation. Defaults to False.
  @type SpellCaps: boolean
  @ivar Stopping: When set to False, only stop commands as a result of a key
      sequence will be sent to the output device. Defaults to True.
  @type Stopping: boolean
  @ivar Trailing: When set to True, the normally blank trailing part of words
      will be additionally processed for output. Defaults to False.
  @type Trailing: boolean
  @ivar Trap: When set to False, methods will not raise a L{Task.Tools.Error} 
      and will instead set an error flag that can be read with 
      L{Task.Tools.Base.TaskTools.getErrorCode}. Otherwise, Task errors are 
      raised and their descriptions are presented to the user.
  @type Trap: boolean
  @ivar Wrap: When set to True, application content is treated as if it were
      a single Item. Defaults to False.
  @type Wrap: boolean
  @ivar Format: Set to FORMAT_TEXT to present words without additional
      spelling. Set to FORMAT_PRONOUNCE to spell punctuation characters for
      presentation. Set to FORMAT_SPELL to spell all characters for 
      presentation. Set to FORMAT_PHONETIC to spell all characters phonetically 
      for presentation. Defaults to FORMAT_TEXT.
  @type Format: integer
  @ivar MaxRepeat: Specifies the minimum number of times a character must be
      repeated before the word containing it will be spelled. The spelling of 
      the repeated character will be a string indicating the character and it's
      count. This does not apply to repeated capital letters when L{CapExpand}
      is True. If the repeated character is in the trailing part of a word, the
      repeated character will only be presented when {Trailing} is True and the
      count is greater than the value of L{Spaces}. Defaults to 4.
  @type MaxRepeat: integer
  @ivar Mode: Defines whether L{Task.Tools} methods without a referenced L{POR}
      will be relative to the application focus, the Pointer POR, or the POR
      delivered with the last event. Defaults to MODE_EVENT.
  @type Mode: integer
  @ivar Spaces: Defines the minimum number of characters needed in the trailing
      part of a word for those characters to be spelled when L{Trailing} is 
      True. Defaults to 2.
  @type Spaces: integer
  @ivar WordDef: Set to NON_BLANK to define the main part of a word to only
      include non-blank characters. Set to ALPHABETIC to define the main part of
      a word to only include alphabetic characters. Set to ALPHA_NUMERIC to
      define the main part of a word to only include alphabetic and numeric
      characters. Set to ALPHA_NUMERIC_PUNCT to define the main part of a word
      to only include characters that are alphabetic, numeric, or punctuation.
      Defaults to NON_BLANK.
  @type WordDef: integer
  @ivar CapString: Set to the string that should be presented prior to the
      spelling of capitalized letters and presented twice before spelling fully
      capitalized words. Defaults to 'cap'.
  @type CapString: string
  @ivar Ignore: Set to a list of characters that should be treated as if they
      were blank. Defaults to an empty list.
  @type Ignore: list
  @ivar Pointer: The pointer mode L{POR}. This POR tracks the pointer mode
      POR within the L{Tier}. L{Task}s need it to perform user actions based on
      the pointer mode POR. L{Tier} needs it to set the "current position"
      when the Mode setting (Pointer/Focus) is in Pointer mode, prior to Task
      execution. L{Perk}s need it during their initialization for the same 
      reason Tasks do.  
  @type Pointer: L{POR}
  @ivar MayStop: Flag indicating whether the next call to 
    L{Task.Tools.Output.mayStop} will send a stop to an output device or simply
    reset this flag. This flag should not be persisted across LSR instances.
  @type MayStop: boolean
  '''
  def __init__(self):
    '''
    Initializes basic state variables to their default values.
    '''
    # init the boolean defaults
    self.Blank = True
    self.CapExpand = True
    self.Caps = True
    self.Dictionary = True
    self.ForceSpell = True
    self.NumExpand = False
    self.SpellCaps = False
    self.Stopping = True
    self.Trailing = False
    self.Trap = True
    self.Wrap = False

    # init the integer defaults
    self.Format = FORMAT_TEXT
    self.MaxRepeat = 4
    self.Mode = MODE_EVENT
    self.Spaces = 2
    self.WordDef = WORD_NON_BLANK

    # init the string defaults
    self.CapString = 'cap'
    self.Ignore = []

    # init the Pointer to a dummy POR
    self.Pointer = POR()
    
    # init MayStop to True
    self.MayStop = True
    
  def getSettings(self):
    '''
    Provides metadata about the setting values in this object. Useful for
    generating configuration dialogs.
    
    @return: Group of settings
    @rtype: L{AEState.Setting.Group}
    '''
    g = self._newGroup()
    g.newBool('Blank', _('Speak blanks'))
    g.newBool('CapExpand', _('Expand capitals'))
    g.newBool('Caps', _('Speak capitals'))
    g.newBool('Dictionary', _('Use custom dictionary'))
    g.newBool('ForceSpell', _('Force spelling'))
    g.newBool('NumExpand', _('Expand numbers'))
    g.newBool('SpellCaps', _('Spell capitals'))
    g.newBool('Stopping', _('Stop automatically'))
    g.newBool('Trailing', _('Process trailing word parts'))
    g.newBool('Trap', _('Trap and announce Task exception'))
    g.newBool('Wrap', _('Wrap items during review'))
    g.newInt('MaxRepeat', _('Max character repeat before spelled'))
    g.newInt('Spaces', _('Min trailing size before spelled'))
    g.newChoice('Mode', _('Pointer mode'), {MODE_FOCUS : _('Focus'),
                                            MODE_POINTER : _('Pointer'),
                                            MODE_EVENT : _('Event')})
    g.newChoice('WordDef', ('Word definition'), 
                {FORMAT_PHONETIC: _('Phonetic'),
                 FORMAT_PRONOUNCE: _('Pronounce'),
                 FORMAT_SPELL: _('Spell'),
                 FORMAT_TEXT: _('Text')})
    g.newString('CapString', _('Capital prefix'))
    g.newSplitList('Ignore', _('Blank characters'), ' ')
    return g
    
  def getNonPersistent(self):
    '''
    Gets a list of instance variables that should not be written to persistent
    storage if/when this object is serialized.
    
    @return: List of variable names
    @rtype: list of string
    '''
    return ['Pointer', 'MayStop']
  
  def initNonPersistent(self):
    '''
    Reinitializes non-persistent instance variables when this object is 
    unserialized.
    '''
    self.Pointer = POR()
    self.MayStop = True